package com.cogknit.fovea.services;

import android.annotation.TargetApi;
import android.app.IntentService;
import android.content.Intent;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.provider.Browser;
import android.provider.CallLog;
import android.provider.ContactsContract.PhoneLookup;

import com.cogknit.fovea.FoveaSharedPreferences;
import com.cogknit.fovea.dataModel.Call;
import com.cogknit.fovea.dataModel.History;
import com.cogknit.fovea.dataModel.Person;
import com.cogknit.fovea.dataModel.Sms;
import com.cogknit.fovea.providers.FoveaContract;
import com.cogknit.fovea.providers.FoveaDatabaseManager;
import com.cogknit.fovea.receivers.AlarmReceiver;
import com.cogknit.fovea.utils.FoveaLog;
import com.cogknit.fovea.utils.SchedulerConstants;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;

/**
 * Class responsible for fetching locally available data (SMS, History, Call
 * Logs) and updating the GoGlocalProvider
 *
 * @author Chirag
 */
public class LocalDataFetchService extends IntentService {

    private static final String TAG = "LDFS";
    private static final int TOP_CONNECTED_PEOPLE_COUNT = 25;

    public LocalDataFetchService() {
        super("LocalDataFetchService");

    }

    @Override
    protected void onHandleIntent(Intent intent) {

        if (intent.getAction() == SchedulerConstants.LocalFetch.ACTION_CALLLOGS) {
            FoveaLog.v(TAG, "Call logs fetch begin!");
            updateCallLogs();
            deleteOldCallLogs();
            FoveaLog.v(TAG, "Call logs Updated!");
        }
        if (intent.getAction() == SchedulerConstants.LocalFetch.ACTION_PEOPLE) {
            FoveaLog.v(TAG, "People fetch begin!");
            repopulatePeopleTables();
            FoveaLog.v(TAG, "People Updated!");
        } else if (intent.getAction() == SchedulerConstants.LocalFetch.ACTION_HISTORY) {
            updateHistoryProvider();
            FoveaLog.v(TAG, "History Updated!");
        } else if (intent.getAction() == SchedulerConstants.LocalFetch.ACTION_SMS) {
            updateSmsProvider();
            FoveaLog.v(TAG, "Messages Updated!");
        }

        AlarmReceiver.completeWakefulIntent(intent);

    }

	/*
     * Call log Related methods:
	 */

    /**
     * Updates the call log directory with new entries from the android provider
     */
    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    /**
     * Function to retrieve call logs which are new from the Android provider
     */
    private void updateCallLogs() {
        // Retrieve Latest time-stamp from CallLogProvider
        Uri mUri = FoveaContract.Calls.CONTENT_CALLS_URI;
        long latestEntry = getLatestTimeStamp(mUri);

        String[] projection = null;
        // Query Android's CallProvider for logs newer than the latest
        // time-stamp already in the DB
        if (Build.VERSION.SDK_INT < 21) {
            String[] p = {CallLog.Calls.NUMBER, CallLog.Calls.DURATION,
                    CallLog.Calls.CACHED_NAME, CallLog.Calls.DATE};
            projection = p;

        } else {
            String[] p = {CallLog.Calls.NUMBER, CallLog.Calls.DURATION,
                    CallLog.Calls.CACHED_NAME, CallLog.Calls.CACHED_LOOKUP_URI,
                    CallLog.Calls.DATE};
            projection = p;
        }
        // Filter out logs older than the latest and those with no duration
        // (missed calls)
        String selection = CallLog.Calls.DATE + ">" + latestEntry + " AND "
                + CallLog.Calls.DURATION + ">" + "0";

        Cursor c = getContentResolver().query(
                CallLog.Calls.CONTENT_URI, projection,
                selection, null,
                CallLog.Calls.DEFAULT_SORT_ORDER);

        // Transfer the new logs to the call log content provider
        ArrayList<Call> newLogs = new ArrayList<Call>();
        if (c.moveToNext()) {
            do {

                String name = c.getString(c
                        .getColumnIndex(CallLog.Calls.CACHED_NAME));
                if (name == null || name == "")
                    name = "Unknown";

                String number = c.getString(c
                        .getColumnIndex(CallLog.Calls.NUMBER));

                long duration = c.getLong(c
                        .getColumnIndex(CallLog.Calls.DURATION));

                long timeStamp = c
                        .getLong(c.getColumnIndex(CallLog.Calls.DATE));


                String uri = null;
                if (Build.VERSION.SDK_INT >= 21) {
                    uri = c.getString(c
                            .getColumnIndex(CallLog.Calls.CACHED_LOOKUP_URI));
                } else {
                    uri = "";
                }
                Call log = new Call(name, duration, number, timeStamp, uri, "");
                newLogs.add(log);
            } while (c.moveToNext());

        }
        c.close();

        // Insert values into CallLogProvider
        for (Call log : newLogs) {
            /*getContentResolver().insert(
                    FoveaContract.Calls.CONTENT_CALLS_URI,
                    log.getContentValues());*/
            FoveaDatabaseManager.insert(this.getApplicationContext(),
                    FoveaContract.Calls.CONTENT_CALLS_URI,
                    log.getContentValues());
        }
        setLatestTimeStamp(mUri);

    }

    /**
     * Helper method to find the latest entry already present in the calls.
     * directory
     *
     * @return The newest time stamp of items already in the directory.
     */
    private long getLatestTimeStamp(Uri uri) {

        SharedPreferences prefs = getSharedPreferences(FoveaSharedPreferences.LastMined.PREFS_NAME, MODE_PRIVATE);
        long timestamp=prefs.getLong(uri.getLastPathSegment(), 0);
        //FoveaLog.d(TAG,"GET -> KEY: "+uri.getLastPathSegment()+" TIMESTAMP: "+timestamp);
        return timestamp ;

    }

    private void setLatestTimeStamp(Uri uri) {
        long timestamp = 0;
        SharedPreferences prefs = getSharedPreferences(FoveaSharedPreferences.LastMined.PREFS_NAME, MODE_PRIVATE);
        String[] projection = {FoveaContract.CommonColumns.TIMESTAMP};
        String sortOrder = FoveaContract.CommonColumns.TIMESTAMP + " DESC";
        String selection = FoveaContract.CommonColumns.TIMESTAMP
                + "=(SELECT MAX(" + FoveaContract.CommonColumns.TIMESTAMP
                + ") from " + uri.getLastPathSegment() + ")";


        Cursor c = FoveaDatabaseManager.query(this.getApplicationContext(), uri, projection, selection, null,
                sortOrder);
        if (c != null && c.moveToNext()) {
            timestamp = c.getLong(c
                    .getColumnIndex(FoveaContract.CommonColumns.TIMESTAMP));
            c.close();
        }

        prefs.edit()
                .putLong(uri.getLastPathSegment(), timestamp)
                .commit();
        FoveaLog.d(TAG, "SET -> KEY: " + uri.getLastPathSegment() + " TIMESTAMP: " + timestamp);

    }

    private void setLatestSmsTimeStamp(Uri uri, long timestamp) {

        SharedPreferences prefs = getSharedPreferences(FoveaSharedPreferences.LastMined.PREFS_NAME, MODE_PRIVATE);
        prefs.edit()
                .putLong(uri.getLastPathSegment(), timestamp)
                .commit();
        FoveaLog.d(TAG, "SET -> KEY: " + uri.getLastPathSegment() + " TIMESTAMP: " + timestamp);

    }

    private void repopulatePeopleTables() {
        Cursor c;
        ArrayList<Person> people = new ArrayList<Person>();
        long after;
        long now = new Date().getTime();
        long day = 1000 * 60 * 60 * 24;
        long week = 7 * day;
        long month = 30 * day;
        long quarter = 3 * month;

        // purge old table
        /*getContentResolver().delete(FoveaContract.People.CONTENT_PEOPLE_URI,
                null, null);*/
        FoveaDatabaseManager.delete(this.getApplicationContext(), FoveaContract.People.CONTENT_PEOPLE_URI,
                null, null);

        // For each timeframe, get the top 5 people with longest overall call
        // duration over that timeframe
        /*String[] timeframes = {FoveaContract.People.DAILY,
                FoveaContract.People.WEEKLY,
                FoveaContract.People.MONTHLY,
                FoveaContract.People.QUARTERLY};*/
        String[] timeframes = {
                FoveaContract.People.QUARTERLY,
                };
        for (String timeframe : timeframes) {
            if (timeframe == FoveaContract.People.DAILY)
                after = now - day;
            else if (timeframe == FoveaContract.People.WEEKLY)
                after = now - week;
            else if (timeframe == FoveaContract.People.MONTHLY)
                after = now - month;
            else
                after = now - quarter;
            String[] projection = {FoveaContract.Calls.Columns.NAME,
                    FoveaContract.Calls.Columns.NUMBER,
                    FoveaContract.Calls.Columns.TIMESTAMP,
                    FoveaContract.Calls.Columns.DURATION,
                    FoveaContract.Calls.Columns.CONTACT_URI};
            String selection = FoveaContract.Calls.Columns.TIMESTAMP + ">"
                    + after;
            // Make query to get all logs for current time-frame
            /*c = getContentResolver().query(
                    FoveaContract.Calls.CONTENT_CALLS_URI, projection,
                    selection, null, null);*/
            c = FoveaDatabaseManager.query(this.getApplicationContext(),
                    FoveaContract.Calls.CONTENT_CALLS_URI, projection,
                    selection, null, null);

            HashMap<String, Person> map = new HashMap<String, Person>();
            double max_frequency = 0;
            double max_duration = 0;
            while (c.moveToNext()) {
                // Get the name of the current entry
                String name = c.getString(c
                        .getColumnIndex(FoveaContract.Calls.Columns.NAME));
                // check if it exists in the hash-map
                if (map.containsKey(name)) {
                    // If it exists, increment its frequency and duration from
                    // values from the cursor
                    long duration = c
                            .getLong(c
                                    .getColumnIndex(FoveaContract.Calls.Columns.DURATION));

                    Person person = map.get(name);
                    person.addDuration(duration);
                    person.addFrequency(1);
                    max_frequency = (person.getFrequency() > max_frequency) ? person.getFrequency():max_frequency;
                    max_duration = (person.getDuration() > max_duration) ? person.getDuration():max_duration;
                } else {
                    // If not, create a new object and add it with the name as
                    // key using values from cursor
                    long duration = c
                            .getLong(c
                                    .getColumnIndex(FoveaContract.Calls.Columns.DURATION));
                    String number = c
                            .getString(c
                                    .getColumnIndex(FoveaContract.Calls.Columns.NUMBER));
                    String uri = c
                            .getString(c
                                    .getColumnIndex(FoveaContract.Calls.Columns.CONTACT_URI));
                    Person person = new Person(name, number, duration, 1, 0.0, uri,
                            timeframe, "");
                    map.put(name, person);
                    max_frequency = (person.getFrequency() > max_frequency) ? person.getFrequency():max_frequency;
                    max_duration = (person.getDuration() > max_duration) ? person.getDuration():max_duration;
                }

            }
            c.close();
            //FoveaLog.v(TAG, "Max duration =" + max_duration + " frequency =" + max_frequency);

            for (HashMap.Entry<String, Person> entry : map.entrySet())
            {
                Person person = entry.getValue();
                double norm_freq = person.getFrequency()/max_frequency;
                double norm_duration = person.getDuration()/max_duration;
                double sqrSum = Math.pow(norm_freq, 2) + Math.pow(norm_duration, 2);
                double weight = Math.sqrt(sqrSum)/2;
                person.setWeight(weight*100);
                //FoveaLog.v(TAG, "Weight =" + person.getWeight());
                people.add(person);
            }

            // Sort people by duration
            /*Collections.sort(people, new Comparator<Person>() {
                @Override
                public int compare(Person p1, Person p2) {
                    return (int) ((int) p2.getDuration() - p1.getDuration()); // Descending
                }

            });*/
            Collections.sort(people, new Comparator<Person>() {
                @Override
                public int compare(Person p1, Person p2) {
                    return  (int) ((int)p2.getWeight() - p1.getWeight()); // Descending
                }

            });

            // Insert the top twenty people for this timeframe into the people
            // directory
            byte count = 0;
            for (Person person : people) {
                if (person.isNameKnown()) {
                    /*getContentResolver().insert(
                            FoveaContract.People.CONTENT_PEOPLE_URI,
                            person.getContentValues());*/
                    FoveaDatabaseManager.insert(this.getApplicationContext(),
                            FoveaContract.People.CONTENT_PEOPLE_URI,
                            person.getContentValues());
                    count++;
                }
                if (count == TOP_CONNECTED_PEOPLE_COUNT)
                    break;
            }

            people.clear();
        }
    }

    private void deleteOldCallLogs() {
        // Clear all entries older than 3 months from the CallS Directory
        long now = new Date().getTime();
        long day = 1000 * 60 * 60 * 24;
        long month = 30 * day;
        long quarter = 3 * month;
        long before = now - quarter;
        String selection = FoveaContract.Calls.Columns.TIMESTAMP + "<"
                + before;
        /*getContentResolver().delete(FoveaContract.Calls.CONTENT_CALLS_URI,
                selection, null);*/
        FoveaDatabaseManager.delete(this.getApplicationContext(), FoveaContract.Calls.CONTENT_CALLS_URI,
                selection, null);

    }

    /*
     * History Related methods:
     */
    private void updateHistoryProvider() {
        ArrayList<History> historyList = new ArrayList<History>();
        // Ensure only newer history than the newest history already in the DB
        // are collected
        Uri mUri = FoveaContract.History.CONTENT_HISTORY_URI;
        long latestEntry = getLatestTimeStamp(mUri);
        String selection = Browser.BookmarkColumns.CREATED
                + ">"
                + latestEntry;

        String[] projection = {Browser.BookmarkColumns.TITLE,
                Browser.BookmarkColumns.URL, Browser.BookmarkColumns.BOOKMARK,
                Browser.BookmarkColumns.CREATED};
        Cursor c = getContentResolver().query(Browser.BOOKMARKS_URI,
                projection, selection, null, null);
        String title = null;
        String url = null;
        int bookmarks = 0;
        String timestamp = null;
        while (c.moveToNext()) {
            title = c
                    .getString(c.getColumnIndex(Browser.BookmarkColumns.TITLE));
            url = c.getString(c.getColumnIndex(Browser.BookmarkColumns.URL));
            bookmarks = c.getInt(c
                    .getColumnIndex(Browser.BookmarkColumns.BOOKMARK));
            timestamp = c.getString(c
                    .getColumnIndex(Browser.BookmarkColumns.CREATED));

            historyList.add(new History(title, url, timestamp, bookmarks, ""));

        }
        c.close();
        // Add all new history from known people to our provider
        //ContentResolver cr = getContentResolver();
        for (History history : historyList) {
            /*cr.insert(FoveaContract.History.CONTENT_HISTORY_URI,
                    history.getContentValues());*/
            FoveaDatabaseManager.insert(this.getApplicationContext(), FoveaContract.History.CONTENT_HISTORY_URI,
                    history.getContentValues());

        }
        setLatestTimeStamp(mUri);

    }


    /*
     * SMS Related methods:
     */
    private boolean contactExists(String number) {

        Uri uri = Uri.withAppendedPath(PhoneLookup.CONTENT_FILTER_URI,
                Uri.encode(number));
        Cursor c = getContentResolver().query(uri, null, null, null, null);
        boolean val = c.moveToNext();
        c.close();
        return val;

    }

    private void updateSmsProvider() {
        Uri CONTENT_SMS_INBOX = Uri.parse("content://sms/inbox");
        Uri CONTENT_SMS_SENT = Uri.parse("content://sms/sent");
        String SMS_COLUMN_DATE = "date";
        String SMS_COLUMN_NUMBER = "address";
        String SMS_COLUMN_BODY = "body";
        ArrayList<Sms> smsList = new ArrayList<Sms>();

        // Ensure only newer sms than the newest sms already in the DB are
        // collected
        Uri mUri = FoveaContract.Sms.CONTENT_SMS_URI;
        Uri dirs[] = {CONTENT_SMS_INBOX, CONTENT_SMS_SENT};
        String[] projection = {SMS_COLUMN_DATE, SMS_COLUMN_BODY,
                SMS_COLUMN_NUMBER};
        for (Uri uri : dirs) {
            long latestEntry = getLatestTimeStamp(uri);

            String selection = SMS_COLUMN_DATE + " > "
                    + latestEntry;
            Cursor c = getContentResolver().query(uri, projection, selection,
                    null, null);
            String number = null;
            String body = null;
            String timestamp = null;
            while (c.moveToNext()) {
                number = c.getString(c.getColumnIndex(SMS_COLUMN_NUMBER));
                body = c.getString(c.getColumnIndex(SMS_COLUMN_BODY));
                timestamp = c.getString(c.getColumnIndex(SMS_COLUMN_DATE));
                long ts=Long.parseLong(timestamp);
                latestEntry=latestEntry>ts?latestEntry:ts;

                // Ensure the SMS is from a person on the contact list
                if (contactExists(number)) {
                    smsList.add(new Sms(number, body, timestamp, ""));
                }
            }
            c.close();
            setLatestSmsTimeStamp(uri,latestEntry);

        }

        // Add all new sms from known people to our provider
        //ContentResolver cr = getContentResolver();
        for (Sms sms : smsList) {
            /*cr.insert(FoveaContract.Sms.CONTENT_SMS_URI,
                    sms.getContentValues());*/
            FoveaDatabaseManager.insert(this.getApplicationContext(), FoveaContract.Sms.CONTENT_SMS_URI,
                    sms.getContentValues());
        }


    }


}
