package com.cogknit.fovea.services;

import android.app.IntentService;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;

import com.android.volley.Response;
import com.android.volley.VolleyError;
import com.android.volley.VolleyLog;
import com.cogknit.fovea.FoveaSharedPreferences;
import com.cogknit.fovea.receivers.AlarmReceiver;
import com.cogknit.fovea.remote.APIConstants;
import com.cogknit.fovea.remote.FoveaRequestResponseManager;
import com.cogknit.fovea.utils.FoveaLog;
import com.cogknit.fovea.utils.SchedulerConstants;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * IntentService to source ME, LIKES or FRIENDS of a logged in user and Sync with Fovea Server
 * Created by Chirag on 08-09-2015.
 */
public class FacebookVolleySyncService extends IntentService {

    /**
     * Debug Tag
     */
    private static final String TAG = "FBVolleySyncService";

    /**
     * Static instance of FoveaRequestResponseManager to manage networking
     */
    private static FoveaRequestResponseManager remote;

    private Intent intent;

    /**
     * Constructor; Initialize Remote interface if required.
     */
    public FacebookVolleySyncService() {
        super("FacebookSyncService");

    }

    /**
     * Scynvhronize required by inspecting intent action.
     *
     * @param intent The intent which triggered the IntentService
     */
    @Override
    protected void onHandleIntent(Intent intent) {
        if (remote == null)
            remote = new FoveaRequestResponseManager(this.getApplicationContext());
        this.intent=intent;

        FoveaLog.d(TAG, TAG + " invoked");
        if (isUserLoggedin()) {
            synchronize(intent.getAction());
        } else {
            FoveaLog.d(TAG, "Unable to Sync Facebook Data: No user logged in!");
        }

    }

    /**
     * Helper method to invoke required synchronization procedure
     *
     * @param dataPoint Endpoint requested for synchronization
     */
    private void synchronize(String dataPoint) {

        switch (dataPoint) {
            case SchedulerConstants.FacebookSync.ACTION_FB_ME:
                getMe();
                break;
            case SchedulerConstants.FacebookSync.ACTION_FB_FRIENDS:
                getFriends();
                break;
            case SchedulerConstants.FacebookSync.ACTION_FB_LIKES:
                getLikes();
                break;
            default:
                FoveaLog.e(TAG, "synchronize(): Invalid datapoint value.");
        }


    }

    /**
     * Helper method to mine User Profile information
     */
    private void getMe() {

        final String URL = GraphApiHelper.ME.getUrlWithParams(getCurrentAccessToken());

        remote.executeJsonRequest(URL, null, new SimpleRequestCallback(new InitiateSyncCallback(APIConstants.getAPIClientURL(this) + APIConstants.URL_FB_ME)), new ErrorCallback());

    }

    /**
     * Helper method to mine User Likes
     */
    private void getLikes() {
        //TODO Get only latest likes
        final String URL = GraphApiHelper.LIKES.getUrlWithParams(getCurrentAccessToken());
        remote.executeJsonRequest(URL, null, new PagedRequestCallback(new InitiateSyncCallback(APIConstants.getAPIClientURL(this) + APIConstants.URL_FB_LIKES)), new ErrorCallback());

    }

    /**
     * Helper method to mine User Friends
     */
    private void getFriends() {
        //TODO Get only latest friends
        final String URL = GraphApiHelper.FRIENDS.getUrlWithParams(getCurrentAccessToken());

        remote.executeJsonRequest(URL, null, new PagedRequestCallback(new InitiateSyncCallback(APIConstants.getAPIClientURL(this) + APIConstants.URL_FB_FRIENDS)), new ErrorCallback());
    }

    /**
     * Helper method to check if token exists
     *
     * @return true if token is available
     */
    private boolean isUserLoggedin() {
        String token = getCurrentAccessToken();
        return (!token.isEmpty());
    }

    /**
     * A helper method to retrieve the current access token from SharedPrefs
     * @return The access_token as a String or "" if no token exists.
     */
    private String getCurrentAccessToken() {
        SharedPreferences prefs = this.getSharedPreferences(FoveaSharedPreferences.FB.PREFS_NAME, Context.MODE_PRIVATE);
        return prefs.getString(FoveaSharedPreferences.FB.ACCESS_TOKEN, "");
    }

    //Facebook Parsing Callback Classes

    /**
     * A Consolidated Callback which is responsible for relaying mined FB Data to Fovea
     */
    private class InitiateSyncCallback implements ConsolidatedCallback {
        private String url;


        public InitiateSyncCallback(String url) {
            this.url = url;
        }

        @Override
        public void onCompleted(String payload) {
            makeVolleyRequest(this.url, payload);
            AlarmReceiver.completeWakefulIntent(intent);
        }


        public void onError(Exception e) {
            e.printStackTrace();
        }


        private void makeVolleyRequest(String urlString, String payload) {
            FoveaLog.d(TAG, "Sending to FOVEA: " + payload);
            remote.sendFbData(urlString, payload);

        }
    }

    /**
     * A Volley Response handler to handle non-paged JSON responses
     */
    private class SimpleRequestCallback implements Response.Listener<JSONObject> {
        /**
         * A Consolidated Callback which is invoked with the data received
         */
        ConsolidatedCallback callback;

        public SimpleRequestCallback(ConsolidatedCallback callback) {
            this.callback = callback;
        }

        @Override
        public void onResponse(JSONObject response) {
            //FoveaLog.d(TAG, response.toString());
            callback.onCompleted(response.toString());
        }
    }

    /**
     * A Volley Response handler for paged requests recursively
     */
    private class PagedRequestCallback implements Response.Listener<JSONObject> {

        /**
         * A Consolidated Callback which is invoked with the data received
         */
        private ConsolidatedCallback callback;

        /**
         * A counter to track number of pages received
         */
        private int count;

        /**
         * A JSON Array to hold the merged array
         */
        private JSONArray jsonPayload;

        public PagedRequestCallback(ConsolidatedCallback callback) {
            this.callback = callback;
            jsonPayload = new JSONArray();
            count = 0;
        }


        /**
         * Procsess the response and if there is a next page, extract the link and make a new request to the next page
         *
         * @param response JSON Response as returned by Facebook
         */
        @Override
        public void onResponse(JSONObject response) {
            FoveaLog.d(TAG, "Page " + ++count + ": " + response.toString());

            try {
                this.jsonPayload = mergeArrays(this.jsonPayload, response.getJSONArray("data"));

                //Check if more pages exist
                if (hasNext(response)) {
                    remote.executeJsonRequest(getNextUrl(response), null, this, new ErrorCallback());
                } else {
                    callback.onCompleted(jsonPayload.toString());
                }
            } catch (JSONException e) {
                e.printStackTrace();
                callback.onError(e);
            }
        }

        /**
         * Inspects the reponse to check if a "next" field exists
         *
         * @param response JSON Response as returned by Facebook
         * @return true if "next" field exists in the response
         * @throws JSONException
         */
        private boolean hasNext(JSONObject response) throws JSONException {
            return response.has("paging") && response.getJSONObject("paging").has("next");
        }

        /**
         * Extracts the url from the "next field to make a request for the next page
         *
         * @param response JSON Response as returned by Facebook
         * @return The URL to the next page of current response
         * @throws JSONException if the "next" field doesn't exist
         */
        private String getNextUrl(JSONObject response) throws JSONException {
            return response.getJSONObject("paging").getString("next");

        }

        /**
         * Combine two JSONArrays (Used to merge results across a paged response
         *
         * @param a First array
         * @param b Second array
         * @return A JSON Array containing all elements from a and b
         */
        private JSONArray mergeArrays(JSONArray a, JSONArray b) {
            JSONArray merged = new JSONArray();
            try {
                for (int i = 0; i < a.length(); i++) {
                    merged.put(a.get(i));
                }
                for (int i = 0; i < b.length(); i++) {
                    merged.put(b.get(i));
                }

            } catch (JSONException e) {
                e.printStackTrace();
            }
            return merged;
        }
    }

    /**
     * Simple wrapper class for Volley error callback
     */
    private class ErrorCallback implements Response.ErrorListener {

        @Override
        public void onErrorResponse(VolleyError error) {
            VolleyLog.e("Error: ", error.getMessage());
        }
    }

    /**
     * Interface to define callbacks for responses for implementing custom logic in a modular fashion through them.
     * Currently used to forward FB results to Fovea
     */
    private interface ConsolidatedCallback {
        void onCompleted(String payload);

        void onError(Exception e);
    }

    /**
     * Enum listing different endpoints to FB Graph API and helper methods to generate URL with parameters
     */
    public enum GraphApiHelper {
        ME, LIKES, FRIENDS;
        public static final String BASE_URL = "https://graph.facebook.com";
        public static final String URL_ME = BASE_URL + "/me";
        public static final String URL_LIKES = URL_ME + "/likes";
        public static final String URL_FRIENDS = URL_ME + "/friends";
        public static String TOKEN_FORMAT = "access_token=%s";

        public String getUrlWithParams(String token) {
            String url = "";
            switch (this) {
                case ME:
                    url = URL_ME;
                    break;
                case LIKES:
                    url = URL_LIKES;
                    break;
                case FRIENDS:
                    url = URL_FRIENDS;
                    break;
            }
            return url + "?" + getAccessTokenParam(token) + "&" + getFieldsParam();
        }

        private String getAccessTokenParam(String token) {
            return String.format(TOKEN_FORMAT, token);
        }

        private String getFieldsParam() {
            String param = "";
            switch (this) {
                case ME:
                    param = "fields=id,name,link,gender,location,religion,age_range,email,sports,work,devices,birthday,bio,about";
                    break;
                case LIKES:
                case FRIENDS:
                    param = "fields=id";
                    break;
            }
            return param;
        }

    }


}
