package com.cogknit.fovea;

import android.content.Context;
import android.content.SharedPreferences;

import com.cogknit.fovea.providers.FoveaAppInvitee;
import com.cogknit.fovea.utils.DeviceUtility;
import com.cogknit.fovea.utils.FoveaLog;

import java.util.List;

/**
 * This class is the main interface for the Fovea Context Engine
 */
public class Fovea {
    private static FoveaManager foveaManager = null;
    private static final String TAG="Fovea";

    /**
     * This is the method responsible for initialize the Fovea SDK with the Appkey, Client key and Mining Contexts.
     * @param context
     * @param appKey
     * @param clientKey
     * @param services
     * @param foveaCallback
     */
    public static void initialize(Context context, String appKey, String clientKey, FoveaConstants.Services[] services, FoveaCallback foveaCallback){
        if (null==foveaManager)
        {
            foveaManager = new FoveaManager(appKey,clientKey,context);
        }

        if (services != null)
        {
            for(FoveaConstants.Services service : services)
            {
                if (DeviceUtility.checkIfPermissionsGrantedAndEnableContexts(context, service))
                {
                    service.activateService(context);
                }
                else
                {
                    //raise an exception
                    foveaCallback.onFailure(new Exception("Necessary permissions for "+service.name()+" service is missing from the manifest or not granted by the user! Please add and retry."));
                }
            }

            for(FoveaConstants.Services service : FoveaConstants.Services.values()){
                FoveaLog.d(TAG, service.name() + ": " + service.getStatus(context));
            }
        }


        foveaManager.initialize(context, foveaCallback);
    }

    /**
     ** This method will initialize the Customer profile for the logged in device user.
     * @param customerProfile object
     * @param callback
     */
    public static void setCustomerProfileDetails(FoveaUserProfile customerProfile, FoveaCallback callback){
        if (null==foveaManager)
        {
            callback.onFailure(new UninitializedException());
        }
        foveaManager.setUserProfileDetails(customerProfile);
        callback.onCompletion();
    }

    /**
     ** This method will initialize Fovea with the Facebook access token of the App user
     * @param context
     * @param token object
     * Raises exception, in case if Fovea is not initialized
     */
    public static void setFacebookAccessToken(Context context, String token) throws UninitializedException {
        //1. Check if SDK is authorized
        SharedPreferences settings = context.getSharedPreferences(
                FoveaSharedPreferences.FOVEA_PREFS, Context.MODE_PRIVATE);
        Boolean hasAuthenticated = settings.getBoolean(FoveaSharedPreferences.UserProfile.isSDKAuthorized, false);
        if (!hasAuthenticated)
        {
            throw new Fovea.UninitializedException();
        }

        if (null==foveaManager)
        {
            initializeFoveaManager(context);
        }
        foveaManager.setFbAccessToken(token);
    }

    /**
     ** This method returns the AppInvitees for the referral
     * Raises exception, in case if Fovea is not initialized
     * @param context
     */
    public static List<FoveaAppInvitee> getAppInvitees(Context context) throws UninitializedException {
        //1. Check if SDK is authorized
        SharedPreferences settings = context.getSharedPreferences(
                FoveaSharedPreferences.FOVEA_PREFS, Context.MODE_PRIVATE);
        Boolean hasAuthenticated = settings.getBoolean(FoveaSharedPreferences.UserProfile.isSDKAuthorized, false);
        if (!hasAuthenticated)
        {
            throw new Fovea.UninitializedException();
        }

        if (null==foveaManager)
        {
            initializeFoveaManager(context);
        }
        return foveaManager.getAppInvitees();
    }

    /**
     ** Invoke this method, to proceed with AppInvitations
     * @param context
     * @param approvedInvitees - List of contacts that user has approved
     * @param deniedInvitees - List of contacts that user has denied
     * @param inviteMessage - The App referral message to be sent out to the invitees
     * Raises UninitializedException, in case if Fovea is not initialized
     * Raises AppInviteException, in case if message length exceeds the 140 char length limit or
     * if Telephone services are not available on the platform
     */
    public static void proceedWithAppInvites(Context context, List<FoveaAppInvitee> approvedInvitees,
                                            List<FoveaAppInvitee> deniedInvitees,
                                            String inviteMessage) throws UninitializedException, Fovea.AppInviteException {
        //1. Check if SDK is authorized
        SharedPreferences settings = context.getSharedPreferences(
                FoveaSharedPreferences.FOVEA_PREFS, Context.MODE_PRIVATE);
        Boolean hasAuthenticated = settings.getBoolean(FoveaSharedPreferences.UserProfile.isSDKAuthorized, false);
        if (!hasAuthenticated)
        {
            throw new Fovea.UninitializedException();
        }

        if (null==foveaManager)
        {
            initializeFoveaManager(context);
        }
        foveaManager.feedbackOnAppInvites(approvedInvitees, deniedInvitees, inviteMessage);
    }

    /**
    ** This class represents the standard Fovea uninitialized exception
    **/
    public static class UninitializedException extends Exception{
        public UninitializedException() {
            super("Please invoke initialize and successfully authorize with keys before accessing this service");
        }

    }

    /**
     ** This class represents the standard Fovea App invite exception
     **/
    public static class AppInviteException extends Exception{
        public AppInviteException(String reason) {
            super(reason);
        }

    }

    private static void initializeFoveaManager(Context context)
    {
        SharedPreferences settings = context.getSharedPreferences(FoveaSharedPreferences.FOVEA_PREFS, Context.MODE_PRIVATE);
        String appkey = settings.getString(FoveaSharedPreferences.UserProfile.AppKey, "");
        String clientKey = settings.getString(FoveaSharedPreferences.UserProfile.ClientKey, "");
        foveaManager = new FoveaManager(appkey,clientKey,context);
    }




}


