package com.coder.mario.android.widget.refresh;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.ValueAnimator;
import android.content.Context;
import android.os.Handler;
import android.os.Message;
import android.support.annotation.IntDef;
import android.support.v4.view.NestedScrollingChild;
import android.support.v4.view.NestedScrollingChildHelper;
import android.support.v4.view.NestedScrollingParent;
import android.support.v4.view.NestedScrollingParentHelper;
import android.support.v4.view.ViewCompat;
import android.util.AttributeSet;
import android.util.Log;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.LinearLayout;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Created by CoderMario on 2018-01-19.
 */
public class NestedRefreshLayout extends LinearLayout implements NestedScrollingParent, NestedScrollingChild {

    /**
     * 标记
     * */
    public static final int STATE_CURRENT_RESET = 0;

    /**
     * 标记
     * */
    public static final int STATE_CURRENT_LOADING = 1;

    /**
     * 标记
     * */
    public static final int STATE_RELEASE_NOTHING = 2;

    /**
     * 标记
     * */
    public static final int STATE_LOADING_COMPLETE = 3;

    /**
     * 标记
     * */
    public static final int STATE_RELEASE_TO_LOADING = 4;

    /***
     * */
    @IntDef({STATE_CURRENT_RESET, STATE_CURRENT_LOADING, STATE_RELEASE_NOTHING, STATE_LOADING_COMPLETE, STATE_RELEASE_TO_LOADING})
    @Retention(RetentionPolicy.SOURCE)
    public @interface State {}

    /**
     * */
    private int mState = STATE_CURRENT_RESET;

    /**
     * */
    private INestedRefreshState mRefreshState;

    /**
     * */
    private NestedScrollingParentHelper mParentHelper;

    /**
     * */
    private NestedScrollingChildHelper mChildHelper;

    /**
     * */
    private AtomicBoolean mNestedTag = new AtomicBoolean(false);

    /**
     * */
    private ValueAnimator mBackAnimator;

    /**
     * */
    private float dyHolder;

    /**
     * */
    private int Range_Max = 520;

    /***
     * */
    private FrameLayout mHeadHolder;

    /***
     * */
    private INestedRefreshLoad mINestedRefreshLoad;

    /***
     * */
    private LoadStartListener mStartListener;

    /**
     * */
    private IdleState mIdleState;

    /**
     * */
    private WorkState mWorkState;

    /**
     * */
    private long mDuration = 1600L;

    /**
     * */
    private DurationHandler mDurationHandler;

    /**
     * */
    private List<ScrollChangeListener> mScrollChangeListenerSet;

    /**
     * */
    private float mResistance = 0.12f;




    public NestedRefreshLayout(Context context) {
        super(context);
        init();
    }

    public NestedRefreshLayout(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    public NestedRefreshLayout(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    private void init() {
        setRefreshState(getIdleState());
        this.setOrientation(LinearLayout.VERTICAL);

        mHeadHolder = new FrameLayout(getContext());
        LayoutParams layoutParams = new LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT);
        addView(mHeadHolder, layoutParams);
    }

    private INestedRefreshLoad getINestedRefreshLoad() {
        return this.mINestedRefreshLoad;
    }

    private IdleState getIdleState() {
        if (null == this.mIdleState) {
            this.mIdleState = new IdleState();
        }
        return this.mIdleState;
    }

    private WorkState getWorkState() {
        if (null == this.mWorkState) {
            this.mWorkState = new WorkState();
        }
        return this.mWorkState;
    }

    private int getState() {
        return this.mState;
    }

    private void setState(int state) {
        if (this.mState == state) {
            return;
        }
        this.mState = state;
        onStateChanged(state);
    }

    public void setRefreshLoad(INestedRefreshLoad refreshLoad) {
        this.mINestedRefreshLoad = refreshLoad;
        if (null == refreshLoad) {
            return;
        }
        int viewHeight = refreshLoad.getViewHeight();
        FrameLayout.LayoutParams layoutParams01 = new FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.MATCH_PARENT, viewHeight);
        mHeadHolder.addView(refreshLoad.getView(), layoutParams01);
        LayoutParams layoutParams02 = (LayoutParams) mHeadHolder.getLayoutParams();
        layoutParams02.setMargins(0, 0 - refreshLoad.getViewHeight(), 0, 0);
        mHeadHolder.setLayoutParams(layoutParams02);
    }

    public final void addView(View child) {
        if(null == child) {
            throw new NullPointerException("the direct child couldn't be null");
        } else if(this.getChildCount() > 1) {
            throw new IllegalStateException("NestedRefreshLayout can host only one direct child");
        } else {
            super.addView(child);
        }
    }

    public final void addView(View child, int index) {
        if(null == child) {
            throw new NullPointerException("the direct child couldn't been null");
        } else if(this.getChildCount() > 1) {
            throw new IllegalStateException("NestedRefreshLayout can host only one direct child");
        } else {
            super.addView(child, index);
        }
    }

    public final void addView(View child, LayoutParams params) {
        if(null == child) {
            throw new NullPointerException("the direct child couldn't been null");
        } else if(this.getChildCount() > 1) {
            throw new IllegalStateException("NestedRefreshLayout can host only one direct child");
        } else {
            super.addView(child, params);
        }
    }

    public final void addView(View child, int width, int height) {
        if(null == child) {
            throw new NullPointerException("the direct child couldn't been null");
        } else if(this.getChildCount() > 1) {
            throw new IllegalStateException("NestedRefreshLayout can host only one direct child");
        } else {
            super.addView(child, width, height);
        }
    }

    public final void addView(View child, int index, LayoutParams params) {
        if(null == child) {
            throw new NullPointerException("the direct child couldn't been null");
        } else if(this.getChildCount() > 1) {
            throw new IllegalStateException("NestedRefreshLayout can host only one direct child");
        } else {
            super.addView(child, index, params);
        }
    }

    @Override
    public boolean dispatchTouchEvent(MotionEvent event) {
        int action = null == event ? 0 - 1 : event.getAction();
        if (MotionEvent.ACTION_DOWN == action) {
            ValueAnimator animator = getBackAnimatorWithoutInit();
            if (null != animator && animator.isRunning()) {
                animator.cancel();
            }
        }
        return super.dispatchTouchEvent(event);
    }

    private NestedScrollingParentHelper getParentHelper() {
        if (null == mParentHelper) {
            mParentHelper = new NestedScrollingParentHelper(this);
        }
        return mParentHelper;
    }

    private NestedScrollingChildHelper getChildHelper() {
        if (null == mChildHelper) {
            mChildHelper = new NestedScrollingChildHelper(this);
        }
        return mChildHelper;
    }

    @Override
    public boolean isNestedScrollingEnabled() {
        return true;
    }

    @Override
    public boolean onStartNestedScroll(View child, View target, int nestedScrollAxes) {
        if (ViewCompat.SCROLL_AXIS_VERTICAL == nestedScrollAxes) {
            return getRefreshState().onStartNestedScroll(child, target);
        }
        return false;
    }

    @Override
    public void onNestedPreScroll(View target, int dx, int dy, int[] consumed) {
        boolean set = mNestedTag.compareAndSet(false, true);
        if (set) {
            getDurationHandler().remove();
        }
        int scrollY = getScrollY();
        if (0 < dy && 0 > scrollY) {
            getRefreshState().onNestedPreScroll(target, dx, dy, consumed);
        }
    }

    @Override
    public void onNestedScroll(View target, int dxConsumed, int dyConsumed, int dxUnconsumed, int dyUnconsumed) {
        if (0 != Math.abs(dyUnconsumed)) {
            getRefreshState().onNestedScroll(target, dxUnconsumed, dyUnconsumed);
        }
    }

    @Override
    public void onStopNestedScroll(View child) {
        if (!mNestedTag.get() || 0 == getScrollY()) {
            return;
        }
        int scrollY = getScrollY();
        mNestedTag.compareAndSet(true, false);
        INestedRefreshState refreshState = getRefreshState();
        if (null == refreshState) {
            return;
        }
        INestedRefreshLoad refreshLoad = getINestedRefreshLoad();
        if (null == refreshLoad) {
            refreshState.onStopNestedScroll(child, Integer.MAX_VALUE, scrollY);
        } else {
            refreshState.onStopNestedScroll(child, refreshLoad.getViewHeight(), scrollY);
        }
    }

    @Override
    public boolean onNestedPreFling(View target, float velocityX, float velocityY) {
        if (0 == velocityY) {
            return true;
        }
        int dir = (int) (velocityY / Math.abs(velocityY));
        return (0 > getScrollY() || canScrollVertically(dir));
    }

    public void setDuration(long duration) {
        this.mDuration = duration;
    }

    /**
     * */
    private void setRefreshState(INestedRefreshState state) {
        this.mRefreshState = state;
        if (null == state) {
            return;
        }
        dyHolder = state.parse2Offset(getScrollY());
    }

    /**
     * */
    private INestedRefreshState getRefreshState() {
        return this.mRefreshState;
    }

    @Override
    public void scrollTo(int x, int y) {
        super.scrollTo(x, y);
        int scrollY = getScrollY();
        INestedRefreshLoad refreshLoad = getINestedRefreshLoad();
        if (null != refreshLoad) {
            refreshLoad.onOffsetChanged(Math.abs(scrollY));
        }
        INestedRefreshState refreshState = getRefreshState();
        if (null != refreshState) {
            refreshState.onScrollChanged(scrollY);
        }
        List<ScrollChangeListener> set = getScrollChangeListenerSet();
        if (null != set && 0 <= set.size()) {
            for (ScrollChangeListener listener : set) {
                if (null == listener) {
                    continue;
                }
                listener.onScrollChanged(y);
            }
        }
    }

    /**
     * */
    private abstract class BaseState implements INestedRefreshState {

        @Override
        public void scrollBy(int dy) {
            NestedRefreshLayout.this.scrollBy(0, dy);
        }
    }

    /**
     * */
    private void onStateChanged(int state) {
        INestedRefreshState refreshState = getRefreshState();
        if (null != refreshState) {
            refreshState.onStateChanged(state);
        }
        if (STATE_CURRENT_LOADING == state) {
            setRefreshState(getWorkState());
            if (null != mStartListener) {
                mStartListener.onStart();
            }
        }
    }

    /**
     * */
    private class IdleState extends BaseState {

        @Override
        public void onAnimatorEnd() {
            int scrollY = getScrollY();
            dyHolder = parse2Offset(scrollY);
            INestedRefreshLoad refreshLoad = getINestedRefreshLoad();
            if (null == refreshLoad) {
                return;
            }
            int height = refreshLoad.getViewHeight();
            if (0 == scrollY) {
                setState(STATE_CURRENT_RESET);
            } else if (0 >= scrollY && height == Math.abs(scrollY)) {
                setState(STATE_CURRENT_LOADING);
            }
        }

        @Override
        public int parse2ScrollY(float offset) {
            if (0 == offset) {
                return 0;
            }
            int dir = (int) (Math.abs(offset) / offset);
            offset = Math.abs(offset);
            float dyMax = Range_Max / mResistance;
            float value = 1.f * Math.min(offset, dyMax) / dyMax;
            float rate = (float) (1.0f - Math.pow((1.0f - value), 4.f));
            int scrollY = (int) (rate * Range_Max);
            return dir * scrollY;
        }

        @Override
        public int parse2Offset(float scrollY) {
            if (0 == scrollY) {
                return 0;
            }
            int dir = (int) (Math.abs(scrollY) / scrollY);
            scrollY = Math.abs(scrollY);
            float value = 1.f - 1.f * scrollY / Range_Max;
            float rate = (float) (1.f - Math.sqrt(Math.sqrt(value)));
            int offset = (int) (rate * (Range_Max / mResistance));
            return dir * offset;
        }

        @Override
        public void onStateChanged(int state) {
            INestedRefreshLoad nestedRefreshLoad = getINestedRefreshLoad();
            if (null != nestedRefreshLoad) {
                nestedRefreshLoad.onStateChanged(getState());
            }
        }

        @Override
        public void onScrollChanged(int scrollY) {
            INestedRefreshLoad nestedRefreshLoad = getINestedRefreshLoad();
            if (null == nestedRefreshLoad) {
                return;
            }
            int height = nestedRefreshLoad.getViewHeight();
            if (0 >= scrollY && height > Math.abs(scrollY)) {
                setState(STATE_RELEASE_NOTHING);
            } else if (0 >= scrollY && height <= Math.abs(scrollY)) {
                setState(STATE_RELEASE_TO_LOADING);
            }
        }

        @Override
        public void onStopNestedScroll(View child, int height, int scrollY) {
            int dir = scrollY / Math.abs(scrollY);
            if (height <= Math.abs(scrollY)) {
                smoothScrollTo(dir * height);
            } else {
                smoothScrollTo(0);
            }
        }

        @Override
        public void onNestedScroll(View target, int dx, int dy) {
            dyHolder += dy;
            if (0 == dyHolder) {
                scrollTo(0, 0);
                return;
            }
            int scrollY = parse2ScrollY(dyHolder);
            scrollTo(0, scrollY);
        }

        @Override
        public boolean onStartNestedScroll(View child, View target) {
            return true;
        }

        @Override
        public void onNestedPreScroll(View target, int dx, int dy, int[] consumed) {
            int scrollY = getScrollY();
            if(scrollY + dy >= 0) {
                consumed[1] = 0 - scrollY;
                scrollTo(0, 0);
            } else {
                consumed[1] = dy;
                scrollBy(consumed[1]);
            }
            dyHolder = parse2Offset(getScrollY());
        }
    }

    /**
     * */
    private class WorkState extends BaseState {

        @Override
        public void onAnimatorEnd() {
            dyHolder = parse2Offset(getScrollY());
            int scrollY = getScrollY();
            if (0 == scrollY && STATE_LOADING_COMPLETE == getState()) {
                setRefreshState(getIdleState());
                setState(STATE_CURRENT_RESET);
            }
        }

        @Override
        public int parse2ScrollY(float offset) {
            if (0 == offset) {
                return 0;
            }
            int dir = (int) (Math.abs(offset) / offset);
            offset = Math.abs(offset);
            float dyMax = Range_Max / mResistance;
            float value = 1.f * Math.min(offset, dyMax) / dyMax;
            float rate = (float) (1.0f - Math.pow((1.0f - value), 4.f));
            int scrollY = (int) (rate * Range_Max);
            return dir * scrollY;
        }

        @Override
        public int parse2Offset(float scrollY) {
            if (0 == scrollY) {
                return 0;
            }
            int dir = (int) (Math.abs(scrollY) / scrollY);
            scrollY = Math.abs(scrollY);
            float value = 1.f - 1.f * scrollY / Range_Max;
            float rate = (float) (1.f - Math.sqrt(Math.sqrt(value)));
            int offset = (int) (rate * (Range_Max / mResistance));
            return dir * offset;
        }

        @Override
        public void onStateChanged(int state) {
            if (STATE_LOADING_COMPLETE != state && STATE_CURRENT_RESET != state) {
                return;
            }
            INestedRefreshLoad nestedRefreshLoad = getINestedRefreshLoad();
            if (null != nestedRefreshLoad) {
                nestedRefreshLoad.onStateChanged(getState());
            }
        }

        @Override
        public void onScrollChanged(int scrollY) {

        }

        @Override
        public void onStopNestedScroll(View child, int height, int scrollY) {
            int dir = scrollY / Math.abs(scrollY);
            int state = getState();
            if (STATE_LOADING_COMPLETE == state) {
                smoothScrollTo(0);
                return;
            }
            INestedRefreshLoad refreshLoad = getINestedRefreshLoad();
            if (null == refreshLoad || 0 < scrollY) {
                smoothScrollTo(0);
                return;
            }
            if (height <= Math.abs(scrollY)) {
                height = refreshLoad.getViewHeight();
                smoothScrollTo(dir * height);
            } else {
                smoothScrollTo(0);
            }
        }

        @Override
        public void onNestedScroll(View target, int dx, int dy) {
            dyHolder += dy;
            if (0 == dyHolder) {
                scrollTo(0, 0);
                return;
            }
            int scrollY = parse2ScrollY(dyHolder);
            scrollTo(0, scrollY);
        }

        @Override
        public boolean onStartNestedScroll(View child, View target) {
            return true;
        }

        @Override
        public void onNestedPreScroll(View target, int dx, int dy, int[] consumed) {
            int scrollY = getScrollY();
            if(scrollY + dy >= 0) {
                consumed[1] = 0 - scrollY;
                scrollTo(0, 0);
            } else {
                consumed[1] = dy;
                scrollBy(consumed[1]);
            }
            dyHolder = parse2Offset(getScrollY());
        }
    }

    private List<ScrollChangeListener> getScrollChangeListenerSet() {
        if (null == mScrollChangeListenerSet) {
            mScrollChangeListenerSet = new ArrayList<>();
        }
        return mScrollChangeListenerSet;
    }

    public void addScrollChangeListener(ScrollChangeListener listener) {
        if (null == listener || getScrollChangeListenerSet().contains(listener)) {
            return;
        }
        getScrollChangeListenerSet().add(listener);
    }

    public void removeScrollChangeListener(ScrollChangeListener listener) {
        if (null == listener || !getScrollChangeListenerSet().contains(listener)) {
            return;
        }
        getScrollChangeListenerSet().remove(listener);
    }

    public void setStartListener(LoadStartListener listener) {
        this.mStartListener = listener;
    }

    private ValueAnimator getBackAnimatorWithoutInit() {
        return mBackAnimator;
    }

    private void smoothScrollTo(int target) {
        int scrollY = getScrollY();
        ValueAnimator animator = getBackAnimator();
        animator.setIntValues(scrollY, target);
        animator.setDuration(160);

        animator.start();
    }

    private ValueAnimator getBackAnimator() {
        if (null == mBackAnimator) {
            mBackAnimator = new ValueAnimator();
            mBackAnimator.addListener(new InnerSmoothScrollListener());
            mBackAnimator.addUpdateListener(new InnerUpdateListener());
        }
        return mBackAnimator;
    }

    private class InnerUpdateListener implements ValueAnimator.AnimatorUpdateListener {

        @Override
        public void onAnimationUpdate(ValueAnimator animation) {
            int value = (int) animation.getAnimatedValue();
            scrollTo(0, value);
        }
    }

    /**
     * */
    private class InnerSmoothScrollListener extends AnimatorListenerAdapter {

        @Override
        public void onAnimationEnd(Animator animation) {
            INestedRefreshState refreshState = getRefreshState();
            refreshState.onAnimatorEnd();
        }
    }

    public void completeLoad() {
        INestedRefreshState refreshState = getRefreshState();
        if (!(refreshState instanceof WorkState)) {
            return;
        }
        setState(STATE_LOADING_COMPLETE);
        if (!mNestedTag.get()) {
            getDurationHandler().send();
        }
    }

    private DurationHandler getDurationHandler() {
        if (null == mDurationHandler) {
            mDurationHandler = new DurationHandler();
        }
        return mDurationHandler;
    }

    private class DurationHandler extends Handler {

        public void send() {
            removeMessages(0x264);
            sendEmptyMessageDelayed(0x264, mDuration);
        }

        public void remove() {
            removeMessages(0x264);
        }

        @Override
        public void handleMessage(Message msg) {
            if (!mNestedTag.get()) {
                smoothScrollTo(0);
            }
        }
    }

    /**
     * */
    public interface ScrollChangeListener {

        void onScrollChanged(int scrollY);
    }

    /**
     * */
    public interface LoadStartListener {

        void onStart();
    }
}
