/*
 * Decompiled with CFR 0.152.
 */
package com.clickhouse.client.api;

import com.clickhouse.client.ClickHouseClient;
import com.clickhouse.client.ClickHouseClientBuilder;
import com.clickhouse.client.ClickHouseConfig;
import com.clickhouse.client.ClickHouseNode;
import com.clickhouse.client.ClickHouseNodeSelector;
import com.clickhouse.client.ClickHouseProtocol;
import com.clickhouse.client.ClickHouseRequest;
import com.clickhouse.client.ClickHouseResponse;
import com.clickhouse.client.api.ClientException;
import com.clickhouse.client.api.OperationStatistics;
import com.clickhouse.client.api.Protocol;
import com.clickhouse.client.api.insert.DataSerializationException;
import com.clickhouse.client.api.insert.InsertResponse;
import com.clickhouse.client.api.insert.InsertSettings;
import com.clickhouse.client.api.insert.POJOSerializer;
import com.clickhouse.client.api.insert.SerializerNotFoundException;
import com.clickhouse.client.api.internal.SerializerUtils;
import com.clickhouse.client.api.internal.SettingsConverter;
import com.clickhouse.client.api.internal.TableSchemaParser;
import com.clickhouse.client.api.internal.ValidationUtils;
import com.clickhouse.client.api.metadata.TableSchema;
import com.clickhouse.client.api.query.QueryResponse;
import com.clickhouse.client.api.query.QuerySettings;
import com.clickhouse.client.config.ClickHouseClientOption;
import com.clickhouse.data.ClickHouseColumn;
import com.clickhouse.data.ClickHouseDataConfig;
import com.clickhouse.data.ClickHouseDataStreamFactory;
import com.clickhouse.data.ClickHouseFormat;
import com.clickhouse.data.ClickHousePipedOutputStream;
import com.clickhouse.data.format.BinaryStreamUtils;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class Client {
    public static final int TIMEOUT = 30000;
    private Set<String> endpoints;
    private Map<String, String> configuration;
    private List<ClickHouseNode> serverNodes = new ArrayList<ClickHouseNode>();
    private Map<Class<?>, List<POJOSerializer>> serializers;
    private Map<Class<?>, Map<String, Method>> getterMethods;
    private Map<Class<?>, Boolean> hasDefaults;
    private static final Logger LOG = LoggerFactory.getLogger(Client.class);
    private ExecutorService queryExecutor;
    private Map<String, OperationStatistics.ClientStatistics> globalClientStats = new ConcurrentHashMap<String, OperationStatistics.ClientStatistics>();
    private static final Set<String> COMPRESS_ALGORITHMS = ValidationUtils.whiteList("LZ4", "LZ4HC", "ZSTD", "ZSTDHC", "NONE");
    private static final Set<String> OUTPUT_FORMATS = Client.createFormatWhitelist("output");
    private static final Set<String> INPUT_FORMATS = Client.createFormatWhitelist("input");
    private static final String INTERNAL_OPERATION_ID = "operationID";

    private Client(Set<String> endpoints, Map<String, String> configuration) {
        this.endpoints = endpoints;
        this.configuration = configuration;
        this.endpoints.forEach(endpoint -> this.serverNodes.add(ClickHouseNode.of((String)endpoint, this.configuration)));
        this.serializers = new HashMap();
        this.getterMethods = new HashMap();
        this.hasDefaults = new HashMap();
        int numThreads = Integer.parseInt(configuration.getOrDefault(ClickHouseClientOption.MAX_THREADS_PER_CLIENT.getKey(), "3"));
        this.queryExecutor = Executors.newFixedThreadPool(numThreads, r -> {
            Thread t = new Thread(r);
            t.setName("ClickHouse-Query-Executor");
            t.setUncaughtExceptionHandler((t1, e) -> LOG.error("Uncaught exception in thread {}", (Object)t1.getName(), (Object)e));
            return t;
        });
        LOG.debug("Query executor created with {} threads", (Object)numThreads);
    }

    private ClickHouseNode getServerNode() {
        return this.serverNodes.get(0);
    }

    public boolean ping() {
        return this.ping(30000);
    }

    public boolean ping(int timeout) {
        ClickHouseClient clientPing = ClickHouseClient.newInstance((ClickHouseProtocol[])new ClickHouseProtocol[]{ClickHouseProtocol.HTTP});
        return clientPing.ping(this.getServerNode(), timeout);
    }

    public void register(Class<?> clazz, TableSchema schema) {
        LOG.debug("Registering POJO: {}", (Object)clazz.getName());
        ArrayList<POJOSerializer> serializers = new ArrayList<POJOSerializer>();
        HashMap<String, Method> getterMethods = new HashMap<String, Method>();
        for (Method method : clazz.getMethods()) {
            String methodName = method.getName();
            if (methodName.startsWith("get") || methodName.startsWith("has")) {
                methodName = methodName.substring(3).toLowerCase();
                getterMethods.put(methodName, method);
            }
            if (!methodName.startsWith("is")) continue;
            methodName = methodName.substring(2).toLowerCase();
            getterMethods.put(methodName, method);
        }
        this.getterMethods.put(clazz, getterMethods);
        for (ClickHouseColumn column : schema.getColumns()) {
            String columnName = column.getColumnName().toLowerCase().replace("_", "");
            serializers.add((obj, stream) -> {
                if (!getterMethods.containsKey(columnName)) {
                    LOG.warn("No getter method found for column: {}", (Object)columnName);
                    return;
                }
                Method getterMethod = this.getterMethods.get(clazz).get(columnName);
                Object value = getterMethod.invoke(obj, new Object[0]);
                boolean hasDefaults = this.hasDefaults.get(clazz);
                if (value == null) {
                    if (hasDefaults && !column.hasDefault()) {
                        BinaryStreamUtils.writeNonNull((OutputStream)stream);
                    }
                    BinaryStreamUtils.writeNull((OutputStream)stream);
                    return;
                }
                if (hasDefaults) {
                    BinaryStreamUtils.writeNonNull((OutputStream)stream);
                }
                if (column.isNullable()) {
                    BinaryStreamUtils.writeNonNull((OutputStream)stream);
                }
                SerializerUtils.serializeData(stream, value, column);
            });
        }
        this.serializers.put(clazz, serializers);
        this.hasDefaults.put(clazz, schema.hasDefaults());
    }

    public Future<InsertResponse> insert(String tableName, List<Object> data, InsertSettings settings) {
        boolean hasDefaults;
        String operationId = this.startOperation();
        settings.setSetting(INTERNAL_OPERATION_ID, operationId);
        this.globalClientStats.get(operationId).start("serialization");
        if (data == null || data.isEmpty()) {
            throw new IllegalArgumentException("Data cannot be empty");
        }
        if (settings == null) {
            settings = new InsertSettings();
        }
        if (hasDefaults = this.hasDefaults.get(data.get(0).getClass()).booleanValue()) {
            settings.setFormat(ClickHouseFormat.RowBinaryWithDefaults);
        } else {
            settings.setFormat(ClickHouseFormat.RowBinary);
        }
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        List<POJOSerializer> serializers = this.serializers.get(data.get(0).getClass());
        if (serializers == null || serializers.isEmpty()) {
            throw new SerializerNotFoundException(data.get(0).getClass());
        }
        for (Object obj : data) {
            for (POJOSerializer serializer : serializers) {
                try {
                    serializer.serialize(obj, stream);
                }
                catch (IOException | IllegalAccessException | InvocationTargetException e) {
                    throw new DataSerializationException(obj, serializer, e);
                }
            }
        }
        this.globalClientStats.get(operationId).stop("serialization");
        LOG.debug("Total serialization time: {}", (Object)this.globalClientStats.get(operationId).getElapsedTime("serialization"));
        return this.insert(tableName, new ByteArrayInputStream(stream.toByteArray()), settings);
    }

    public Future<InsertResponse> insert(String tableName, InputStream data, InsertSettings settings) {
        String operationId = (String)settings.getSetting(INTERNAL_OPERATION_ID);
        if (operationId == null) {
            operationId = this.startOperation();
            settings.setSetting(INTERNAL_OPERATION_ID, operationId);
        }
        OperationStatistics.ClientStatistics clientStats = this.globalClientStats.remove(operationId);
        clientStats.start("insert");
        CompletableFuture<InsertResponse> responseFuture = new CompletableFuture<InsertResponse>();
        try (ClickHouseClient client = this.createClient();){
            ClickHouseRequest.Mutation request = this.createMutationRequest(client.write(this.getServerNode()), tableName, settings).format(settings.getFormat());
            CompletableFuture future = null;
            try (ClickHousePipedOutputStream stream = ClickHouseDataStreamFactory.getInstance().createPipedOutputStream((ClickHouseDataConfig)request.getConfig());){
                int bytesRead;
                future = request.data(stream.getInputStream()).execute();
                byte[] buffer = new byte[settings.getInputStreamBatchSize()];
                while ((bytesRead = data.read(buffer)) != -1) {
                    stream.write(buffer, 0, bytesRead);
                }
            }
            catch (IOException e) {
                responseFuture.completeExceptionally(new ClientException("Failed to write data to the output stream", e));
            }
            if (!responseFuture.isCompletedExceptionally()) {
                try {
                    InsertResponse response = new InsertResponse(client, (ClickHouseResponse)future.get(), clientStats);
                    responseFuture.complete(response);
                }
                catch (InterruptedException | ExecutionException e) {
                    responseFuture.completeExceptionally(new ClientException("Operation has likely timed out.", e));
                }
            }
            clientStats.stop("insert");
            LOG.debug("Total insert (InputStream) time: {}", (Object)clientStats.getElapsedTime("insert"));
        }
        return responseFuture;
    }

    public Future<QueryResponse> query(String sqlQuery, Map<String, Object> qparams, QuerySettings settings) {
        OperationStatistics.ClientStatistics clientStats = new OperationStatistics.ClientStatistics();
        clientStats.start("query");
        ClickHouseClient client = this.createClient();
        ClickHouseRequest request = client.read(this.getServerNode());
        ExecutorService executor = this.queryExecutor;
        if (settings.getExecutorService() != null) {
            executor = settings.getExecutorService();
        }
        request.options(SettingsConverter.toRequestOptions(settings.getAllSettings()));
        request.settings(SettingsConverter.toRequestSettings(settings.getAllSettings()));
        request.query(sqlQuery, settings.getQueryID());
        ClickHouseFormat format = ClickHouseFormat.valueOf((String)settings.getFormat());
        request.format(format);
        if (qparams != null && !qparams.isEmpty()) {
            request.params(qparams, new Object[0]);
        }
        CompletableFuture<QueryResponse> future = new CompletableFuture<QueryResponse>();
        executor.submit(() -> {
            MDC.put((String)"queryId", (String)settings.getQueryID());
            LOG.trace("Executing request: {}", (Object)request);
            try {
                QueryResponse queryResponse = new QueryResponse(client, request.execute(), settings, format, clientStats);
                queryResponse.ensureDone();
                future.complete(queryResponse);
            }
            catch (Exception e) {
                future.completeExceptionally(e);
            }
            finally {
                MDC.remove((String)"queryId");
            }
        });
        return future;
    }

    public TableSchema getTableSchema(String table) {
        return this.getTableSchema(table, "default");
    }

    public TableSchema getTableSchema(String table, String database) {
        try (ClickHouseClient clientQuery = this.createClient();){
            ClickHouseRequest request = clientQuery.read(this.getServerNode());
            request.query("DESCRIBE TABLE " + table + " FORMAT " + ClickHouseFormat.TSKV.name());
            try {
                TableSchema tableSchema = new TableSchemaParser().createFromBinaryResponse((ClickHouseResponse)clientQuery.execute(request).get(), table, database);
                return tableSchema;
            }
            catch (Exception e) {
                throw new ClientException("Failed to get table schema", e);
            }
        }
    }

    private ClickHouseClient createClient() {
        ClickHouseConfig clientConfig = new ClickHouseConfig(new ClickHouseConfig[0]);
        ClickHouseClientBuilder clientV1 = ClickHouseClient.builder().config(clientConfig).nodeSelector(ClickHouseNodeSelector.of((ClickHouseProtocol)ClickHouseProtocol.HTTP, (ClickHouseProtocol[])new ClickHouseProtocol[0]));
        return clientV1.build();
    }

    private ClickHouseRequest.Mutation createMutationRequest(ClickHouseRequest.Mutation request, String tableName, InsertSettings settings) {
        if (settings == null) {
            return (ClickHouseRequest.Mutation)request.table(tableName);
        }
        if (settings.getSetting("query_id") != null) {
            request.table(tableName, settings.getSetting("query_id").toString());
        } else {
            request.table(tableName);
        }
        if (settings.getSetting("insert_deduplication_token") != null) {
            request.set("insert_deduplication_token", settings.getSetting("insert_deduplication_token").toString());
        }
        return request;
    }

    public static Set<String> getCompressAlgorithms() {
        return COMPRESS_ALGORITHMS;
    }

    public static Set<String> getOutputFormats() {
        return OUTPUT_FORMATS;
    }

    private static Set<String> createFormatWhitelist(String shouldSupport) {
        HashSet<String> formats = new HashSet<String>();
        boolean supportOutput = "output".equals(shouldSupport);
        boolean supportInput = "input".equals(shouldSupport);
        boolean supportBoth = "both".equals(shouldSupport);
        for (ClickHouseFormat format : ClickHouseFormat.values()) {
            if (!(supportOutput && format.supportsOutput() || supportInput && format.supportsInput()) && !supportBoth) continue;
            formats.add(format.name());
        }
        return Collections.unmodifiableSet(formats);
    }

    private String startOperation() {
        String operationId = UUID.randomUUID().toString();
        this.globalClientStats.put(operationId, new OperationStatistics.ClientStatistics());
        return operationId;
    }

    public static class Builder {
        private Set<String> endpoints = new HashSet<String>();
        private Map<String, String> configuration = new HashMap<String, String>();

        public Builder() {
            this.setConnectTimeout(30L, ChronoUnit.SECONDS).setSocketTimeout(2L, ChronoUnit.SECONDS).setSocketRcvbuf(804800L).setSocketSndbuf(804800L).enableCompression(true).enableDecompression(false);
        }

        public Builder addEndpoint(String endpoint) {
            this.endpoints.add(endpoint);
            return this;
        }

        public Builder addEndpoint(Protocol protocol, String host, int port) {
            String endpoint = String.format("%s://%s:%d", protocol.toString().toLowerCase(), host, port);
            this.addEndpoint(endpoint);
            return this;
        }

        public Builder addConfiguration(String key, String value) {
            this.configuration.put(key, value);
            return this;
        }

        public Builder addUsername(String username) {
            this.configuration.put("user", username);
            return this;
        }

        public Builder addPassword(String password) {
            this.configuration.put("password", password);
            return this;
        }

        public Builder addAccessToken(String accessToken) {
            this.configuration.put("access_token", accessToken);
            return this;
        }

        public Builder setConnectTimeout(long size) {
            this.configuration.put("connect_timeout", String.valueOf(size));
            return this;
        }

        public Builder setConnectTimeout(long amount, ChronoUnit unit) {
            this.setConnectTimeout(Duration.of(amount, unit).toMillis());
            return this;
        }

        public Builder setSocketTimeout(long size) {
            this.configuration.put("socket_timeout", String.valueOf(size));
            return this;
        }

        public Builder setSocketTimeout(long amount, ChronoUnit unit) {
            this.setSocketTimeout(Duration.of(amount, unit).toMillis());
            return this;
        }

        public Builder setSocketRcvbuf(long size) {
            this.configuration.put("socket_rcvbuf", String.valueOf(size));
            return this;
        }

        public Builder setSocketSndbuf(long size) {
            this.configuration.put("socket_sndbuf", String.valueOf(size));
            return this;
        }

        public Builder setSocketReuseaddr(boolean value) {
            this.configuration.put("socket_reuseaddr", String.valueOf(value));
            return this;
        }

        public Builder setSocketKeepalive(boolean value) {
            this.configuration.put("socket_keepalive", String.valueOf(value));
            return this;
        }

        public Builder setSocketTcpNodelay(boolean value) {
            this.configuration.put("socket_tcp_nodelay", String.valueOf(value));
            return this;
        }

        public Builder setSocketLinger(int secondsToWait) {
            this.configuration.put("socket_linger", String.valueOf(secondsToWait));
            return this;
        }

        public Builder enableCompression(boolean enabled) {
            this.configuration.put("compress", String.valueOf(enabled));
            return this;
        }

        public Builder enableDecompression(boolean enabled) {
            this.configuration.put("decompress", String.valueOf(enabled));
            return this;
        }

        public Client build() {
            if (this.endpoints.isEmpty()) {
                throw new IllegalArgumentException("At least one endpoint is required");
            }
            if (!(this.configuration.containsKey("access_token") || this.configuration.containsKey("user") && this.configuration.containsKey("password"))) {
                throw new IllegalArgumentException("Username and password are required");
            }
            return new Client(this.endpoints, this.configuration);
        }
    }
}

