/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

/**
 * This entity class holds information of the current state of the ping.
 * <p/>
 * The way the ping event is designed is: Maintain a ticker that ticks every
 * two seconds. When this timer matures (total of two minutes), then send the ping
 * event. If any changes are done to the session, this object is destroyed
 * and created again. The same logic goes on, until this object is set to null,
 * which is the case when the ping timer has matured and the event is fired.
 */
final class PingEntity {
    private int tickCount;

    PingEntity() {
        tickCount = 0;
    }

    public void doTick() {
        tickCount++;
    }

    int getTickCount() {
        return tickCount;
    }

    /**
     * Checks whether this ping timer has matured or now.
     *
     * @param seconds The delay in seconds to be qualified as matured
     * @return Whether the ping entity has matured or not
     */
    public boolean hasMatured(int seconds) {
        // Calculate the number of required ticks, and then check if we've crossed it
        int requiredTicks = (seconds * 1000) / Constants.PING_TICK_INTERVAL;
        return getTickCount() >= requiredTicks;
    }
}
