package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * User: Jude Pereira
 * Date: 07/07/2015
 * Time: 12:23
 */
final class LocalDataStore {
    private static final String eventNamespace = "local_events";
    private static final String PERSONALISATION_ENABLED = "personalisationEnabledBool";

    private static final HashMap<String, Object> PROFILE_FIELDS_IN_THIS_SESSION = new HashMap<String, Object>();

    private static boolean personalizationEnabled = false;

    private static final Object personalizationEnabledLock = new Object();

    static void setPersonalisationEnabled(final Context context, final boolean enable) {
        synchronized (personalizationEnabledLock) {
            personalizationEnabled = enable;
        }

        CleverTapAPI.postAsyncSafely("LocalDataStore#persistPersonalizationEnabled", new Runnable() {
            @Override
            public void run() {
                StorageHelper.putBoolean(context, PERSONALISATION_ENABLED, enable);
            }
        });
    }

    static boolean isPersonalisationEnabled(final Context context) {
        synchronized (personalizationEnabledLock) {
            return personalizationEnabled;
        }
    }

    /**
     * Whenever a profile field is updated, in the session, put it here.
     * The value must be an epoch until how long it is valid for (using existing TTL).
     * <p/>
     * When upstream updates come in, check whether or not to update the field.
     */
    private static final HashMap<String, Integer> PROFILE_EXPIRY_MAP = new HashMap<String, Integer>();

    // our sqlite db wrapper
    private static DBAdapter dbAdapter;

    /**
     * CleverTapAPI calls this in its singleton constructor.
     */
    static void initializeWithContext(Context context) {
        dbAdapter = new DBAdapter(context);
        inflateLocalProfileAsync(context);
    }

    static void changeUser(final Context context) {
        resetLocalProfileSync(context);
    }

    static void setDataSyncFlag(Context context, JSONObject event) {
        try {
            // Check the personalisation flag
            boolean enablePersonalisation = StorageHelper.getBoolean(context, PERSONALISATION_ENABLED, false);
            if (!enablePersonalisation) {
                event.put("dsync", false);
                return;
            }

            // Always request a dsync when the App Launched event is recorded
            final String eventType = event.getString("type");
            if ("event".equals(eventType)) {
                final String evtName = event.getString("evtName");
                if (Constants.APP_LAUNCHED_EVENT.equals(evtName)) {
                    Logger.logFine("Local cache needs to be updated (triggered by App Launched)");
                    event.put("dsync", true);
                    return;
                }
            }

            // If a profile event, then blindly set it to true
            if ("profile".equals(eventType)) {
                event.put("dsync", true);
                Logger.logFine("Local cache needs to be updated (profile event)");
                return;
            }

            // Default to expire in 20 minutes
            final int now = (int) (System.currentTimeMillis() / 1000);

            int expiresIn = getLocalCacheExpiryInterval(context, 20 * 60);

            int lastUpdate = StorageHelper.getInt(context, "local_cache_last_update", now);

            if (lastUpdate + expiresIn < now) {
                event.put("dsync", true);
                Logger.logFine("Local cache needs to be updated");
            } else {
                event.put("dsync", false);
                Logger.logFine("Local cache doesn't need to be updated");
            }
        } catch (Throwable t) {
            Logger.error("Failed to sync with upstream", t);
        }
    }

    static void persistEvent(Context context, JSONObject event, int type) {

        if (event == null) return;

        try {
            if (type == Constants.RAISED_EVENT) {
                persistEvent(context, event);
            }
        } catch (Throwable t) {
            Logger.error("Failed to sync with upstream", t);
        }
    }

    static void syncWithUpstream(Context context, JSONObject response) {
        try {
            JSONObject eventUpdates = null;
            JSONObject profileUpdates = null;

            if (!response.has("evpr")) return;

            JSONObject evpr = response.getJSONObject("evpr");
            if (evpr.has("profile")) {
                JSONObject profile = evpr.getJSONObject("profile");
                if (profile.has("_custom")) {
                    JSONObject custom = profile.getJSONObject("_custom");
                    profile.remove("_custom");
                    Iterator keys = custom.keys();
                    while (keys.hasNext()) {
                        String next = keys.next().toString();

                        Object value = null;
                        try {
                            value = custom.getJSONArray(next);
                        } catch (Throwable t) {
                            try {
                                value = custom.get(next);
                            } catch (JSONException e) {
                                //no-op
                            }
                        }

                        if (value != null) {
                            profile.put(next, value);
                        }
                    }
                }

                profileUpdates = syncProfile(context, profile);
            }

            if (evpr.has("events")) {
                eventUpdates = syncEventsFromUpstream(context, evpr.getJSONObject("events"));
            }

            if (evpr.has("expires_in")) {
                int expiresIn = evpr.getInt("expires_in");
                setLocalCacheExpiryInterval(context, expiresIn);
            }

            StorageHelper.putInt(context, "local_cache_last_update", (int) (System.currentTimeMillis() / 1000));

            Boolean profileUpdatesNotEmpty = (profileUpdates != null && profileUpdates.length() > 0);
            Boolean eventsUpdatesNotEmpty = (eventUpdates != null && eventUpdates.length() > 0);
            if (profileUpdatesNotEmpty || eventsUpdatesNotEmpty) {
                JSONObject updates = new JSONObject();

                if (profileUpdatesNotEmpty) {
                    updates.put("profile", profileUpdates);
                }

                if (eventsUpdatesNotEmpty) {
                    updates.put("events", eventUpdates);
                }

                SyncListener syncListener = CleverTapAPI.getInstance(context).getSyncListener();
                if (syncListener != null) {
                    try {
                        syncListener.profileDataUpdated(updates);
                    } catch (Throwable t) {
                        Logger.error("Execution of sync listener failed", t);
                    }
                }
            }
        } catch (Throwable t) {
            Logger.error("Failed to sync with upstream", t);
        }
    }

    @SuppressLint("CommitPrefEdits")
    private static JSONObject syncEventsFromUpstream(Context context, JSONObject events) {
        try {
            JSONObject eventUpdates = null;
            SharedPreferences prefs = StorageHelper.getPreferences(context, eventNamespace);
            Iterator keys = events.keys();
            SharedPreferences.Editor editor = prefs.edit();
            while (keys.hasNext()) {
                String event = keys.next().toString();
                String encoded = prefs.getString(event, encodeEventDetails(0, 0, 0));

                EventDetail ed = decodeEventDetails(event, encoded);

                JSONArray upstream = events.getJSONArray(event);
                if (upstream == null || upstream.length() < 3) {
                    Logger.error("Corrupted upstream event detail");
                    continue;
                }

                int upstreamCount, first, last;
                try {
                    upstreamCount = upstream.getInt(0);
                    first = upstream.getInt(1);
                    last = upstream.getInt(2);
                } catch (Throwable t) {
                    Logger.error("Failed to parse upstream event message: " + upstream.toString());
                    continue;
                }

                if (upstreamCount > ed.getCount()) {
                    editor.putString(event, encodeEventDetails(first, last, upstreamCount));
                    Logger.logFine("Accepted update for event " + event + " from upstream");

                    try {
                        if (eventUpdates == null) {
                            eventUpdates = new JSONObject();
                        }

                        JSONObject evUpdate = new JSONObject();

                        JSONObject countUpdate = new JSONObject();
                        countUpdate.put("oldValue", ed.getCount());
                        countUpdate.put("newValue", upstreamCount);
                        evUpdate.put("count", countUpdate);

                        JSONObject firstUpdate = new JSONObject();
                        firstUpdate.put("oldValue", ed.getFirstTime());
                        firstUpdate.put("newValue", upstream.getInt(1));
                        evUpdate.put("firstTime", firstUpdate);

                        JSONObject lastUpdate = new JSONObject();
                        lastUpdate.put("oldValue", ed.getLastTime());
                        lastUpdate.put("newValue", upstream.getInt(2));
                        evUpdate.put("lastTime", lastUpdate);

                        eventUpdates.put(event, evUpdate);

                    } catch (Throwable t) {
                        Logger.error("Couldn't set event updates", t);
                    }

                } else {
                    Logger.logFine("Rejected update for event " + event + " from upstream");
                }
            }
            StorageHelper.persist(editor);
            return eventUpdates;
        } catch (Throwable t) {
            Logger.error("Couldn't sync events from upstream", t);
            return null;
        }
    }

    @SuppressLint("CommitPrefEdits")
    private static JSONObject syncProfile(Context context, JSONObject remoteProfile) {

        // Will hold the changes to be returned
        JSONObject profileUpdates = new JSONObject();

        if (remoteProfile == null || remoteProfile.length() <= 0) return profileUpdates;

        try {

            // will hold the updated fields that need to be written to the local profile
            JSONObject fieldsToUpdateLocally = new JSONObject();

            // cache the current time for shouldPreferLocalUpdateForKey check
            final int now = (int) (System.currentTimeMillis() / 1000);

            // walk the remote profile and compare values against the local profile values
            // prefer the remote profile value unless we have set a still-valid expiration time for the local profile value
            final Iterator keys = remoteProfile.keys();

            while (keys.hasNext()) {
                try {

                    String key = keys.next().toString();

                    if (shouldPreferLocalProfileUpdateForKeyForTime(key, now)) {
                        // We shouldn't accept the upstream value, as our map
                        // forces us to use the local
                        Logger.logFine("Rejecting upstream value for key " + key + " " +
                                "because our local cache prohibits it");
                        continue;
                    }

                    Object localValue = getProfileValueForKey(key);

                    Object remoteValue = remoteProfile.get(key);

                    // if remoteValue is empty (empty string or array) treat it as removed, so null it out here
                    // all later tests handle null values
                    if (profileValueIsEmpty(remoteValue)) {
                        remoteValue = null;
                    }

                    // handles null values
                    if (!profileValuesAreEqual(remoteValue, localValue)) {
                        try {
                            // Update required as we prefer the remote value once we've passed the local expiration time check

                            // add the new value to be written to the local profile
                            // if empty send a remove message
                            if (remoteValue != null) {
                                fieldsToUpdateLocally.put(key, remoteValue);
                            } else {
                                removeProfileField(context, key, true);
                            }

                            // add the changed values to the dictionary to be returned
                            // handles null values
                            JSONObject changesObject = buildChangeFromOldValueToNewValue(localValue, remoteValue);
                            if (changesObject != null) {
                                profileUpdates.put(key, changesObject);
                            }

                        } catch (Throwable t) {
                            Logger.logFine("Failed to set profile updates", t);
                        }
                    }

                } catch (Throwable t) {
                    Logger.logFine("Failed to update profile field", t);
                }
            }

            // save the changed fields locally
            if (fieldsToUpdateLocally.length() > 0) {
                setProfileFields(context, fieldsToUpdateLocally, true);
            }

            return profileUpdates;

        } catch (Throwable t) {
            Logger.error("Failed to sync remote profile", t);
            return null;
        }
    }

    @SuppressLint("CommitPrefEdits")
    private static void persistEvent(Context context, JSONObject event) {
        try {
            String evtName = event.getString("evtName");
            if (evtName == null) return;

            SharedPreferences prefs = StorageHelper.getPreferences(context, eventNamespace);

            int now = (int) (System.currentTimeMillis() / 1000);

            String encoded = prefs.getString(evtName, encodeEventDetails(now, now, 0));
            EventDetail ed = decodeEventDetails(evtName, encoded);

            String updateEncoded = encodeEventDetails(ed.getFirstTime(), now, ed.getCount() + 1);
            SharedPreferences.Editor editor = prefs.edit();
            editor.putString(evtName, updateEncoded);
            StorageHelper.persist(editor);
        } catch (Throwable t) {
            Logger.error("Failed to persist event locally", t);
        }
    }

    private static String encodeEventDetails(int first, int last, int count) {
        return count + "|" + first + "|" + last;
    }

    private static EventDetail decodeEventDetails(String name, String encoded) {
        if (encoded == null) return null;

        String[] parts = encoded.split("\\|");
        return new EventDetail(Integer.parseInt(parts[0]), Integer.parseInt(parts[1]),
                Integer.parseInt(parts[2]), name);
    }

    static Map<String, EventDetail> getEventHistory(Context context) {
        try {
            SharedPreferences prefs = StorageHelper.getPreferences(context, eventNamespace);
            Map<String, ?> all = prefs.getAll();
            Map<String, EventDetail> out = new HashMap<String, EventDetail>();
            for (String eventName : all.keySet()) {
                out.put(eventName, decodeEventDetails(eventName, all.get(eventName).toString()));
            }
            return out;
        } catch (Throwable t) {
            Logger.error("Failed to retrieve local event history", t);
            return null;
        }
    }

    static EventDetail getEventDetail(Context context, String eventName) {
        try {
            if (!isPersonalisationEnabled(context)) return null;

            SharedPreferences prefs = StorageHelper.getPreferences(context, eventNamespace);
            return decodeEventDetails(eventName, prefs.getString(eventName, null));
        } catch (Throwable t) {
            Logger.error("Failed to retrieve local event detail", t);
            return null;
        }
    }

    // profile getters and setters
    // note only use for external purposes
    static Object getProfileProperty(Context context, String key) {
        return getProfileValueForKey(key);
    }

    // use for internal purposes, skips the personalization check
    static Object getProfileValueForKey(String key) {
        return _getProfileProperty(key);
    }

    private static Object _getProfileProperty(String key) {

        if (key == null) return null;

        synchronized (PROFILE_FIELDS_IN_THIS_SESSION) {
            try {
                return PROFILE_FIELDS_IN_THIS_SESSION.get(key);

            } catch (Throwable t) {
                Logger.error("Failed to retrieve local profile property", t);
                return null;
            }
        }
    }

    static void setProfileFields(Context context, JSONObject fields) {
        setProfileFields(context, fields, false);
    }

    static void setProfileFields(Context context, JSONObject fields, Boolean fromUpstream) {
        if (fields == null) return;

        try {
            final Iterator keys = fields.keys();

            while (keys.hasNext()) {
                String key = keys.next().toString();
                setProfileField(context, key, fields.get(key), fromUpstream);
            }

        } catch (Throwable t) {
            Logger.error("Failed to set profile fields", t);
        }
    }

    static void setProfileField(Context context, String key, Object value) {
        setProfileField(context, key, value, false);
    }

    static void setProfileField(Context context, String key, Object value, Boolean fromUpstream) {
        if (key == null || value == null) return;

        try {
            _setProfileField(key, value);

            if (!fromUpstream) {
                updateLocalProfileKeyExpiryTime(context, key);
            }
        } catch (Throwable t) {
            // no-op
        }

        persistLocalProfileAsync(context);
    }

    private static void _setProfileField(String key, Object value) {
        if (key == null || value == null) return;

        synchronized (PROFILE_FIELDS_IN_THIS_SESSION) {
            PROFILE_FIELDS_IN_THIS_SESSION.put(key, value);
        }
    }

    static void removeProfileFields(Context context, ArrayList<String> fields) {
        if (fields == null) return;
        removeProfileFields(context, fields, false);
    }

    static void removeProfileFields(Context context, ArrayList<String> fields, Boolean fromUpstream) {
        if (fields == null) return;

        for (String key : fields) {
            removeProfileField(context, key, fromUpstream);
        }
    }

    static void removeProfileField(Context context, String key) {
        removeProfileField(context, key, false);
    }

    static void removeProfileField(Context context, String key, Boolean fromUpstream) {

        if (key == null) return;

        try {
            _removeProfileField(key);

            // even though its a remove add an expiration time for the local key, as we still need it in the sync
            if (!fromUpstream) {
                updateLocalProfileKeyExpiryTime(context, key);
            }

        } catch (Throwable t) {
            // no-op
        }

        persistLocalProfileAsync(context);
    }

    private static void _removeProfileField(String key) {

        if (key == null) return;

        synchronized (PROFILE_FIELDS_IN_THIS_SESSION) {
            try {
                PROFILE_FIELDS_IN_THIS_SESSION.remove(key);

            } catch (Throwable t) {
                Logger.error("Failed to remove local profile value for key " + key, t);
            }
        }
    }

    // local profile persistence operations

    private static String getUserProfileID(final Context context) {
        return CleverTapAPI.getAccountID(context);
    }

    private static void resetLocalProfileSync(final Context context) {

        synchronized (PROFILE_EXPIRY_MAP) {
            PROFILE_EXPIRY_MAP.clear();
        }

        synchronized (PROFILE_FIELDS_IN_THIS_SESSION) {
            PROFILE_FIELDS_IN_THIS_SESSION.clear();
        }

        final String accountID = getUserProfileID(context);
        dbAdapter.removeUserProfile(accountID);
    }

    private static void inflateLocalProfileAsync(final Context context) {

        final String accountID = getUserProfileID(context);

        CleverTapAPI.postAsyncSafely("LocalDataStore#inflateLocalProfileAsync", new Runnable() {
            @Override
            public void run() {
                synchronized (PROFILE_FIELDS_IN_THIS_SESSION) {
                    try {
                        JSONObject profile = dbAdapter.fetchUserProfileById(accountID);

                        if (profile == null) return;

                        Iterator<?> keys = profile.keys();
                        while (keys.hasNext()) {
                            try {
                                String key = (String) keys.next();
                                Object value = profile.get(key);
                                if (value instanceof JSONObject) {
                                    JSONObject jsonObject = profile.getJSONObject(key);
                                    PROFILE_FIELDS_IN_THIS_SESSION.put(key, jsonObject);
                                } else if (value instanceof JSONArray) {
                                    JSONArray jsonArray = profile.getJSONArray(key);
                                    PROFILE_FIELDS_IN_THIS_SESSION.put(key, jsonArray);
                                } else {
                                    PROFILE_FIELDS_IN_THIS_SESSION.put(key, value);
                                }
                            } catch (JSONException e) {
                                // no-op
                            }
                        }

                        Logger.logFine("Inflated local profile " + PROFILE_FIELDS_IN_THIS_SESSION.toString());

                    } catch (Throwable t) {
                        //no-op
                    }
                }

                synchronized (personalizationEnabledLock) {
                    personalizationEnabled = StorageHelper.getBoolean(context, PERSONALISATION_ENABLED, false);
                }
            }
        });

    }

    private static void persistLocalProfileAsync(final Context context) {

        final String profileID = getUserProfileID(context);

        CleverTapAPI.postAsyncSafely("LocalDataStore#persistLocalProfileAsync", new Runnable() {
            @Override
            public void run() {
                synchronized (PROFILE_FIELDS_IN_THIS_SESSION) {
                    long status = dbAdapter.storeUserProfile(profileID, new JSONObject(PROFILE_FIELDS_IN_THIS_SESSION));
                    Logger.logFine("Persist Local Profile complete with status " + status + " for id " + profileID);
                }
            }
        });
    }

    // local cache/profile key expiry handling

    private static void updateLocalProfileKeyExpiryTime(Context context, String key) {
        if (key == null) return;

        synchronized (PROFILE_EXPIRY_MAP) {
            PROFILE_EXPIRY_MAP.put(key, calculateLocalKeyExpiryTime(context));
        }
    }

    private static void removeLocalProfileKeyExpiryTime(String key) {
        if (key == null) return;

        synchronized (PROFILE_EXPIRY_MAP) {
            PROFILE_EXPIRY_MAP.remove(key);
        }
    }

    private static Integer getLocalProfileKeyExpiryTimeForKey(String key) {
        if (key == null) return 0;

        synchronized (PROFILE_EXPIRY_MAP) {
            return PROFILE_EXPIRY_MAP.get(key);
        }
    }

    private static int getLocalCacheExpiryInterval(final Context context, int defaultInterval) {
        return StorageHelper.getInt(context, "local_cache_expires_in", defaultInterval);
    }

    private static void setLocalCacheExpiryInterval(final Context context, final int ttl) {
        StorageHelper.putInt(context, "local_cache_expires_in", ttl);
    }

    private static int calculateLocalKeyExpiryTime(Context context) {
        final int now = (int) (System.currentTimeMillis() / 1000);
        return (now + getLocalCacheExpiryInterval(context, 0));
    }

    // checks whether we have a local update expiration time and its greater than specified time (or defaults to the current time)
    // if so we prefer the local update to a remote update of the specified key

    private static Boolean shouldPreferLocalProfileUpdateForKeyForTime(String key, int time) {
        final int now = (time <= 0) ? (int) (System.currentTimeMillis() / 1000) : time;
        Integer keyValidUntil = getLocalProfileKeyExpiryTimeForKey(key);
        return (keyValidUntil != null && keyValidUntil > now);
    }

    // helpers

    private static JSONObject buildChangeFromOldValueToNewValue(Object oldValue, Object newValue) {

        if (oldValue == null && newValue == null) return null;

        JSONObject keyUpdates = new JSONObject();

        try {
            // if newValue is null means its been removed, represent that as -1
            Object _newVal = (newValue != null) ? newValue : -1;
            keyUpdates.put("newValue", _newVal);

            if (oldValue != null) {
                keyUpdates.put("oldValue", oldValue);
            }

        } catch (Throwable t) {
            Logger.logFine("Failed to create profile changed values object", t);
            return null;
        }

        return keyUpdates;
    }

    private static Boolean profileValueIsEmpty(Object value) {

        if (value == null) return true;

        Boolean isEmpty = false;

        if (value instanceof String) {
            isEmpty = ((String) value).trim().length() == 0;
        }

        if (value instanceof JSONArray) {
            isEmpty = ((JSONArray) value).length() <= 0;
        }

        return isEmpty;
    }

    private static Boolean profileValuesAreEqual(Object value1, Object value2) {
        // convert to strings and compare
        // stringify handles null values
        return stringify(value1).equals(stringify(value2));
    }

    private static String stringify(Object value) {
        return (value == null) ? "" : value.toString();
    }

}
