/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.Uri;
import android.util.Base64;

import com.clevertap.android.sdk.exceptions.CleverTapMetaDataNotFoundException;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import javax.net.ssl.HttpsURLConnection;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.Iterator;
import java.util.Map;

/**
 * Provides various methods to communicate with the CleverTap servers.
 */
final class CommsManager {
    static final String NAMESPACE_ARP = "ARP";

    static void flushQueueAsync(final Context context) {
        CleverTapAPI.postAsyncSafely("CommsManager#flushQueueAsync", new Runnable() {
            @Override
            public void run() {
                flushQueueSync(context);
            }
        });
    }

    static void flushQueueSync(final Context context) {
        flushDBQueue(context);
    }

    private static void flushDBQueue(final Context context) {

        Logger.logFine("Somebody has invoked me to send the queue to CleverTap servers");

        if (!canSendQueue(context)) {
            Logger.logFine("Unable to send queue, bailing");
            return;
        }

        QueueManager.QueueCursor cursor;
        QueueManager.QueueCursor previousCursor = null;
        boolean loadMore = true;
        while (loadMore) {
            cursor = QueueManager.getQueuedEvents(context, 50, previousCursor);

            if (cursor == null || cursor.isEmpty()) {
                Logger.logFine("No events in the queue, bailing");
                break;
            }

            previousCursor = cursor;
            JSONArray queue = cursor.getData();

            if (queue == null || queue.length() <= 0) {
                Logger.logFine("No events in the queue, bailing");
                break;
            }

            loadMore = (sendQueue(context, queue) == 200);
        }
    }

    private static int sendQueue(final Context context, final JSONArray queue) {

        int status = -1;

        if (queue == null || queue.length() <= 0) return status;

        HttpsURLConnection conn = null;
        try {
            URL url = new URL(getEndpoint(context));
            conn = (HttpsURLConnection) url.openConnection();
            conn.setConnectTimeout(10000);
            conn.setReadTimeout(10000);
            conn.setRequestProperty("Content-Type", "application/json; charset=utf-8");
            conn.setInstanceFollowRedirects(false);

            final String body;

            synchronized (CommsManager.class) {

                final String req = insertHeader(context, queue);
                Logger.logFine("Send queue contains " + queue.length() + " items: " + req);
                conn.setDoOutput(true);
                conn.getOutputStream().write(req.getBytes("UTF-8"));

                final int responseCode = conn.getResponseCode();
                status = responseCode;

                // Always check for a 200 OK
                if (responseCode != 200) {
                    throw new IOException("Response code is not 200. It is " + responseCode);
                }

                BufferedReader br = new BufferedReader(new InputStreamReader(conn.getInputStream(), "utf-8"));

                StringBuilder sb = new StringBuilder();
                String line;
                while ((line = br.readLine()) != null) {
                    sb.append(line);
                }
                body = sb.toString();
                processResponse(context, body);
                Logger.logFine("Completed successfully");
            }
        } catch (Throwable e) {
            Logger.logFine("An exception occurred while trying to send the queue", e);

        } finally {
            if (conn != null) {
                try {
                    conn.disconnect();
                } catch (Throwable t) {
                    // Ignore
                }
            }
        }
        return status;
    }

    private static boolean canSendQueue(final Context context) {
        return isOnline(context);
    }

    private static String insertHeader(Context context, JSONArray arr) {
        try {
            // Insert our header at the first position
            final JSONObject header = new JSONObject();
            final CleverTapAPI api = CleverTapAPI.getInstance(context);

            String deviceId = api.getCleverTapID();
            if (deviceId != null && !deviceId.equals(""))  {
                header.put("g", deviceId);
            } else {
                Logger.error("CRITICAL: Couldn't finalise on a device ID!");
            }

            header.put("type", "meta");

            JSONObject appFields = api.getAppLaunchedFields();
            header.put("af", appFields);



            String accountId = null, token = null;
            try {
                accountId = ManifestMetaData.getMetaData(context, Constants.LABEL_ACCOUNT_ID);
                token = ManifestMetaData.getMetaData(context, Constants.LABEL_TOKEN);
            } catch (CleverTapMetaDataNotFoundException e) {
                // No account ID/token, be gone now!
                Logger.logFine("Account ID/token not found, will not add to queue header");
            }

            if (accountId != null) {
                header.put("id", accountId);
            }

            if (token != null) {
                header.put("tk", token);
            }

            InAppFCManager.attachToHeader(context, header);

            // Resort to string concat for backward compatibility
            return "[" + header.toString() + ", " + arr.toString().substring(1);
        } catch (Throwable t) {
            Logger.error("CommsManager: Failed to attach header", t);
            return arr.toString();
        }
    }

    private static boolean isOnline(Context context) {
        try {
            ConnectivityManager cm =
                    (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo netInfo = cm.getActiveNetworkInfo();
            return netInfo != null && netInfo.isConnected();
        } catch (Throwable ignore) {
            // Can't decide whether or not the net is available, maybe it just is
            // Therefore, let's wait for a timeout or something to create an IOException later
            return true;
        }
    }

    private static void processResponse(final Context context, final String responseStr) {
        if (responseStr == null) return;

        try {
            Logger.logFine("Trying to process response: " + responseStr);
            JSONObject response = new JSONObject(responseStr);
            try {
                if (Constants.ENABLE_INAPP) {
                    InAppManager.processResponseAsync(response, context);
                }
            } catch (Throwable t) {
                Logger.error("Failed to process in-app notifications from the response!", t);
            }

            // Always look for a GUID in the response, and if present, then perform a force update
            try {
                if (response.has("g")) {
                    final String deviceID = response.getString("g");
                    DeviceInfo.forceUpdateDeviceId(deviceID);
                    Logger.logFine("Got a new device ID: " + deviceID);
                }
            } catch (Throwable t) {
                Logger.error("Failed to update device ID!", t);
            }

            try {
                LocalDataStore.syncWithUpstream(context, response);
            } catch (Throwable t) {
                Logger.error("Failed to sync local cache with upstream", t);
            }

            // Handle "arp" (additional request parameters)
            try {
                if (response.has("arp")) {
                    final JSONObject arp = (JSONObject) response.get("arp");
                    if (arp.length() > 0) {
                        handleARPUpdate(context, arp);
                    }
                }
            } catch (Throwable t) {
                Logger.logFine("Failed to process ARP", t);
            }

            // Handle "console" - print them as info to the console
            try {
                if (response.has("console")) {
                    final JSONArray console = (JSONArray) response.get("console");
                    if (console.length() > 0) {
                        for (int i = 0; i < console.length(); i++) {
                            Logger.log(console.get(i).toString());
                        }
                    }
                }
            } catch (Throwable t) {
                // Ignore
            }

            // Handle server set debug level
            try {
                if (response.has("dbg_lvl")) {
                    final int debugLevel = response.getInt("dbg_lvl");
                    if (debugLevel >= 0) {
                        CleverTapAPI.setDebugLevel(debugLevel);
                        Logger.logFine("Set debug level to " + debugLevel + " for this session (set by upstream)");
                    }
                }
            } catch (Throwable t) {
                // Ignore
            }

            // Handle stale_inapp
            try {
                InAppFCManager.processResponse(context, response);
            } catch (Throwable t) {
                // Ignore
            }
        } catch (Throwable t) {
            Logger.error("Failed to send events to CleverTap", t);
        }
    }

    private static String getEndpoint(Context context) {
        String endpoint = "https://wzrkt.com/a1?os=Android&t=" + BuildInfo.SDK_SVN_REVISION;
        try {
            endpoint += "&z=" + ManifestMetaData.getMetaData(context, Constants.LABEL_ACCOUNT_ID);
        } catch (CleverTapMetaDataNotFoundException ignore) {
            // Ignore
        }

        // Attach ARP as B64 encoded JSON object
        try {
            final JSONObject arp = getARP(context);
            if (arp != null && arp.length() > 0) {
                final String encoded = Uri.encode(Base64.encodeToString(arp.toString().getBytes("UTF-8"), 0));
                if (encoded != null && encoded.length() > 1) {
                    endpoint += "&arp=" + encoded;
                }
            }
        } catch (Throwable t) {
            Logger.error("Failed to attach ARP", t);
        }

        endpoint += "&ts=" + (int) (System.currentTimeMillis() / 1000);

        Logger.logFine("Using endpoint " + endpoint);
        return endpoint;
    }

    /**
     * The ARP is additional request parameters, which must be sent once
     * received after any HTTP call. This is sort of a proxy for cookies.
     *
     * @return A JSON object containing the ARP key/values. Can be null.
     */
    private static JSONObject getARP(final Context context) {
        try {
            final SharedPreferences prefs = StorageHelper.getPreferences(context, NAMESPACE_ARP);
            final Map<String, ?> all = prefs.getAll();
            final Iterator<? extends Map.Entry<String, ?>> iter = all.entrySet().iterator();

            while (iter.hasNext()) {
                final Map.Entry<String, ?> kv = iter.next();
                final Object o = kv.getValue();
                if (o instanceof Number && ((Number) o).intValue() == -1) {
                    iter.remove();
                }
            }

            return new JSONObject(all);
        } catch (Throwable t) {
            Logger.logFine("Failed to construct ARP object", t);
            return null;
        }
    }

    @SuppressLint("CommitPrefEdits")
    static void clearARP(final Context context) {
        final SharedPreferences prefs = StorageHelper.getPreferences(context, NAMESPACE_ARP);
        final SharedPreferences.Editor editor = prefs.edit();
        editor.clear();
        StorageHelper.persist(editor);
    }

    @SuppressLint("CommitPrefEdits")
    private static void handleARPUpdate(final Context context, final JSONObject arp) {
        if (arp == null || arp.length() == 0) return;

        final SharedPreferences prefs = StorageHelper.getPreferences(context, NAMESPACE_ARP);
        final SharedPreferences.Editor editor = prefs.edit();

        final Iterator<String> keys = arp.keys();
        while (keys.hasNext()) {
            final String key = keys.next();
            try {
                final Object o = arp.get(key);
                if (o instanceof Number) {
                    final int update = ((Number) o).intValue();
                    editor.putInt(key, update);
                } else if (o instanceof String) {
                    if (((String) o).length() < 100) {
                        editor.putString(key, (String) o);
                    } else {
                        Logger.logFine("ARP update for key " + key + " rejected (string value too long)");
                    }
                } else if (o instanceof Boolean) {
                    editor.putBoolean(key, (Boolean) o);
                } else {
                    Logger.logFine("ARP update for key " + key + " rejected (invalid data type)");
                }
            } catch (JSONException e) {
                // Ignore
            }
        }
        StorageHelper.persist(editor);
    }
}
