/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.support.v4.content.ContextCompat;
import com.clevertap.android.sdk.exceptions.CleverTapException;
import com.clevertap.android.sdk.exceptions.CleverTapPermissionsNotSatisfied;

import java.util.UUID;

import static com.clevertap.android.sdk.Logger.logExtraFine;

/**
 * Provides various methods to access the device.
 */
final class DeviceInfo {
    private static String tmpProfileID = null;

    private static final Object deviceIDLock = new Object();

    private static boolean notifiedAboutPermanentGUID = false;

    static String getDeviceID(final Context context, boolean autoGenerateIfRequired, boolean makeAutoGeneratedGUIDPermanent) {
        synchronized (deviceIDLock) {
            String trueID = StorageHelper.getString(context, Constants.DEVICE_ID_TAG, null);

            if (autoGenerateIfRequired && (trueID == null || trueID.trim().equals(""))) {
                if (tmpProfileID == null) {
                    // Generate one and keep
                    tmpProfileID = UUID.randomUUID().toString().replace("-", "");

                    // Append the Android automatic generated GUID
                    tmpProfileID = "__" + tmpProfileID;
                }

                trueID = tmpProfileID;

                if (makeAutoGeneratedGUIDPermanent) {
                    forceUpdateDeviceId(context, trueID);
                }
            }

            if (!notifiedAboutPermanentGUID && makeAutoGeneratedGUIDPermanent) {
                notifiedAboutPermanentGUID = true;
                final String finalTrueID = trueID;
                CleverTapAPI.postAsyncSafely("DeviceInfo#getDeviceID-notify-listener", new Runnable() {
                    @Override
                    public void run() {
                        final SyncListener sl;
                        try {
                            sl = CleverTapAPI.getInstance(context).getSyncListener();
                            if (sl != null) {
                                sl.profileDidInitialize(finalTrueID);
                            }
                        } catch (CleverTapException e) {
                            // Ignore
                        }
                    }
                });
            }

            return trueID;
        }
    }

    /**
     * Updates the device ID, if required.
     * <p>
     * This is used internally by the SDK, there is no need to call this explicitly.
     * </p>
     *
     * @param context The Android {@link Context}
     */
    @SuppressLint("CommitPrefEdits")
    static void updateDeviceIdIfRequired(Context context) {
        String deviceID = DeviceInfo.getDeviceID(context, false, false);

        // If the device ID is already present, simply return
        if (deviceID != null) {
            return;
        }

        String foundDeviceID = null;

        // Note: The reason this works is because the async task to retrieve the ad ID
        // has been executed just before this task, as we have a serial executor
        final String adID = CleverTapAPI.getGoogleAdID();
        if (adID != null && adID.trim().length() > 2) {
            foundDeviceID = Constants.GUID_PREFIX_GOOGLE_AD_ID + adID;
        }

        if (foundDeviceID != null) {
            synchronized (deviceIDLock) {
                String currentGUID = getDeviceID(context, false, false);
                if (currentGUID != null && currentGUID.trim().length() > 2) {
                    Logger.error("The SDK has found an existing device ID, but wasn't able to use it as one has already been set!");
                    return;
                }
                forceUpdateDeviceId(context, foundDeviceID);
                logExtraFine("Retrieved a GUID for this installation - " + foundDeviceID);
            }
        }
    }

    /**
     * Force updates the device ID, with the ID specified.
     * <p>
     * This is used internally by the SDK, there is no need to call this explicitly.
     * </p>
     *
     * @param context The Android {@link Context}
     * @param id      The new device ID
     */
    @SuppressLint("CommitPrefEdits")
    static void forceUpdateDeviceId(Context context, String id) {
        Logger.logExtraFine("Force updating the device ID to " + id);
        synchronized (deviceIDLock) {
            StorageHelper.putString(context, Constants.DEVICE_ID_TAG, id);
        }
    }

    /**
     * Tests whether a particular permission is available or not.
     *
     * @param context    The Android {@link Context}
     * @param permission The fully qualified Android permission name
     * @throws CleverTapPermissionsNotSatisfied
     */
    static void testPermission(Context context, String permission)
            throws CleverTapPermissionsNotSatisfied {
        if (!hasPermission(context, permission))
            throw new CleverTapPermissionsNotSatisfied("Permission required: " + permission);
    }

    @SuppressWarnings("WeakerAccess")
    static boolean hasPermission(Context context, String permission) {
        try {
            return PackageManager.PERMISSION_GRANTED == ContextCompat.checkSelfPermission(context, permission);
        } catch (Throwable t) {
            return false;
        }
    }

    /**
     * Returns the integer identifier for the default app icon.
     *
     * @param context The Android context
     * @return The integer identifier for the image resource
     */
    static int getAppIconAsIntId(Context context) {
        ApplicationInfo ai = context.getApplicationInfo();
        return ai.icon;
    }
}
