/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Bundle;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

/**
 * Manages in-app notifications.
 */
@SuppressLint("CommitPrefEdits")
public final class InAppManager {
    private InAppManager() {
        // No instantiation
    }

    private static void createAndShow(Context context, JSONObject inapp) {
        final Bundle b = getBundleFromJsonObject(inapp);
        if (!CleverTapAPI.isAppForeground()) {
            Logger.error("Cannot show in-app notification. App isn't in foreground. " +
                    "Please check your integration if your app is actually in foreground.");
            return;
        }

        if (!InAppFCManager.canShow(context, inapp)) {
            Logger.logFine("InAppFCManager rejected an inapp");
            return;
        }

        InAppFCManager.didShow(context, inapp);

        Logger.logFine("Will show new notification shortly: " + b.toString());
        if (!validateNotifBundle(b)) {
            // Discard - the format wasn't as expected

            Logger.logFine("Notification was not formatted correctly. Discarding");
            return;
        }

        Intent i = new Intent(context, InAppNotificationActivity.class);
        i.putExtras(b);
        // Start our transparent activity
        try {
            if (CleverTapAPI.currentActivity == null) {
                throw new IllegalStateException("Current activity reference not found");
            }

            CleverTapAPI.currentActivity.startActivity(i);
        } catch (Throwable t) {
            Logger.error("InAppManager: Please verify the integration of your app." +
                    " It is not setup to support in-app notifications yet.", t);
        }
    }

    static void processResponseAsync(final JSONObject response, final Context context) {
        CleverTapAPI.postAsyncSafely("InAppManager#processResponseAsync", new Runnable() {
            @Override
            public void run() {
                try {
                    processResponse(response, context);
                } catch (Throwable t) {
                    Logger.error("Failed to process response!", t);
                }
            }
        });
    }

    private static void processResponse(final JSONObject response, final Context context) {
        try {
            Logger.logFine("InAppManager: Processing response");

            if (!response.has("inapp_notifs")) {
                Logger.logFine("InAppManager: Response JSON object doesn't contain the inapp key, bailing");
                return;
            }

            int perSession = 10;
            int perDay = 10;
            if (response.has(Constants.INAPP_MAX_PER_SESSION) && response.get(Constants.INAPP_MAX_PER_SESSION) instanceof Integer) {
                perSession = response.getInt(Constants.INAPP_MAX_PER_SESSION);
            }

            if (response.has("imp") && response.get("imp") instanceof Integer) {
                perDay = response.getInt("imp");
            }

            InAppFCManager.updateLimits(context, perDay, perSession);

            JSONArray inappNotifs;
            try {
                inappNotifs = response.getJSONArray(Constants.INAPP_JSON_RESPONSE_KEY);
            } catch (JSONException e) {
                Logger.logFine("InAppManager: In-app key didn't contain a valid JSON array");
                return;
            }

            // Add all the new notifications to the queue
            SharedPreferences prefs = StorageHelper.getPreferences(context);
            SharedPreferences.Editor editor = prefs.edit();
            try {
                JSONArray inappsFromPrefs = new JSONArray(prefs.getString(Constants.PREFS_INAPP_KEY, "[]"));

                // Now add the rest of them :)
                if (inappNotifs != null && inappNotifs.length() > 0) {
                    for (int i = 0; i < inappNotifs.length(); i++) {
                        try {
                            JSONObject inappNotif = inappNotifs.getJSONObject(i);
                            inappsFromPrefs.put(inappNotif);
                        } catch (JSONException e) {
                            Logger.logFine("InAppManager: Malformed inapp notification");
                        }
                    }
                }

                // Commit all the changes
                editor.putString(Constants.PREFS_INAPP_KEY, inappsFromPrefs.toString());
                StorageHelper.persist(editor);
            } catch (Throwable e) {
                Logger.logFine("InAppManager: Failed to parse the in-app notifications properly");
                Logger.error("InAppManager: Reason: " + e.getMessage(), e);
            }
            // Fire the most recent notification, if any
            showNotificationIfAvailable(context);
        } catch (Throwable t) {
            Logger.error("InAppManager: Failed to parse response", t);
        }
    }

    public static void showNotificationIfAvailable(Context context) {
        SharedPreferences prefs = StorageHelper.getPreferences(context);
        try {
            if (!CleverTapAPI.getInstance(context).canShowInAppOnActivity()) {
                Logger.logExtraFine("Not showing notification on blacklisted activity");
                return;
            }

            JSONArray inapps = new JSONArray(prefs.getString(Constants.PREFS_INAPP_KEY, "[]"));
            if (inapps.length() < 1) {
                return;
            }

            ArrayList<Integer> indicesToIgnore = new ArrayList<Integer>();
            int qualifiedIndex = -1;

            final InAppNotificationListener listener = CleverTapAPI.getInstance(context).getInAppNotificationListener();

            for (int i = 0; i < inapps.length(); i++) {
                JSONObject inapp = inapps.getJSONObject(i);

                if (!InAppFCManager.canShow(context, inapp)) {
                    indicesToIgnore.add(i);
                    continue;
                }

                final boolean goFromListener;

                if (listener != null) {
                    final HashMap<String, Object> kvs;
                    final JSONObject data = inapp.getJSONObject("d");
                    if (data.has("kv")) {
                        kvs = Utils.convertJSONObjectToHashMap(data.getJSONObject("kv"));
                    } else {
                        kvs = new HashMap<String, Object>();
                    }

                    goFromListener = listener.beforeShow(kvs);
                } else {
                    goFromListener = true;
                }

                if (goFromListener) {
                    qualifiedIndex = i;
                    break;
                } else {
                    Logger.error("Application has decided to not show this in-app notification: " + inapp.optString(Constants.NOTIFICATION_ID_TAG, "<unknown ID>"));
                    indicesToIgnore.add(i);
                }
            }

            if (qualifiedIndex != -1) {
                createAndShow(context, inapps.getJSONObject(qualifiedIndex));
            }

            // JSON array doesn't have the feature to remove a single element,
            // so we have to copy over the entire array, but the first element
            JSONArray inappsUpdated = new JSONArray();
            for (int i = 0; i < inapps.length(); i++) {
                if (indicesToIgnore.contains(i) || qualifiedIndex == i) continue;

                inappsUpdated.put(inapps.get(i));
            }
            SharedPreferences.Editor editor = prefs.edit().putString(Constants.PREFS_INAPP_KEY, inappsUpdated.toString());
            StorageHelper.persist(editor);
        } catch (Throwable t) {
            // We won't get here
            Logger.logFine("InAppManager: Couldn't parse JSON array string from prefs", t);
        }
    }

    private static boolean validateNotifBundle(Bundle notif) {
        try {
            final Bundle w = notif.getBundle("w");
            final Bundle d = notif.getBundle("d");
            if (w == null || d == null) return false;

            // Check that either xdp or xp is set
            if (!isKeyValid(w, Constants.INAPP_X_DP, Integer.class))
                if (!isKeyValid(w, Constants.INAPP_X_PERCENT, Integer.class))
                    return false;

            // Check that either ydp or yp is set
            if (!isKeyValid(w, Constants.INAPP_Y_DP, Integer.class))
                if (!isKeyValid(w, Constants.INAPP_Y_PERCENT, Integer.class))
                    return false;

            // Check that dk is set
            if (!(isKeyValid(w, Constants.INAPP_NOTIF_DARKEN_SCREEN, Boolean.class)))
                return false;

            // Check that sc is set
            if (!(isKeyValid(w, Constants.INAPP_NOTIF_SHOW_CLOSE, Boolean.class)))
                return false;

            // Check that html is set
            if (!(isKeyValid(d, Constants.INAPP_DATA_TAG, String.class)))
                return false;

            // Check that pos contains the right value
            if ((isKeyValid(w, Constants.INAPP_POSITION, String.class))) {
                //noinspection ConstantConditions
                char pos = w.getString(Constants.INAPP_POSITION).charAt(0);
                switch (pos) {
                    case Constants.INAPP_POSITION_TOP:
                        break;
                    case Constants.INAPP_POSITION_RIGHT:
                        break;
                    case Constants.INAPP_POSITION_BOTTOM:
                        break;
                    case Constants.INAPP_POSITION_LEFT:
                        break;
                    case Constants.INAPP_POSITION_CENTER:
                        break;
                    default:
                        return false;
                }
            } else
                return false;

            // All is good
            //noinspection Contract
            return true;
        } catch (Throwable t) {
            Logger.error("Failed to parse in-app notification!", t);
            return false;
        }
    }

    private static boolean isKeyValid(Bundle b, String key, Class<?> type) {
        //noinspection ConstantConditions
        return b.containsKey(key) && b.get(key).getClass().equals(type);
    }

    private static Bundle getBundleFromJsonObject(JSONObject notif) {
        Bundle b = new Bundle();
        Iterator iterator = notif.keys();
        while (iterator.hasNext()) {
            String key = (String) iterator.next();
            try {
                Object value = notif.get(key);
                if (value instanceof String)
                    b.putString(key, (String) value);
                else if (value instanceof Character)
                    b.putChar(key, (Character) value);
                else if (value instanceof Integer)
                    b.putInt(key, (Integer) value);
                else if (value instanceof Float)
                    b.putFloat(key, (Float) value);
                else if (value instanceof Double)
                    b.putDouble(key, (Double) value);
                else if (value instanceof Long)
                    b.putLong(key, (Long) value);
                else if (value instanceof Boolean)
                    b.putBoolean(key, (Boolean) value);
                else if (value instanceof JSONObject)
                    b.putBundle(key, getBundleFromJsonObject((JSONObject) value));
            } catch (JSONException e) {
                // Strange value?
                Logger.logFine("InAppManager: Key had foreign object. Discarding");
            }
        }
        return b;
    }
}
