/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.accounts.Account;
import android.accounts.AccountManager;
import android.content.Context;
import android.telephony.TelephonyManager;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.ParseException;
import java.util.*;

/**
 * Handles all push events which describe a user's profile.
 */
public final class ProfileHandler {
    private final Context context;
    private static final String ENUM_PREFIX = "$E_";

    /**
     * Please do not create an object of this explicitly.
     *
     * @param context The Android context
     * @see CleverTapAPI
     */
    ProfileHandler(Context context) {
        this.context = context;
    }

    enum KnownFields {
        Name(Constants.USER_NAME), Email(Constants.USER_EMAIL), Education(Constants.USER_EDUCATION),
        Married(Constants.USER_RELATIONSHIP_STATUS), DOB(Constants.USER_DOB), Gender(Constants.USER_GENDER),
        Phone(Constants.USER_PHONE), Age(Constants.USER_AGE);

        private final String storageValue;

        KnownFields(String storageValue) {
            this.storageValue = storageValue;
        }
    }

    /**
     * Set a collection of unique values as a multi-value user profile property, any existing value will be overwritten.
     * Max 100 values, on reaching 100 cap, oldest value(s) will be removed.
     * Values must be Strings and are limited to 40 bytes.
     *
     * @param key    String
     * @param values {@link ArrayList} with String values
     */
    public void setMultiValuesForKey(String key, ArrayList<String> values) {
        _handleMultiValues(values, key, Constants.COMMAND_SET);
    }

    /**
     * Add a unique value to a multi-value user profile property
     * If the property does not exist it will be created
     * <p/>
     * Max 100 values, on reaching 100 cap, oldest value(s) will be removed.
     * Values must be Strings and are limited to 40 bytes.
     * <p/>
     * If the key currently contains a scalar value, the key will be promoted to a multi-value property
     * with the current value cast to a string and the new value(s) added
     *
     * @param key   String
     * @param value String
     */
    public void addMultiValueForKey(String key, String value) {

        if (value == null || value.isEmpty()) {
            _generateInvalidMultiValueError(key, value);
            return;
        }

        addMultiValuesForKey(key, new ArrayList<String>(Collections.singletonList(value)));
    }

    /**
     * Add a collection of unique values to a multi-value user profile property
     * If the property does not exist it will be created
     * <p/>
     * Max 100 values, on reaching 100 cap, oldest value(s) will be removed.
     * Values must be Strings and are limited to 40 bytes.
     * <p/>
     * If the key currently contains a scalar value, the key will be promoted to a multi-value property
     * with the current value cast to a string and the new value(s) added
     *
     * @param key    String
     * @param values {@link ArrayList} with String values
     */
    public void addMultiValuesForKey(String key, ArrayList<String> values) {

        String command = (LocalDataStore.getProfileValueForKey(key) != null) ? Constants.COMMAND_ADD : Constants.COMMAND_SET;
        _handleMultiValues(values, key, command);
    }

    /**
     * Remove a unique value from a multi-value user profile property
     * <p/>
     * If the key currently contains a scalar value, prior to performing the remove operation
     * the key will be promoted to a multi-value property with the current value cast to a string.
     * If the multi-value property is empty after the remove operation, the key will be removed.
     *
     * @param key   String
     * @param value String
     */
    public void removeMultiValueForKey(String key, String value) {

        if (value == null || value.isEmpty()) {
            _generateInvalidMultiValueError(key, value);
            return;
        }

        removeMultiValuesForKey(key, new ArrayList<String>(Collections.singletonList(value)));
    }

    /**
     * Remove a collection of unique values from a multi-value user profile property
     * <p/>
     * If the key currently contains a scalar value, prior to performing the remove operation
     * the key will be promoted to a multi-value property with the current value cast to a string.
     * <p/>
     * If the multi-value property is empty after the remove operation, the key will be removed.
     *
     * @param key    String
     * @param values {@link ArrayList} with String values
     */
    public void removeMultiValuesForKey(String key, ArrayList<String> values) {
        _handleMultiValues(values, key, Constants.COMMAND_REMOVE);
    }

    /**
     * Remove the user profile property value specified by key from the user profile
     *
     * @param key String
     */
    public void removeValueForKey(String key) {

        key = (key == null) ? "" : key; // so we will generate a validation error later on

        try {
            // validate the key
            ValidationResult vr;

            vr = Validator.cleanObjectKey(key);
            key = vr.getObject().toString();

            if (key.isEmpty()) {
                ValidationResult error = new ValidationResult();
                error.setErrorCode(512);
                error.setErrorDesc("Key is empty, profile removeValueForKey aborted.");
                pushValidationResult(error);
                Logger.log("Key is empty, profile removeValueForKey aborted");
                // Abort
                return;
            }
            // Check for an error
            if (vr.getErrorCode() != 0) {
                pushValidationResult(vr);
            }

            // remove from the local profile
            LocalDataStore.removeProfileField(context, key);

            // send the delete command
            JSONObject command = new JSONObject().put(Constants.COMMAND_DELETE, true);
            JSONObject update = new JSONObject().put(key, command);
            pushBasicProfile(update);

            Logger.logFine("removing value for key " + key + " from user profile");

        } catch (Throwable t) {
            Logger.error("Failed to remove profile value for key " + key, t);
        }
    }

    /**
     * Push a profile update.
     *
     * @param profile A {@link Map}, with keys as strings, and values as {@link String},
     *                {@link Integer}, {@link Long}, {@link Boolean}, {@link Float}, {@link Double},
     *                {@link java.util.Date}, or {@link Character}
     */
    public void push(Map<String, Object> profile) {
        if (profile == null || profile.isEmpty())
            return;
        try {
            ValidationResult vr;
            JSONObject customProfile = new JSONObject();
            JSONObject fieldsToUpdateLocally = new JSONObject();
            for (String key : profile.keySet()) {
                Object value = profile.get(key);

                vr = Validator.cleanObjectKey(key);
                key = vr.getObject().toString();
                // Check for an error
                if (vr.getErrorCode() != 0) {
                    pushValidationResult(vr);
                }

                try {
                    vr = Validator.cleanObjectValue(value, true, true);
                } catch (Exception e) {
                    // The object was neither a String, Boolean, or any number primitives
                    ValidationResult error = new ValidationResult();
                    error.setErrorCode(512);
                    error.setErrorDesc("Object value wasn't a primitive. Last event aborted.");
                    pushValidationResult(error);
                    Logger.log("Object value isn't a primitive. Profile update aborted");
                    // Abort
                    return;
                }
                value = vr.getObject();
                // Check for an error
                if (vr.getErrorCode() != 0) {
                    pushValidationResult(vr);
                }

                // Check for known keys
                if (key.equalsIgnoreCase("Phone")) {
                    try {
                        Validator.validatePhone(value);
                        value = ((Number) value).longValue();
                    } catch (Exception e) {
                        pushValidationResult(new ValidationResult(512, "Invalid phone number"));
                        Logger.log("Invalid phone number");
                        continue;
                    }
                }

                if (key.equalsIgnoreCase("Age")) {
                    if (value instanceof Integer) {
                        int age = (Integer) value;
                        if (!(age > 0 && age < 120)) {
                            Logger.log("Invalid age supplied");
                            pushValidationResult(new ValidationResult(512, "Invalid age"));
                            continue;
                        }
                    } else {
                        Logger.log("Age looks to be of an unsupported data type");
                        pushValidationResult(new ValidationResult(512, "Invalid age (unknown data type)"));
                        continue;
                    }
                }

                // add to the local profile update object
                fieldsToUpdateLocally.put(key, value);

                try {
                    KnownFields kf = KnownFields.valueOf(key);
                    if (kf == null) throw new Exception();

                    keepProperty(kf.storageValue, value);
                } catch (Throwable t) {
                    customProfile.put(key, value);
                }
            }

            Logger.logFine("Constructed custom profile: " + customProfile.toString());

            // update local profile values
            if (fieldsToUpdateLocally.length() > 0) {
                LocalDataStore.setProfileFields(context, fieldsToUpdateLocally);
            }

            pushBasicProfile(customProfile);

        } catch (Throwable t) {
            // Will not happen
            Logger.error("Failed to push profile", t);
        }
    }

    private String getGraphUserPropertySafely(JSONObject graphUser, String key, String def) {
        try {
            String prop = (String) graphUser.get(key);
            if (prop != null)
                return prop;
            else
                return def;
        } catch (Throwable t) {
            return def;
        }
    }

    /**
     * Pushes everything available in the JSON object returned by the Facebook GraphRequest
     *
     * @param graphUser The object returned from Facebook
     */
    public void pushFacebookUser(JSONObject graphUser) {
        try {
            if (graphUser == null)
                return;
            // Note: No validations are required here, as everything is controlled
            String name = getGraphUserPropertySafely(graphUser, "name", "");
            try {
                // Certain users have nasty looking names - unicode chars, validate for any
                // not allowed chars
                ValidationResult vr = Validator.cleanObjectValue(name, false, false);
                name = vr.getObject().toString();

                if (vr.getErrorCode() != 0) {
                    pushValidationResult(vr);
                }
            } catch (IllegalArgumentException e) {
                // Weird name, wasn't a string, or any number
                // This would never happen with FB
                name = "";
            }

            String gender = getGraphUserPropertySafely(graphUser, "gender", null);
            // Convert to WR format
            if (gender != null) {
                if (gender.toLowerCase().startsWith("m"))
                    gender = "M";
                else if (gender.toLowerCase().startsWith("f"))
                    gender = "F";
                else
                    gender = "";
            } else {
                gender = null;
            }
            String email = getGraphUserPropertySafely(graphUser, "email", "");

            String birthday = getGraphUserPropertySafely(graphUser, "birthday", null);
            if (birthday != null) {
                // Some users don't have the year of birth mentioned
                // FB returns only MM/dd in those cases
                if (birthday.matches("^../..")) {
                    // This means that the year is not available(~30% of the times)
                    // Ignore
                    birthday = "";
                } else {
                    try {
                        Date date = Constants.FB_DOB_DATE_FORMAT.parse(birthday);
                        // WR format = $D_yyyyMMdd
                        birthday = "$D_" + Constants.DOB_DATE_FORMAT.format(date);
                    } catch (ParseException e) {
                        // Differs from the specs
                        birthday = "";
                    }
                }
            }

            String work;
            try {
                JSONArray workArray = graphUser.getJSONArray("work");
                work = (workArray.length() > 0) ? "Y" : "N";
            } catch (Throwable t) {
                work = "";
            }

            String education;
            try {
                JSONArray eduArray = graphUser.getJSONArray("education");
                // FB returns the education levels in a descending order - highest = last entry
                String fbEdu = eduArray.getJSONObject(eduArray.length() - 1).getString("type");
                if (fbEdu.toLowerCase().contains("high school"))
                    education = "School";
                else if (fbEdu.toLowerCase().contains("college"))
                    education = "College";
                else if (fbEdu.toLowerCase().contains("graduate school"))
                    education = "Graduate";
                else
                    education = "";
            } catch (Throwable t) {
                // No education info available
                education = "";
            }

            String id = getGraphUserPropertySafely(graphUser, "id", "");

            String married = getGraphUserPropertySafely(graphUser, "relationship_status", null);
            if (married != null) {
                if (married.equalsIgnoreCase("married")) {
                    married = "Y";
                } else {
                    married = "N";
                }
            }

            keepProperty(Constants.FB_ID, id);
            keepProperty(Constants.FB_NAME, name);
            keepProperty(Constants.FB_EMAIL, email);
            keepProperty(Constants.FB_GENDER, gender);
            keepProperty(Constants.FB_EDUCATION, education);
            keepProperty(Constants.FB_EMPLOYED, work);
            keepProperty(Constants.FB_DOB, birthday);
            keepProperty(Constants.FB_RELATIONSHIP_STATUS, married);

            pushBasicProfile(null);
        } catch (Throwable t) {
            Logger.error("Failed to parse graph user object successfully", t);
        }
    }

    private void keepProperty(String key, Object value) {
        if (key == null || key.equals("")
                || value == null) {
            return;
        }
        if (value instanceof String) {
            StorageHelper.putString(context, key, (String) value);
        } else if (value instanceof Integer) {
            StorageHelper.putInt(context, key, (Integer) value);
        } else if (value instanceof Long) {
            StorageHelper.putLong(context, key, (Long) value);
        } else {
            Logger.log("Ignored value due to unsupported type: " + value.getClass().getName());
        }
    }

    /**
     * Pushes default user information, if not already set.
     * <p/>
     * This is initiated from CleverTapAPI. There is no need to call this method
     * explicitly.
     */

    void pushProfileDefaults(Context context, JSONObject defaults) {
        try {
            Logger.logFine("Handling Profile Defaults " + defaults.toString());

            if (defaults.length() <= 0) {
                return;
            }

            JSONObject profile = new JSONObject();

            Iterator i = defaults.keys();
            while (i.hasNext()) {
                String next = i.next().toString();
                try {
                    if (LocalDataStore.getProfileValueForKey(next) != null) {
                        Logger.logFine("pushProfileDefaults: Have value for " + next + " ignoring default value " + defaults.get(next).toString());
                        continue;
                    }
                    profile.put(next, defaults.get(next));

                } catch (Exception e) {
                    // Ignore
                }
            }

            if (profile.length() <= 0) {
                return;
            }

            try {
                Logger.logFine("Pushing Profile Defaults event " + profile.toString());
                pushBasicProfile(profile);
            } catch (Exception e) {
                // We won't get here
                Logger.logFine("FATAL: Pushing Profile Defaults event failed!");
            }
        } catch (Throwable t) {
            Logger.error("pushProfileDefaults", t);
        }
    }

    /**
     * Pushes basic user information, if available.
     * <p/>
     * Basic information is pushed daily. There is no need to call this method
     * explicitly.
     */
    void pushBasicProfile(JSONObject baseProfile) {
        try {
            JSONObject profileEvent = new JSONObject();

            if (baseProfile != null && baseProfile.length() > 0) {
                Iterator i = baseProfile.keys();
                while (i.hasNext()) {
                    String next = i.next().toString();

                    // need to handle command-based JSONObject props here now
                    Object value = null;
                    try {
                        value = baseProfile.getJSONObject(next);
                    } catch (Throwable t) {
                        try {
                            value = baseProfile.get(next);
                        } catch (JSONException e) {
                            //no-op
                        }
                    }

                    if (value != null) {
                        profileEvent.put(next, value);
                    }
                }
            }

            try {
                Account account = getGoogleAccount();
                if (account != null && account.name != null) {
                    ValidationResult vr;
                    try {
                        vr = Validator.cleanObjectValue(account.name, false, false);
                        keepProperty(Constants.SYSTEM_EMAIL, vr.getObject().toString());
                    } catch (IllegalArgumentException e) {
                        // The object was neither a String, or any number primitives
                        // Ignore
                    }
                }

                String carrier = getCarrier();
                if (carrier != null && !carrier.equals("")) {
                    keepProperty(Constants.SYSTEM_CARRIER, carrier);
                }

                String cc = getCountryCode();
                if (cc != null && !cc.equals("")) {
                    keepProperty(Constants.SYSTEM_COUNTRY_CODE, cc);
                }

                // Add the user's timezone
                keepProperty(Constants.SYSTEM_TIMEZONE, TimeZone.getDefault().getID());

                addKeptProfileProperties(context, profileEvent);

                // Attempt to send all other email addresses too
                try {
                    JSONArray allAccounts = getAllAccounts();
                    if (allAccounts != null && allAccounts.length() > 0) {
                        profileEvent.put("Alternate Emails", allAccounts);
                    }
                } catch (Throwable t) {
                    // Ignore
                }

                JSONObject event = new JSONObject();
                event.put("profile", profileEvent);
                QueueManager.addToQueue(context, event, Constants.PROFILE_EVENT);
            } catch (JSONException e) {
                // We won't get here
                Logger.logFine("FATAL: Creating basic profile update event failed!");
            }
        } catch (Throwable t) {
            Logger.error("Basic profile sync", t);
        }
    }

    private void addPropertyFromStoreIfExists(Context context, String profileEventKey, JSONObject profile, DataType dataType, String... preferenceKeys) {
        for (String pk : preferenceKeys) {
            try {
                Object value;
                switch (dataType) {
                    case Integer:
                        try {
                            value = StorageHelper.getInt(context, pk, Integer.MIN_VALUE);
                            if ((Integer) value == Integer.MIN_VALUE) continue;
                        } catch (ClassCastException c) {
                            value = StorageHelper.getLong(context, pk, Long.MIN_VALUE);
                            if ((Long) value == Long.MIN_VALUE) continue;
                        }
                        break;
                    case Long:
                        try {
                            value = StorageHelper.getLong(context, pk, Long.MIN_VALUE);
                            if ((Long) value == Long.MIN_VALUE) continue;
                        } catch (ClassCastException c) {
                            value = StorageHelper.getInt(context, pk, Integer.MIN_VALUE);
                            if ((Integer) value == Integer.MIN_VALUE) continue;
                        }
                        break;
                    case String:
                        value = StorageHelper.getString(context, pk, null);
                        if (value == null) continue;
                        break;
                    default:
                        value = null;
                }

                //noinspection ConstantConditions
                if (value == null) continue;

                try {
                    profile.put(profileEventKey, value);

                    // add to the local profile
                    LocalDataStore.setProfileField(context, profileEventKey, value.toString());

                    return;
                } catch (JSONException e) {
                    // Ignore
                }
            } catch (Exception e) {
                // Ignore
            }
        }
    }

    /**
     * Adds the stored user populated/FB/G+ data, if available.
     * Preference is given to user populated, followed by FB, and then G+, if all are available.
     * <p/>
     * Adds:
     * 1. Name
     * 2. Gender
     * 3. Education
     * 4. Employed
     * 5. Married
     * 6. DOB
     * 7. FBID
     * 8. GPID
     * 10. Phone
     * 11. Age
     * 12. Email
     * 13. tz
     * 14. Carrier
     * 15. cc
     */
    private void addKeptProfileProperties(Context context, JSONObject profile) {
        addPropertyFromStoreIfExists(context, "Name", profile, DataType.String, Constants.USER_NAME, Constants.FB_NAME, Constants.GP_NAME);
        addPropertyFromStoreIfExists(context, "Gender", profile, DataType.String, Constants.USER_GENDER, Constants.FB_GENDER, Constants.GP_GENDER);
        addPropertyFromStoreIfExists(context, "Education", profile, DataType.String, Constants.USER_EDUCATION, Constants.FB_EDUCATION);
        addPropertyFromStoreIfExists(context, "Employed", profile, DataType.String, Constants.USER_EMPLOYED, Constants.FB_EMPLOYED, Constants.GP_EMPLOYED);
        addPropertyFromStoreIfExists(context, "Married", profile, DataType.String, Constants.USER_RELATIONSHIP_STATUS, Constants.FB_RELATIONSHIP_STATUS, Constants.GP_RELATIONSHIP_STATUS);
        addPropertyFromStoreIfExists(context, "DOB", profile, DataType.String, Constants.USER_DOB, Constants.FB_DOB, Constants.GP_DOB);
        addPropertyFromStoreIfExists(context, "FBID", profile, DataType.String, Constants.FB_ID);
        addPropertyFromStoreIfExists(context, "GPID", profile, DataType.String, Constants.GP_ID);
        addPropertyFromStoreIfExists(context, "Phone", profile, DataType.Long, Constants.USER_PHONE);
        addPropertyFromStoreIfExists(context, "Age", profile, DataType.Integer, Constants.USER_AGE);
        addPropertyFromStoreIfExists(context, "Email", profile, DataType.String, Constants.USER_EMAIL, Constants.FB_EMAIL, Constants.SYSTEM_EMAIL);
        addPropertyFromStoreIfExists(context, "tz", profile, DataType.String, Constants.SYSTEM_TIMEZONE);
        addPropertyFromStoreIfExists(context, "Carrier", profile, DataType.String, Constants.SYSTEM_CARRIER);
        addPropertyFromStoreIfExists(context, "cc", profile, DataType.String, Constants.SYSTEM_COUNTRY_CODE);

        // add the secondary email
        addSecondaryEmail(context, profile);
    }

    private void addSecondaryEmail(Context context, JSONObject profile) {
        try {
            String originalEmail = profile.getString("Email");
            if (originalEmail == null || originalEmail.equals("")) {
                addPropertyFromStoreIfExists(context, "Email", profile, DataType.String, Constants.FB_EMAIL);
            } else {
                String email2 = StorageHelper.getString(context, Constants.FB_EMAIL, "");
                if (email2 == null || email2.equals(originalEmail) || email2.equals("")) {
                    return;
                }
                profile.put("Email2", email2);
            }
        } catch (JSONException ignore) {
            // Ignore
        }
    }

    /**
     * Pushes an enum type of property for a user.
     * <p/>
     * Enum types show up in the CleverTap dashboard, when filtering down to a certain
     * group of people.
     * <p/>
     * You can have up to four enums, with eight values for each of them.
     * <p/>
     * For example, if you have different levels amongst your customer,
     * say, Gold, Silver, and Platinum, then you could use the key, "Type",
     * and the value corresponding to the type of customer, i.e. Gold, Silver, or Platinum.
     *
     * @param key   The enum key
     * @param value The value for this profile
     */
    @Deprecated
    public void pushEnum(String key, String value) {
        JSONObject enumEvent = new JSONObject();
        try {
            ValidationResult vr;
            vr = Validator.cleanObjectKey(key);
            key = vr.getObject().toString();
            if (key == null || key.equals(""))
                // Abort
                return;

            vr = Validator.cleanObjectValue(value, false, false);
            value = vr.getObject().toString();
            if (value == null || value.equals(""))
                // Abort
                return;

            enumEvent.put(key, ENUM_PREFIX + value);

            JSONObject event = new JSONObject();
            event.put("profile", enumEvent);
            QueueManager.addToQueue(context, event, Constants.PROFILE_EVENT);
        } catch (Throwable e) {
            // We won't get here
            Logger.logFine("FATAL: Creating enum for profile update event failed!");
        }
    }

    private String getCarrier() {
        try {
            TelephonyManager tm = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            String carrier = tm.getSimOperatorName();
            ValidationResult vr = Validator.cleanObjectValue(carrier, false, false);
            if (vr.getErrorCode() == 0)
                return (String) vr.getObject();
        } catch (Throwable ignore) {
        }
        return "";
    }

    Account getGoogleAccount() {
        if (context == null) return null;

        // Respect disable email privacy
        try {
            String value = ManifestMetaData.getMetaData(context, Constants.LABEL_PRIVACY_MODE);
            if (value.contains(Constants.PRIVACY_MODE_DISABLE_EMAIL)) {
                return null;
            }
        } catch (Throwable t) {
            // Okay cool, so the default behaviour is to capture Google account
        }

        try {
            AccountManager manager = AccountManager.get(context);
            Account[] accounts = manager.getAccountsByType("com.google");
            if (accounts != null && accounts.length > 0)
                return accounts[0];
            else
                return null;
        } catch (Throwable ignore) {
            return null;
        }
    }

    private JSONArray getAllAccounts() {
        if (context == null) return null;

        // Respect disable email privacy
        try {
            String value = ManifestMetaData.getMetaData(context, Constants.LABEL_PRIVACY_MODE);
            if (value.contains(Constants.PRIVACY_MODE_DISABLE_EMAIL)) {
                return null;
            }
        } catch (Throwable t) {
            // Okay cool, so the default behaviour is to capture Google account
        }

        try {
            final JSONArray all = new JSONArray();
            AccountManager am = AccountManager.get(context);
            Account[] accounts = am.getAccounts();
            for (Account account : accounts) {
                final String name = account.name;
                if (name != null && name.contains("@")) {
                    all.put(name);
                }
            }
            return all;
        } catch (Throwable t) {
            return null;
        }
    }

    private String getCountryCode() {
        try {
            TelephonyManager tm = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            return tm.getSimCountryIso();
        } catch (Throwable ignore) {
            return "";
        }
    }

    /**
     * Pushes everything useful within the Google Plus
     * {@link com.google.android.gms.plus.model.people.Person} object.
     *
     * @param person The {@link com.google.android.gms.plus.model.people.Person} object
     * @see com.google.android.gms.plus.model.people.Person
     */
    public void pushGooglePlusPerson(com.google.android.gms.plus.model.people.Person person) {
        if (person == null) {
            return;
        }
        try {
            // Note: No validations are required here, as everything is controlled
            String name = "";
            if (person.hasDisplayName()) {
                try {
                    // Certain users have nasty looking names - unicode chars, validate for any
                    // not allowed chars
                    name = person.getDisplayName();
                    ValidationResult vr = Validator.cleanObjectValue(name, false, false);
                    name = vr.getObject().toString();

                    if (vr.getErrorCode() != 0) {
                        pushValidationResult(vr);
                    }
                } catch (Throwable t) {
                    // Weird name, wasn't a string, or any number
                    // This would never happen with G+
                    name = "";
                }
            }

            String gender = "";
            if (person.hasGender()) {
                if (person.getGender() == com.google.android.gms.plus.model.people.Person.Gender.MALE) {
                    gender = "M";
                } else if (person.getGender() == com.google.android.gms.plus.model.people.Person.Gender.FEMALE) {
                    gender = "F";
                }
            }

            String birthday = "";

            if (person.hasBirthday()) {
                // We have the string as YYYY-MM-DD
                try {
                    Date date = Constants.GP_DOB_DATE_FORMAT.parse(person.getBirthday());
                    // WR format = $D_yyyyMMdd
                    birthday = "$D_" + Constants.DOB_DATE_FORMAT.format(date);
                } catch (Throwable t) {
                    // Differs from the specs
                    birthday = "";
                }
            }

            String work = "";
            if (person.hasOrganizations()) {
                List<com.google.android.gms.plus.model.people.Person.Organizations> organizations = person.getOrganizations();
                for (com.google.android.gms.plus.model.people.Person.Organizations o : organizations) {
                    if (o.getType() == com.google.android.gms.plus.model.people.Person.Organizations.Type.WORK) {
                        work = "Y";
                        break;
                    }
                }
            }

            String id = "";
            if (person.hasId()) {
                id = person.getId();
            }

            String married = "";
            if (person.hasRelationshipStatus()) {
                if (person.getRelationshipStatus() == com.google.android.gms.plus.model.people.Person.RelationshipStatus.MARRIED) {
                    married = "Y";
                } else {
                    married = "N";
                }
            }

            // Construct json object from the data
            keepProperty(Constants.GP_ID, id);
            keepProperty(Constants.GP_NAME, name);
            keepProperty(Constants.GP_GENDER, gender);
            keepProperty(Constants.GP_EMPLOYED, work);
            keepProperty(Constants.GP_DOB, birthday);
            keepProperty(Constants.GP_RELATIONSHIP_STATUS, married);
            pushBasicProfile(null);
        } catch (Throwable t) {
            // We won't get here
            Logger.logFine("FATAL: Creating G+ profile update event failed!");
        }
    }

    /**
     * Return the user profile property value for the specified key
     *
     * @param name String
     * @return {@link JSONArray}, String or null
     */
    public Object getProperty(String name) {
        return LocalDataStore.getProfileProperty(context, name);
    }

    // private multi-value handlers and helpers

    private void _handleMultiValues(ArrayList<String> values, String key, String command) {
        if (key == null) return;

        if (values == null || values.isEmpty()) {
            _generateInvalidMultiValueError(key, values);
            return;
        }

        ValidationResult vr;

        // validate the key
        vr = Validator.cleanMultiValuePropertyKey(key);

        // Check for an error
        if (vr.getErrorCode() != 0) {
            pushValidationResult(vr);
        }

        // reset the key
        Object _key = vr.getObject();
        String cleanKey = (_key != null) ? vr.getObject().toString() : null;

        // if key is empty generate an error and return
        if (cleanKey == null || cleanKey.isEmpty()) {
            _generateInvalidMultiValueKeyError(key);
            return;
        }

        key = cleanKey;

        try {
            JSONArray currentValues = _constructExistingMultiValue(key, command);
            JSONArray newValues = _cleanMultiValues(values, key);
            _validateAndPushMultiValue(currentValues, newValues, values, key, command);

        } catch (Throwable t) {
            Logger.logFine("Error handling multi value operation for key " + key, t);
        }
    }

    private JSONArray _constructExistingMultiValue(String key, String command) {

        Boolean remove = command.equals(Constants.COMMAND_REMOVE);
        Boolean add = command.equals(Constants.COMMAND_ADD);

        // only relevant for add's and remove's; a set overrides the existing value, so return a new array
        if (!remove && !add) return new JSONArray();

        Object existing = LocalDataStore.getProfileProperty(context, key);

        // if there is no existing value
        if (existing == null) {
            // if its a remove then return null to abort operation
            // no point in running remove against a nonexistent value
            if (remove) return null;

            // otherwise return an empty array
            return new JSONArray();
        }

        // value exists

        // the value should only ever be a JSONArray or scalar (String really)

        // if its already a JSONArray return that
        if (existing instanceof JSONArray) return (JSONArray) existing;

        // handle a scalar value as the existing value
        /*
            if its an add, our rule is to promote the scalar value to multi value and include the cleaned stringified
            scalar value as the first element of the resulting array

            NOTE: the existing scalar value is currently limited to 120 bytes; when adding it to a multi value
            it is subject to the current 40 byte limit

            if its a remove, our rule is to delete the key from the local copy
            if the cleaned stringified existing value is equal to any of the cleaned values passed to the remove method

            if its an add, return an empty array as the default,
            in the event the existing scalar value fails stringifying/cleaning

            returning null will signal that a remove operation should be aborted,
            as there is no valid promoted multi value to remove against
         */

        JSONArray _default = (add) ? new JSONArray() : null;

        String stringified = _stringifyAndCleanScalarProfilePropValue(existing);

        return (stringified != null) ? new JSONArray().put(stringified) : _default;
    }

    private String _stringifyScalarProfilePropValue(Object value) {
        String val = null;

        try {
            val = value.toString();
        } catch (Exception e) {
            // no-op
        }

        return val;
    }

    private String _stringifyAndCleanScalarProfilePropValue(Object value) {
        String val = _stringifyScalarProfilePropValue(value);

        if (val != null) {
            ValidationResult vr = Validator.cleanMultiValuePropertyValue(val);

            // Check for an error
            if (vr.getErrorCode() != 0) {
                pushValidationResult(vr);
            }

            Object _value = vr.getObject();
            val = (_value != null) ? vr.getObject().toString() : null;
        }

        return val;
    }

    private JSONArray _cleanMultiValues(ArrayList<String> values, String key) {

        try {
            if (values == null || key == null) return null;

            JSONArray cleanedValues = new JSONArray();
            ValidationResult vr;

            // loop through and clean the new values
            for (String value : values) {
                value = (value == null) ? "" : value;  // so we will generate a validation error later on

                // validate value
                vr = Validator.cleanMultiValuePropertyValue(value);

                // Check for an error
                if (vr.getErrorCode() != 0) {
                    pushValidationResult(vr);
                }

                // reset the value
                Object _value = vr.getObject();
                value = (_value != null) ? vr.getObject().toString() : null;

                // if value is empty generate an error and return
                if (value == null || value.isEmpty()) {
                    _generateInvalidMultiValueError(key, value);
                    // Abort
                    return null;
                }
                // add to the newValues to be merged
                cleanedValues.put(value);
            }

            return cleanedValues;

        } catch (Throwable t) {
            Logger.logFine("Error cleaning multi values for key " + key, t);
            _generateInvalidMultiValueError(key, values);
            return null;
        }
    }

    private void _validateAndPushMultiValue(JSONArray currentValues, JSONArray newValues, ArrayList<String> originalValues, String key, String command) {

        try {

            // if any of these are null, indicates some problem along the way so abort operation
            if (currentValues == null || newValues == null || originalValues == null || key == null || command == null)
                return;

            String mergeOperation = command.equals(Constants.COMMAND_REMOVE) ? Validator.REMOVE_VALUES_OPERATION : Validator.ADD_VALUES_OPERATION;

            // merge currentValues and newValues
            ValidationResult vr = Validator.mergeMultiValuePropertyForKey(currentValues, newValues, mergeOperation, key);

            // Check for an error
            if (vr.getErrorCode() != 0) {
                pushValidationResult(vr);
            }

            // set the merged local values array
            JSONArray localValues = (JSONArray) vr.getObject();

            // update local profile
            // remove an empty array
            if (localValues == null || localValues.length() <= 0) {
                LocalDataStore.removeProfileField(context, key);
            } else {
                // not empty so save to local profile
                LocalDataStore.setProfileField(context, key, localValues);
            }

            // push to server
            JSONObject commandObj = new JSONObject();
            commandObj.put(command, new JSONArray(originalValues));

            JSONObject fields = new JSONObject();
            fields.put(key, commandObj);

            pushBasicProfile(fields);

            Logger.logFine("Constructed multi-value profile push: " + fields.toString());

        } catch (Throwable t) {
            Logger.logFine("Error pushing multiValue for key " + key, t);
        }
    }

    private void _generateInvalidMultiValueError(String key, Object value) {
        ValidationResult error = new ValidationResult();
        String msg = "Invalid multi value: " + value.toString() + " for key " + key + ", profile multi value operation aborted.";
        error.setErrorCode(512);
        error.setErrorDesc(msg);
        pushValidationResult(error);
        Logger.log(msg);
    }

    private void _generateInvalidMultiValueKeyError(String key) {
        ValidationResult error = new ValidationResult();
        error.setErrorCode(523);
        error.setErrorDesc("Invalid multi-value property key " + key);
        pushValidationResult(error);
        Logger.log("Invalid multi-value property key " + key + " profile multi value operation aborted");
    }

    private void pushValidationResult(ValidationResult vr) {
        CleverTapAPI.pushValidationResult(vr);
    }
}