/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import com.clevertap.android.sdk.exceptions.CleverTapMetaDataNotFoundException;
import com.clevertap.android.sdk.exceptions.CleverTapPermissionsNotSatisfied;
import org.json.JSONObject;

/**
 * Broadcast receiver for INSTALL_REFERRAL intents.
 */
public final class InstallReferrerBroadcastReceiver extends BroadcastReceiver {
    public void onReceive(Context context, Intent intent) {
        try {
            CleverTapAPI wr = CleverTapAPI.getInstance(context);
            wr.pushInstallReferrer(intent);
        } catch (CleverTapMetaDataNotFoundException ignore) {
            // Ignore
        } catch (CleverTapPermissionsNotSatisfied ignore) {
            // Ignore
        }
    }

    /*package*/
    static JSONObject getUrchinFromUri(Context context, Uri uri) {
        JSONObject referrer = new JSONObject();
        try {
            // Don't care for null values - they won't be added anyway
            String source = getUtmOrWzrkValue("source", uri);
            String medium = getUtmOrWzrkValue("medium", uri);
            String campaign = getUtmOrWzrkValue("campaign", uri);

            CleverTapAPI.getInstance(context).session.setSource(source);
            CleverTapAPI.getInstance(context).session.setMedium(medium);
            CleverTapAPI.getInstance(context).session.setCampaign(campaign);

            referrer.put("us", source);
            referrer.put("um", medium);
            referrer.put("uc", campaign);

            String wm = getWzrkValueForKey("medium", uri);
            if (wm != null && wm.matches("^email$|^social$|^search$"))
                referrer.put("wm", wm);

            Logger.logFine("Referrer data: " + referrer.toString(4));
        } catch (Throwable ignore) {
            // Won't happen
        }
        return referrer;
    }

    private static String getUtmOrWzrkValue(String utmKey, Uri uri) {
        // Give preference to utm_*, else, try to look for wzrk_*
        String value;
        if ((value = getUtmValueForKey(utmKey, uri)) != null
                || (value = getWzrkValueForKey(utmKey, uri)) != null)
            return value;
        else
            return null;
    }

    private static String getWzrkValueForKey(String key, Uri uri) {
        key = "wzrk_" + key;
        return getValueForKey(key, uri);
    }

    private static String getUtmValueForKey(String key, Uri uri) {
        key = "utm_" + key;
        return getValueForKey(key, uri);
    }

    private static String getValueForKey(String key, Uri uri) {
        if (key == null || uri == null) return null;
        try {
            String value = uri.getQueryParameter(key);
            if (value == null) return null;
            if (value.getBytes("UTF-8").length > 120)
                return Validator.fastTrim(value, 120);
            else
                return value;
        } catch (Throwable t) {
            Logger.logFine("Couldn't parse the install referrer URI", t);
            return null;
        }
    }
}
