/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.graphics.Bitmap;
import android.location.Location;
import android.location.LocationManager;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.support.v4.app.NotificationCompat;
import com.clevertap.android.sdk.exceptions.CleverTapMetaDataNotFoundException;
import com.clevertap.android.sdk.exceptions.CleverTapPermissionsNotSatisfied;
import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesUtil;
import com.google.android.gms.gcm.GoogleCloudMessaging;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.URLDecoder;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Manipulates the CleverTap SDK.
 */
public final class CleverTapAPI {
    // Static fields
    public static final String CHARGED_EVENT = "Charged";
    public static final String NOTIFICATION_TAG = "wzrk_pn";
    private static final String PROPERTY_APP_VERSION = "appVersionCode";
    private static final Handler handlerUsingMainLooper = new Handler(Looper.getMainLooper());
    private static final ExecutorService es = Executors.newFixedThreadPool(1);
    private static int debugLevel = 0;
    private static boolean appForeground = false;
    private static CleverTapAPI ourInstance = null;
    static Runnable pendingInappRunnable = null;
    static String localAccountID = null, localToken = null;

    static int activityCount = 0;
    static ValidationResult pendingValidationResult = null;
    static Activity currentActivity;

    // Non static fields
    public final EventHandler event;
    public final DataHandler data;
    public final ProfileHandler profile;
    public final SessionHandler session;
    private Runnable sessionTimeoutRunnable = null;
    private GoogleCloudMessaging gcm;
    private final Context context;
    private long appLastSeen;
    private SyncListener syncListener = null;

    /**
     * Note: You don't need to use a lock to access this variable - everything happens in
     * a serial execution manner with the async handler
     */
    private boolean pushedGcmId = false;

    /**
     * Returns the generic handler object which is used to post
     * runnables. The returned value will never be null.
     *
     * @return The static generic handler
     * @see Handler
     */
    static Handler getHandlerUsingMainLooper() {
        return handlerUsingMainLooper;
    }

    /**
     * Use this to safely post a runnable to the async handler.
     * It adds try/catch blocks around the runnable and the handler itself.
     */
    static void postAsyncSafely(final String name, final Context context, final Runnable runnable) {
        try {
            es.submit(new Runnable() {
                @Override
                public void run() {
                    try {
                        Logger.logFine("Executor service: Starting task - " + name);
                        final long start = System.currentTimeMillis();
                        runnable.run();
                        final long time = System.currentTimeMillis() - start;
                        Logger.logFine("Executor service: Task completed successfully in " + time + "ms (" + name + ")");
                    } catch (Throwable t) {
                        Logger.logFine("Executor service: Failed to complete the scheduled task", t);
                    }
                }
            });
        } catch (Throwable t) {
            Logger.logFine("Failed to submit task to the executor service", t);
        }
    }

    /**
     * Returns whether or not the app is in the foreground.
     *
     * @return The foreground status
     */
    static boolean isAppForeground() {
        return appForeground;
    }

    /**
     * This method is used internally.
     */
    public static void setAppForeground(boolean appForeground) {
        CleverTapAPI.appForeground = appForeground;
    }

    public void setSyncListener(SyncListener syncListener) {
        this.syncListener = syncListener;
    }

    public SyncListener getSyncListener() {
        return syncListener;
    }

    /**
     * Returns an instance of the CleverTap SDK.
     *
     * @param context The Android context
     * @return The {@link CleverTapAPI} object
     * @throws CleverTapMetaDataNotFoundException
     * @throws CleverTapPermissionsNotSatisfied
     */
    public static synchronized CleverTapAPI getInstance(final Context context)
            throws CleverTapMetaDataNotFoundException, CleverTapPermissionsNotSatisfied {
        if (ourInstance == null && context != null) {
            DeviceInfo.updateDeviceIdIfRequired(context.getApplicationContext());
            ourInstance = new CleverTapAPI(context.getApplicationContext());
        }
        return ourInstance;
    }

    private CleverTapAPI(final Context context)
            throws CleverTapMetaDataNotFoundException, CleverTapPermissionsNotSatisfied {
        this.context = context;
        event = new EventHandler(context);
        profile = new ProfileHandler(context);
        data = new DataHandler(context);
        session = new SessionHandler(context);
        DeviceInfo.testPermission(context, "android.permission.INTERNET");
        // Test for account ID, and token
        ManifestMetaData.getMetaData(context, Constants.LABEL_ACCOUNT_ID);
        ManifestMetaData.getMetaData(context, Constants.LABEL_TOKEN);


        Logger.logFine("New instance of CleverTapAPI created");

        // Try to automatically register to GCM
        initPushHandling();
    }

    public static void changeCredentials(String accountID, String token) {
        localAccountID = accountID;
        localToken = token;
    }

    private final String PREFS_LAST_DAILY_PUSHED_EVENTS_DATE = "lastDailyEventsPushedDate";

    @SuppressLint("CommitPrefEdits")
    private void pushDailyEventsAsync() {
        postAsyncSafely("CleverTapAPI#pushDailyEventsAsync", null, new Runnable() {
            @Override
            public void run() {
                try {
                    Date d = new Date();
                    SharedPreferences prefs = StorageHelper.getPreferences(context);
                    Calendar cal = Calendar.getInstance();
                    cal.setTime(d);
                    if (hasAppVersionChanged() || prefs.getInt(PREFS_LAST_DAILY_PUSHED_EVENTS_DATE, 0) != cal.get(Calendar.DATE)) {

                        Logger.logFine("Queuing daily events");
                        profile.pushBasicProfile(null);
                        updateAdvertisingID();
                        if (!pushedGcmId) {
                            String gcmRegId = getRegistrationId();
                            if (gcmRegId != null && !gcmRegId.equals(""))
                                data.pushGcmRegistrationId(gcmRegId, GcmManager.isGcmEnabled(context));
                        } else {
                            Logger.logFine("Skipped push of the GCM ID. Somebody already sent it.");
                        }
                        pushedGcmId = true;
                    }
                    SharedPreferences.Editor editor = prefs.edit().putInt(PREFS_LAST_DAILY_PUSHED_EVENTS_DATE, cal.get(Calendar.DATE));
                    StorageHelper.persist(editor);
                    updateAppVersionInPrefs();
                } catch (Throwable t) {
                    Logger.error("Daily profile sync failed", t);
                }
            }
        });
    }

    /**
     * Enables or disables debugging. If enabled, see debug messages in Android's logcat utility.
     * Debug messages are tagged as CleverTap.
     *
     * @param level Can be one of the following - 0(disables all debugging),
     *              1(show some debug output)
     */
    public static void setDebugLevel(int level) {
        debugLevel = level;
    }

    public static int getDebugLevel() {
        return debugLevel;
    }

    /**
     * Tells CleverTap that the current Android activity was paused. Add a call to this
     * method in the onPause() method of every activity.
     *
     * @param activity The calling activity
     */
    public void activityPaused(Activity activity) {
        setAppForeground(false);
        // Just update it
        currentActivity = activity;
        appLastSeen = System.currentTimeMillis();
        // Create the runnable, if it is null
        if (sessionTimeoutRunnable == null)
            sessionTimeoutRunnable = new Runnable() {
                @Override
                public void run() {
                    long now = System.currentTimeMillis();
                    if (!isAppForeground()
                            && (now - appLastSeen) > Constants.SESSION_LENGTH_MINS * 60 * 1000) {
                        Logger.logFine("Session timeout reached");
                        SessionManager.destroySession();

                        Logger.logFine("Current activity set to null");
                        CleverTapAPI.currentActivity = null;
                    }
                }
            };
        // Remove any existing session timeout runnable object
        getHandlerUsingMainLooper().removeCallbacks(sessionTimeoutRunnable);
        getHandlerUsingMainLooper().postDelayed(sessionTimeoutRunnable,
                Constants.SESSION_LENGTH_MINS * 60 * 1000);
        Logger.logFine("Foreground activity gone to background");
    }


    private static HashSet<String> inappActivityExclude = null;

    /**
     * Notifies CleverTap that a new activity was created. Add a call to this method
     * in the onCreate() method of every activity.
     */
    private void notifyActivityChanged(Activity activity) {
        currentActivity = activity;
        activityCount++;

        //noinspection ConstantConditions
        if (activity != null) {
            Logger.logFine("Activity changed: " + activity.getLocalClassName());
        }

        if (inappActivityExclude == null) {
            inappActivityExclude = new HashSet<String>();
            try {
                String activities = ManifestMetaData.getMetaData(context, Constants.LABEL_INAPP_EXCLUDE);
                if (activities != null) {
                    String[] split = activities.split(",");
                    for (String a : split) {
                        inappActivityExclude.add(a.trim());
                    }
                }
            } catch (Throwable t) {
                // Ignore
            }
            Logger.log("In-app notifications will not be shown on " + Arrays.toString(inappActivityExclude.toArray()));
        }

        if (Constants.ENABLE_INAPP) {
            // We MUST loop through the set and do a contains on each and every entry there
            boolean activityBlacklisted = false;

            for (String blacklistedActivity : inappActivityExclude) {
                if (currentActivity != null && currentActivity.getLocalClassName().contains(blacklistedActivity)) {
                    activityBlacklisted = true;
                    break;
                }
            }


            if (!activityBlacklisted) {
                if (pendingInappRunnable != null) {
                    Logger.logFine("Found a pending inapp runnable. Scheduling it");
                    getHandlerUsingMainLooper().postDelayed(pendingInappRunnable, 200);
                    pendingInappRunnable = null;
                } else {
                    // Show an in-app notification, if available
                    InAppManager.showNotificationIfAvailable(context);
                }
            } else {
                Logger.log("In-app notifications will not be shown for this activity (" + activity.getLocalClassName() + ")");
            }
        }

        pushDailyEventsAsync();

        event.pushDeviceDetails();
        SessionManager.activityChanged(getCurrentActivityName());
    }

    /**
     * Tells CleverTap that the current Android activity was resumed. Add a call to this
     * method in the onResume() method of every activity.
     *
     * @param activity The calling activity
     */
    public void activityResumed(Activity activity) {
        setAppForeground(true);
        boolean newLaunch = currentActivity == null;
        // Check if this is a different activity from the last one
        if (currentActivity == null
                || !currentActivity.getLocalClassName().equals(activity.getLocalClassName()))
            notifyActivityChanged(activity);

        Logger.logFine("Background activity in foreground");

        // Start a lazy App Launched
        if (newLaunch) {
            getHandlerUsingMainLooper().postDelayed(new Runnable() {
                @Override
                public void run() {
                    // Raise the App Launched event
                    pushAppLaunchedEvent("delayed generic handler");
                }
            }, 500);
        }
    }

    private void updateAdvertisingID() {
        postAsyncSafely("CleverTapAPI#updateAdvertisingID", null, new Runnable() {
            @Override
            public void run() {
                try {
                    AdvertisingIdClient.Info adInfo;
                    adInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
                    final String id = adInfo.getId();
                    final boolean dontTrack = adInfo.isLimitAdTrackingEnabled();
                    HashMap<String, Object> m = new HashMap<String, Object>();
                    if (dontTrack) {
                        m.put("GoogleAdID", "");
                    } else if (id != null) {
                        m.put("GoogleAdID", id);
                    }
                    profile.push(m);
                } catch (Throwable t) {
                    Logger.logFine("Failed to update advertising ID", t);
                    StorageHelper.putString(context, Constants.ADVERTISER_ID, "");
                }
            }
        });
    }

    synchronized void pushAppLaunchedEvent(String source) {
        if (SessionManager.isAppLaunchedBeenPushed()) {
            Logger.logFine("App Launched has already been triggered. Will not trigger it; source = " + source);
            return;
        } else {
            Logger.logFine("Firing App Launched event; source = " + source);
        }
        SessionManager.setAppLaunchedBeenPushed(true);
        JSONObject event = new JSONObject();
        try {
            event.put("evtName", Constants.APP_LAUNCHED_EVENT);
            JSONObject evtData = new JSONObject();
            PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            evtData.put("Build", pInfo.versionCode + "");
            evtData.put("Version", pInfo.versionName);
            evtData.put("OS Version", android.os.Build.VERSION.RELEASE);
            evtData.put("SDK Version", BuildInfo.SDK_SVN_REVISION);

            // Respect disable location privacy
            boolean tryLocation = true;
            try {
                String value = ManifestMetaData.getMetaData(context, Constants.LABEL_PRIVACY_MODE);
                if (value.contains(Constants.PRIVACY_MODE_DISABLE_LOCATION)) {
                    tryLocation = false;
                }
            } catch (Throwable t) {
                // Okay cool, so the default behaviour is to capture Google account
            }


            if (tryLocation) {
                Location loc = getLocation();
                if (loc != null) {
                    evtData.put("Latitude", loc.getLatitude());
                    evtData.put("Longitude", loc.getLongitude());
                } else {
                    Logger.logFine("Location object is null");
                }
            }


            try {
                // Device data
                String make = Build.MANUFACTURER;
                String model = Build.MODEL;
                model = model.replace(make, "");
                evtData.put("Make", make.trim());
                evtData.put("Model", model.trim());
            } catch (Throwable t) {
                // Ignore
            }
            event.put("evtData", evtData);
        } catch (Throwable t) {
            // We won't get here
        }
        QueueManager.addToQueue(context, event, Constants.RAISED_EVENT);
    }

    /**
     * Sends all the events in the event queue.
     */
    public void flush() {
        new CommsManager().drainQueueAsync(context);
    }

    /**
     * Check the device to make sure it has the Google Play Services APK.
     *
     * @return The result of the request
     */
    private boolean checkPlayServices() {
        int resultCode;
        try {
            resultCode = GooglePlayServicesUtil.isGooglePlayServicesAvailable(context);
        } catch (Throwable e) {
            resultCode = -1;
        }

        Logger.logFine("Google Play services availability: " +
                (resultCode == ConnectionResult.SUCCESS));
        return resultCode == ConnectionResult.SUCCESS;
    }

    /**
     * Initializes the GCM architecture built into the CleverTap SDK.
     * <p/>
     * If you'd like to handle the messages yourself, please refer our documentation.
     */
    void initPushHandling() {
        String senderID;
        try {
            senderID = ManifestMetaData.getMetaData(context, Constants.LABEL_SENDER_ID);
            if (senderID != null) {
                senderID = senderID.replace("id:", "");
            }
        } catch (CleverTapMetaDataNotFoundException e) {
            // Ignore
            return;
        } catch (Throwable t) {
            Logger.error("Failed to automatically register to GCM", t);
            return;
        }
        Logger.log("Requesting a GCM registration ID for project ID(s) - " + senderID);
        if (checkPlayServices() && getRegistrationId().equals("")) {
            Logger.logFine("Google Play services available, and no valid registration ID found. Initializing");
            doGcmRegistration(senderID);
        }
    }

    /**
     * Gets the current registration ID for application on GCM service.
     * <p/>
     * If the result is empty, the app needs to register.
     *
     * @return registration ID, or empty string if there is no existing
     * registration ID.
     */
    public String getRegistrationId() {
        final SharedPreferences prefs = getPreferences(context);
        String registrationId = prefs.getString(Constants.GCM_PROPERTY_REG_ID, "");
        if (registrationId.equals("")) {
            if (getDebugLevel() >= Constants.DEBUG_FINEST)
                Logger.logFine("GCM registration ID not found");
            return "";
        }
        // Check if app was updated; if so, it must clear the registration ID
        // since the existing regID is not guaranteed to work with the new
        // app version
        int registeredVersion = prefs.getInt(PROPERTY_APP_VERSION, Integer.MIN_VALUE);
        int currentVersion = getAppVersion();
        if (registeredVersion != currentVersion) {
            if (getDebugLevel() >= Constants.DEBUG_FINEST)
                Logger.logFine("App version changed");
            return "";
        }
        return registrationId;
    }

    /**
     * @return Application's {@code SharedPreferences}.
     */
    private SharedPreferences getPreferences(Context context) {
        return StorageHelper.getPreferences(context);
    }

    /**
     * @return Application's version code from the {@code PackageManager}.
     */
    private int getAppVersion() {
        try {
            PackageInfo packageInfo = context.getPackageManager()
                    .getPackageInfo(context.getPackageName(), 0);
            return packageInfo.versionCode;
        } catch (Throwable ignore) {
            // Should never happen
            return 0;
        }
    }

    /**
     * This is meant for use with regard to non-GCM related data.
     * For GCM, use {@code PROPERTY_APP_VERSION}.
     */
    private final String APP_VERSION_CODE_TAG = "avc";

    /**
     * Compares the current app version and the recorded app version.
     *
     * @return Whether the app version was changed - upgraded, or downgraded
     */
    private boolean hasAppVersionChanged() {
        int currentVer = getAppVersion();
        SharedPreferences prefs = getPreferences(context);
        int prefsVer = prefs.getInt(APP_VERSION_CODE_TAG, -1);
        return currentVer != prefsVer;
    }

    /**
     * Updates the stored app version code to the current one.
     */
    @SuppressLint("CommitPrefEdits")
    private void updateAppVersionInPrefs() {
        int currentVer = getAppVersion();
        SharedPreferences prefs = getPreferences(context);
        SharedPreferences.Editor editor = prefs.edit().putInt(APP_VERSION_CODE_TAG, currentVer);
        StorageHelper.persist(editor);
    }

    /**
     * Registers the application with the GCM servers asynchronously.
     * <p/>
     * Stores the registration ID and app versionCode in the application's
     * shared preferences.
     */
    private void doGcmRegistration(final String senderId) {
        postAsyncSafely("CleverTapAPI#doGcmRegistration", null, new Runnable() {
            @Override
            public void run() {
                try {
                    if (gcm == null) {
                        gcm = GoogleCloudMessaging.getInstance(context);
                    }
                    Logger.logFine("Trying to register against GCM servers");
                    String gcmRegId = gcm.register(senderId);
                    Logger.logFine("Registered successfully");

                    // Send the ID to CleverTap servers
                    data.pushGcmRegistrationId(gcmRegId, true);
                    pushedGcmId = true;

                    // Persist the regID - no need to register again.
                    storeRegistrationId(gcmRegId);
                } catch (Throwable ex) {
                    Logger.logFine("Exception while registering with GCM servers: " + ex.toString());
                    ex.printStackTrace();
                }
            }
        });
    }

    /**
     * Stores the registration ID and app versionCode in the application's
     * {@code SharedPreferences}.
     *
     * @param regId registration ID
     */
    @SuppressLint("CommitPrefEdits")
    private void storeRegistrationId(String regId) {
        final SharedPreferences prefs = getPreferences(context);
        int appVersion = getAppVersion();
        if (CleverTapAPI.getDebugLevel() > Constants.DEBUG_FINEST)
            Logger.logFine("Saving GCM registration ID on app version " + appVersion);
        SharedPreferences.Editor editor = prefs.edit();
        editor.putString(Constants.GCM_PROPERTY_REG_ID, regId);
        editor.putInt(PROPERTY_APP_VERSION, appVersion);
        StorageHelper.persist(editor);
    }

    String getCurrentActivityName() {
        if (currentActivity != null) {
            return currentActivity.getLocalClassName();
        }
        return null;
    }

    /**
     * Checks whethor this notification is from CleverTap.
     *
     * @param extras The payload from the GCM intent
     * @return See {@link NotificationInfo}
     */
    public static NotificationInfo getNotificationInfo(final Bundle extras) {
        if (extras == null) return new NotificationInfo(false, false);

        boolean fromCleverTap = extras.containsKey(NOTIFICATION_TAG);
        boolean shouldRender = fromCleverTap && extras.containsKey("nm");
        return new NotificationInfo(fromCleverTap, shouldRender);
    }

    /**
     * Launches an asynchronous task to download the notification icon from CleverTap,
     * and create the Android notification.
     * <p/>
     * If your app is using CleverTap SDK's built in GCM message handling,
     * this method does not need to be called explicitly.
     * <p/>
     * Use this method when implementing your own GCM handling mechanism. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param context A reference to an Android context
     * @param extras  The {@link Bundle} object received by the broadcast receiver
     */
    public static void createNotification(final Context context, final Bundle extras) {
        //noinspection ConstantConditions
        if (extras == null || extras.get(NOTIFICATION_TAG) == null) {
            return;
        }

        try {
            postAsyncSafely("CleverTapAPI#createNotification", context, new Runnable() {
                @Override
                public void run() {
                    try {
                        // Check if this is a test notification
                        if (extras.containsKey(Constants.DEBUG_KEY)
                                && "y".equals(extras.getString(Constants.DEBUG_KEY))) {
                            int r = (int) (Math.random() * 10);
                            if (r != 8) {
                                // Discard acknowledging this notif
                                return;
                            }
                            JSONObject event = new JSONObject();
                            try {
                                JSONObject actions = new JSONObject();
                                for (String x : extras.keySet()) {
                                    Object value = extras.get(x);
                                    actions.put(x, value);
                                }
                                event.put("evtName", "wzrk_d");
                                event.put("evtData", actions);
                                QueueManager.addToQueue(context, event, Constants.RAISED_EVENT);
                            } catch (JSONException ignored) {
                                // Won't happen
                            }
                            // Drop further processing
                            return;
                        }
                        String notifTitle = extras.getString("nt");
                        // If not present, set it to the app name
                        notifTitle = (notifTitle != null) ? notifTitle : context.getApplicationInfo().name;
                        String notifMessage = extras.getString("nm");
                        if (notifMessage == null) {
                            // What else is there to show then?
                            return;
                        }
                        String icoPath = extras.getString("ico");
                        Intent launchIntent;

                        if (extras.containsKey(Constants.DEEP_LINK_KEY)) {
                            launchIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(extras.getString(Constants.DEEP_LINK_KEY)));
                        } else {
                            launchIntent = context.getPackageManager().getLaunchIntentForPackage(context.getPackageName());
                        }

                        PendingIntent pIntent;

                        // Take all the properties from the notif and add it to the intent
                        launchIntent.putExtras(extras);
                        launchIntent.putExtra(Constants.NOTIFICATION_RECEIVED_EPOCH_TAG, System.currentTimeMillis());
                        launchIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP);
                        pIntent = PendingIntent.getActivity(context, (int) System.currentTimeMillis(),
                                launchIntent, PendingIntent.FLAG_UPDATE_CURRENT);

                        NotificationCompat.Style style;
                        String bigPictureUrl = extras.getString("wzrk_bp");
                        if (bigPictureUrl != null && bigPictureUrl.startsWith("http")) {
                            try {
                                Bitmap bpMap = Utils.getNotificationBitmap(bigPictureUrl, false, context);

                                if (bpMap == null) throw new Exception("Failed to fetch big picture!");

                                style = new NotificationCompat.BigPictureStyle()
                                        .setSummaryText(notifMessage)
                                        .bigPicture(bpMap);
                            } catch (Throwable t) {
                                style = new NotificationCompat.BigTextStyle()
                                        .bigText(notifMessage);
                                Logger.error("Falling back to big text notification, couldn't fetch big picture", t);
                            }
                        } else {
                            style = new NotificationCompat.BigTextStyle()
                                    .bigText(notifMessage);
                        }

                        int smallIcon;
                        try {
                            String x = ManifestMetaData.getMetaData(context, Constants.LABEL_NOTIFICATION_ICON);
                            smallIcon = context.getResources().getIdentifier(x, "drawable", context.getPackageName());
                            if (smallIcon == 0) throw new IllegalArgumentException();
                        } catch (Throwable t) {
                            smallIcon = DeviceInfo.getAppIconAsIntId(context);
                        }

                        NotificationCompat.Builder nb = new NotificationCompat.Builder(context)
                                .setContentTitle(notifTitle)
                                .setContentText(notifMessage)
                                .setLargeIcon(Utils.getNotificationBitmap(icoPath, true, context))
                                .setContentIntent(pIntent)
                                .setAutoCancel(true)
                                .setStyle(style)
                                .setSmallIcon(smallIcon);

                        try {
                            if (extras.containsKey("wzrk_sound")) {
                                Object o = extras.get("wzrk_sound");
                                if ((o instanceof String && o.equals("true")) || (o instanceof Boolean && (Boolean) o)) {
                                    Uri defaultSoundUri = RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION);
                                    nb.setSound(defaultSoundUri);
                                }
                            }
                        } catch (Throwable t) {
                            Logger.error("Could not process sound parameter", t);
                        }

                        Notification n = nb.build();

                        NotificationManager notificationManager =
                                (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);

                        notificationManager.notify((int) (Math.random() * 100), n);
                    } catch (Throwable t) {
                        // Occurs if the notification image was null
                        // Let's return, as we couldn't get a handle on the app's icon
                        // Some devices throw a PackageManager* exception too
                        Logger.error("Couldn't render notification!", t);
                    }
                }
            });
        } catch (Throwable t) {
            Logger.error("Failed to process GCM notification", t);
        }
    }

    /**
     * This method is used internally.
     */
    public void pushDeepLink(Uri uri) {
        pushDeepLink(uri, false);
    }

    synchronized void pushDeepLink(Uri uri, boolean install) {
        if (uri == null)
            return;

        try {
            JSONObject referrer = InstallReferrerBroadcastReceiver.getUrchinFromUri(context, uri);
            referrer.put("referrer", uri.toString());
            if (install) {
                referrer.put("install", true);
            }
            event.pushDeviceDetailsWithExtras(referrer);
        } catch (Throwable t) {
            Logger.error("Failed to push deep link", t);
        }
    }

    public synchronized void pushInstallReferrer(String source, String medium, String campaign) {
        if (source == null && medium == null && campaign == null) return;
        try {
            // If already pushed, don't send it again
            int status = StorageHelper.getInt(context, "app_install_status", 0);
            if (status != 0) {
                Logger.log("Install referrer has already been set. Will not override it");
                return;
            }
            StorageHelper.putInt(context, "app_install_status", 1);

            if (source != null) source = Uri.encode(source);
            if (medium != null) medium = Uri.encode(medium);
            if (campaign != null) campaign = Uri.encode(campaign);

            String uriStr = "wzrk://track?install=true";
            if (source != null) uriStr += "&utm_source=" + source;
            if (medium != null) uriStr += "&utm_medium=" + medium;
            if (campaign != null) uriStr += "&utm_campaign=" + campaign;

            Uri uri = Uri.parse(uriStr);
            pushDeepLink(uri, true);
        } catch (Throwable t) {
            Logger.error("Failed to push install referrer", t);
        }
    }

    private static final HashMap<String, Integer> installReferrerMap = new HashMap<String, Integer>(8);

    public void enablePersonalization() {
        LocalDataStore.setPersonalisationEnabled(context, true);
    }

    public void disablePersonalization() {
        LocalDataStore.setPersonalisationEnabled(context, false);
    }

    public void pushInstallReferrer(Intent intent) {
        try {
            final Bundle extras = intent.getExtras();
            // Preliminary checks
            if (extras == null || !extras.containsKey("referrer")) {
                return;
            }
            final String url;
            try {
                url = URLDecoder.decode(extras.getString("referrer"), "UTF-8");

                Logger.logFine("Referrer received: " + url);
            } catch (Throwable e) {
                // Could not decode
                return;
            }
            if (url == null) {
                return;
            }
            int now = (int) (System.currentTimeMillis() / 1000);

            if (installReferrerMap.containsKey(url) && now - installReferrerMap.get(url) < 10) {
                Logger.logFine("Skipping install referrer due to duplicate within 10 seconds");
                return;
            }

            installReferrerMap.put(url, now);

            Uri uri = Uri.parse("wzrk://track?install=true&" + url);

            pushDeepLink(uri, true);
        } catch (Throwable t) {
            // Weird
        }
    }

    private Location getLocation() {
        try {
            LocationManager lm = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);
            List<String> providers = lm.getProviders(true);
            Location bestLocation = null;
            for (String provider : providers) {
                Location l = lm.getLastKnownLocation(provider);
                if (l == null) {
                    continue;
                }
                if (bestLocation == null || l.getAccuracy() < bestLocation.getAccuracy()) {
                    bestLocation = l;
                }
            }
            return bestLocation;
        } catch (Throwable t) {
            Logger.logFine("Couldn't get user's location", t);
            return null;
        }
    }

    Context getContext() {
        return context;
    }
}
