/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import org.json.JSONArray;

import java.io.UnsupportedEncodingException;
import java.util.BitSet;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

/**
 * Provides methods to validate various entities.
 */
final class Validator {
    private static final String[] eventNameCharsNotAllowed = {".", ":", "$", "'", "\"", "\\"};
    private static final String[] objectKeyCharsNotAllowed = {".", ":", "$", "'", "\"", "\\"};
    private static final String[] objectValueCharsNotAllowed = {"'", "\"", "\\"};
    private static final String[] restrictedNames = {"Stayed", "Notification Clicked",
            "Notification Viewed", "UTM Visited", "Notification Sent", "App Launched", "wzrk_d",
            "App Uninstalled", "Notification Bounced"};


    private static int getMaxNumberMultiValuePropertyValues() {
        return 100;
    }

    private static int getMaxMultiValuePropertyValueByteLength() {
        return 40;
    }

    public static final String ADD_VALUES_OPERATION = "multiValuePropertyAddValues";
    public static final String REMOVE_VALUES_OPERATION = "multiValuePropertyRemoveValues";

    /**
     * Cleans the event name to the following guidelines:
     * <p>
     * The following characters are removed:
     * dot, colon, dollar sign, single quote, double quote, and backslash.
     * Additionally, the event name is limited to 32 characters.
     * </p>
     *
     * @param name The event name to be cleaned
     * @return The {@link ValidationResult} object containing the object,
     * and the error code(if any)
     */
    static ValidationResult cleanEventName(String name) {
        ValidationResult vr = new ValidationResult();

        name = name.trim();
        for (String x : eventNameCharsNotAllowed)
            name = name.replace(x, "");

        if (name.length() > 32) {
            name = name.substring(0, 31);
            vr.setErrorDesc(name.trim() + "... exceeds the limit of 32 characters. Trimmed");
            vr.setErrorCode(510);
        }

        vr.setObject(name.trim());
        return vr;
    }


    /**
     * Cleans the object key.
     *
     * @param name Name of the object key
     * @return The {@link ValidationResult} object containing the object,
     * and the error code(if any)
     */
    static ValidationResult cleanObjectKey(String name) {
        ValidationResult vr = new ValidationResult();
        name = name.trim();
        for (String x : objectKeyCharsNotAllowed)
            name = name.replace(x, "");

        if (name.length() > 32) {
            name = name.substring(0, 31);
            vr.setErrorDesc(name.trim() + "... exceeds the limit of 32 characters. Trimmed");
            vr.setErrorCode(520);
        }

        vr.setObject(name.trim());

        return vr;
    }

    /**
     * Cleans a multi-value property key.
     *
     * @param name Name of the property key
     * @return The {@link ValidationResult} object containing the key,
     * and the error code(if any)
     * <p/>
     * First calls cleanObjectKey
     * Known property keys are reserved for multi-value properties, subsequent validation is done for those
     */
    static ValidationResult cleanMultiValuePropertyKey(String name) {
        ValidationResult vr = cleanObjectKey(name);

        name = (String) vr.getObject();

        // make sure its not a known property key (reserved in the case of multi-value)

        try {
            ProfileHandler.KnownFields kf = ProfileHandler.KnownFields.valueOf(name);
            if (kf != null) {
                vr.setErrorDesc(name + "... is a restricted key for multi-value properties. Operation aborted.");
                vr.setErrorCode(523);
                vr.setObject(null);
            }
        } catch (Throwable t) {
            //no-op
        }

        return vr;
    }

    /**
     * Cleans a multi-value property value.
     * <p/>
     * trims whitespace, forces lowercase
     * removes reserved characters
     * trims byte len to currently 40 bytes
     *
     * @param value the property value
     * @return The {@link ValidationResult} object containing the value,
     * and the error code(if any)
     */
    static ValidationResult cleanMultiValuePropertyValue(String value) {
        ValidationResult vr = new ValidationResult();

        // trim whitespace and force lowercase
        value = value.trim().toLowerCase();

        // remove reserved characters
        for (String x : objectValueCharsNotAllowed) {
            value = value.replace(x, "");
        }

        // check byte len
        int maxLen = getMaxMultiValuePropertyValueByteLength();
        try {
            if (value.getBytes("UTF-8").length > maxLen) {
                value = fastTrim(value, maxLen);
                vr.setErrorDesc(value + "... exceeds the limit of " + maxLen + " bytes. Trimmed");
                vr.setErrorCode(521);
            }
        } catch (UnsupportedEncodingException ignore) {
            // We really shouldn't get here
            // Ignore
        }

        vr.setObject(value);

        return vr;

    }

    /**
     * Merges a multi-value property JSONArray.
     * <p/>
     * trims to max length currently 100 items, on a FIFO basis
     * <p/>
     * please clean the key and newValues values before calling this
     *
     * @param currentValues current JSONArray property value
     * @param newValues     JSONArray of new values
     * @param action        String the action to take relative to the new values ($add, $remove)
     * @param key           String the property key
     * @return The {@link ValidationResult} object containing the merged value,
     * and the error code(if any)
     */
    static ValidationResult mergeMultiValuePropertyForKey(JSONArray currentValues, JSONArray newValues, String action, String key) {
        ValidationResult vr = new ValidationResult();
        Boolean remove = REMOVE_VALUES_OPERATION.equals(action);
        vr = _mergeListInternalForKey(key, currentValues, newValues, remove, vr);
        return vr;
    }

    /**
     * Cleans the object value, only if it is a string, otherwise, it simply returns the object.
     * <p/>
     * It also accepts a {@link java.util.Date} object, and converts it to a CleverTap
     * specific date format.
     * <p/>
     * In addition, if the object represents a number in a string format, it will convert
     * and return an Integer object.
     *
     * @param o Object to be cleaned(only if it is a string)
     * @return The cleaned object
     */
    static ValidationResult cleanObjectValue(Object o, boolean useOldDate, boolean autoConvert)
            throws IllegalArgumentException {
        ValidationResult vr = new ValidationResult();
        // If it's any type of number, send it back
        if (o instanceof Integer
                || o instanceof Float
                || o instanceof Boolean
                || o instanceof Double) {
            // No support for double
            if (o instanceof Double) {
                vr.setObject(((Double) o).floatValue());
            } else {
                vr.setObject(o);
            }
            return vr;
        } else if (o instanceof Long) {
            if (!autoConvert) {
                // This signifies that it's an event attribute - no long support here
                vr.setObject(o.toString());
            } else {
                vr.setObject(o);
            }
            return vr;
        } else if (o instanceof String || o instanceof Character) {
            String value;
            if (o instanceof Character)
                value = String.valueOf(o);
            else
                value = (String) o;
            value = value.trim();
            if (autoConvert) {
                // Try to convert the value to a long first, if not,
                // then treat it as a string and continue validation
                try {
                    // Return a Long object
                    Long i = Long.parseLong(value);
                    if (i.toString().equals(value)) {
                        // It's an integer alright
                        vr.setObject(i);
                        return vr;
                    }
                } catch (NumberFormatException e) {
                    // Okay, proceed
                }

                // Try to convert the value to an double first, if not,
                // then treat it as a string and continue validation
                try {
                    // Return a float object
                    Double d = Double.parseDouble(value);
                    // No support for double, hence float - loss of precision - OK
                    vr.setObject(d.floatValue());
                    return vr;
                } catch (NumberFormatException e) {
                    // Okay, proceed
                }

                // Try to convert the value to an boolean first, if not,
                // then treat it as a string and continue validation
                try {
                    // Return an Boolean object
                    if (value.equalsIgnoreCase("true"))
                        vr.setObject(true);
                    else if (value.equalsIgnoreCase("false"))
                        vr.setObject(false);
                    else
                        throw new Exception();
                    return vr;
                } catch (Exception e) {
                    // Okay, proceed
                }
            }

            for (String x : objectValueCharsNotAllowed) {
                value = value.replace(x, "");
            }

            try {
                if (value.getBytes("UTF-8").length > 120) {
                    value = fastTrim(value, 120);
                    vr.setErrorDesc(value.trim() + "... exceeds the limit of 120 bytes. Trimmed");
                    vr.setErrorCode(521);
                }
            } catch (UnsupportedEncodingException ignore) {
                // We really shouldn't get here
                // Ignore
            }
            vr.setObject(value.trim());
            return vr;
        } else if (o instanceof Date) {
            String date;
            if (useOldDate) {
                date = "$D_" + Constants.DOB_DATE_FORMAT.format(((Date) o));
            } else {
                date = "$D_" + ((Date) o).getTime() / 1000;
            }
            vr.setObject(date);
            return vr;
        } else {
            throw new IllegalArgumentException("Not a String, Boolean, Long, Integer, Float, Double, or Date");
        }
    }

    /**
     * Checks if the phone number is less than 8 characters.
     *
     * @param value The object(usually a {@link String} containing the phone number
     */
    static void validatePhone(String value) {
        try {
            if (value.length() >= 8) {
                return;
            }
        } catch (Throwable e) {
            // Ignore
        }
        Logger.error("Invalid phone number specified - " + value);
        throw new IllegalArgumentException("Invalid phone number");
    }

    /**
     * Checks whether the specified event name is restricted. If it is,
     * then create a pending error, and abort.
     *
     * @param name The event name
     * @return Boolean indication whether the event name is restricted
     */
    static boolean isRestrictedEventName(String name) {
        if (name == null) return false;
        for (String x : restrictedNames)
            if (name.equalsIgnoreCase(x)) {
                // The event name is restricted
                ValidationResult error = new ValidationResult();
                error.setErrorCode(513);
                error.setErrorDesc(name + " is a restricted event name. Last event aborted.");
                CleverTapAPI.pushValidationResult(error);
                Logger.error(name + " is a restricted system event name. Last event aborted.");
                return true;
            }
        return false;
    }

    static String fastTrim(String input, int byteLen) {
        try {
            byte[] data = input.getBytes("UTF-8");
            if (data.length <= byteLen) return input;

            int total = 0;
            //noinspection ForLoopReplaceableByForEach
            for (int i = 0; i < data.length; i++) {
                int charc = 0;
                if (data[i] >= 0) {
                    charc = 1;
                } else {
                    int mask = data[i] & 0xFF;
                    if (mask >> 4 == 0x0F) { /* 4 bytes */
                        charc = 4;
                    } else if (mask >> 5 == 0x07) { /* 3 bytes */
                        charc = 3;
                    } else if (mask >> 6 == 0x03) { /* 2 bytes */
                        charc = 2;
                    }
                }

                if (total + charc <= byteLen) {
                    total += charc;
                } else {
                    break;
                }
            }

            if (total == data.length) return input;

            byte[] out = new byte[total];
            System.arraycopy(data, 0, out, 0, out.length);

            return new String(out, "UTF-8");
        } catch (UnsupportedEncodingException ex) {
            return "";
        }
    }

    // multi-value list operations

    /**
     * scans right to left until max to maintain latest max values for the multi-value property specified by key.
     *
     * @param key    the property key
     * @param left   original list
     * @param right  new list
     * @param remove if remove new list from original
     * @param vr     ValidationResult for error and merged list return
     */
    private static ValidationResult _mergeListInternalForKey(String key, JSONArray left,
                                                             JSONArray right, boolean remove, ValidationResult vr) {

        if (left == null) {
            vr.setObject(null);
            return vr;
        }

        if (right == null) {
            vr.setObject(left);
            return vr;
        }

        int maxValNum = getMaxNumberMultiValuePropertyValues();

        JSONArray mergedList = new JSONArray();

        HashSet<String> set = new HashSet<String>();

        int lsize = left.length(), rsize = right.length();

        BitSet dupSetForAdd = null;

        if (!remove)
            dupSetForAdd = new BitSet(lsize + rsize);

        int lidx = 0;

        int ridx = scan(right, set, dupSetForAdd, lsize);

        if (!remove && set.size() < maxValNum) {
            lidx = scan(left, set, dupSetForAdd, 0);
        }

        for (int i = lidx; i < lsize; i++) {
            try {
                if (remove) {
                    String _j = (String) left.get(i);

                    if (!set.contains(_j)) {
                        mergedList.put(_j);
                    }
                } else if (!dupSetForAdd.get(i)) {
                    mergedList.put(left.get(i));
                }

            } catch (Throwable t) {
                //no-op
            }
        }

        if (!remove && mergedList.length() < maxValNum) {

            for (int i = ridx; i < rsize; i++) {

                try {
                    if (!dupSetForAdd.get(i + lsize)) {
                        mergedList.put(right.get(i));
                    }
                } catch (Throwable t) {
                    //no-op
                }
            }
        }

        // check to see if the list got trimmed in the merge
        if (ridx > 0 || lidx > 0) {
            vr.setErrorDesc("Multi value property for key " + key + " exceeds the limit of " + maxValNum + " items. Trimmed");
            vr.setErrorCode(521);
        }

        vr.setObject(mergedList);

        return vr;
    }


    private static int scan(JSONArray list, Set<String> set, BitSet dupSetForAdd, int off) {

        if (list != null) {

            int maxValNum = getMaxNumberMultiValuePropertyValues();

            for (int i = list.length() - 1; i >= 0; i--) {

                try {
                    Object obj = list.get(i);

                    String n = obj != null ? obj.toString() : null;

                    if (dupSetForAdd == null) { /* remove */
                        if (n != null) set.add(n);
                    } else {
                        if (n == null || set.contains(n)) {
                            dupSetForAdd.set(i + off, true);
                        } else {
                            set.add(n);

                            if (set.size() == maxValNum) {
                                return i;
                            }
                        }
                    }

                } catch (Throwable t) {
                    // no-op
                }
            }
        }

        return 0;
    }
}
