/*
 *
 *    Copyright 2014 Citrus Payment Solutions Pvt. Ltd.
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 * /
 */

package com.citrus.sdk.payment;

import android.text.TextUtils;

import com.citrus.sdk.classes.PGHealth;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import static com.citrus.sdk.payment.CardOption.CardScheme;

/**
 * Created by salil on 12/5/15.
 */
public class MerchantPaymentOption {

    private Set<CardScheme> creditCardSchemeSet = null;
    private Set<CardScheme> debitCardSchemeSet = null;
    private ArrayList<NetbankingOption> netbankingOptionList = null;
    private Set<String> bankCIDSet = null;

    private MerchantPaymentOption(Set<CardScheme> creditCardSchemeSet,
                                  Set<CardScheme> debitCardSchemeSet,
                                  ArrayList<NetbankingOption> netbankingOptionList,
                                  Set<String> bankCIDSet) {
        this.creditCardSchemeSet = creditCardSchemeSet;
        this.debitCardSchemeSet = debitCardSchemeSet;
        this.netbankingOptionList = netbankingOptionList;
        this.bankCIDSet = bankCIDSet;
    }

    public Set<CardScheme> getCreditCardSchemeSet() {
        return creditCardSchemeSet;
    }

    public Set<CardScheme> getDebitCardSchemeSet() {
        return debitCardSchemeSet;
    }

    public ArrayList<NetbankingOption> getNetbankingOptionList() {
        return netbankingOptionList;
    }

    public Set<String> getBankCIDSet() {
        return bankCIDSet;
    }

    public static MerchantPaymentOption getMerchantPaymentOptions(JsonObject merchantPaymentOptionsObj) {
        return getMerchantPaymentOptions(merchantPaymentOptionsObj, null);
    }

    public static MerchantPaymentOption getMerchantPaymentOptions(JsonObject merchantPaymentOptionsObj, Map<String, PGHealth> pgHealthMap) {
        MerchantPaymentOption merchantPaymentOption;
        Set<CardScheme> debitCardSchemeSet = null;
        Set<CardScheme> creditCardSchemeSet = null;
        ArrayList<NetbankingOption> netbankingOptionList = null;
        Set<String> bankCIDSet = null;

        JsonArray bankArray = merchantPaymentOptionsObj.getAsJsonArray("netBanking");
        JsonArray creditCardArray = merchantPaymentOptionsObj.getAsJsonArray("creditCard");
        JsonArray debitCardArray = merchantPaymentOptionsObj.getAsJsonArray("debitCard");

        int size = -1;
        // Parse credit card scheme
        size = creditCardArray.size();
        for (int i = 0; i < size; i++) {
            JsonElement element = creditCardArray.get(i);
            String cardScheme = element.getAsString();

            if (creditCardSchemeSet == null) {
                creditCardSchemeSet = new HashSet<>();
            }

            if (CardOption.CardScheme.getCardScheme(cardScheme) != null) {
                creditCardSchemeSet.add(CardOption.CardScheme.getCardScheme(cardScheme));
            }
        }

        // Parse debit card scheme
        size = debitCardArray.size();
        for (int i = 0; i < size; i++) {
            JsonElement element = debitCardArray.get(i);
            String cardScheme = element.getAsString();

            if (debitCardSchemeSet == null) {
                debitCardSchemeSet = new HashSet<>();
            }

            if (CardOption.CardScheme.getCardScheme(cardScheme) != null) {
                debitCardSchemeSet.add(CardOption.CardScheme.getCardScheme(cardScheme));
            }
        }

        // Parse netbanking options
        size = bankArray.size();
        for (int i = 0; i < size; i++) {
            JsonElement element = bankArray.get(i);
            if (element.isJsonObject()) {
                JsonObject bankOption = element.getAsJsonObject();

                element = bankOption.get("bankName");
                String bankName = element.getAsString();

                element = bankOption.get("issuerCode");
                String issuerCode = element.getAsString();

                if (!TextUtils.isEmpty(bankName) && !TextUtils.isEmpty(issuerCode)) {
                    NetbankingOption netbankingOption = new NetbankingOption(bankName, issuerCode);

                    if (pgHealthMap != null) {
                        netbankingOption.setPgHealth(pgHealthMap.get(issuerCode));
                    }

                    if (netbankingOptionList == null) {
                        netbankingOptionList = new ArrayList<>();
                    }

                    if (bankCIDSet == null) {
                        bankCIDSet = new HashSet<>();
                    }

                    netbankingOptionList.add(netbankingOption);
                    bankCIDSet.add(netbankingOption.getBankCID());
                }
            }
        }

        merchantPaymentOption = new MerchantPaymentOption(creditCardSchemeSet, debitCardSchemeSet, netbankingOptionList, bankCIDSet);

        return merchantPaymentOption;
    }


    public static MerchantPaymentOption getMerchantPaymentOptions(JSONObject merchantPaymentOptionsObj) {
        return getMerchantPaymentOptions(merchantPaymentOptionsObj, null);
    }

    public static MerchantPaymentOption getMerchantPaymentOptions(JSONObject merchantPaymentOptionsObj, Map<String, PGHealth> pgHealthMap) {
        MerchantPaymentOption merchantPaymentOption = null ;
        try {
            Set<CardScheme> debitCardSchemeSet = null;
            Set<CardScheme> creditCardSchemeSet = null;
            ArrayList<NetbankingOption> netbankingOptionList = null;
            Set<String> bankCIDSet = null;

            JSONArray bankArray         = merchantPaymentOptionsObj.getJSONArray("netBanking");
            JSONArray creditCardArray   = merchantPaymentOptionsObj.getJSONArray("creditCard");
            JSONArray debitCardArray    = merchantPaymentOptionsObj.getJSONArray("debitCard");

            int size = -1;
            // Parse credit card scheme
            size = creditCardArray.length();

            try {
                for (int i = 0; i < size; i++) {
                    // JSONObject element = creditCardArray.getString(i);
                    final String cardScheme = creditCardArray.getString(i);

                    if (creditCardSchemeSet == null) {
                        creditCardSchemeSet = new HashSet<>();
                    }

                    if (CardScheme.getCardScheme(cardScheme) != null) {
                        creditCardSchemeSet.add(CardScheme.getCardScheme(cardScheme));
                    }
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }

            // Parse debit card scheme
            try {
                size = debitCardArray.length();
                for (int i = 0; i < size; i++) {
                    //JSONObject element = debitCardArray.getJSONObject(i);
                    // String cardScheme = element.toString();
                    final String cardScheme = debitCardArray.getString(i);

                    if (debitCardSchemeSet == null) {
                        debitCardSchemeSet = new HashSet<>();
                    }

                    if (CardScheme.getCardScheme(cardScheme) != null) {
                        debitCardSchemeSet.add(CardScheme.getCardScheme(cardScheme));
                    }
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }

            // Parse netbanking options
            try {
                size = bankArray.length();
                for (int i = 0; i < size; i++) {
                    JSONObject bankOption = bankArray.getJSONObject(i);
                    // if (element.isJsonObject())
                    {
                        // JsonObject bankOption = element.getAsJsonObject();
                        // JSONObject bankOption = element ;

                        // JSONObject element = bankOption.getJSONObject( "bankName" );
                        final String bankName = bankOption.getString( "bankName" );


                        // element = bankOption.getJSONObject( "issuerCode" );
                        final String issuerCode = bankOption.getString( "issuerCode" );

                        if (!TextUtils.isEmpty(bankName) && !TextUtils.isEmpty(issuerCode)) {
                            NetbankingOption netbankingOption = new NetbankingOption(bankName, issuerCode);

                            if (pgHealthMap != null) {
                                netbankingOption.setPgHealth(pgHealthMap.get(issuerCode));
                            }

                            if (netbankingOptionList == null) {
                                netbankingOptionList = new ArrayList<>();
                            }

                            if (bankCIDSet == null) {
                                bankCIDSet = new HashSet<>();
                            }

                            netbankingOptionList.add(netbankingOption);
                            bankCIDSet.add(netbankingOption.getBankCID());
                        }
                    }
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }

            merchantPaymentOption = new MerchantPaymentOption(creditCardSchemeSet, debitCardSchemeSet, netbankingOptionList, bankCIDSet);
        } catch ( Exception e) {
            e.printStackTrace();
            // merchantPaymentOption = null ;
        }

        return merchantPaymentOption;
    }

    @Override
    public String toString() {
        return "MerchantPaymentOption{" +
                "creditCardSchemeSet=" + creditCardSchemeSet +
                ", debitCardSchemeSet=" + debitCardSchemeSet +
                ", netbankingOptionList=" + netbankingOptionList +
                '}';
    }
}
