/*
   Copyright 2014 Citrus Payment Solutions Pvt. Ltd.
   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at
     http://www.apache.org/licenses/LICENSE-2.0
   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package com.citrus.sdk.payment;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.os.Parcel;
import android.text.TextUtils;

import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.classes.Month;
import com.citrus.sdk.classes.PGHealth;
import com.citrus.sdk.classes.Utils;
import com.citrus.sdk.classes.Year;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by salil on 13/2/15.
 */
public abstract class CardOption extends PaymentOption {

    protected String cardHolderName = null;
    protected String cardNumber = null;
    protected String cardCVV = null;
    protected String cardExpiry = null;
    protected String cardExpiryMonth = null;
    protected String cardExpiryYear = null;
    protected CardScheme cardScheme = null;
    protected String nickName = null;

    /**
     * Use this for wallet PG Payment
     *
     * @param transactionAmount
     * @param cardHolderName
     * @param cardNumber
     * @param cardCVV
     * @param cardExpiryMonth
     * @param cardExpiryYear
     */
    CardOption(Amount transactionAmount, String cardHolderName, String cardNumber, String cardCVV, Month cardExpiryMonth, Year cardExpiryYear) {
        this(cardHolderName, cardNumber, cardCVV, cardExpiryMonth, cardExpiryYear);
        this.transactionAmount = transactionAmount;
    }

    /**
     * @param cardHolderName  - Name of the card holder.
     * @param cardNumber      - Card number.
     * @param cardCVV         - CVV of the card. We do not store CVV at our end.
     * @param cardExpiryMonth - Card Expiry Month 01 to 12 e.g. 01 for January.
     * @param cardExpiryYear  - Card Expiry Year in the form of YYYY e.g. 2015.
     */
    CardOption(String cardHolderName, String cardNumber, String cardCVV, Month cardExpiryMonth, Year cardExpiryYear) {
        if (!TextUtils.isEmpty(cardHolderName)) {
            this.cardHolderName = Utils.removeSpecialCharacters(cardHolderName);
        } else {
            this.cardHolderName = "Card Holder Name";
        }

        this.cardNumber = normalizeCardNumber(cardNumber);
        this.cardCVV = cardCVV;
        this.cardScheme = CardScheme.getCardSchemeUsingNumber(cardNumber);

        if (cardExpiryMonth != null) {
            this.cardExpiryMonth = cardExpiryMonth.toString();
        }
        if (cardExpiryYear != null) {
            this.cardExpiryYear = cardExpiryYear.toString();
        }

        if (!TextUtils.isEmpty(this.cardExpiryMonth) && !TextUtils.isEmpty(this.cardExpiryYear)) {
            this.cardExpiry = cardExpiryMonth + "/" + cardExpiryYear;
        }
    }

    /**
     * @param token
     * @param cardCVV
     * @deprecated use {@link CardOption#setCardCVV(String)} instead
     */
    CardOption(String token, String cardCVV) {
        this.token = token;
        this.cardCVV = cardCVV;
    }

    /**
     * @param cardNumber
     * @param cardScheme
     * @deprecated use {@link CardOption#setCardScheme(CardScheme)} instead.
     */
    CardOption(String cardNumber, CardScheme cardScheme) {
        this.cardNumber = cardNumber;
        this.cardScheme = cardScheme;
    }

    CardOption() {
    }

    /**
     * Get the type of the card, i.e. DEBIT or CREDIT.
     *
     * @return type of the card in string, i.e. debit or credit.
     */
    public abstract String getCardType();

    public String getCardHolderName() {
        return cardHolderName;
    }

    public String getNickName() {
        String name = getName();

        if ("---".equalsIgnoreCase(name)) {
            name = String.format((this instanceof CreditCardOption ? "Credit Card (%s)" : "Debit Card (%s)"), getLast4Digits());
        }

        return name;
        // return getName();
    }

    public String getCardExpiryYear() {
        return cardExpiryYear;
    }

    public String getCardExpiry() {
        return cardExpiry;
    }

    public String getCardExpiryMonth() {

        return cardExpiryMonth;
    }

    public String getCardCVV() {

        return cardCVV;
    }

    public void setCardCVV(String cardCVV) {
        this.cardCVV = cardCVV;
    }

    public String getCardNumber() {
        return cardNumber;
    }

    public CardScheme getCardScheme() {
        return cardScheme;
    }

    public void setCardScheme(CardScheme cardScheme) {
        this.cardScheme = cardScheme;
    }

    public void setNickName(String nickName) {
        this.nickName = Utils.removeSpecialCharacters(nickName);
    }

    @Override
    public PGHealth getPgHealth() {
        // Currently PGHealth for card schemes is not supported. Hence returning GOOD everytime.
        return PGHealth.GOOD;
    }

    private static String normalizeCardNumber(String number) {
        if (number == null) {
            return null;
        }
        return number.trim().replaceAll("\\s+|-", "");
    }

    public String getLast4Digits() {
        String last4Digits = null;

        if (!TextUtils.isEmpty(cardNumber)) {
            int length = cardNumber.length();

            if (length - 4 > 0) {
                last4Digits = cardNumber.substring(length - 4, length);
            }
        }

        return last4Digits;
    }

    @Override
    public Drawable getOptionIcon(Context context) {
        // Return the icon depending upon the scheme of the card.
        Drawable drawable = null;

        int resourceId = 0;
        if (cardScheme == CardScheme.VISA) {
            resourceId = context.getResources().getIdentifier("visa", "drawable", context.getPackageName());
        } else if (cardScheme == CardScheme.MASTER_CARD) {
            resourceId = context.getResources().getIdentifier("mcrd", "drawable", context.getPackageName());
        } else if (cardScheme == CardScheme.MAESTRO) {
            resourceId = context.getResources().getIdentifier("mtro", "drawable", context.getPackageName());
        } else if (cardScheme == CardScheme.DINERS) {
            resourceId = context.getResources().getIdentifier("dinerclub", "drawable", context.getPackageName());
        } else if (cardScheme == CardScheme.JCB) {
            resourceId = context.getResources().getIdentifier("jcb", "drawable", context.getPackageName());
        } else if (cardScheme == CardScheme.AMEX) {
            resourceId = context.getResources().getIdentifier("amex", "drawable", context.getPackageName());
        } else if (cardScheme == CardScheme.RPAY) {
            resourceId = context.getResources().getIdentifier("rupay", "drawable", context.getPackageName());
            if (resourceId == 0) {
                resourceId = context.getResources().getIdentifier("rpay", "drawable", context.getPackageName());
            }
        } else if (cardScheme == CardScheme.DISCOVER) {
            resourceId = context.getResources().getIdentifier("discover", "drawable", context.getPackageName());
        }

        if (resourceId == 0) {
            if ((resourceId = context.getResources().getIdentifier("default_card", "drawable", context.getPackageName())) != 0) {
                drawable = context.getResources().getDrawable(resourceId);
            }
        } else {
            drawable = context.getResources().getDrawable(resourceId);
        }

        return drawable;
    }

    @Override
    public String toString() {
        return super.toString() + "CardOption{" +
                "cardHolderName='" + cardHolderName + '\'' +
                ", cardNumber='" + cardNumber + '\'' +
                ", cardCVV='" + cardCVV + '\'' +
                ", cardExpiry='" + cardExpiry + '\'' +
                ", cardExpiryMonth='" + cardExpiryMonth + '\'' +
                ", cardExpiryYear='" + cardExpiryYear + '\'' +
                ", cardScheme='" + cardScheme + '\'' +
                '}';
    }

    /**
     * Returns the no of digits of CVV for that particular card.
     * <p/>
     * Only AMEX has 4 digit CVV, else all cards have 3 digit CVV.
     *
     * @return return 3 or 4 depending upon the card scheme.
     */
    public int getCVVLength() {
        int cvvLength = 3;

        if (cardScheme == CardScheme.AMEX) {
            cvvLength = 4;
        }

        return cvvLength;
    }

    public boolean validateCard() {
        // For tokenized payments, check for cvv validity.
        if (!TextUtils.isEmpty(token)) {
            // Check for cvv if the card is not maestro
            if (cardScheme != CardScheme.MAESTRO) {
                return validateCVV();
            }
        }

        // In case of maestro card, cvv and expiry is empty.
        if (cardScheme == CardScheme.MAESTRO) {

            // If token is empty, then check for card number.
            if (TextUtils.isEmpty(token)) {
                // If for maestro card cvv is provided then it should be only 3 digits and also check for valid card number.
                if (!TextUtils.isEmpty(cardCVV)) {
                    return (cardCVV.length() == 3) && validateCardNumber();
                }
                // Since it is not saved card, then check for card number. And since cvv is empty, skip cvv validation.
                return validateCardNumber();
            } else {
                // Since this is saved card and cvv is given then check cvv validity.
                if (!TextUtils.isEmpty(cardCVV)) {
                    // If for maestro card cvv is provided then it should be only 3 digits and also check for valid card number.
                    return cardCVV.length() == 3;
                }

                // If card token for maestro card and cvv is not given so no validation is required.
                return true;
            }
        }

        // In case of cards other than maestro check for card number, validity and cvv.
        return validateCardNumber() && validateExpiryDate() && validateCVV();
    }

    /**
     * Use this method to check for card validity while saving the card.
     * <p/>
     * While saving the card, cvv is not stored and is optional. So do not check for cvv while validating the card.
     *
     * @return
     */
    public boolean validateForSaveCard() {
        if (cardScheme == CardScheme.MAESTRO) {
            return validateCardNumber();
        }

        return validateCardNumber() && validateExpiryDate();
    }

    public boolean validateCardNumber() {
        if (TextUtils.isEmpty(cardNumber)) {
            return false;
        }

        if (TextUtils.isEmpty(cardNumber) || !TextUtils.isDigitsOnly(cardNumber) || !isValidLuhnNumber(cardNumber)) {
            return false;
        }

        // Check for length of card number.
        if (cardScheme == CardScheme.AMEX) {
            if (cardNumber.length() != 15) {
                return false;
            }
        } else if (cardScheme == CardScheme.VISA) {
            // VISA cards length either 13 or 16.
            if (cardNumber.length() != 13 && cardNumber.length() != 16) {
                return false;
            }
        } else if (cardScheme != CardScheme.MAESTRO) {
            if (cardNumber.length() != 16) {
                return false;
            }
        } else {
            // MAESTRO will have length 12-19.
            if (cardNumber.length() < 12 || cardNumber.length() > 19) {
                return false;
            }
        }

        return true;
    }

    public boolean validateExpiryDate() {

        if (cardScheme == CardScheme.MAESTRO && TextUtils.isEmpty(cardExpiryMonth) && TextUtils.isEmpty(cardExpiryYear)) {
            return true;
        }

        if (!validateExpMonth()) {
            return false;
        }
        if (!validateExpYear()) {
            return false;
        }
        return !Utils.hasMonthPassed(Integer.valueOf(cardExpiryYear), Integer.valueOf(cardExpiryMonth));
    }

    private boolean validateExpMonth() {
        if (cardExpiryMonth == null) {
            return false;
        }
        return (Integer.valueOf(cardExpiryMonth) >= 1 && Integer.valueOf(cardExpiryMonth) <= 12);
    }

    private boolean validateExpYear() {
        if (cardExpiryYear == null) {
            return false;
        }
        return !Utils.hasYearPassed(Integer.valueOf(cardExpiryYear));
    }

    public boolean validateCVV() {
        if (TextUtils.isEmpty(cardCVV)) {
            return false;
        }

        // Check the length of the CVV
        if (cardScheme == CardScheme.AMEX) {
            // In case of AMEX, the cvv length is 4.
            return (cardCVV.length() == 4);
        } else {
            // If not AMEX, the CVV length should be 3.
            return (cardCVV.length() == 3);
        }
    }

    /**
     * Get reasons for the validity failure for save card only.
     * We do not need CVV while saving the card, hence bypass the check for cvv while checking validity as well reasons.
     *
     * @return
     */
    public String getCardValidityFailureReasonsForSaveCard() {
        String reason = null;
        if (!validateForSaveCard()) {
            StringBuilder builder = new StringBuilder();
            // Avoid this check in case of tokenized payment
            if (TextUtils.isEmpty(token) && !validateCardNumber()) {
                builder.append(" Invalid Card Number. ");
            }

            // Avoid this check in case of tokenized payment
            if (TextUtils.isEmpty(token) && !validateExpiryDate()) {
                builder.append(" Invalid Expiry Date. ");
            }

            reason = builder.toString();
        }

        return reason;
    }

    public String getCardValidityFailureReasons() {
        String reason = null;
        if (!validateCard()) {
            StringBuilder builder = new StringBuilder();
            // Avoid this check in case of tokenized payment
            if (TextUtils.isEmpty(token) && !validateCardNumber()) {
                builder.append(" Invalid Card Number. ");
            }

            // Avoid this check in case of tokenized payment
            if (TextUtils.isEmpty(token) && !validateExpiryDate()) {
                builder.append(" Invalid Expiry Date. ");
            }

            // If maestro card and cvv is not empty, then check for cvv, else skip.
            if (cardScheme == CardScheme.MAESTRO) {
                // If cvv is empty then do not check for cvv.
                if (!TextUtils.isEmpty(cardCVV)) {
                    if (!validateCVV()) {
                        builder.append(" Invalid CVV. ");
                    }
                }
            } else {
                // For cards other than maestro, validate cvv.
                if (!validateCVV()) {
                    builder.append(" Invalid CVV. ");
                }
            }

            reason = builder.toString();
        }

        return reason;
    }

    @Override
    public String getSavePaymentOptionObject() {
        JSONObject object = null;
        try {
            object = new JSONObject();
            JSONArray paymentOptions = new JSONArray();

            JSONObject option = new JSONObject();
            option.put("owner", cardHolderName);
            // Set nickname
            if (!TextUtils.isEmpty(nickName)) {
                option.put("name", nickName);
            }
            option.put("number", cardNumber);
            option.put("scheme", cardScheme.toString());
            if (TextUtils.isEmpty(cardExpiry) && cardScheme == CardScheme.MAESTRO) {
                option.put("expiryDate", "12/2049");
            } else {
                option.put("expiryDate", cardExpiry);
            }
            option.put("type", getCardType());
            paymentOptions.put(option);

            object.put("paymentOptions", paymentOptions);
            object.put("type", "payment");
        } catch (JSONException e) {
            e.printStackTrace();
        }

        return object.toString();
    }

    // @formatter:off

    /**
     * This will return the json object for moto or new make payment call.
     *
     * @return For Cards the format is
     * <p/>
     * <p/>
     * {
     * "id":"4e361d717bf26359f1b5ac6f33edda37",
     * "type":"paymentOptionIdToken"
     * "cvv":"223"
     * }
     * OR
     * {
     * "type":"paymentOptionToken",
     * "paymentMode":{
     * "cvv":"221",
     * "holder":"Card Holder Name",
     * "number":"4111111111111111",
     * "scheme":"VISA",
     * "type":"debit",
     * "expiry":"12\/2044"
     * }
     * }
     */
    // @formatter:on
    @Override
    public JSONObject getMOTOPaymentOptionObject() throws JSONException {
        JSONObject jsonObject = new JSONObject();

        if (isTokenizedPayment()) {
            jsonObject.put("type", "paymentOptionIdToken");
            jsonObject.put("id", token);

            // In case of MAESTRO card if cvv is empty put the dummy value.
            if (CardScheme.MAESTRO == cardScheme && TextUtils.isEmpty(cardCVV)) {
                jsonObject.put("cvv", "123");
            } else {
                jsonObject.put("cvv", cardCVV);
            }
        } else {
            jsonObject.put("type", "paymentOptionToken");
            // PaymentMode
            JSONObject paymentMode = new JSONObject();
            paymentMode.put("holder", cardHolderName);
            paymentMode.put("number", cardNumber);
            paymentMode.put("scheme", cardScheme.getName().toUpperCase());
            paymentMode.put("type", getCardType());

            // In case of MAESTRO card if cvv is empty put the dummy value.
            if (CardScheme.MAESTRO == cardScheme && TextUtils.isEmpty(cardCVV)) {
                paymentMode.put("cvv", "123");
            } else {
                paymentMode.put("cvv", cardCVV);
            }

            // In case of MAESTRO card if expiry is empty put the dummy value.
            if (CardScheme.MAESTRO == cardScheme && TextUtils.isEmpty(cardExpiry)) {
                paymentMode.put("expiry", "11/2049");
            } else {
                paymentMode.put("expiry", cardExpiry);
            }

            jsonObject.put("paymentMode", paymentMode);
        }

        return jsonObject;
    }

    @Override
    /**
     * This will return the json required for wallet charge api.
     * The json format is
     *
     * {
     "paymentMode": "CREDIT_CARD",
     "name": "Test",
     "cardNumber": "4028530052708001",
     "cardExpiryDate": "02/2017",
     "cardScheme": "VISA",
     "amount": "10.00",
     "currency": "INR",
     "cvv": "018"
     }
     */
    public JSONObject getWalletChargePaymentOptionObject() throws JSONException {
        JSONObject jsonObject = null;
        if (transactionAmount != null) {

            jsonObject = new JSONObject();
            if (this instanceof CreditCardOption) {
                jsonObject.put("paymentMode", "CREDIT_CARD");
            } else {
                jsonObject.put("paymentMode", "DEBIT_CARD");
            }

            if (isTokenizedPayment()) {
                jsonObject.put("name", "");
                jsonObject.put("cardNumber", "");
                jsonObject.put("cardExpiryDate", "");
                jsonObject.put("cardScheme", "");
                jsonObject.put("cvv", cardCVV);
                jsonObject.put("savedCardToken", token);
            } else {
                jsonObject.put("name", cardHolderName);
                jsonObject.put("cardNumber", cardNumber);
                jsonObject.put("cardScheme", cardScheme.getName().toUpperCase());

                // In case of MAESTRO card if expiry is empty put the dummy value.
                if (CardScheme.MAESTRO == cardScheme && TextUtils.isEmpty(cardExpiry)) {
                    jsonObject.put("cardExpiryDate", "11/2049");
                } else {
                    jsonObject.put("cardExpiryDate", cardExpiry);
                }

                // In case of MAESTRO card if cvv is empty put the dummy value.
                if (CardScheme.MAESTRO == cardScheme && TextUtils.isEmpty(cardCVV)) {
                    jsonObject.put("cvv", "123");
                } else {
                    jsonObject.put("cvv", cardCVV);
                }
            }

            jsonObject.put("amount", String.valueOf(transactionAmount.getValueAsDouble()));
            jsonObject.put("currency", transactionAmount.getCurrency());

        }
        return jsonObject;
    }

    private boolean isValidLuhnNumber(String number) {
        boolean isOdd = true;
        int sum = 0;

        for (int index = number.length() - 1; index >= 0; index--) {
            char c = number.charAt(index);
            if (!Character.isDigit(c)) {
                return false;
            }
            int digitInteger = Integer.parseInt("" + c);
            isOdd = !isOdd;

            if (isOdd) {
                digitInteger *= 2;
            }

            if (digitInteger > 9) {
                digitInteger -= 9;
            }

            sum += digitInteger;
        }

        return sum % 10 == 0;
    }

    /**
     * Denotes the type of the card. i.e. Credit or Debit.
     */
    public enum CardType {
        DEBIT {
            public String getCardType() {
                return "debit";
            }
        }, CREDIT {
            public String getCardType() {
                return "credit";
            }
        };

        /**
         * Get the type of the card in string, i.e. debit or credit.
         *
         * @return type of the card in string, i.e. debit or credit.
         */
        public abstract String getCardType();
    }

    public enum CardScheme {
        VISA("4") {
            public String getName() {
                return "visa";
            }
        }, MASTER_CARD("5") {
            public String getName() {
                return "mcrd";
            }
        }, MAESTRO("502260", "504433",
                "504434", "504435", "504437", "504645", "504681",
                "504753", "504775", "504809", "504817", "504834",
                "504848", "504884", "504973", "504993", "508125",
                "508126", "508159", "508192", "508227", "56",
                "600206", "603123", "603741", "603845", "622018",
                "67") {
            public String getName() {
                return "mtro";
            }
        }, DINERS("30", "36", "38", "39") {
            public String getName() {
                return "DINERS";
            }
        }, JCB("35") {
            public String getName() {
                return "jcb";
            }
        }, AMEX("34", "37") {
            public String getName() {
                return "amex";
            }
        }, RPAY("5085", "5086", "5087", "5088", "6069", "607", "6081", "6521", "6522", "6524") {
            public String getName() {
                return "RPAY";
            }
        },
        DISCOVER("60", "62", "64", "65") {
            public String getName() {
                return "DISCOVER";
            }
        },
        UNKNOWN("0") {
            public String getName() {
                return "UNKNOWN";
            }
        };

        private final String[] pattern;

        private CardScheme(String... pattern) {
            this.pattern = pattern;
        }

        public abstract String getName();

        public String getIconName() {
            return getName().toLowerCase();
        }

        public static CardScheme getCardScheme(String cardScheme) {
            if ("visa".equalsIgnoreCase(cardScheme)) {
                return VISA;
            } else if ("mcrd".equalsIgnoreCase(cardScheme) || "Master Card".equalsIgnoreCase(cardScheme)) {
                return MASTER_CARD;
            } else if ("mtro".equalsIgnoreCase(cardScheme) || "Maestro Card".equalsIgnoreCase(cardScheme)) {
                return MAESTRO;
            } else if ("DINERS".equalsIgnoreCase(cardScheme)) {
                return DINERS;
            } else if ("jcb".equalsIgnoreCase(cardScheme)) {
                return JCB;
            } else if ("amex".equalsIgnoreCase(cardScheme)) {
                return AMEX;
            } else if ("DISCOVER".equalsIgnoreCase(cardScheme)) {
                return DISCOVER;
            } else if ("RPAY".equalsIgnoreCase(cardScheme) || ("RuPay".equalsIgnoreCase(cardScheme))) {
                return RPAY;
            } else {
                return null;
            }
        }

        public static CardScheme getCardSchemeUsingNumber(String cardNumber) {

            CardScheme cardScheme = UNKNOWN;

            for (CardScheme scheme : values()) {
                // If the given card number matches the patterns of any of the card scheme, break and return the scheme.
                if (Utils.hasAnyPrefix(cardNumber, scheme.pattern)) {
                    cardScheme = scheme;

                    break;
                }
            }

            return cardScheme;
        }

        public static int getCVVLength(String cardNumber) {
            CardScheme scheme = getCardSchemeUsingNumber(cardNumber);

            if (scheme == AMEX) {
                return 4;
            } else {
                return 3;
            }
        }

        public static int getFilterLength(CardScheme scheme) {
            if (scheme == AMEX) {
                return 18;
            } else if (scheme == MAESTRO) {
                return 23;
            } else {
                return 19;
            }
        }
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        super.writeToParcel(dest, flags);
        dest.writeString(this.cardHolderName);
        dest.writeString(this.cardNumber);
        dest.writeString(this.cardCVV);
        dest.writeString(this.cardExpiry);
        dest.writeString(this.cardExpiryMonth);
        dest.writeString(this.cardExpiryYear);
        dest.writeInt(this.cardScheme == null ? -1 : this.cardScheme.ordinal());
        dest.writeString(this.nickName);
        dest.writeParcelable(this.transactionAmount, 0);
    }

    protected CardOption(Parcel in) {
        super(in);
        this.cardHolderName = in.readString();
        this.cardNumber = in.readString();
        this.cardCVV = in.readString();
        this.cardExpiry = in.readString();
        this.cardExpiryMonth = in.readString();
        this.cardExpiryYear = in.readString();
        int tmpCardScheme = in.readInt();
        this.cardScheme = tmpCardScheme == -1 ? null : CardScheme.values()[tmpCardScheme];
        this.nickName = in.readString();
        this.transactionAmount = in.readParcelable(Amount.class.getClassLoader());
    }

    public static final Creator<CardOption> CREATOR = new Creator<CardOption>() {
        public CardOption createFromParcel(Parcel source) {
            return null;
        }

        public CardOption[] newArray(int size) {
            return new CardOption[size];
        }
    };
}
