package com.citrus.sdk.network.request;

import android.content.Context;
import android.util.Log;

import com.android.volley.AuthFailureError;
import com.android.volley.DefaultRetryPolicy;
import com.android.volley.Request;
import com.android.volley.Response;
import com.android.volley.VolleyError;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.StringRequest;
import com.citrus.sdk.Callback;
import com.citrus.sdk.network.BaseApi;
import com.citrus.sdk.network.HttpMethod;

import org.json.JSONObject;

import java.lang.reflect.Type;
import java.util.Map;

/**
 * Created by vinay on 1/4/16.
 */
public class ApiExecutor {

    private static final String TAG = ApiExecutor.class.getSimpleName();

    private static final String CANCEL_TAG = "com.sdk.citrus.volley.TAG";

    private static ApiExecutor instance;
    private VolleyNetworkManager volleyNetworkManager;

    private ApiExecutor(final Context context) {
        volleyNetworkManager = VolleyNetworkManager.getInstance(context);
    }

    public static ApiExecutor getInstance(final Context context) {
        if (instance == null) {
            synchronized (ApiExecutor.class) {
                if (instance == null) {
                    instance = new ApiExecutor(context);
                }
            }
        }
        return instance;
    }

    public void executeJsonApi(final Context context, final BaseApi baseApi,
                               final ApiRequest request, final Callback callback) {
        Log.v(TAG, TAG + ".executeJsonApi(): request = " + request + ", baseApi = " + baseApi);

        final int method = getVolleyMethod(request.getHttpMethod());
        final String URL = getUrl(baseApi, request);

        JSONObject body = null;
        final RequestBody requestBody = request.getRequestBody();
        if (requestBody != null && requestBody.getRequestBodyType() == REQUEST_BODY_TYPE.JSON) {
            body = (JSONObject) request.getRequestBody().getBody();
        }

        Log.v(TAG, TAG + ".executeJsonApi(): body = " + body);

        final JsonObjectRequest jsonObjectRequest = new JsonObjectRequest(method, URL, body,
                new Response.Listener<JSONObject>() {
                    @Override
                    public void onResponse(JSONObject response) {
                        Log.v(TAG, TAG + ".executeJsonApi().onResponse(): response = " + response);
                        baseApi.sendResponse(callback, response);
                    }
                }, new Response.ErrorListener() {
            @Override
            public void onErrorResponse(VolleyError error) {
                Log.e(TAG, TAG + ".executeJsonApi().onErrorResponse(): error = " + error);
                baseApi.sendError(callback, error);
            }
        }) {
            @Override
            public Map<String, String> getHeaders() throws AuthFailureError {
                final Map<String, String> headers = request.getHeaders();
                Log.d(TAG, TAG + ".executeJsonApi().getHeaders(): headers = " + headers);

                if (headers == null || headers.isEmpty()) {
                    return super.getHeaders();
                }

                return headers;
            }
        };

        jsonObjectRequest.setTag(CANCEL_TAG);
        setDefaultRetryPolicy(jsonObjectRequest);

        volleyNetworkManager.addToRequestQueue(jsonObjectRequest);
    }


    public void executeCustomJsonApi(final BaseApi baseApi,
                                     final ApiRequest request, final Callback<JSONObject> callback) {

        Log.v(TAG, TAG);
        Log.v(TAG, TAG + ".executeCustomJsonApi(): request = " + request + ", baseApi = " + baseApi);

        final int method = getVolleyMethod(request.getHttpMethod());
        final String URL = getUrl(baseApi, request);
        final String body = getVolleyBodyStr(request);

        final CustomJsonRequest customRequest = new CustomJsonRequest(method, URL,
                request.getParams(), new Response.Listener<JSONObject>() {
            @Override
            public void onResponse(JSONObject response) {
                Log.v(TAG, TAG + ".executeCustomJsonApi().onResponse(): response = " + response);
                baseApi.sendResponse(callback, response);
            }
        }, new Response.ErrorListener() {
            @Override
            public void onErrorResponse(VolleyError error) {
                Log.e(TAG, TAG + ".executeCustomJsonApi().onErrorResponse(): error = " + error);
                baseApi.sendError(callback, error);
            }
        }, body, request.getHeaders(), request.getBodyContentType());

        customRequest.setTag(CANCEL_TAG);

        setDefaultRetryPolicy(customRequest);

        volleyNetworkManager.addToRequestQueue(customRequest);
    }


    public void executeStringApi(final BaseApi baseApi, final ApiRequest request, final Callback callback) {
        Log.v(TAG, TAG);
        Log.v(TAG, TAG + ".executeStringApi(): request = " + request + ", baseApi = " + baseApi);

        final int method = getVolleyMethod(request.getHttpMethod());
        final String URL = getUrl(baseApi, request);
        final byte[] body = getVolleyBody(request);

        final StringRequest stringRequest = new CustomStringRequest(method, URL, new Response.Listener<String>() {
            @Override
            public void onResponse(String response) {
                Log.v(TAG, TAG + ".executeStringApi().onResponse(): response = " + response);
                baseApi.sendResponse(callback, response);
            }
        }, new Response.ErrorListener() {
            @Override
            public void onErrorResponse(VolleyError error) {
                Log.e(TAG, TAG + ".executeStringApi().onErrorResponse(): error = " + error);
                baseApi.sendError(callback, error);
            }
        }, request.getParams(), body, request.getHeaders(), request.getBodyContentType());


        stringRequest.setTag(CANCEL_TAG);

        setDefaultRetryPolicy(stringRequest);

        volleyNetworkManager.addToRequestQueue(stringRequest);
    }


    public void executeCustomObjectApi(final BaseApi baseApi,
                                       final ApiRequest request, final Callback callback) {
        Log.v(TAG, TAG);
        Log.v(TAG, TAG + ".executeCustomObjectApi(): request = " + request + ", baseApi = " + baseApi);
        executeCustomObjectApi( baseApi , request, null, callback );
    }

    public void executeCustomObjectApi(final BaseApi baseApi,
                                       final ApiRequest request, Type type , final Callback callback) {
        Log.v(TAG, TAG);
        Log.v(TAG, TAG + ".executeCustomObjectApi(): request = " + request +", type = " + type+ ", baseApi = " + baseApi);

        final int method = getVolleyMethod(request.getHttpMethod());
        final String URL = getUrl(baseApi, request);

        final String body = getVolleyBodyStr(request);

        final GsonRequest gsonRequest ;
        if( type == null ){
            gsonRequest = new GsonRequest(URL, request.getClazz(), method,
                    request.getHeaders(), request.getParams(), request.getBodyContentType(), new Response.Listener() {
                @Override
                public void onResponse(Object response) {
                    Log.v(TAG, TAG + ".executeCustomObjectApi().onResponse(): response = " + response);
                    if (response != null) {
                        Log.v(TAG, TAG + ".executeCustomObjectApi().onResponse(): response class = " + response.getClass().getName());
                    }
                    baseApi.sendResponse(callback, response);
                }
            }, new Response.ErrorListener() {
                @Override
                public void onErrorResponse(VolleyError error) {
                    Log.e(TAG, TAG + ".executeCustomObjectApi().onResponse(): error = " + error);
                    baseApi.sendError(callback, error);
                }
            }, body);
        }
        else {
            gsonRequest = new GsonRequest(URL, type, method,
                    request.getHeaders(), request.getParams(), request.getBodyContentType(), new Response.Listener() {
                @Override
                public void onResponse(Object response) {
                    Log.v(TAG, TAG + ".executeCustomObjectApi().onResponse(): response = " + response);
                    if (response != null) {
                        Log.v(TAG, TAG + ".executeCustomObjectApi().onResponse(): response class = " + response.getClass().getName());
                    }
                    baseApi.sendResponse(callback, response);
                }
            }, new Response.ErrorListener() {
                @Override
                public void onErrorResponse(VolleyError error) {
                    Log.e(TAG, TAG + ".executeCustomObjectApi().onResponse(): error = " + error);
                    baseApi.sendError(callback, error);
                }
            }, body);
        }

        gsonRequest.setTag(CANCEL_TAG);

        setDefaultRetryPolicy(gsonRequest);

        volleyNetworkManager.addToRequestQueue(gsonRequest);
    }

    public void cancelAllApiRequests() {
        volleyNetworkManager.cancelRequests(CANCEL_TAG);
    }

    private static String getUrl(final BaseApi baseApi, final ApiRequest request) {
        final String relativeUrl = request.getRelativeUrl();
        if (relativeUrl == null || relativeUrl.isEmpty()) {
            return baseApi.getBaseUrl();
        }

        return baseApi.getBaseUrl() + relativeUrl;
    }

    private static int getVolleyMethod(HttpMethod httpMethod) {
        int method;
        switch (httpMethod) {
            default:
            case GET:
                method = Request.Method.GET;
                break;

            case POST:
                method = Request.Method.POST;
                break;

            case PUT:
                method = Request.Method.PUT;
                break;

            case DELETE:
                method = Request.Method.DELETE;
                break;
        }
        return method;
    }


    private static byte[] getVolleyBody(ApiRequest request) {
        final String body = getVolleyBodyStr(request);
        return (body == null) ? null : body.getBytes();
    }

    private static String getVolleyBodyStr(ApiRequest request) {
        String body = null;
        final RequestBody requestBody = request.getRequestBody();
        if (requestBody != null) {
            switch (requestBody.getRequestBodyType()) {
                case JSON:
                    JSONObject jsonBody = (JSONObject) requestBody.getBody();
                    body = jsonBody.toString();
                    break;

                case STRING:
                    body = (String) requestBody.getBody();
                    break;
            }
        }

        return body;
    }

    private Request<?> setDefaultRetryPolicy(Request<?> request) {
        request.setRetryPolicy(new DefaultRetryPolicy(DefaultRetryPolicy.DEFAULT_TIMEOUT_MS * 5,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));
        return request;
    }
}
