package com.citrus.sdk.network;

import android.content.Context;
import android.util.Log;

import com.citrus.sdk.Callback;
import com.citrus.sdk.Environment;
import com.citrus.sdk.ResponseMessages;
import com.citrus.sdk.classes.AccessToken;
import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.network.request.ApiExecutor;
import com.citrus.sdk.network.request.ApiRequest;
import com.citrus.sdk.network.request.ApiRequestBuilder;
import com.citrus.sdk.network.request.RequestBody;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusResponse;
import com.citrus.sdk.walletpg.WalletConsumerProfile;
import com.citrus.sdk.walletpg.WalletPGPaymentResponse;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;


/**
 * Created by vinay on 10/4/16.
 */
public class WalletPgApi extends BaseApi {

    private static final String TAG = WalletPgApi.class.getSimpleName();

    private static WalletPgApi instance = null;
    final ApiExecutor executor = ApiExecutor.getInstance(mContext);

    public static WalletPgApi getInstance(Context context, Environment environment) {
        if (instance == null) {
            synchronized (PgApi.class) {
                if (instance == null) {
                    instance = new WalletPgApi(context, environment);
                }
            }
        }
        return instance;
    }

    private WalletPgApi(Context context, Environment environment) {
        super(context, environment);
    }

    @Override
    public String getBaseUrl() {
        return environment.getWalletPGBaseUrl();
    }

    public void makeWalletPGPayment(final AccessToken accessToken, final String walletPGPaymentJSON,
                                    final Callback<WalletPGPaymentResponse> callback) {
        if (validate()) {
            if (accessToken != null && accessToken.getHeaderAccessToken() != null) {
                final ApiRequest walletPgChargeApi = getWalletPgChargeApi(accessToken.getHeaderAccessToken(), walletPGPaymentJSON);
                executor.executeCustomJsonApi(this, walletPgChargeApi, new Callback<JSONObject>() {
                    @Override
                    public void success(JSONObject jsonObject) {
                        if (jsonObject != null) {
                            WalletPGPaymentResponse walletPGPaymentResponse = WalletPGPaymentResponse.fromJSON(jsonObject.toString());
                            sendResponse(callback, walletPGPaymentResponse);
                        } else {
                            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_WALLET_CHARGE, CitrusResponse.Status.FAILED));
                        }
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            }
        }
    }


    private ApiRequest getWalletPgChargeApi(final String headerAccessToken, final String body) {
        final RequestBody requestBody = ApiRequestBuilder.buildJsonRequestBody(body);
        final ApiRequest walletPgChargeApi = ApiRequestBuilder.buildApi(Api.WALLET_PG_CHARGE, null,
                headerAccessToken, null, requestBody);

        return walletPgChargeApi;
    }

    public void fetchConsumerProfile(final Amount amount, final AccessToken accessToken,
                                     final Callback<WalletConsumerProfile> callback) {
        if (validate()) {
            if (accessToken != null && accessToken.getHeaderAccessToken() != null) {
                final ApiRequest fetchConsumerProfileApi = getFetchConsumerProfileApi(accessToken.getHeaderAccessToken(),
                        String.valueOf(amount.getValueAsDouble()),
                        amount.getCurrency());
                executor.executeCustomJsonApi(this, fetchConsumerProfileApi, new Callback<JSONObject>() {

                    @Override
                    public void success(JSONObject jsonObject) {
                        Log.i(TAG, TAG + ".executeCustomJsonApi().success(): jsonObject = " + jsonObject);

                        try {
                            final WalletConsumerProfile walletConsumerProfile = WalletConsumerProfile.fromJSON(jsonObject.toString());
                            if (walletConsumerProfile != null) {
                                // setCitrusUser(walletConsumerProfile.getCitrusUser());
                                sendResponse(callback, walletConsumerProfile);
                            } else {
                                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_GET_WALLET, CitrusResponse.Status.FAILED));
                            }
                        } catch (JSONException e) {
                            e.printStackTrace();
                            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_GET_WALLET, CitrusResponse.Status.FAILED));
                        }
                    }

                    @Override
                    public void error(CitrusError error) {
                        Log.e(TAG, TAG + ".executeCustomJsonApi().error(): error = " + error);
                        sendError(callback, error);
                    }
                });
            }
        }
    }


    private ApiRequest getFetchConsumerProfileApi(final String headerAccessToken, final String amount, final String currency) {
        final Map<String, String> params = new HashMap<>(2);
        params.put("amount", amount);
        params.put("currency", currency);

        final ApiRequest fetchConsumerProfileApi = ApiRequestBuilder.buildApi(Api.WALLET_PG_FETCH_CONSUMER_PROFILE, null,
                headerAccessToken, params, null);
        return fetchConsumerProfileApi;
    }
}
