package com.citrus.sdk.network;

import android.content.Context;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Log;
import android.webkit.CookieManager;
import android.webkit.CookieSyncManager;

import com.citrus.cash.PersistentConfig;
import com.citrus.citrususer.RandomPassword;
import com.citrus.mobile.OAuth2GrantType;
import com.citrus.mobile.OauthToken;
import com.citrus.sdk.Callback;
import com.citrus.sdk.CitrusUser;
import com.citrus.sdk.Constants;
import com.citrus.sdk.Environment;
import com.citrus.sdk.ResponseMessages;
import com.citrus.sdk.UMCallback;
import com.citrus.sdk.classes.AccessToken;
import com.citrus.sdk.classes.CitrusUMResponse;
import com.citrus.sdk.classes.LinkUserExtendedResponse;
import com.citrus.sdk.classes.LinkUserResponse;
import com.citrus.sdk.classes.LinkUserVerifyEOTPUpdateMobile;
import com.citrus.sdk.classes.MemberInfo;
import com.citrus.sdk.classes.UserBind;
import com.citrus.sdk.classes.VerifyMobileResponse;
import com.citrus.sdk.network.request.ApiExecutor;
import com.citrus.sdk.network.request.ApiRequest;
import com.citrus.sdk.network.request.ApiRequestBuilder;
import com.citrus.sdk.network.request.RequestBody;
import com.citrus.sdk.response.BindUserResponse;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusResponse;
import com.orhanobut.logger.Logger;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;


/**
 * Created by vinay on 10/4/16.
 */
public class UmApi extends BaseApi {

    private static final String TAG = UmApi.class.getSimpleName();

    private static UmApi instance = null;

    //We will persist in life time of the app
    private AccessToken signUpToken = null;
    private CitrusUser citrusUser = null;
    private ApiExecutor executor;

    public static UmApi getInstance(Context context, Environment environment) {
        if (instance == null) {
            synchronized (UmApi.class) {
                if (instance == null) {
                    instance = new UmApi(context, environment);
                }
            }
        }
        return instance;
    }

    private UmApi(Context context, Environment environment) {
        super(context, environment);
        executor = ApiExecutor.getInstance(context);
    }

    @Override
    public String getBaseUrl() {
        return environment.getBaseUrl();
    }


    public synchronized void getSignUpToken(final Callback<AccessToken> callback) {
        Log.i(TAG, TAG + ".getSignUpToken(callback)");
        if (validate()) {
            if (signUpToken == null) {
                final ApiRequest accessTokenApi = getSignUpTokenApi(tokenUtils.getSignupId(), tokenUtils.getSignupSecret(),
                        OAuth2GrantType.implicit.toString());
                executor.executeCustomObjectApi(this, accessTokenApi, new Callback<AccessToken>() {
                    @Override
                    public void success(AccessToken accessToken) {
                        if (accessToken != null) {
                            signUpToken = accessToken;
                            sendResponse(callback, accessToken);
                        } else {
                            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_SIGNUP_TOKEN, CitrusResponse.Status.FAILED));
                        }
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            } else {
                sendResponse(callback, signUpToken);
            }
        }
    }


    private ApiRequest getSignUpTokenApi(final String clientId, final String clientSecret,
                                         final String grantType) {
        Log.i(TAG, TAG + ".getSignUpTokenApi(final String clientId, final String clientSecret,\n" +
                "final String grantType)");

        final Map<String, String> params = new HashMap<>();
        params.put("client_id",     clientId);
        params.put("client_secret", clientSecret);
        params.put("grant_type",    grantType);

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.GET_SIGNUP_TOKEN, null,
                                                                            null, params, null);
        return apiRequest;
    }


    public void getUserNameToken(final String userName, final Callback<AccessToken> callback) {
        Log.i(TAG, TAG + ".getUserNameToken(final userName , callback ");

        if (validate()) {
            final ApiRequest userNameTokenApi = getUserNameTokenApi(tokenUtils.getSigninId(), tokenUtils.getSigninSecret(),
                    userName, OAuth2GrantType.username.toString());
            executor.executeCustomObjectApi(this, userNameTokenApi, new Callback<AccessToken>() {
                @Override
                public void success(AccessToken accessToken) {
                    if (accessToken != null) {
                        tokenUtils.saveToken(Constants.TokenType.SIGNIN_TOKEN, accessToken);///grant Type username token saved
                        sendResponse(callback, accessToken);
                    } else {
                        sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_SIGNUP_TOKEN, CitrusResponse.Status.FAILED));
                    }
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }
    }


    private ApiRequest getUserNameTokenApi(final String clientId, final String clientSecret,
                                           final String username, final String grantType) {
        Log.i(TAG, TAG + ".getUserNameTokenApi(final clientId , clientSecret, username, granttype");

        final Map<String, String> params = new HashMap<>(4);
        params.put("client_id", clientId);
        params.put("client_secret", clientSecret);
        params.put("grant_type", grantType);
        params.put("username", username);

        // Calls same API as GetSignUpToken, but passes different params
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.GET_SIGNUP_TOKEN, null,
                null, params, null);
        return apiRequest;
    }

    public void getPrepaidToken(final String email, final String password, final Callback<AccessToken> callback) {
        Log.i(TAG, TAG + ".getPrepaidToken(final email , password, callback");
        getPrepaidToken(email, password, OAuth2GrantType.password, callback);
    }

    public void getPrepaidToken(final String email, final String password,
                                OAuth2GrantType grantType, final Callback<AccessToken> callback) {
        Log.i(TAG, TAG + ".getPrepaidToken(final email , password, granttype, callback)");
        Log.i(TAG, TAG + ". calling getPrepaidToken(final email , password, granttype, callback, true)");
        getPrepaidToken(email, password, grantType, callback, true);
    }

    public void getPrepaidToken(final String email, final String password,
                                OAuth2GrantType grantType, final Callback<AccessToken> callback,
                                final boolean saveToken) {
        Log.i(TAG, TAG + ".getPrepaidToken(final email , password, granttype, callback, " + saveToken);
        if (validate()) {
            final ApiRequest prepaidTokenApi = getPrepaidTokenApi(tokenUtils.getSigninId(), tokenUtils.getSigninSecret(),
                    email, password, grantType.toString());
            if (saveToken) {
                executor.executeCustomObjectApi(this, prepaidTokenApi, new Callback<AccessToken>() {
                    @Override
                    public void success(AccessToken accessToken) {
                        if (accessToken != null) {
                            tokenUtils.saveToken(Constants.TokenType.PREPAID_TOKEN, accessToken);///grant Type username token saved
                            sendResponse(callback, accessToken);
                        } else {
                            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_SIGNUP_TOKEN, CitrusResponse.Status.FAILED));
                        }
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            } else {
                executor.executeCustomObjectApi(this, prepaidTokenApi, callback);
            }
        }
    }


    private ApiRequest getPrepaidTokenApi(final String clientId, final String clientSecret,
                                          final String username, final String password,
                                          final String grantType) {
        Log.i(TAG, TAG + ".getPrepaidTokenApi(final clientId , clientSecret, username, password, grantType)");

        final Map<String, String> params = new HashMap<>(5);
        params.put("client_id", clientId);
        params.put("client_secret", clientSecret);
        params.put("username", username);
        params.put("password", password);
        params.put("grant_type", grantType);

        // Calls same API as GetSignUpToken, but passes different params
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.GET_SIGNUP_TOKEN, null,
                null, params, null);
        return apiRequest;
    }


    private void getRefreshToken(final String tokenType, final String refreshToken, final Callback<AccessToken> callback) {
        Log.i(TAG, TAG + ".getRefreshToken(final tokenType , refreshToken,callback");
        if (validate()) {

            final ApiRequest refreshTokenApi = getRefreshTokenApi(tokenUtils.getSigninId(), tokenUtils.getSigninSecret(),
                    OAuth2GrantType.refresh_token.toString(), refreshToken);
            executor.executeCustomObjectApi(this, refreshTokenApi, new com.citrus.sdk.Callback<AccessToken>() {
                @Override
                public void success(AccessToken accessToken) {
                    tokenUtils.saveToken(Constants.TokenType.valueOf(tokenType), accessToken);
                    sendResponse(callback, accessToken);
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }
    }

    private ApiRequest getRefreshTokenApi(final String client_ID, final String client_Secret,
                                          final String grantType, final String refreshToken) {
        final Map<String, String> params = new HashMap<>(4);
        params.put("client_id", client_ID);
        params.put("client_secret", client_Secret);
        params.put("grant_type", grantType);
        params.put("refresh_token", refreshToken);

        // Calls same API as GetSignUpToken, but passes different params
        final ApiRequest refreshTokenApi = ApiRequestBuilder.buildApi(Api.GET_SIGNUP_TOKEN, null,
                null, params, null);

        return refreshTokenApi;
    }


    public void getUserNameToken(final Callback<AccessToken> callback) {
        Log.i(TAG, TAG + ".getUserNameToken( callback");
        getSavedToken(Constants.TokenType.SIGNIN_TOKEN, callback);
    }

    private void getSavedToken(final Constants.TokenType tokenType, final com.citrus.sdk.Callback<AccessToken> callback) {
        Log.i(TAG, TAG + ".getSavedToken( tokenType, callback)");
        getSavedToken(false, tokenType, callback);
    }

    private void getSavedToken(final boolean isPrepaidPayRequest, final Constants.TokenType tokenType,
                               final Callback<AccessToken> callback) {
        Log.i(TAG, TAG + ".getSavedToken( isPrepaidPayRequest, tokenType, callback");
        super.tokenUtils.getToken(isPrepaidPayRequest, tokenType, new UMCallback<AccessToken>() {
            @Override
            public void expired(String token) { //this is refresh token
                getRefreshToken(tokenType.toString(), token, callback);
            }

            @Override
            public void success(Object accessToken) {
                sendResponse(callback, accessToken);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }


    public void getPrepaidToken(final Callback<AccessToken> callback) {
        Log.i(TAG, TAG + ".getPrepaidToken(  callback)");
        getSavedToken(Constants.TokenType.PREPAID_TOKEN, callback);
    }

    public void getPrepaidPayToken(final Callback<AccessToken> callback) {
        Log.i(TAG, TAG + ".getPrepaidPayToken(  callback");
        getSavedToken(true, Constants.TokenType.PREPAID_TOKEN, callback);
    }


    public synchronized void createUser(final String emailId, final String mobileNo, final Callback<String> callback) {
        Log.i(TAG, TAG + ".createUser( )");
        if (validate()) {
            getSignUpToken(new Callback<AccessToken>() {
                @Override
                public void success(final AccessToken signupToken) {

                    if (signupToken != null && signupToken.getHeaderAccessToken() != null) {
                        final OauthToken signuptoken = new OauthToken(UmApi.this.mContext, Constants.SIGNUP_TOKEN);
                        signuptoken.createToken(signupToken.getJSON()); //Oauth Token received

                        final ApiRequest bindUserApi = getBindResponseApi(signupToken.getHeaderAccessToken(), emailId, mobileNo);
                        executor.executeCustomObjectApi(UmApi.this, bindUserApi, new Callback<UserBind>() {
                            @Override
                            public void success(final UserBind userBind) {
                                if (userBind != null) {
                                    getUserNameToken(userBind.getUsername(), new Callback<AccessToken>() {
                                        @Override
                                        public void success(final AccessToken accessToken) {
                                            if (accessToken != null && accessToken.getHeaderAccessToken() != null) {
                                                final OauthToken token = new OauthToken(UmApi.this.mContext, Constants.SIGNIN_TOKEN);
                                                token.createToken(accessToken.getJSON());
                                                token.saveUserDetails(emailId, mobileNo);//save email and mobile No of the user//
                                                sendResponse(callback, ResponseMessages.SUCCESS_MESSAGE_USER_BIND);
                                            } else {
                                                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BIND_USER, CitrusResponse.Status.FAILED));
                                            }
                                        }

                                        @Override
                                        public void error(CitrusError error) {
                                            sendError(callback, error);
                                        }
                                    });
                                } else {
                                    sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BIND_USER, CitrusResponse.Status.FAILED));
                                }
                            }

                            @Override
                            public void error(CitrusError error) {
                                sendError(callback, error);
                            }
                        });
                    } else {
                        sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_SIGNUP_TOKEN, CitrusResponse.Status.FAILED));
                    }
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }
    }

    private ApiRequest getBindResponseApi(final String accessToken, final String email,
                                          final String mobile) {
        Log.i(TAG, TAG + ".getBindResponseApi( )");

        final Map<String, String> params = new HashMap<>(2);
        params.put("email", email);
        params.put("mobile", mobile);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.GET_BIND_RESPONSE, null,
                accessToken, params, null);
        return apiRequest;
    }

    public synchronized void isCitrusMember(final String emailId, final String mobileNo, final Callback<Boolean> callback) {
        Log.i(TAG, TAG + ".isCitrusMember( )");
        if (validate()) {
            createUser(emailId, mobileNo, new Callback<String>() {
                @Override
                public void success(String s) {
                    if (ResponseMessages.SUCCESS_MESSAGE_USER_BIND.equalsIgnoreCase(s)) { //bind Successful
                        RandomPassword pwd = new RandomPassword();

                        String random_pass = pwd.generate(emailId, mobileNo);

                        // TODO: check retrofit version. How to determine 200 OK
                        getPrepaidToken(emailId, random_pass, OAuth2GrantType.password, new Callback<AccessToken>() {
                            @Override
                            public void success(AccessToken accessToken) {
                                Logger.d("User Not A Citrus Member. Please Sign Up User.");
                                sendResponse(callback, false);
                            }

                            @Override
                            public void error(CitrusError error) {
                                //TODO:  check if other Status codes
                                /*if (error.getStatusCode() == null) {

                                }*/
                                Logger.d("User Already A Citrus Member. Please Sign In User.");
                                sendResponse(callback, true);
                            }
                        }, false);
                    } else {
                        sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BIND_USER, CitrusResponse.Status.FAILED));
                    }
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }
    }


    public synchronized void getMemberInfo(final String emailId, final String mobileNo,
                                           final Callback<MemberInfo> callback) {
        Log.i(TAG, TAG + ".getMemberInfo( )");
        if (validate()) {
            getSignUpToken(new Callback<AccessToken>() {
                @Override
                public void success(final AccessToken accessToken) {
                    if (accessToken != null && accessToken.getHeaderAccessToken() != null) {
                        JSONObject jsonObject = new JSONObject();
                        OauthToken signuptoken = new OauthToken(UmApi.this.mContext, Constants.SIGNUP_TOKEN); //save this token
                        signuptoken.createToken(accessToken.getJSON()); //Oauth Token received
                        try {
                            jsonObject.put("email", emailId);
                            jsonObject.put("mobile", mobileNo);

                            final ApiRequest memberInfoApi = getMemberInfoApi(accessToken.getHeaderAccessToken(), jsonObject.toString());
                            executor.executeCustomJsonApi(UmApi.this, memberInfoApi, new Callback<JSONObject>() {
                                @Override
                                public void success(JSONObject jsonObject) {
                                    if (jsonObject != null) {
                                        final MemberInfo memberInfo = MemberInfo.fromJSON(jsonObject.toString());
                                        sendResponse(callback, memberInfo);
                                    } else {
                                        sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_MEMBER_INFO, CitrusResponse.Status.FAILED));
                                    }
                                }

                                @Override
                                public void error(CitrusError error) {
                                    sendError(callback, error);
                                }
                            });

                        } catch (JSONException e) {
                            e.printStackTrace();
                            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_MEMBER_INFO, CitrusResponse.Status.FAILED));
                        }
                    } else {
                        sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_SIGNUP_TOKEN, CitrusResponse.Status.FAILED));
                    }
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }
    }


    private ApiRequest getMemberInfoApi(final String accessToken, final String body) {
        Log.i(TAG, TAG + ".getMemberInfoApi( )");
        final RequestBody requestBody = ApiRequestBuilder.buildJsonRequestBody(body);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.GET_MEMBER_INFO, null,
                accessToken, null, requestBody);

        return apiRequest;
    }


    public void bindUserByMobile(final String emailId, final String mobileNo,
                                 final Callback<BindUserResponse> callback) {
        Log.i(TAG, TAG + ".bindUserByMobile( )");
        if (validate()) {
            getMemberInfo(emailId, mobileNo, new Callback<MemberInfo>() {
                @Override
                public void success(final MemberInfo memberInfo) {
                    getSignUpToken(new Callback<AccessToken>() {
                        @Override
                        public void success(final AccessToken accessToken) {
                            final ApiRequest bindUserByMobileApi = getbindUserByMobileApi(accessToken.getHeaderAccessToken(), emailId, mobileNo);
                            executor.executeCustomObjectApi(UmApi.this, bindUserByMobileApi, new Callback<UserBind>() {
                                @Override
                                public void success(UserBind userBind) {
                                    final BindUserResponse bindUserResponse;
                                    // If the user is fresh user then send the password reset link.
                                    if (memberInfo.getProfileByMobile() == null && memberInfo.getProfileByEmail() == null) {
                                        bindUserResponse = new BindUserResponse(BindUserResponse.RESPONSE_CODE_NEW_USER_BOUND);

                                        resetPassword(emailId, null);
                                    } else {
                                        bindUserResponse = new BindUserResponse(BindUserResponse.RESPONSE_CODE_EXISTING_USER_BOUND);
                                    }

                                    getUserNameToken(userBind.getUsername(), new Callback<AccessToken>() {
                                        @Override
                                        public void success(AccessToken accessToken) {
                                            sendResponse(callback, bindUserResponse);
                                        }

                                        @Override
                                        public void error(CitrusError error) {
                                            sendError(callback, error);
                                        }
                                    });
                                }

                                @Override
                                public void error(CitrusError error) {
                                    sendError(callback, error);
                                }
                            });
                        }

                        @Override
                        public void error(CitrusError error) {
                            sendError(callback, error);
                        }
                    });
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }
    }


    private ApiRequest getbindUserByMobileApi(final String accessToken, final String email,
                                              final String mobile) {
        Log.i(TAG, TAG + ".getbindUserByMobileApi( )");

        final Map<String, String> params = new HashMap<>(2);
        params.put("email", email);
        params.put("mobile", mobile);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_BIND_USER_BY_MOBILE, null,
                accessToken, params, null);
        return apiRequest;
    }

    public void resetPassword(final String emailId, final @NonNull Callback<CitrusResponse> callback) {
        Log.i(TAG, TAG + ".resetPassword( )");

        getSignUpToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                if (accessToken != null) {
                    final ApiRequest resetPasswordApi = getResetPasswordApi(accessToken.getHeaderAccessToken(), emailId);
                    executor.executeCustomJsonApi(UmApi.this, resetPasswordApi, new Callback<JSONObject>() {
                        @Override
                        public void success(JSONObject o) {
                            sendResponse(callback, new CitrusResponse(ResponseMessages.SUCCESS_MESSAGE_RESET_PASSWORD, CitrusResponse.Status.SUCCESSFUL));
                        }

                        @Override
                        public void error(CitrusError error) {
                            sendError(callback, error);
                        }
                    });
                } else {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_RESET_PASSWORD, CitrusResponse.Status.FAILED));
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }


    private ApiRequest getResetPasswordApi(final String accessToken, final String username) {
        final Map<String, String> params = new HashMap<>(1);
        params.put("username", username);

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_RESET_PASSWORD, null,
                accessToken, params, null);
        return apiRequest;
    }


    public void linkUserWithOTP(final String emailId, final String mobileNo,
                                final boolean forceMobileVerification, final Callback<LinkUserResponse> callback) {
        Log.i(TAG, TAG + ".linkUserWithOTP( )");
        if (validate()) {
            getSignUpToken(new Callback<AccessToken>() {
                @Override
                public void success(AccessToken accessToken) {
                    final JSONObject jsonObject = new JSONObject();
                    try {
                        jsonObject.put("email", emailId);
                        jsonObject.put("mobile", mobileNo);
                        jsonObject.put("force_mobile_verification", forceMobileVerification);

                        final ApiRequest linkUserResponseApi = getLinkUserApi(accessToken.getHeaderAccessToken(), jsonObject);
                        executor.executeCustomObjectApi(UmApi.this, linkUserResponseApi, callback);
                    } catch (JSONException e) {
                        e.printStackTrace();
                        sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_LINK_USER, CitrusResponse.Status.FAILED));
                    }
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }
    }


    private ApiRequest getLinkUserApi(final String signUpToken, final JSONObject jsonBody) {
        Log.i(TAG, TAG + ".getLinkUserApi( )");
        final RequestBody requestBody = ApiRequestBuilder.buildJsonRequestBody(jsonBody);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_LINK_USER, null,
                signUpToken, null, requestBody);

        return apiRequest;
    }

    public synchronized void getUsernameAndPrepaidToken(final String emailId, final String password,
                                                        final Callback<AccessToken> callback) {
        Log.i(TAG, TAG + ".getUsernameAndPrepaidToken( email, pwd, callback)");
        getUsernameAndPrepaidToken(emailId, password, OAuth2GrantType.password, callback);
    }

    public synchronized void getUsernameAndPrepaidToken(final String emailId, final String password,
                                                        final OAuth2GrantType grantType, final Callback<AccessToken> callback) {
        Log.i(TAG, TAG + ".getUsernameAndPrepaidToken(email, pwd, grantytype, callback )");
        getUserNameToken(emailId, new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                getPrepaidToken(emailId, password, grantType, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void signUp(final String emailId, final String mobileNo,
                                    final String password, final Callback<CitrusResponse> callback) {
        Log.i(TAG, TAG + ".signUp( )");
        if (validate()) {
            getUserNameToken(new Callback<AccessToken>() {
                @Override
                public void success(AccessToken accessToken) {
                    final RandomPassword pwd = new RandomPassword();
                    final String random_pass = pwd.generate(emailId, mobileNo);

                    final ApiRequest setPasswordApi = getSetPasswordApi(accessToken.getHeaderAccessToken(), random_pass, password);
                    executor.executeCustomJsonApi(UmApi.this, setPasswordApi, new Callback<JSONObject>() {

                        @Override
                        public void success(JSONObject jsonObject) {
                            sendResponse(callback, new CitrusResponse(ResponseMessages.SUCCESS_MESSAGE_SIGN_UP, CitrusResponse.Status.SUCCESSFUL));
                        }

                        @Override
                        public void error(CitrusError error) {
                            sendError(callback, error);
                        }
                    });
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }
    }


    private ApiRequest getSetPasswordApi(final String accessToken, final String oldPassword,
                                         final String newPassword) {
        final Map<String, String> params = new HashMap<>(2);
        params.put("old", oldPassword);
        params.put("new", newPassword);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_SET_PASSWORD, null,
                accessToken, params, null);
        return apiRequest;
    }

    public synchronized void updateMobile(final String mobileNo, final Callback<String> callback) {
        Log.i(TAG, TAG + ".updateMobile( )");
        if (validate()) {
            if (!TextUtils.isEmpty(mobileNo)) {
                getUserNameToken(new Callback<AccessToken>() {
                    @Override
                    public void success(AccessToken accessToken) {
                        if (accessToken != null) {
                            final JSONObject jsonObject = new JSONObject();
                            try {
                                jsonObject.put("mobile", mobileNo);

                                final ApiRequest updateMobileApi = getUpdateMobileApi(accessToken.getHeaderAccessToken(), jsonObject);
                                executor.executeCustomObjectApi(UmApi.this, updateMobileApi, callback);
                            } catch (JSONException e) {
                                e.printStackTrace();
                                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_UPDATE_MOBILE, CitrusResponse.Status.FAILED));
                            }
                        } else {
                            sendError(callback, new CitrusError(ResponseMessages.ERROR_SIGNIN_TOKEN_NOT_FOUND, CitrusResponse.Status.FAILED));
                        }
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);

                    }
                });
            } else {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_MOBILE_NO, CitrusResponse.Status.FAILED));
            }
        }
    }


    private ApiRequest getUpdateMobileApi(final String accessToken, final JSONObject jsonBody) {
        Log.i(TAG, TAG + ".getUpdateMobileApi( )");
        final RequestBody requestBody = ApiRequestBuilder.buildJsonRequestBody(jsonBody);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_UPDATE_MOBILE, null,
                accessToken, null, requestBody);

        return apiRequest;
    }

    public synchronized void verifyMobile(final String verificationCode, final Callback<String> callback) {
        Log.i(TAG, TAG + ".verifyMobile( )");


        if (validate()) {
            if (!TextUtils.isEmpty(verificationCode)) {
                getPrepaidToken(new Callback<AccessToken>() {
                    @Override
                    public void success(AccessToken accessToken) {
                        if (accessToken != null) {
                            JSONObject jsonObject = new JSONObject();
                            try {
                                jsonObject.put("verificationCode", verificationCode);

                                final ApiRequest verifyMobileApi = getverifyMobileApi(accessToken.getHeaderAccessToken(), jsonObject);
                                executor.executeCustomObjectApi(UmApi.this, verifyMobileApi, new Callback<VerifyMobileResponse>() {

                                    @Override
                                    public void success(VerifyMobileResponse verifyMobileResponse) {
                                        if (verifyMobileResponse != null &&
                                                verifyMobileResponse.getResponseCode() == VerifyMobileResponse.MOBILE_VERIFIED_SUCCESSFULLY) {
                                            sendResponse(callback, verifyMobileResponse.getResponseMessage());

                                            //TODO: check why following code
                                            // Removed the cached user information.
                                            citrusUser = null;
                                            getProfileInfo(null);
                                        } else {
                                            sendError(callback, new CitrusError(verifyMobileResponse.getResponseMessage(), CitrusResponse.Status.FAILED));
                                        }
                                    }

                                    @Override
                                    public void error(CitrusError error) {
                                        sendError(callback, error);
                                    }
                                });
                            } catch (JSONException e) {
                                e.printStackTrace();
                                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_UPDATE_MOBILE, CitrusResponse.Status.FAILED));
                            }

                        } else {
                            sendError(callback, new CitrusError(ResponseMessages.ERROR_SIGNIN_TOKEN_NOT_FOUND, CitrusResponse.Status.FAILED));
                        }
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            } else {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_VERIFICATION_CODE, CitrusResponse.Status.FAILED));
            }
        }
    }


    private ApiRequest getverifyMobileApi(final String accessToken, final JSONObject jsonBody) {
        Log.i(TAG, TAG + ".getverifyMobileApi( )");
        final RequestBody requestBody = ApiRequestBuilder.buildJsonRequestBody(jsonBody);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_VERIFY_MOBILE, null,
                accessToken, null, requestBody);

        return apiRequest;
    }

    public void getProfileInfo(final Callback<CitrusUser> callback) {
        Log.i(TAG, TAG + ".getProfileInfo( )");
        if (validate()) {
            if (citrusUser == null) {
                getUserNameToken(new Callback<AccessToken>() {
                    @Override
                    public void success(AccessToken accessToken) {

                        final ApiRequest profileInfoApi = getProfileInfoApi(accessToken.getHeaderAccessToken());
                        executor.executeCustomJsonApi(UmApi.this, profileInfoApi, new Callback<JSONObject>() {

                            @Override
                            public void success(JSONObject jsonObject) {
                                if (jsonObject != null) {
                                    final String profileInfo = jsonObject.toString();
                                    citrusUser = CitrusUser.fromJSON(profileInfo);
                                    sendResponse(callback, citrusUser);
                                } else {
                                    sendError(callback, new CitrusError(ResponseMessages.ERROR_UPDATE_MEMBER_INFO, CitrusResponse.Status.FAILED));
                                }
                            }

                            @Override
                            public void error(CitrusError error) {
                                sendError(callback, error);
                            }
                        });
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            } else {
                sendResponse(callback, citrusUser);
            }
        }
    }

    private ApiRequest getProfileInfoApi(final String accessToken) {
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_GET_PROFILE_INFO, null,
                accessToken, null, null);
        return apiRequest;
    }


    public void changePassword(final String oldPassword, final String newPassword,
                               final Callback<CitrusUMResponse> callback) {
        Log.i(TAG, TAG + ".changePassword( )");

        if (validate()) {
            getUserNameToken(new Callback<AccessToken>() {
                @Override
                public void success(AccessToken accessToken) {

                    final ApiRequest changePasswordApi = getChangePasswordApi(accessToken.getHeaderAccessToken(),
                            oldPassword, newPassword);
                    executor.executeCustomObjectApi(UmApi.this, changePasswordApi, callback);
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }
    }


    private ApiRequest getChangePasswordApi(final String accessToken, final String oldPassword,
                                            final String newPassword) {
        final Map<String, String> params = new HashMap<>(2);
        params.put("old", oldPassword);
        params.put("new", newPassword);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_CHANGE_PASSWORD, null,
                accessToken, params, null);
        return apiRequest;
    }

    public void updateProfileInfo(final String firstName, final String lastName,
                                  final Callback<CitrusUMResponse> callback) {
        Log.i(TAG, TAG + ".updateProfileInfo( )");

        if (validate()) {
            getPrepaidToken(new Callback<AccessToken>() {
                                @Override
                                public void success(AccessToken accessToken) {
                                    final JSONObject profileInfoJSON = new JSONObject();
                                    try {
                                        profileInfoJSON.put("firstName", firstName);
                                        profileInfoJSON.put("lastName", lastName);

                                        final ApiRequest updateUserProfileDetailsApi = getUpdatUserProfileDetailsApi(accessToken.getHeaderAccessToken(), profileInfoJSON);
                                        executor.executeCustomObjectApi(UmApi.this,
                                                updateUserProfileDetailsApi, new Callback<CitrusUMResponse>() {

                                                    @Override
                                                    public void success(CitrusUMResponse citrusUMResponse) {
                                                        //TODO: why this code ?
                                                        citrusUser = null;
                                                        getProfileInfo(null);
                                                        sendResponse(callback, citrusUMResponse);
                                                    }

                                                    @Override
                                                    public void error(CitrusError error) {
                                                        sendError(callback, error);
                                                    }
                                                });
                                    } catch (JSONException e) {
                                        sendError(callback, new CitrusError(ResponseMessages.ERROR_UPDATE_MEMBER_INFO, CitrusResponse.Status.FAILED));
                                    }
                                }

                                @Override
                                public void error(CitrusError error) {
                                    sendError(callback, error);
                                }
                            }

            );
        }
    }


    private ApiRequest getUpdatUserProfileDetailsApi(final String accessToken, final JSONObject jsonBody) {
        final RequestBody requestBody = ApiRequestBuilder.buildJsonRequestBody(jsonBody);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_UPDATE_USER_PROFILE, null,
                accessToken, null, requestBody);

        return apiRequest;
    }


    public void sendOneTimePassword(String source, String otpType,
                                    String identity, final Callback<CitrusUMResponse> callback) {
        Log.i(TAG, TAG + ".sendOneTimePassword( )");

        if (validate()) {
            final JSONObject rawObject = new JSONObject();
            try {
                rawObject.put("source", source);
                rawObject.put("otpType", otpType);
                rawObject.put("identity", identity);

                final ApiRequest sendOneTimePasswordApi = getSendOneTimePasswordApi(rawObject);
                executor.executeCustomObjectApi(UmApi.this, sendOneTimePasswordApi, callback);
            } catch (JSONException e) {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_SEND_OTP, CitrusResponse.Status.FAILED));
            }
        }
    }

    private ApiRequest getSendOneTimePasswordApi(final JSONObject jsonBody) {
        final RequestBody requestBody = ApiRequestBuilder.buildJsonRequestBody(jsonBody);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_SEND_ONE_TIME_PASSWORD, null,
                null, null, requestBody);

        return apiRequest;
    }


    public void resetUserPassword(final String emailId, final @NonNull Callback<CitrusUMResponse> callback) {
        Log.i(TAG, TAG + ".resetUserPassword( )");

        getSignUpToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                if (accessToken != null) {
                    final ApiRequest resetUserPasswordApi = getResetUserPasswordApi(accessToken.getHeaderAccessToken(), emailId);
                    executor.executeCustomObjectApi(UmApi.this, resetUserPasswordApi, callback);
                } else {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_RESET_PASSWORD, CitrusResponse.Status.FAILED));
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }


    private ApiRequest getResetUserPasswordApi(final String accessToken, final String username) {
        final Map<String, String> params = new HashMap<>(1);
        params.put("username", username);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_RESET_USER_PASSWORD, null,
                accessToken, params, null);
        return apiRequest;
    }

    public void signUpUser(final String email, final String mobile, final String password,
                           final String firstName, final String lastName, final String sourceType,
                           final boolean markMobileVerified, final boolean markEmailVerified,
                           final Callback<CitrusResponse> callback) {
        Log.i(TAG, TAG + ".signUpUser( )");

        if (validate()) {
            getSignUpToken(new Callback<AccessToken>() {
                @Override
                public void success(AccessToken accessToken) {

                    final ApiRequest signUpUserApi = getsignUpUserApi(accessToken.getHeaderAccessToken(), email,
                            mobile, password, firstName, lastName, sourceType, String.valueOf(markMobileVerified), String.valueOf(markEmailVerified));
                    executor.executeCustomObjectApi(UmApi.this, signUpUserApi, new Callback<CitrusUMResponse>() {

                        @Override
                        public void success(CitrusUMResponse citrusUMResponse) {
                            if (citrusUMResponse.getResponseCode().equalsIgnoreCase(Constants.SIGNUP_SUCCESS_CODE)) {
                                sendResponse(callback, new CitrusResponse(ResponseMessages.SUCCESS_MESSAGE_SIGN_UP, CitrusResponse.Status.SUCCESSFUL));
                            } else {
                                JSONObject errorJSON = new JSONObject();
                                try {
                                    errorJSON.put("responseCode", citrusUMResponse.getResponseCode());
                                    errorJSON.put("responseMessage", citrusUMResponse.getResponseMessage());

                                    sendError(callback, new CitrusError(errorJSON.toString(), CitrusResponse.Status.FAILED));
                                } catch (JSONException e) {
                                    sendError(callback, new CitrusError(ResponseMessages.ERROR_SIGN_UP, CitrusResponse.Status.FAILED));
                                }
                            }
                        }

                        @Override
                        public void error(CitrusError error) {
                            sendError(callback, error);
                        }
                    });
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }
    }


    private ApiRequest getsignUpUserApi(final String accessToken, final String email,
                                        final String mobile, final String password,
                                        final String firstName, final String lastName,
                                        final String sourceType, final String markMobileVerified,
                                        final String markEmailVerified) {
        final Map<String, String> params = new HashMap<>(8);
        params.put("email", email);
        params.put("mobile", mobile);
        params.put("password", password);
        params.put("firstName", firstName);
        params.put("lastName", lastName);
        params.put("sourceType", sourceType);
        params.put("markMobileVerified", markMobileVerified);
        params.put("markEmailVerified", markEmailVerified);

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_SIGN_UP_USER, null,
                accessToken, params, null);
        return apiRequest;
    }

    public void linkUserExtended(final String emailId, final String mobileNo,
                                 final Callback<LinkUserExtendedResponse> callback) {
        Log.i(TAG, TAG + ".linkUserExtended( )");

        if (validate()) {
            getSignUpToken(new Callback<AccessToken>() {
                @Override
                public void success(AccessToken accessToken) {
                    if (accessToken != null && accessToken.getHeaderAccessToken() != null) {
                        try {

                            JSONObject jsonObject = new JSONObject();
                            if (!TextUtils.isEmpty(emailId)) {
                                jsonObject.put("email", emailId);
                            }
                            jsonObject.put("mobile", mobileNo);

                            final ApiRequest linkUserExtendedApi = getlinkUserExtendedApi(accessToken.getHeaderAccessToken(), jsonObject);
                            executor.executeCustomJsonApi(UmApi.this, linkUserExtendedApi, new Callback<JSONObject>() {

                                @Override
                                public void success(JSONObject linkUserJsonObject) {
                                    final LinkUserExtendedResponse linkUserExtendedResponse = LinkUserExtendedResponse.fromJSON(linkUserJsonObject.toString());
                                    linkUserExtendedResponse.setInputEmail(emailId);
                                    linkUserExtendedResponse.setInputMobile(mobileNo);
                                    sendResponse(callback, linkUserExtendedResponse);
                                }

                                @Override
                                public void error(CitrusError error) {
                                    sendError(callback, error);
                                }
                            });
                        } catch (Exception e) {
                            e.printStackTrace();
                            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_LINK_USER, CitrusResponse.Status.FAILED));
                        }
                    } else {
                        sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_SIGNUP_TOKEN, CitrusResponse.Status.FAILED));
                    }
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }
    }


    private ApiRequest getlinkUserExtendedApi(final String signUpToken, final JSONObject jsonBody) {
        Log.i(TAG, TAG + ".getlinkUserExtendedApi( )");

        final RequestBody requestBody = ApiRequestBuilder.buildJsonRequestBody(jsonBody);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_LINK_USER_EXTENDED, null,
                signUpToken, null, requestBody);

        return apiRequest;
    }


    public void linkUserExtendedVerifyEOTPAndUpdateMobile(final String emailId, final String signInGrantType,
                                                          final String linkUserPassword, final LinkUserExtendedResponse linkUserExtended,
                                                          final Callback<CitrusResponse> callback) {
        Log.i(TAG, TAG + ".linkUserExtendedVerifyEOTPAndUpdateMobile( )");

        if (validate()) {
            getSignUpToken(new Callback<AccessToken>() {
                               @Override
                               public void success(final AccessToken accessToken) {
                                   if (accessToken != null && accessToken.getHeaderAccessToken() != null) {

                                       OauthToken token = new OauthToken(mContext, Constants.SIGNIN_TOKEN);
                                       token.createToken(accessToken.getJSON());///grant Type username token saved
                                       try {
                                           final ApiRequest linkUserExtendedVerifyEOTPAndUpdateMobileApi = getlinkUserExtendedVerifyEOTPAndUpdateMobileApi(accessToken.getHeaderAccessToken(),
                                                   signInGrantType, tokenUtils.getSigninId(), tokenUtils.getSigninSecret(), emailId, linkUserPassword, linkUserExtended.getRequestedMobile());
                                           executor.executeCustomObjectApi(UmApi.this,
                                                   linkUserExtendedVerifyEOTPAndUpdateMobileApi, new Callback<LinkUserVerifyEOTPUpdateMobile>() {

                                                       @Override
                                                       public void success(LinkUserVerifyEOTPUpdateMobile linkUserVerifyEOTPUpdateMobile) {
                                                           if (linkUserVerifyEOTPUpdateMobile != null) {
                                                               AccessToken linkUserToken = linkUserVerifyEOTPUpdateMobile.getOAuth2AccessToken();
                                                               if (linkUserToken != null && linkUserToken.getHeaderAccessToken() != null) {
                                                                   tokenUtils.saveToken(Constants.TokenType.PREPAID_TOKEN, accessToken);///grant Type password token saved
                                                                   String input;
                                                                   if (linkUserExtended.getInputEmail().equalsIgnoreCase("")) {
                                                                       input = linkUserExtended.getInputMobile();
                                                                   } else {
                                                                       input = linkUserExtended.getInputEmail();
                                                                   }
                                                                   getUserNameToken(input, new Callback<AccessToken>() {
                                                                       @Override
                                                                       public void success(AccessToken accessToken) {
                                                                           sendResponse(callback, null);
                                                                       }

                                                                       @Override
                                                                       public void error(CitrusError error) {
                                                                           sendError(callback, error);
                                                                       }
                                                                   });
                                                               } else {
                                                                   final CitrusError citrusError = new CitrusError(ResponseMessages.ERROR_SOMETHING_WENT_WRONG,
                                                                           CitrusResponse.Status.FAILED);
                                                                   sendError(callback, citrusError);
                                                               }
                                                           } else {
                                                               final CitrusError citrusError = new CitrusError(ResponseMessages.ERROR_SOMETHING_WENT_WRONG,
                                                                       CitrusResponse.Status.FAILED);
                                                               sendError(callback, citrusError);
                                                           }
                                                       }

                                                       @Override
                                                       public void error(CitrusError error) {
                                                           sendError(callback, error);
                                                       }
                                                   });
                                       } catch (Exception e) {
                                           e.printStackTrace();
                                           sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_LINK_USER, CitrusResponse.Status.FAILED));
                                       }
                                   } else {
                                       sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_SIGNUP_TOKEN, CitrusResponse.Status.FAILED));
                                   }
                               }

                               @Override
                               public void error(CitrusError error) {
                                   sendError(callback, error);
                               }
                           }

            );
        }
    }

    private ApiRequest getlinkUserExtendedVerifyEOTPAndUpdateMobileApi(final String signUpToken, final String grant_type,
                                                                       final String client_ID, final String client_Secret,
                                                                       final String username, final String password,
                                                                       final String requestedMobile) {
        final Map<String, String> params = new HashMap<>(6);
        params.put("grant_type", grant_type);
        params.put("client_id", client_ID);
        params.put("client_secret", client_Secret);
        params.put("username", username);
        params.put("password", password);
        params.put("requestedMobile", requestedMobile);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_LINK_USER_EXTENDED_VERIFY_EOTP_AND_UPDATE_MOBILE, null,
                signUpToken, params, null);

        return apiRequest;
    }


    public synchronized void linkUserExtendedVerifyMobileAndSignIn(final LinkUserExtendedResponse linkUserExtendedResponse,
                                                                   final String verificationCode, final Callback<CitrusResponse> callback) {
        Log.i(TAG, TAG + ".linkUserExtendedVerifyMobileAndSignIn( )");
        if (validate()) {
            getSignUpToken(new Callback<AccessToken>() {
                @Override
                public void success(final AccessToken accessToken) {
                    if (accessToken != null && accessToken.getHeaderAccessToken() != null) {
                        final OauthToken token = new OauthToken(mContext, Constants.SIGNIN_TOKEN);
                        token.createToken(accessToken.getJSON());///grant Type username token saved
                        try {

                            final JSONObject jsonObject = new JSONObject();
                            jsonObject.put("identity", linkUserExtendedResponse.getLinkUserUUID());
                            jsonObject.put("verificationCode", verificationCode);
                            jsonObject.put("client_id", tokenUtils.getSigninId());
                            jsonObject.put("client_secret", tokenUtils.getSigninSecret());

                            final ApiRequest api = getlinkUserExtendedVerifyMobileAndSignInApi(accessToken.getHeaderAccessToken(), jsonObject);
                            executor.executeCustomObjectApi(UmApi.this, api, new Callback<AccessToken>() {

                                @Override
                                public void success(AccessToken accessToken) {
                                    Logger.d("SIGN IN RESPONSE " + accessToken.getJSON().toString());
                                    if (accessToken.getHeaderAccessToken() != null) {
                                        tokenUtils.saveToken(Constants.TokenType.PREPAID_TOKEN, accessToken);///grant Type password token saved
                                        getUserNameToken(linkUserExtendedResponse.getInputMobile(), new com.citrus.sdk.Callback<AccessToken>() {
                                            @Override
                                            public void success(AccessToken accessToken) {
                                                sendResponse(callback, null);
                                            }

                                            @Override
                                            public void error(CitrusError error) {
                                                sendError(callback, error);
                                            }
                                        });
                                    } else {
                                        final CitrusError citrusError = new CitrusError(ResponseMessages.ERROR_SOMETHING_WENT_WRONG,
                                                CitrusResponse.Status.FAILED);
                                        sendError(callback, citrusError);
                                    }
                                }

                                @Override
                                public void error(CitrusError error) {
                                    sendError(callback, error);
                                }
                            });
                        } catch (Exception e) {
                            e.printStackTrace();
                            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_LINK_USER, CitrusResponse.Status.FAILED));
                        }
                    } else {
                        sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_SIGNUP_TOKEN, CitrusResponse.Status.FAILED));
                    }
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }
    }

    public void signOut(com.citrus.sdk.Callback<Boolean> callback) {
        Log.i(TAG, TAG + ".signOut( )");

        // TODO clear the tokens and other things
        CookieSyncManager.createInstance(this.mContext);
        CookieManager.getInstance().setCookie(environment.getBaseUrl(), Constants.CITRUS_PREPAID_COOKIE);//remove App Cookie
        new PersistentConfig(this.mContext).clearToken(); //clear stored cookies
        OauthToken token = new OauthToken(this.mContext, "");
        boolean isUserSignedOut = false;
        isUserSignedOut = token.clearToken(); //clear stored oauth token
        citrusUser = null;
        signUpToken = null;
        callback.success(isUserSignedOut);
    }


    private ApiRequest getlinkUserExtendedVerifyMobileAndSignInApi(final String signUpToken, final JSONObject jsonBody) {
        final RequestBody requestBody = ApiRequestBuilder.buildJsonRequestBody(jsonBody);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi(Api.UM_LINK_USER_EXTENDED_VERIFY_MOBILE_AND_SIGN_IN, null,
                signUpToken, null, requestBody);
        return apiRequest;
    }

    public CitrusUser getCitrusUser() {
        return citrusUser;
    }

    public void setCitrusUser(CitrusUser citrusUser) {
        this.citrusUser = citrusUser;
    }
}
