package com.citrus.sdk.network;

import android.content.Context;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.citrus.sdk.Callback;
import com.citrus.sdk.CitrusUser;
import com.citrus.sdk.Constants;
import com.citrus.sdk.Environment;
import com.citrus.sdk.ResponseMessages;
import com.citrus.sdk.SubscriptionRequest;
import com.citrus.sdk.classes.AccessToken;
import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.classes.CashoutInfo;
import com.citrus.sdk.classes.CitrusPrepaidBill;
import com.citrus.sdk.classes.UpdateSubscriptionRequest;
import com.citrus.sdk.classes.Utils;
import com.citrus.sdk.network.request.ApiExecutor;
import com.citrus.sdk.network.request.ApiRequest;
import com.citrus.sdk.network.request.ApiRequestBuilder;
import com.citrus.sdk.network.request.RequestBody;
import com.citrus.sdk.payment.PaymentType;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusResponse;
import com.citrus.sdk.response.PaymentResponse;
import com.citrus.sdk.response.SubscriptionResponse;
import com.google.gson.reflect.TypeToken;

import org.json.JSONObject;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * Created by vinay on 10/4/16.
 */
public class PrepaidApi extends BaseApi {

    private static PrepaidApi instance;
    private ApiExecutor executor ;

    public static PrepaidApi getInstance(Context context, Environment environment) {
        if (instance == null) {
            synchronized ( PrepaidApi.class ) {
                if (instance == null) {
                    instance = new PrepaidApi(context, environment);
                }
            }
        }
        return instance;
    }

    private PrepaidApi(Context context, Environment environment) {
        super(context, environment);
        executor = ApiExecutor.getInstance( context );
    }

    @Override
    public String getBaseUrl() {
        return environment.getBaseUrl() ;
    }

    public void getPrepaymentTokenValidity(AccessToken signupToken, AccessToken prepaidToken, String scope, final Callback<Boolean> callback) {
        final String prepaymentToken = prepaidToken.getPrepaidPayToken() != null ? prepaidToken.getPrepaidPayToken().getHeaderAccessToken() : "";

        final ApiRequest prepaymentTokenValidityApi = getPrepaymentTokenValidityApi( signupToken.getHeaderAccessToken(), prepaymentToken, scope ) ;
        executor.executeCustomJsonApi(this, prepaymentTokenValidityApi, new Callback<JSONObject>() {

            @Override
            public void success(JSONObject jsonObject) {
                boolean valid = false;
                if (jsonObject != null && !TextUtils.isEmpty(jsonObject.toString())) {
                    try {
                        final String validity = jsonObject.optString("expiration");

                        valid = Utils.isTokenValid(validity);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                    sendResponse(callback, valid);
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }


    /**   @GET( "/service/v2/token/validate" )
    Call<JsonElement> getPrepaymentTokenValidity(@Header("Authorization") String signupToken, @Header("OwnerAuthorization") String prepaymentToken,
                            @Header("OwnerScope") String scope);
     */
    private ApiRequest getPrepaymentTokenValidityApi(final String signupToken, final String prepaymentToken ,
                                                     final String scope){
        final Map<String,String> headers = new HashMap<>(2);
        headers.put( "OwnerAuthorization",  prepaymentToken );
        headers.put( "OwnerScope",          scope);

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_IS_PREPAID_TOKEN_VALID, headers, signupToken, null, null);
        return apiRequest ;
    }

    public void getPrepaidBill(AccessToken accessToken, Amount amount, String returnUrl, final Callback<CitrusPrepaidBill> callback) {

        final ApiRequest prepaidBillApi = getPrepaidBillApi(accessToken.getHeaderAccessToken(), amount.getValue(),
                                amount.getCurrency(), returnUrl) ;
        executor.executeCustomObjectApi(this, prepaidBillApi, callback);
    }

    private ApiRequest getPrepaidBillApi(final String header, final String amount ,
                                            final String currency, final String redirectUrl ){
        final Map<String,String> params = new HashMap<>(3);
        params.put( "amount" ,      amount);
        params.put( "currency" ,    currency);
        params.put( "redirect" ,    redirectUrl );

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_GET_PREPAID_BILL, null,
                                                                        header, params, null );
        return apiRequest ;
    }


    public synchronized void getBalance(final AccessToken accessToken, final Callback<Amount> callback) {
        if (validate()) {

            final ApiRequest getBalanceApi = getGetBalanceApi(accessToken.getHeaderAccessToken(), "" );
            executor.executeCustomObjectApi( this, getBalanceApi, callback );
        }
    }


    private ApiRequest getGetBalanceApi(final String header, final String dummyObject ){
        final Map<String,String> params = new HashMap<>(3);
        params.put( "dummy" , dummyObject);
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_GET_BALANCE, null,
                                                                                    header, params, null );
        return apiRequest ;
    }


    public synchronized void activatePrepaidUser(final AccessToken accessToken, final Callback<Amount> callback) {
        if (validate()) {
            final ApiRequest activatePrepaidUserApi = getActivatePrepaidUserApi(accessToken.getHeaderAccessToken());
            executor.executeCustomObjectApi(this, activatePrepaidUserApi, callback);
        }
    }

    private ApiRequest getActivatePrepaidUserApi(final String header ){
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_ACTIVATE_PREPAID_USER, null,
                                                                                        header, null, null );
        return apiRequest ;
    }

    public void newPrepaidPay(AccessToken accessToken, PaymentType.CitrusCash citrusCash,
                              final Callback<PaymentResponse> callback) {

        final ApiRequest newPrepaidApi = getNewPrepaidPayApi(accessToken.getPrepaidPayToken().getHeaderAccessToken(), citrusCash.getPaymentJSON() );
        executor.executeCustomJsonApi(this, newPrepaidApi, new Callback<JSONObject>(){

            @Override
            public void success(JSONObject jsonObject) {
                PaymentResponse paymentResponse;
                if (jsonObject != null && (paymentResponse = PaymentResponse.fromJSON(jsonObject.toString())) != null) {
                    sendResponse(callback, paymentResponse);
                } else {
                    sendError(callback, new CitrusError("Error while making payment", CitrusResponse.Status.FAILED));
                }
            }

            @Override
            public void error(CitrusError error) {

            }
        });
    }

    private ApiRequest getNewPrepaidPayApi(final String header, final String body ){
        final RequestBody requestBody   = ApiRequestBuilder.buildJsonRequestBody( body);
        final ApiRequest apiRequest     = ApiRequestBuilder.buildApi( Api.PREPAID_NEW_PREPAID_PAY, null,
                                                                                header, null, requestBody );
        return apiRequest ;
    }


    public synchronized void sendMoney(final AccessToken accessToken, final Amount amount,
                                       final CitrusUser toUser, final String message,
                                       final Callback<PaymentResponse> callback) {
        if (validate()) {
            if (amount == null || TextUtils.isEmpty(amount.getValue())) {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_AMOUNT, CitrusResponse.Status.FAILED));
                return;
            }

            if ((toUser == null) || (TextUtils.isEmpty(toUser.getEmailId()) && TextUtils.isEmpty(toUser.getMobileNo()))) {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_EMAIL_ID_MOBILE_NO, CitrusResponse.Status.FAILED));
                return;
            }

            if (!TextUtils.isEmpty(toUser.getEmailId())) {
                final ApiRequest sendMoneyByEmailApi = getSendMoneyByEmailApi(accessToken.getHeaderAccessToken(), amount.getValue(),
                                                                amount.getCurrency(), message, toUser.getEmailId());
                executor.executeCustomObjectApi(this, sendMoneyByEmailApi, callback);
            } else {
                long mobileNo = Utils.isValidMobileNumber(toUser.getMobileNo());
                if (mobileNo != -1) {
                    final ApiRequest sendMoneyByMobileApi = getSendMoneyByMobileApi(accessToken.getHeaderAccessToken(), amount.getValue(),
                                                                        amount.getCurrency(), message, String.valueOf(mobileNo));
                    executor.executeCustomObjectApi(this, sendMoneyByMobileApi , callback);
                } else {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_INVALID_MOBILE_NO, CitrusResponse.Status.FAILED));
                }
            }
        }
    }


    private ApiRequest getSendMoneyByEmailApi(final String header, final String amount, final String currency,
                                              final String message, final String emailId ){
        final Map<String,String> params = new HashMap<>(4);
        params.put("amount",    amount);
        params.put("currency",  currency );
        params.put( "message" , message );
        params.put( "to" ,      emailId );

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_SEND_MONEY_BY_EMAIL, null,
                                                                                    header, params, null );
        return apiRequest ;
    }

    private ApiRequest getSendMoneyByMobileApi(final String header, final String amount, final String currency,
                                              final String message, final String mobileNo ){
        final Map<String,String> params = new HashMap<>(4);
        params.put( "amount",    amount);
        params.put( "currency",  currency );
        params.put( "message" ,  message );
        params.put( "to" ,       mobileNo );

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_SEND_MONEY_BY_MOBILE, null,
                                                                            header, params, null );
        return apiRequest ;
    }


    public synchronized void sendMoneyToMoblieNo(final AccessToken accessToken, final Amount amount,
                                                 final String mobileNo, final String message,
                                                 final Callback<PaymentResponse> callback) {
        if (validate()) {

            if (amount == null || TextUtils.isEmpty(amount.getValue())) {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_AMOUNT, CitrusResponse.Status.FAILED));
                return;
            }

            if (TextUtils.isEmpty(mobileNo)) {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_MOBILE_NO, CitrusResponse.Status.FAILED));
                return;
            }

            long validMobileNo = Utils.isValidMobileNumber(mobileNo);
            if (validMobileNo != -1) {
                final ApiRequest sendMoneyByMobileApi = getSendMoneyByMobileApi(accessToken.getHeaderAccessToken(), amount.getValue(),
                        amount.getCurrency(), message, String.valueOf(mobileNo));

                executor.executeCustomObjectApi(this, sendMoneyByMobileApi , callback);
            } else {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_INVALID_MOBILE_NO, CitrusResponse.Status.FAILED));
            }
        }
    }


    public synchronized void cashout(final AccessToken accessToken, @NonNull final CashoutInfo cashoutInfo, final Callback<PaymentResponse> callback) {
        // Check whether the balance in the wallet is greater than the transaction amount.
        getBalance(accessToken, new Callback<Amount>() {
            @Override
            public void success(Amount balanceAmount) {
                // If the balance amount is greater than equal to the transaction amount, proceed with the payment.
                if (balanceAmount.getValueAsDouble() >= cashoutInfo.getAmount().getValueAsDouble()) {
                    // Since we have access Token, withdraw the money.
                    final ApiRequest cashoutApi = getCashoutApi(accessToken.getHeaderAccessToken(), cashoutInfo.getAmount().getValue(),
                                                                cashoutInfo.getAmount().getCurrency(), cashoutInfo.getAccountHolderName(),
                                                                cashoutInfo.getAccountNo(), cashoutInfo.getIfscCode() );

                    executor.executeCustomObjectApi( PrepaidApi.this, cashoutApi, callback);
                } else {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_INSUFFICIENT_BALANCE, CitrusResponse.Status.FAILED));
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }


    private ApiRequest getCashoutApi( final String header, final String amount, final String currency,
                                        final String owner, final String accountNo, final String ifscCode ){
        final Map<String,String> params = new HashMap<>(5);
        params.put( "amount",   amount);
        params.put( "currency", currency );
        params.put( "owner" ,   owner );
        params.put( "account" , accountNo );
        params.put( "ifsc" ,    ifscCode );

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_CASHOUT, null,
                                                                                header, params, null );
        return apiRequest ;
    }

    public synchronized void getCashoutInfo(final AccessToken accessToken, final Callback<CashoutInfo> callback) {
        final ApiRequest cashoutInfoApi = getCashoutInfoApi(accessToken.getHeaderAccessToken());
        executor.executeCustomJsonApi( PrepaidApi.this, cashoutInfoApi,  new Callback<JSONObject>(){

            @Override
            public void success(JSONObject jsonObject) {
                CashoutInfo cashoutInfo = CashoutInfo.fromJSON(jsonObject.toString());
                sendResponse(callback, cashoutInfo);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error );
            }
        });
    }


    private ApiRequest getCashoutInfoApi( final String header){
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_GET_CASHOUT_INFO, null,
                                                                    header, null, null );

        return apiRequest ;
    }


    public void saveCashoutInfo(final AccessToken accessToken, final CashoutInfo cashoutInfo, final Callback<CitrusResponse> callback) {
        if (cashoutInfo != null) {
            final ApiRequest saveCashoutInfoApi = getSaveCashoutInfoApi(accessToken.getHeaderAccessToken(), CashoutInfo.toJSON(cashoutInfo));
            executor.executeCustomJsonApi(PrepaidApi.this, saveCashoutInfoApi, new Callback<JSONObject>() {

                @Override
                public void success(JSONObject jsonObject) {
                    sendResponse(callback, new CitrusResponse(ResponseMessages.SUCCESS_MESSAGE_SAVED_CASHOUT_OPTIONS, CitrusResponse.Status.SUCCESSFUL));
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_INVALID_CASHOUT_INFO, CitrusResponse.Status.FAILED));
        }
    }

    private ApiRequest getSaveCashoutInfoApi(final String header, final String body ){
        final RequestBody requestBody   = ApiRequestBuilder.buildJsonRequestBody( body);
        final ApiRequest apiRequest     = ApiRequestBuilder.buildApi( Api.PREPAID_SAVE_CASHOUT_INFO, null,
                                                                                    header, null, requestBody );

        return apiRequest ;
    }

    public void getActiveSubscriptions(AccessToken accessToken, final Callback<SubscriptionResponse> callback) {
        final ApiRequest activeSubscriptionsRequest = getActiveSubscriptionsApi(accessToken.getHeaderAccessToken() );

        final Type type = new TypeToken<ArrayList<SubscriptionResponse>>() {}.getType();
        executor.executeCustomObjectApi( this, activeSubscriptionsRequest,  type, new Callback<List<SubscriptionResponse>>(){

            @Override
            public void success(List<SubscriptionResponse> subscriptionResponses) {
                if(subscriptionResponses == null || subscriptionResponses.isEmpty() ) {
                    sendResponse(callback, null);
                }
                else {
                    final SubscriptionResponse subscriptionResponse = subscriptionResponses.get(0) ;
                    sendResponse(callback, subscriptionResponse );
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    private ApiRequest getActiveSubscriptionsApi( final String token){
        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_GET_ACTIVE_SUBSCRIPTION, null,
                                                                                 token, null, null );

        return apiRequest ;
    }

    public void deActivateSubscription(AccessToken accessToken, String subscriptionID, final Callback<SubscriptionResponse> callback) {
        final ApiRequest deActivateSubscriptionRequest = getDeactivateSubscriptionApi(accessToken.getHeaderAccessToken(), subscriptionID,
                                                                                                Constants.DEACTIVATE_SUBSCRIPTION );

        executor.executeCustomObjectApi( this, deActivateSubscriptionRequest, callback);
    }

    private ApiRequest getDeactivateSubscriptionApi( final String accessToken, final String subscriptionID,
                                                     final String status){

        final Map<String,String> params = new HashMap<>();
        params.put( "subscriptionId", subscriptionID );
        params.put( "status", status );

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_DEACTIVATE_SUBSCRIPTION, null,
                                                                            accessToken, params, null );

        return apiRequest ;
    }

    public void createSubscription(AccessToken accessToken, SubscriptionRequest subscriptionRequest,
                                                final Callback<SubscriptionResponse> callback) {
        final ApiRequest apiRequest ;
        if (subscriptionRequest.isSavedCardSubscription()) {
            //create subscription with saved card
            apiRequest = getCreateSubscriptionWithTokenApi( accessToken.getHeaderAccessToken(),
                                                            subscriptionRequest.getAuthRefId(), subscriptionRequest.getThresholdAmount(),
                                                            subscriptionRequest.getLoadAmount(), subscriptionRequest.getType(),
                                                            subscriptionRequest.getToken()) ;
        } else {
            //create subscription with credit card
            apiRequest = getCreateSubscriptionWithCardApi(   accessToken.getHeaderAccessToken(),
                                                            subscriptionRequest.getPan(), subscriptionRequest.getExpiry(),
                                                            subscriptionRequest.getHolder(), subscriptionRequest.getAuthRefId(),
                                                            subscriptionRequest.getThresholdAmount(), subscriptionRequest.getLoadAmount(),
                                                            subscriptionRequest.getType());
        }

        executor.executeCustomObjectApi( this, apiRequest, callback);
    }

    private ApiRequest getCreateSubscriptionWithTokenApi( final String accessToken,
                                                          final String refID, final String thresholdAmount,
                                                          final String loadAmount, final String type,
                                                          final String token){

        final Map<String,String> params = new HashMap<>();
        params.put( "authRefId", refID );
        params.put( "thresholdAmount", thresholdAmount );
        params.put( "loadAmount", loadAmount );
        params.put( "type", type );
        params.put( "token", token );

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_CREATE_SUBSCRIPTION, null,
                                                                                accessToken, params, null );

        return apiRequest ;
    }

    private ApiRequest getCreateSubscriptionWithCardApi( final String accessToken,
                                                        final String pan, final String expiry,
                                                        final String holder, final String refID,
                                                        final  String thresholdAmount, final String loadAmount,
                                                        final  String type){

        final Map<String,String> params = new HashMap<>();
        params.put( "pan", pan );
        params.put( "expiry", expiry );
        params.put( "holder", holder );
        params.put( "authRefId", refID );
        params.put( "thresholdAmount", thresholdAmount );
        params.put( "loadAmount", loadAmount );
        params.put( "type", type );

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_CREATE_SUBSCRIPTION, null,
                                                                        accessToken, params, null );

        return apiRequest ;
    }

    public void updateSubscriptiontoLowerValue(AccessToken accessToken, UpdateSubscriptionRequest subscriptionRequest,
                                               final Callback<SubscriptionResponse> callback) {

        final ApiRequest updateSubscriptionRequest = getUpdateSubscriptiontoLowerValueApi(
                                            accessToken.getHeaderAccessToken(), subscriptionRequest.getSubscriptionId(),
                                            subscriptionRequest.getThresholdAmount(), subscriptionRequest.getLoadAmount());

        executor.executeCustomObjectApi( this, updateSubscriptionRequest ,callback );

    }

    private ApiRequest getUpdateSubscriptiontoLowerValueApi( final String accessToken, final String subscriptionID,
                                                             final String thresholdAmount, final String loadAmount){

        final Map<String,String> params = new HashMap<>();
        params.put( "subscriptionId",   subscriptionID );
        params.put( "thresholdAmount",  thresholdAmount );
        params.put( "loadAmount",       loadAmount );

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_UPDATE_SUBSCRIPTION, null,
                                                                                    accessToken, params, null );

        return apiRequest ;
    }

    public void updateSubscriptiontoHigherValue(AccessToken accessToken, UpdateSubscriptionRequest subscriptionRequest, final Callback<SubscriptionResponse> callback) {

        final ApiRequest updateSubscriptionRequest = getUpdateSubscriptiontoHigherValueApi(
                                                                accessToken.getHeaderAccessToken(), subscriptionRequest.getSubscriptionId(),
                                                                subscriptionRequest.getThresholdAmount(), subscriptionRequest.getLoadAmount(),
                                                                subscriptionRequest.getAuthRefId());

        executor.executeCustomObjectApi( this, updateSubscriptionRequest ,callback );
    }

    private ApiRequest getUpdateSubscriptiontoHigherValueApi(   final String accessToken, final String subscriptionID,
                                                                final String thresholdAmount, final String loadAmount,
                                                                final String authRefID){

        final Map<String,String> params = new HashMap<>();
        params.put( "subscriptionId",   subscriptionID );
        params.put( "thresholdAmount",  thresholdAmount );
        params.put( "loadAmount",       loadAmount );
        params.put( "authRefId",        authRefID );

        final ApiRequest apiRequest = ApiRequestBuilder.buildApi( Api.PREPAID_UPDATE_SUBSCRIPTION, null,
                accessToken, params, null );

        return apiRequest ;
    }
}

