package com.citrus.sdk.network;

import android.content.Context;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.citrus.sdk.Callback;
import com.citrus.sdk.CitrusUser;
import com.citrus.sdk.Environment;
import com.citrus.sdk.ResponseMessages;
import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.classes.Utils;
import com.citrus.sdk.dynamicPricing.DynamicPricingRequest;
import com.citrus.sdk.dynamicPricing.DynamicPricingRequestType;
import com.citrus.sdk.dynamicPricing.DynamicPricingResponse;
import com.citrus.sdk.network.request.ApiExecutor;
import com.citrus.sdk.network.request.ApiRequest;
import com.citrus.sdk.network.request.ApiRequestBuilder;
import com.citrus.sdk.network.request.RequestBody;
import com.citrus.sdk.payment.PaymentBill;
import com.citrus.sdk.payment.PaymentOption;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusResponse;

import org.json.JSONObject;


/**
 * Created by vinay on 10/4/16.
 */
public class DynamicPricingPgApi extends BaseApi {

    private static DynamicPricingPgApi instance = null;
    final ApiExecutor executor = ApiExecutor.getInstance(mContext);

    public static DynamicPricingPgApi getInstance(Context context, Environment environment) {
        if (instance == null) {
            synchronized (DynamicPricingPgApi.class) {
                if (instance == null) {
                    instance = new DynamicPricingPgApi(context, environment);
                }
            }
        }
        return instance;
    }

    private DynamicPricingPgApi(Context context, Environment environment) {
        super(context, environment);
    }

    @Override
    public String getBaseUrl() {
        return environment.getDynamicPricingBaseUrl();
    }


    public synchronized void performDynamicPricing(final BillingApi billingApi,
                                                   @NonNull final DynamicPricingRequestType dynamicPricingRequestType,
                                                   @NonNull final String billUrl,
                                                   @NonNull final Callback<DynamicPricingResponse> callback) {

        if (validate()) {
            if (billingApi != null && dynamicPricingRequestType != null && !TextUtils.isEmpty(billUrl)) {

                final Amount originalAmount = dynamicPricingRequestType.getOriginalAmount();
                final String format = "#.00";

                String url = Utils.getUrlFormatted(billUrl, originalAmount, format, dynamicPricingRequestType);

                billingApi.getBill(url, originalAmount, format, new Callback<PaymentBill>() {
                    @Override
                    public void success(PaymentBill paymentBill) {
                        performDynamicPricing(dynamicPricingRequestType, paymentBill, callback);
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            }
        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_NULL_DP_PARAMS, CitrusResponse.Status.FAILED));
        }
    }

    public synchronized void performDynamicPricing(@NonNull final DynamicPricingRequestType dynamicPricingRequestType,
                                                   @NonNull final PaymentBill paymentBill,
                                                   @NonNull final Callback<DynamicPricingResponse> callback) {

        if (validate()) {
            if (dynamicPricingRequestType != null && paymentBill != null) {
                final PaymentOption paymentOption = dynamicPricingRequestType.getPaymentOption();
                final CitrusUser citrusUser = dynamicPricingRequestType.getCitrusUser();
                final DynamicPricingRequest request = new DynamicPricingRequest(dynamicPricingRequestType, paymentBill);
                final String dpPath = environment.getDynamicPricingPath();
                final ApiRequest dyanamicPricingApi = getDynamicPricingApi(dpPath, DynamicPricingRequest.toJSON(request));
                executor.executeCustomJsonApi(this, dyanamicPricingApi, new Callback<JSONObject>() {
                    @Override
                    public void success(JSONObject jsonObject) {
                        if (jsonObject != null) {
                            final DynamicPricingResponse dynamicPricingResponse = DynamicPricingResponse.fromJSON(jsonObject.toString());
                            dynamicPricingResponse.setPaymentBill(paymentBill);
                            dynamicPricingResponse.setPaymentOption(paymentOption);
                            dynamicPricingResponse.setCitrusUser(citrusUser);
                            sendResponse(callback, dynamicPricingResponse);
                        } else {
                            sendError(callback, new CitrusError("Unable to apply dynamic pricing", CitrusResponse.Status.FAILED));
                        }
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            } else {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_NULL_DP_PARAMS, CitrusResponse.Status.FAILED));
            }
        }
    }

    private ApiRequest getDynamicPricingApi(final String dpPath, final String body) {
        final RequestBody requestBody = ApiRequestBuilder.buildJsonRequestBody(body);
        final ApiRequest dynamicPricingApi = ApiRequestBuilder.buildApi(Api.DYNAMIC_PRICING_PERFORM, null,
                null, null, requestBody, dpPath);

        return dynamicPricingApi;

    }
}
