/*
 *
 *    Copyright 2014 Citrus Payment Solutions Pvt. Ltd.
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 * /
 */

package com.citrus.sdk;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Build;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.v4.content.LocalBroadcastManager;
import android.text.TextUtils;
import android.widget.Toast;

import com.citrus.mobile.OAuth2GrantType;
import com.citrus.mobile.OauthToken;
import com.citrus.sdk.classes.AccessToken;
import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.classes.BCCancelResponse;
import com.citrus.sdk.classes.BCResponse;
import com.citrus.sdk.classes.BinServiceResponse;
import com.citrus.sdk.classes.CardBinDetails;
import com.citrus.sdk.classes.CashoutInfo;
import com.citrus.sdk.classes.CitrusConfig;
import com.citrus.sdk.classes.CitrusException;
import com.citrus.sdk.classes.CitrusPrepaidBill;
import com.citrus.sdk.classes.CitrusUMResponse;
import com.citrus.sdk.classes.LinkBindUserResponse;
import com.citrus.sdk.classes.LinkUserExtendedResponse;
import com.citrus.sdk.classes.LinkUserPasswordType;
import com.citrus.sdk.classes.LinkUserResponse;
import com.citrus.sdk.classes.MemberInfo;
import com.citrus.sdk.classes.PGHealthResponse;
import com.citrus.sdk.classes.StructResponsePOJO;
import com.citrus.sdk.response.SubscriptionResponse;
import com.citrus.sdk.classes.UpdateSubscriptionRequest;
import com.citrus.sdk.classes.Utils;
import com.citrus.sdk.dynamicPricing.DynamicPricingRequestType;
import com.citrus.sdk.dynamicPricing.DynamicPricingResponse;
import com.citrus.sdk.login.AccessType;
import com.citrus.sdk.login.LoginInfo;
import com.citrus.sdk.login.PasswordType;
import com.citrus.sdk.otp.NetBankForOTP;
import com.citrus.sdk.payment.CardOption;
import com.citrus.sdk.payment.CitrusCash;
import com.citrus.sdk.payment.CreditCardOption;
import com.citrus.sdk.payment.DebitCardOption;
import com.citrus.sdk.payment.MVCOption;
import com.citrus.sdk.payment.MerchantPaymentOption;
import com.citrus.sdk.payment.NetbankingOption;
import com.citrus.sdk.payment.PaymentBill;
import com.citrus.sdk.payment.PaymentOption;
import com.citrus.sdk.payment.PaymentType;
import com.citrus.sdk.response.BindUserResponse;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusLogger;
import com.citrus.sdk.response.CitrusResponse;
import com.citrus.sdk.response.PaymentResponse;
import com.citrus.sdk.walletpg.WalletPGPaymentResponse;
import com.facebook.android.crypto.keychain.SharedPrefsBackedKeyChain;
import com.facebook.crypto.Crypto;
import com.facebook.crypto.Entity;
import com.facebook.crypto.exception.CryptoInitializationException;
import com.facebook.crypto.exception.KeyChainException;
import com.facebook.crypto.util.SystemNativeCryptoLibrary;
import com.orhanobut.logger.Logger;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

import static com.citrus.sdk.classes.Utils.isValidEmail;
import static com.citrus.sdk.classes.Utils.isValidMobile;
import static com.citrus.sdk.response.CitrusResponse.Status;

/**
 * Created by salil on 11/5/15.
 */
public class CitrusClient implements ConfigChangeListener {


    private String signinId;
    private String signinSecret;
    private String signupId;
    private String signupSecret;
    private String vanity;

    private String merchantName;
    private Environment environment = Environment.SANDBOX;
    private Amount balanceAmount;
    private static CitrusClient instance;
    private final Context mContext;
    private MerchantPaymentOption merchantPaymentOption = null;

    private OauthToken oauthToken = null;
    private BroadcastReceiver paymentEventReceiver = null;
    private boolean initialized = false;
    private boolean showDummyScreen = false;
    private boolean autoOtpReading = false;

    private RetrofitAPIWrapper retrofitAPIWrapper = null;

    private PaymentOption persistPaymentOption = null;

    private boolean checkPrepaidTokenValidityCalled = false;

    public Crypto getCrypto() {
        return crypto;
    }

    private Crypto crypto = null;


    public UpdateSubscriptionRequest getUpdateSubscriptionRequest() {
        return updateSubscriptionRequest;
    }

    private UpdateSubscriptionRequest updateSubscriptionRequest = null;

    public SubscriptionRequest getSubscriptionRequest() {
        return subscriptionRequest;
    }

    private SubscriptionRequest subscriptionRequest = null;

    private SubscriptionResponse activeSubscription = null;

    private CitrusClient(Context context) {
        mContext = context;
        retrofitAPIWrapper = RetrofitAPIWrapper.getInstance(context);
        CitrusConfig.getInstance().setConfigChangeListener(this);
    }

    public void enableLog(boolean enable) {
        if (enable) {
            CitrusLogger.enableLogs();
        } else {
            CitrusLogger.disableLogs();
        }
    }

    public void showDummyScreenWhilePayments(boolean showDummyScreen) {
        this.showDummyScreen = showDummyScreen;
    }

    public boolean isShowDummyScreenWhilePayments() {
        return showDummyScreen;
    }

    public boolean isAutoOtpReading() {
        return autoOtpReading;
    }

    public void enableAutoOtpReading(boolean enable) {
        this.autoOtpReading = enable;
    }

    public native String getEncryptionKey();

    public static boolean isCitrusNativeLibraryLoaded = false;

    public static boolean isFacebookNativeLibraryLoaded = false;

    public boolean isUserLoggedIn = false;

    public enum AUTO_LOAD_TYPE {
        QUICK_AUTO_LOAD, UPDATE_AUTO_LOAD, LAZY_AUTO_LOAD

    }

    public AUTO_LOAD_TYPE getAUTO_load_type() {
        return mAUTO_load_type;
    }

    private AUTO_LOAD_TYPE mAUTO_load_type = null;

    public void destroy() {
        initialized = false;
        signinId = null;
        signinSecret = null;
        signupId = null;
        signupSecret = null;
        vanity = null;
        environment = null;
        oauthToken = null;
        isUserLoggedIn = false;
        checkPrepaidTokenValidityCalled = false;

        retrofitAPIWrapper.cancelAllRequests();
    }

    public void cancelAllRequests(){
        retrofitAPIWrapper.cancelAllRequests();
    }

    public void init(@NonNull String signupId, @NonNull String signupSecret, @NonNull String signinId, @NonNull String signinSecret, @NonNull String vanity, @NonNull Environment environment) {
        if (!initialized) {

            oauthToken = new OauthToken(mContext);
            this.signupId = signupId;
            this.signupSecret = signupSecret;
            this.signinId = signinId;
            this.signinSecret = signinSecret;
            this.vanity = vanity;

            if (validate()) {
                // Initialize the wrapper for retrofit clients.
                retrofitAPIWrapper.init(signupId, signupSecret, signinId, signinSecret, vanity, environment);
            }

            if (!CitrusLogger.isEnableLogs()) {
                CitrusLogger.disableLogs();
            }

            if (environment == null) {
                this.environment = Environment.SANDBOX;
            }
            this.environment = environment;
            saveSDKEnvironment();

//            fetchPGHealthForAllBanks();

            //getMerchantPaymentOptions(null);

            // Fetch profile info if the user is signed in.
            // If not signed in the information will be fetched once the user signs in.
            isUserSignedIn(new Callback<Boolean>() {
                @Override
                public void success(Boolean signedIn) {
                    if (signedIn) {
                        //getProfileInfo(null);
                        isUserLoggedIn = true;
                        // Check whether the prepaid token is valid or not.
                        // checkPrepaymentTokenValidity(null);

                    }
                }

                @Override
                public void error(CitrusError error) {
                    // Not required to handle the error.
                }
            });

            initialized = true;

        }
        if (environment != Environment.PRODUCTION) {
            Toast.makeText(mContext, "Environment is *** " + environment.toString(), Toast.LENGTH_LONG).show();
        }
    }

    private void fetchPGHealthForAllBanks() {
        // Fetch the pg health for all the banks which will be used later.
        retrofitAPIWrapper.fetchPGHealthForAllBanks();
    }

    private void saveSDKEnvironment() {
        if (oauthToken.getCurrentEnvironment() == Environment.NONE) { //no environment saved till now
            oauthToken.saveEnvironment(environment);
            Logger.d("NO ENVIRONMENT EXISTS earlier");
        } else if (oauthToken.getCurrentEnvironment() == environment) {
            //dont save new enviroment
            Logger.d("PREVIOUS AND CURRENT ENVIRONMENT IS SAME");
        } else { //environment changed-  logout user, save new environment
            signOut(new Callback<CitrusResponse>() {
                @Override
                public void success(CitrusResponse citrusResponse) {
                    oauthToken.saveEnvironment(environment);
                    Logger.d("ENVIRONMMENT MISMATCH ***" + "user Logging out");

                }

                @Override
                public void error(CitrusError error) {
                    oauthToken.saveEnvironment(environment);
                }
            });
        }
    }

    public static CitrusClient getInstance(Context context) {
        if (instance == null) {
            synchronized (CitrusClient.class) {
                if (instance == null) {
                    instance = new CitrusClient(context);
                }
            }
        }

        return instance;
    }

    private void validateLinkUserCredentials(String email, String mobileNo, final Callback callback) {
        if (TextUtils.isEmpty(email)) {
            sendError(callback, new CitrusError("Email ID should not be null.", Status.FAILED));
        } else if (TextUtils.isEmpty(mobileNo)) {
            sendError(callback, new CitrusError("Mobile number should not be null.", Status.FAILED));
        } else if (!isValidEmail(email) && !isValidMobile(mobileNo)) {
            sendError(callback, new CitrusError("Invalid Credentials entered.", Status.FAILED));
        } else if (!isValidEmail(email)) {
            sendError(callback, new CitrusError("Email is not in proper format.", Status.FAILED));
        } else if (!isValidMobile(mobileNo)) {
            sendError(callback, new CitrusError("Mobile number is not in proper format.", Status.FAILED));
        } else {
            sendResponse(callback, true);
        }
    }

    private void validateLinkBindUserCredentials(String email, String mobileNo, AccessType accessType, final Callback callback) {
        if (!isValidEmail(email) && !isValidMobile(mobileNo)) {
            sendError(callback, new CitrusError("Invalid Credentials entered.", Status.FAILED));
        } else if (!isValidEmail(email)) {
            sendError(callback, new CitrusError("Email is not in proper format.", Status.FAILED));
        } else if (!isValidMobile(mobileNo)) {
            sendError(callback, new CitrusError("Mobile number is not in proper format.", Status.FAILED));
        } else {
            if (TextUtils.isEmpty(mobileNo) && accessType == AccessType.FULL) {
                sendError(callback, new CitrusError("Mobile No is required for Full Access.", Status.FAILED));
            } else {
                sendResponse(callback, true);
            }
        }
    }

    // Public APIS start

//    /**
//     * * Link Bind User Api call
//     *
//     * @param emailId
//     * @param mobileNo
//     * @param callback
//     */
//    public synchronized void linkBindUser(final String emailId, final String mobileNo, final AccessType accessType, final Callback<LinkBindUserResponse> callback) {
//        validateLinkBindUserCredentials(emailId, mobileNo, accessType, new Callback() {
//            @Override
//            public void success(Object o) {
////                retrofitAPIWrapper.linkBindUser(emailId, mobileNo, accessType, callback);
//            }
//
//            @Override
//            public void error(CitrusError error) {
//                sendError(callback, error);
//            }
//        });
//    }


    /**
     * Link User Extended Api call
     *
     * @param emailId
     * @param mobileNo
     * @param callback
     */
    public synchronized void linkUserExtended(final String emailId, final String mobileNo, final Callback<LinkUserExtendedResponse> callback) {

        validateLinkUserCredentials(emailId, mobileNo, new Callback() {
            @Override
            public void success(Object o) {
                retrofitAPIWrapper.linkUserExtended(emailId, mobileNo, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

//    public void linkBindUserSignIn(LoginInfo loginInfo, PasswordType passwordType, String linkUserPassword, final Callback<CitrusResponse> callback) {
//
//        String uuid = loginInfo.getUUID();
//        if (linkUserPassword == null || linkUserPassword.equalsIgnoreCase("")) {
//            sendError(callback, new CitrusError(loginInfo.getMessage(), Status.FAILED));
//        } else if (passwordType == PasswordType.NONE) {
//            sendError(callback, new CitrusError(loginInfo.getMessage(), Status.FAILED));
//        } else {
//            int linkUserCase = loginInfo.getLoginUserCase();
//            CitrusUser citrusUser = loginInfo.getCitrusUser();
//
//            final Callback<CitrusResponse> responseCallback = new Callback<CitrusResponse>() {
//                @Override
//                public void success(CitrusResponse citrusResponse) {
//                    checkPrepaidTokenValidityCalled = true;
//                    isUserLoggedIn = true;
//                    sendResponse(callback, citrusResponse);
//                }
//
//                @Override
//                public void error(CitrusError error) {
//                    sendError(callback, error);
//                }
//            };
//            switch (passwordType) {
//                case OTP:
//                    if (linkUserCase == 1 || linkUserCase == 6 || linkUserCase == 2 || linkUserCase == 7 || linkUserCase == 8 || linkUserCase == 9) {
//                        Logger.d("old sigin, otp");
//                        final String linkUserMobile = citrusUser.getMobileNo();
//                        retrofitAPIWrapper.signInWithOTP(linkUserMobile, linkUserPassword, responseCallback);
//                    } else if (linkUserCase == 3 || linkUserCase == 4 || linkUserCase == 5 || linkUserCase == 10 || linkUserCase == 11 || linkUserCase == 12) {
//                        Logger.d("verify and sigin, otp");
//                        synchronized (this) {
////                            retrofitAPIWrapper.linkBindUserVerifyMobileAndSignIn(loginInfo, linkUserPassword, responseCallback);
//                        }
//
//                    }
//                    break;
//
//                case PASSWORD:
//                    if (linkUserCase == 12 || linkUserCase == 2 || linkUserCase == 5 || linkUserCase == 9) {
//                        Logger.d("old sigin, password");
//                        signIn(uuid, linkUserPassword, callback);
//                    }
//                    break;
//            }
//        }
//    }


    public void linkUserExtendedSignIn(LinkUserExtendedResponse linkUserExtended, LinkUserPasswordType linkUserPasswordType, String linkUserPassword, final Callback<CitrusResponse> callback) {

        String inputEmail = linkUserExtended.getInputEmail();
        if (linkUserPassword == null || linkUserPassword.equalsIgnoreCase("")) {
            sendError(callback, new CitrusError(linkUserExtended.getLinkUserMessage(), Status.FAILED));
        } else if (linkUserPasswordType == LinkUserPasswordType.None) {
            sendError(callback, new CitrusError(linkUserExtended.getLinkUserMessage(), Status.FAILED));
        } else {
            int linkUserCase = linkUserExtended.formatResponseCode();
            switch (linkUserPasswordType) {
                case Otp:
                    if (linkUserCase == 1 || linkUserCase == 6 || linkUserCase == 2 || linkUserCase == 7) {
                        Logger.d("old sigin, otp");
                        signInWithOTP(linkUserExtended, linkUserPassword, callback);
                    } else if (linkUserCase == 3 || linkUserCase == 4 || linkUserCase == 5 || linkUserCase == 10 || linkUserCase == 11 || linkUserCase == 12) {
                        Logger.d("verify and sigin, otp");
                        synchronized (this) {
                            linkUserExtendedVerifyMobileAndSignIn(linkUserExtended, linkUserPassword, callback);
                        }
                    } else if (linkUserCase == 8 || linkUserCase == 9) {
                        Logger.d("Update mobile sigin, e-otp");
                        String signInGrantType = OAuth2GrantType.onetimepass.toString();
                        synchronized (this) {
                            linkUserExtendedVerifyEOTPAndUpdateMobile(inputEmail, signInGrantType, linkUserPassword, linkUserExtended, callback);
                        }
                    }
                    break;

                case Password:
                    if (linkUserCase == 1 || linkUserCase == 6 || linkUserCase == 2 || linkUserCase == 5) {
                        Logger.d("old sigin, password");
                        signIn(inputEmail, linkUserPassword, callback);
                    } else if (linkUserCase == 8) {
                        Logger.d("Update mobile sigin, password");
                        String signInGrantType = OAuth2GrantType.password.toString();
                        synchronized (this) {
                            linkUserExtendedVerifyEOTPAndUpdateMobile(inputEmail, signInGrantType, linkUserPassword, linkUserExtended, callback);
                        }
                    }
                    break;
            }
        }
    }

    /**
     * LinkUserExtended VerifyEOTPAndUpdateMobile API call
     *
     * @param emailId
     * @param linkUserPassword
     * @param linkUserExtended
     * @param callback
     */
    public void linkUserExtendedVerifyEOTPAndUpdateMobile(final String emailId, final String signInGrantType, final String linkUserPassword, final LinkUserExtendedResponse linkUserExtended, final Callback<CitrusResponse> callback) {
        retrofitAPIWrapper.linkUserExtendedVerifyEOTPAndUpdateMobile(emailId, signInGrantType, linkUserPassword, linkUserExtended, callback);

        // Since the user is signed-in we are marking the flag as called and bypass the prepaid validity check
        checkPrepaidTokenValidityCalled = true;
        isUserLoggedIn = true;
    }

    /**
     * * LinkUserExtended VerifyMobileAndSignIn API call
     *
     * @param linkUserExtendedResponse
     * @param verificationCode
     * @param callback
     */
    public synchronized void linkUserExtendedVerifyMobileAndSignIn(final LinkUserExtendedResponse linkUserExtendedResponse, final String verificationCode, final Callback<CitrusResponse> callback) {
        retrofitAPIWrapper.linkUserExtendedVerifyMobileAndSignIn(linkUserExtendedResponse, verificationCode, callback);

        // Since the user is signed-in we are marking the flag as called and bypass the prepaid validity check
        checkPrepaidTokenValidityCalled = true;
        isUserLoggedIn = true;
    }


    /**
     * @param emailId  - emailId of the user
     * @param mobileNo - mobileNo of the user
     * @param callback - callback
     *                 <p/>
     * @deprecated Use {@link CitrusClient#linkUserExtended(String, String, Callback)} instead.
     * <p/>
     * This api will check whether the user is existing user or not. If the user is existing user,
     * then it will return the existing details, else it will create an account internally and
     * then call signUp to set the password and activate the account.
     */
    public synchronized void isCitrusMember(final String emailId, final String mobileNo, final Callback<Boolean> callback) {

        retrofitAPIWrapper.isCitrusMember(emailId, mobileNo, callback);
    }

    public synchronized void getMemberInfo(final String emailId, final String mobileNo, final Callback<MemberInfo> callback) {

        retrofitAPIWrapper.getMemberInfo(emailId, mobileNo, callback);
    }

    /**
     * This method can be used by non prepaid merchants - where email and mobile is enough for save/get Cards.
     * This method will create Citrus Account of the user with email. All the cards will be saved to emailID.
     *
     * @param emailId  - emailId of the user
     * @param mobileNo - mobileNo of the user
     * @param callback - callback
     */
    public synchronized void bindUser(final String emailId, final String mobileNo, final Callback<String> callback) {

        retrofitAPIWrapper.createUser(emailId, mobileNo, callback);

    }

    public synchronized void bindUserByMobile(final String emailId, final String mobileNo, final Callback<BindUserResponse> callback) {

        retrofitAPIWrapper.bindUserByMobile(emailId, mobileNo, callback);
    }

    /**
     * This method will enable the user to Sign in with OTP.
     * <li>
     * <ol>If user's mobileNo no is verified then OTP will be sent on the mobileNo and user will be able to sign in using mOTP. </ol>
     * <ol>If the mobileNo no is not found/unverified and email is found then OTP is sent on the emailId and user will be able to sign in using eOTP. </ol>
     * <ol>If the emailId and mobileNo no are not found then the user's account is created.</ol>
     * </li>
     *
     * @param emailId
     * @param mobileNo
     * @param callback
     * @deprecated Use {@link CitrusClient#linkUserExtended(String, String, Callback)} instead.
     */
    public synchronized void linkUserWithOTP(final String emailId, final String mobileNo, final boolean forceMobileVerification, final Callback<LinkUserResponse> callback) {

        retrofitAPIWrapper.linkUserWithOTP(emailId, mobileNo, forceMobileVerification, callback);
    }

    /**
     * @param emailId
     * @param password
     * @param callback
     * @deprecated Use {@link CitrusClient#linkUserExtended(String, String, Callback) and respective signin } instead.
     */
    public synchronized void signIn(final String emailId, final String password, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.signIn(emailId, password, callback);
        // Since the user is signed-in we are marking the flag as called and bypass the prepaid validity check
        checkPrepaidTokenValidityCalled = true;
        isUserLoggedIn = true;
    }

    /**
     * Signin with mobile no and password.
     * <p/>
     *
     * @param mobileNo
     * @param password
     * @param callback
     * @deprecated Use {@link CitrusClient#linkUserExtended(String, String, Callback) and respective signin } instead.
     */
    public synchronized void signInWithMobileNo(final String mobileNo, final String password, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.signInWithMobileNo(mobileNo, password, callback);
        // Since the user is signed-in we are marking the flag as called and bypass the prepaid validity check
        checkPrepaidTokenValidityCalled = true;
        isUserLoggedIn = true;
    }

    /**
     * Signin with mobile no and password.
     *
     * @param linkUserExtended
     * @param otp
     * @param callback
     * @deprecated Use {@link CitrusClient#linkUserExtended(String, String, Callback) and respective signin } instead.
     */
    public synchronized void signInWithOTP(final LinkUserExtendedResponse linkUserExtended, final String otp, final Callback<CitrusResponse> callback) {
        final String linkUserMobile = linkUserExtended.getLinkUserMobile();
        retrofitAPIWrapper.signInWithOTP(linkUserMobile, otp, callback);
        // Since the user is signed-in we are marking the flag as called and bypass the prepaid validity check
        checkPrepaidTokenValidityCalled = true;
        isUserLoggedIn = true;
    }

    /**
     * Signout the existing logged in user.
     */
    public synchronized void signOut(Callback<CitrusResponse> callback) {

        isUserLoggedIn = false;
        checkPrepaidTokenValidityCalled = false;
        activeSubscription = null;
        retrofitAPIWrapper.signOut(callback);
    }

    /**
     * Set the user password.
     *
     * @param emailId
     * @param mobileNo
     * @param password
     * @param callback
     * @deprecated Use {@link CitrusClient#linkUserExtended(String, String, Callback) } instead.
     */
    public synchronized void signUp(final String emailId, String mobileNo, final String password, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.signUp(emailId, mobileNo, password, callback);
    }

    /**
     * Reset the user password. The password reset link will be sent to the user.
     *
     * @param emailId
     * @param callback
     */
    public synchronized void resetPassword(final String emailId, @NonNull final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.resetPassword(emailId, callback);
    }

    /**
     * Update or Change or Verify mobile no.
     *
     * @param mobileNo
     * @param callback
     */
    public synchronized void updateMobile(final String mobileNo, final Callback<String> callback) {

        retrofitAPIWrapper.updateMobile(mobileNo, callback);
    }

    /**
     * Verify mobile no.
     *
     * @param verificationCode
     * @param callback
     */
    public synchronized void verifyMobile(final String verificationCode, final Callback<String> callback) {

        retrofitAPIWrapper.verifyMobile(verificationCode, callback);
    }

    /**
     * Get the user saved payment options.
     *
     * @param callback - callback
     */
    public synchronized void getWallet(final Callback<List<PaymentOption>> callback) {

        retrofitAPIWrapper.getWallet(callback);
    }


    /**
     * Get the user saved payment options. If you want any specific bank at 0th position pass BankCID
     *
     * @param callback - callback
     */
    public synchronized void getWalletWithDefaultBank(final Callback<List<PaymentOption>> callback, final BankCID bankCID) {

        retrofitAPIWrapper.getWalletWithDefaultBank(callback, bankCID);
    }

    @Deprecated
    /**
     * Activate the prepaid user.
     *@deprecated
     * @param callback
     */
    public synchronized void activatePrepaidUser(final Callback<Amount> callback) {

        retrofitAPIWrapper.activatePrepaidUser(callback);
    }

    /**
     * This will retrun profile of the signed in user such as first name, last name, email and mobile.
     *
     * @param callback
     */
    public synchronized void getProfileInfo(final Callback<CitrusUser> callback) {

        retrofitAPIWrapper.getProfileInfo(callback);
    }

    /**
     * Get the balance of the user.
     *
     * @param callback
     */
    public synchronized void getBalance(final Callback<Amount> callback) {

        retrofitAPIWrapper.getBalance(callback);
    }

    /**
     * Save the paymentOption.
     *
     * @param paymentOption - PaymentOption to be saved.
     * @param callback
     */

    public synchronized void savePaymentOption(final PaymentOption paymentOption, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.savePaymentOption(paymentOption, callback);
    }

    /**
     * Save the card - this is required for auto CVV feature
     *
     * @param paymentOption - PaymentOption to be saved.
     * @param callback
     */

    private synchronized void saveCard(final PaymentOption paymentOption, final Callback<AddCardResponse> callback) {

        // we will not save the card if conceal or citruslibrary is not loaded - this will be called only for auto CVV feature
        if (CitrusConfig.getInstance().isOneTapPaymentEnabled() && isOneTapPaymentSupported()) {
            retrofitAPIWrapper.saveCard(paymentOption, callback);
        }
    }


    /**
     * Deletes the saved Payment Option
     *
     * @param paymentOption
     * @param callback
     */
    public synchronized void deletePaymentOption(final PaymentOption paymentOption, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.deletePaymentOption(paymentOption, callback);
    }

    /**
     * Get the payment bill for the transaction.
     *
     * @param amount   - Transaction amount
     * @param callback
     */
    public synchronized void getBill(String billUrl, Amount amount, String format, final Callback<PaymentBill> callback) {

        retrofitAPIWrapper.getBill(billUrl, amount, format, callback);
    }

    // Dynamic Pricing.

    /**
     * Perform Dynamic Pricing. You can specify one the dynamicPricingRequestType to perform Dynamic Pricing.
     *
     * @param dynamicPricingRequestType - One of the dynamicPricingRequestType from {@link DynamicPricingRequestType}
     * @param billUrl                   - billUrl from where we will fetch the bill.
     * @param callback                  - callback
     */
    public synchronized void performDynamicPricing(@NonNull final DynamicPricingRequestType dynamicPricingRequestType, @NonNull final String billUrl, @NonNull final Callback<DynamicPricingResponse> callback) {

        retrofitAPIWrapper.performDynamicPricing(dynamicPricingRequestType, billUrl, callback);
    }

    /**
     * Perform Dynamic Pricing. You can specify one the dynamicPricingRequestType to perform Dynamic Pricing.
     *
     * @param dynamicPricingRequestType - One of the dynamicPricingRequestType from {@link DynamicPricingRequestType}
     * @param paymentBill               - PaymentBill in case you are fetching bill response from your server.
     * @param callback                  - callback
     */
    public synchronized void performDynamicPricing(@NonNull final DynamicPricingRequestType dynamicPricingRequestType, @NonNull final PaymentBill paymentBill, @NonNull final Callback<DynamicPricingResponse> callback) {

        retrofitAPIWrapper.performDynamicPricing(dynamicPricingRequestType, paymentBill, callback);
    }

    /**
     * Send money to your friend.
     *
     * @param amount   - Amount to be sent
     * @param toUser   - The user detalis. Enter emailId if send by email or mobileNo if send by mobile.
     * @param message  - Optional message
     * @param callback - Callback
     * @deprecated Use {@link CitrusClient#sendMoneyToMoblieNo(Amount, String, String, Callback)} instead.
     */
    public synchronized void sendMoney(final Amount amount, final CitrusUser toUser, final String message, final Callback<PaymentResponse> callback) {

        retrofitAPIWrapper.sendMoney(amount, toUser, message, callback);
    }

    /**
     * @param amount
     * @param mobileNo
     * @param message
     * @param callback
     */
    public synchronized void sendMoneyToMoblieNo(final Amount amount, final String mobileNo, final String message, final Callback<PaymentResponse> callback) {

        retrofitAPIWrapper.sendMoneyToMoblieNo(amount, mobileNo, message, callback);
    }

    /**
     * Returns the access token of the currently logged in user.
     */
    public void getPrepaidToken(final Callback<AccessToken> callback) {

        retrofitAPIWrapper.getPrepaidToken(callback);
    }


    /**
     * Get the merchant available payment options. You need to show the user available payment option in your app.
     *
     * @param callback
     */
    public synchronized void getMerchantPaymentOptions(final Callback<MerchantPaymentOption> callback) {

        retrofitAPIWrapper.getMerchantPaymentOptions(callback);
    }


    /**
     * Get the payment options available for load money. You need to show the user available payment option in your app.
     *
     * @param callback
     */
    public synchronized void getLoadMoneyPaymentOptions(final Callback<MerchantPaymentOption> callback) {

        retrofitAPIWrapper.getLoadMoneyPaymentOptions(callback);
    }

    public synchronized void isUserSignedIn(final Callback<Boolean>
                                                    callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                sendResponse(callback, true);
            }

            @Override
            public void error(CitrusError error) {
                sendResponse(callback, false);
            }
        });
    }

    /**
     * Does some thing in old style.
     *
     * @deprecated use {@link #simpliPay(PaymentType, Callback)} instead.
     */
    public synchronized void loadMoney(final PaymentType.LoadMoney loadMoney, final Callback<TransactionResponse> callback) {
        // Validate the card details before forwarding transaction.
        if (loadMoney != null) {
            PaymentOption paymentOption = loadMoney.getPaymentOption();
            persistPaymentOption = paymentOption;
            // If the CardOption is invalid, check what is incorrect and respond with proper message.
            if (paymentOption instanceof CardOption && !((CardOption) paymentOption).validateCard()) {

                sendError(callback, new CitrusError(((CardOption) paymentOption).getCardValidityFailureReasons(), Status.FAILED));
                subscriptionRequest = null;
                mAUTO_load_type = null;
                return;
            }

            if (!validatePaymentOptionForMerchant(paymentOption, retrofitAPIWrapper.getPaymentOption(loadMoney))) {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_PAYMENT_OPTION_NOT_SUPPORTED, Status.FAILED));
                return;
            }

        }
        if (mAUTO_load_type == null) { //this is normal auto load...
            mAUTO_load_type = AUTO_LOAD_TYPE.LAZY_AUTO_LOAD;
        }
        createAutoLoadRequest(loadMoney.getPaymentOption());//local cache Payment Option which may need after load money transaction.

        registerReceiver(callback, new IntentFilter(loadMoney.getIntentAction()));

        startCitrusActivity(loadMoney, false);
    }


    /**
     * Does some thing in old style.
     *
     * @deprecated use {@link #simpliPay(PaymentType, Callback)} instead.
     */
    public synchronized void loadMoneyWithOneTap(final PaymentType.LoadMoney loadMoney, final Callback<TransactionResponse> callback) {

        // Validate the card details before forwarding transaction.
        if (loadMoney != null) {
            if (loadMoney.getPaymentOption() instanceof CardOption) {
                if (loadMoney.getPaymentOption().getFingerPrint() == null) {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_WITHOUT_SAVED_CARD_CVV_PAYMENT, Status.FAILED)); //token does not exist
                    return;
                }
                if (getCVVOfFingerPrint((CardOption) loadMoney.getPaymentOption()) == null) {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_NO_CVV_FOR_SAVED_CARD, Status.FAILED));
                    return;
                }
                ((CardOption) loadMoney.getPaymentOption()).setCardCVV(getCVVOfFingerPrint((CardOption) loadMoney.getPaymentOption())); // read the saved CVV and update Payment Option

                persistPaymentOption = loadMoney.getPaymentOption();
                loadMoney(loadMoney, callback);
            } else { //One tap should work only for Card Payment
                sendError(callback, new CitrusError(ResponseMessages.ERROR_ONE_TAP_ONLY_CARD, Status.FAILED));
            }

        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_NULL_PAYMENT_OPTION, Status.FAILED));
        }
    }

    /**
     * Does some thing in old style.
     *
     * @deprecated use {@link #simpliPay(PaymentType, Callback)} instead.
     */
    public synchronized void pgPayment(final PaymentType.PGPayment pgPayment, final Callback<TransactionResponse> callback) {
        // Validate the card details before forwarding transaction.
        if (pgPayment != null) {
            PaymentOption paymentOption = pgPayment.getPaymentOption();
            persistPaymentOption = paymentOption;
            // If the CardOption is invalid, check what is incorrect and respond with proper message.
            if (paymentOption instanceof CardOption && !((CardOption) paymentOption).validateCard()) {

                sendError(callback, new CitrusError(((CardOption) paymentOption).getCardValidityFailureReasons(), Status.FAILED));
                return;
            }


            if (!validatePaymentOptionForMerchant(paymentOption, retrofitAPIWrapper.getPaymentOption(pgPayment))) {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_PAYMENT_OPTION_NOT_SUPPORTED, Status.FAILED));
                return;
            }
        }

        registerReceiver(callback, new IntentFilter(pgPayment.getIntentAction()));

        startCitrusActivity(pgPayment, false);
    }

    /**
     * This method will be called only for SAVED CVV
     *
     * @param pgPayment
     * @param callback
     * @deprecated use {@link #simpliPay(PaymentType, Callback)} instead.
     */
    public synchronized void pgPaymentWithOneTap(final PaymentType.PGPayment pgPayment, final Callback<TransactionResponse> callback) {

        // Validate the card details before forwarding transaction.
        if (pgPayment != null) {

            if (pgPayment.getPaymentOption() instanceof CardOption) {
                if (pgPayment.getPaymentOption().getFingerPrint() == null) {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_WITHOUT_SAVED_CARD_CVV_PAYMENT, Status.FAILED)); //token does not exist
                    return;
                }
                if (getCVVOfFingerPrint((CardOption) pgPayment.getPaymentOption()) == null) {
                    //ERROR_NO_CVV_FOR_SAVED_CARD
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_NO_CVV_FOR_SAVED_CARD, Status.FAILED));
                    return;
                }
                ((CardOption) pgPayment.getPaymentOption()).setCardCVV(getCVVOfFingerPrint((CardOption) pgPayment.getPaymentOption())); // read the saved CVV and update Payment Option

                persistPaymentOption = pgPayment.getPaymentOption();
                pgPayment(pgPayment, callback);
            } else { //One tap should work only for Card Payment
                sendError(callback, new CitrusError(ResponseMessages.ERROR_ONE_TAP_ONLY_CARD, Status.FAILED));

            }

        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_NULL_PAYMENT_OPTION, Status.FAILED));
        }

    }

    @Deprecated
    /**
     * Does some thing in old style.
     *
     * @deprecated use {@link #simpliPay(PaymentType, Callback)} instead.
     */
    public synchronized void pgPayment(final DynamicPricingResponse dynamicPricingResponse, final Callback<TransactionResponse> callback) {
        if (dynamicPricingResponse != null) {
            PaymentBill paymentBill = dynamicPricingResponse.getPaymentBill();

            PaymentType.PGPayment pgPayment;
            try {
                pgPayment = new PaymentType.PGPayment(paymentBill, dynamicPricingResponse.getPaymentOption(), dynamicPricingResponse.getCitrusUser());

                registerReceiver(callback, new IntentFilter(pgPayment.getIntentAction()));

                startCitrusActivity(pgPayment, dynamicPricingResponse, false);
            } catch (CitrusException e) {
                e.printStackTrace();
                sendError(callback, new CitrusError(e.getMessage(), Status.FAILED));
            }
        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_NULL_DYNAMIC_RESPONSE, Status.FAILED));
        }
    }

    /**
     * @param walletPGPayment
     * @param callback
     */
    private synchronized void walletPGCharge(final WalletPGPayment walletPGPayment, final Callback<TransactionResponse> callback) {
        PaymentOption otherPaymentOption = walletPGPayment.getOtherPaymentOption();
        if (otherPaymentOption != null) {
            if (otherPaymentOption instanceof CardOption && !((CardOption) otherPaymentOption).validateCard()) {
                sendError(callback, new CitrusError(((CardOption) otherPaymentOption).getCardValidityFailureReasons(), Status.FAILED));
                return;
            }

            if (!validatePaymentOptionForMerchant(otherPaymentOption, retrofitAPIWrapper.getPaymentOption(walletPGPayment))) {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_PAYMENT_OPTION_NOT_SUPPORTED, Status.FAILED));
                return;
            }
        }

        if (!walletPGPayment.validateTotalTransactionAmount()) {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_WALLET_PG_INVALID_TOTAL_AMOUNT, Status.FAILED));
            return;
        }

        // Save the payment option for further use. Mostly to enable one tap payment after successful transaction.
        persistPaymentOption = otherPaymentOption;
        registerReceiver(callback, new IntentFilter(walletPGPayment.getIntentAction()));
        startCitrusActivity(walletPGPayment, false);
    }

    /**
     * Does some thing in old style.
     *
     * @deprecated use {@link #simpliPay(PaymentType, Callback)} instead.
     */
    public synchronized void makePayment(final PaymentType.PGPayment pgPayment, final Callback<TransactionResponse> callback) {
        // Validate the card details before forwarding transaction.
        if (pgPayment != null) {
            PaymentOption paymentOption = pgPayment.getPaymentOption();
            persistPaymentOption = paymentOption;
            // If the CardOption is invalid, check what is incorrect and respond with proper message.
            if (paymentOption instanceof CardOption && !((CardOption) paymentOption).validateCard()) {

                sendError(callback, new CitrusError(((CardOption) paymentOption).getCardValidityFailureReasons(), Status.FAILED));
                return;
            }

            if (!validatePaymentOptionForMerchant(paymentOption, retrofitAPIWrapper.getPaymentOption(pgPayment))) {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_PAYMENT_OPTION_NOT_SUPPORTED, Status.FAILED));
                return;
            }
        }

        registerReceiver(callback, new IntentFilter(pgPayment.getIntentAction()));

        startCitrusActivity(pgPayment, true);
    }

    /**
     * Does some thing in old style.
     *
     * @deprecated use {@link #simpliPay(PaymentType, Callback)} instead.
     */
    public synchronized void makePaymentWithOneTap(final PaymentType.PGPayment pgPayment, final Callback<TransactionResponse> callback) {
        // Validate the card details before forwarding transaction.
        if (pgPayment != null) {
            PaymentOption paymentOption = pgPayment.getPaymentOption();

            if (paymentOption instanceof CardOption) {
                CardOption savedCardOption = ((CardOption) paymentOption);
                if (savedCardOption.getFingerPrint() == null) {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_WITHOUT_SAVED_CARD_CVV_PAYMENT, Status.FAILED)); //token does not exist
                    return;
                }
                if (getCVVOfFingerPrint(savedCardOption) == null) {
                    //ERROR_NO_CVV_FOR_SAVED_CARD
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_NO_CVV_FOR_SAVED_CARD, Status.FAILED));
                    return;
                }
                ((CardOption) paymentOption).setCardCVV(getCVVOfFingerPrint(savedCardOption)); // read the saved CVV and update Payment Option

                persistPaymentOption = paymentOption;
            }

            // If the CardOption is invalid, check what is incorrect and respond with proper message.
            if (paymentOption instanceof CardOption && !((CardOption) paymentOption).validateCard()) {

                sendError(callback, new CitrusError(((CardOption) paymentOption).getCardValidityFailureReasons(), Status.FAILED));
                return;
            }
        }

        registerReceiver(callback, new IntentFilter(pgPayment.getIntentAction()));

        startCitrusActivity(pgPayment, true);
    }

    private void isEnoughPrepaidBalance(final Amount amount, final Callback<Boolean> callback) {

        getBalance(new Callback<Amount>() {
            @Override
            public void success(Amount userBalance) {
                if (userBalance.getValueAsDouble() >= amount.getValueAsDouble()) {
                    sendResponse(callback, true);
                } else {
                    sendResponse(callback, false);
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });

    }

    //this is new implementation of Pay Using Cash API - without hitting PG

    @Deprecated
    /**
     * New Prepaid Pay API.
     * @param citrusCash
     * @param callback
     * @deprecated Please use {@link CitrusClient#simpliPay(PaymentType, Callback)}  instead.
     */
    public synchronized void prepaidPay(final PaymentType.CitrusCash citrusCash, final Callback<PaymentResponse> callback) {
        getProfileInfo(new Callback<CitrusUser>() {
            @Override
            public void success(CitrusUser citrusUser) {
                proceedWithNewPrepaidPay(citrusCash, callback);
            }

            @Override
            public void error(CitrusError error) {
                proceedWithNewPrepaidPay(citrusCash, callback);
            }
        });


    }

    private void proceedWithNewPrepaidPay(final PaymentType.CitrusCash citrusCash, final Callback<PaymentResponse> callback) {
        if (!checkPrepaidTokenValidityCalled) { //check only once
            //check if prepaid token is valid and then pay
            checkPrepaymentTokenValidity(new Callback<Boolean>() {
                @Override
                public void success(Boolean isValid) {
                    if (isValid) {
                        retrofitAPIWrapper.newPrepaidPay(citrusCash, callback);
                    } else {
                        sendError(callback, new CitrusError("User's cookie has expired. Please signin.", CitrusResponse.Status.FAILED));
                    }

                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, new CitrusError(error.getMessage(), CitrusResponse.Status.FAILED));
                }
            });
            checkPrepaidTokenValidityCalled = true;
        } else {
            retrofitAPIWrapper.newPrepaidPay(citrusCash, callback); //proceed to prepaidpay
        }
    }

    private void checkPrepaymentTokenValidity(final Callback<Boolean> callback) {

        retrofitAPIWrapper.checkPrepaymentTokenValidity(callback);
    }

    // Cashout Related APIs
    public synchronized void cashout(@NonNull final CashoutInfo cashoutInfo, final Callback<PaymentResponse> callback) {

        retrofitAPIWrapper.cashout(cashoutInfo, callback);
    }

    public synchronized void getCashoutInfo(final Callback<CashoutInfo> callback) {

        retrofitAPIWrapper.getCashoutInfo(callback);
    }

    public synchronized void saveCashoutInfo(final CashoutInfo cashoutInfo, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.saveCashoutInfo(cashoutInfo, callback);
    }

    // PG Health.

    /**
     * It returns {@link com.citrus.sdk.classes.PGHealth} which denotes the health of the PG for in
     * If the health is bad merchants can warn user to use another payment method.
     *
     * @param paymentOption
     * @param callback
     */
    public synchronized void getPGHealth(PaymentOption paymentOption, final Callback<PGHealthResponse> callback) {

        retrofitAPIWrapper.getPGHealth(paymentOption, callback);
    }

    /**
     * Internal.
     * Returns the bank details using the card number.
     *
     * @param cardOption
     */
    public void getBINDetails(CardOption cardOption, final Callback<BinServiceResponse> callback) {

        retrofitAPIWrapper.getBINDetails(cardOption, callback);
    }

    /**
     * Internal.
     * Returns the netbank for OTP which will be used to detect the bank for which auto OTP is being processed.
     */
    public NetBankForOTP getNetBankForOTP() {
        return retrofitAPIWrapper.getNetBankForOTP();
    }

    /**
     * Internal.
     * Set the netbank
     */
    public void resetNetBankForOTP() {
        retrofitAPIWrapper.resetNetBankForOTP();
    }

    public synchronized String getUserEmailId() {
        CitrusUser citrusUser = getCitrusUser();
        if (citrusUser != null) {
            return citrusUser.getEmailId();
        } else {
            return null;
        }
    }

    public synchronized String getUserMobileNumber() {
        CitrusUser citrusUser = getCitrusUser();
        if (citrusUser != null) {
            return citrusUser.getMobileNo();
        } else {
            return null;
        }
    }

    public synchronized CitrusUser getCitrusUser() {
        return retrofitAPIWrapper.getCitrusUser();
    }

    // Public APIS end

    private void unregisterReceiver(BroadcastReceiver receiver) {
        LocalBroadcastManager.getInstance(mContext).unregisterReceiver(receiver);
    }

    private void startCitrusActivity(final PaymentType paymentType, final DynamicPricingResponse dynamicPricingResponse, final boolean useNewAPI) {
        getProfileInfo(new Callback<CitrusUser>() {
            @Override
            public void success(CitrusUser citrusUser) {
                Intent intent = new Intent(mContext, CitrusActivity.class);
                intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                intent.putExtra(Constants.INTENT_EXTRA_PAYMENT_TYPE, paymentType);
                intent.putExtra(Constants.INTENT_EXTRA_DYNAMIC_PRICING_RESPONSE, dynamicPricingResponse);
                intent.putExtra(Constants.INTENT_EXTRA_USE_NEW_API, useNewAPI);

                mContext.startActivity(intent);
            }

            @Override
            public void error(CitrusError error) {
                Intent intent = new Intent(mContext, CitrusActivity.class);
                intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                intent.putExtra(Constants.INTENT_EXTRA_PAYMENT_TYPE, paymentType);
                intent.putExtra(Constants.INTENT_EXTRA_DYNAMIC_PRICING_RESPONSE, dynamicPricingResponse);
                intent.putExtra(Constants.INTENT_EXTRA_USE_NEW_API, useNewAPI);

                mContext.startActivity(intent);
            }
        });
    }

    private void startCitrusActivity(final PaymentType paymentType, boolean useNewAPI) {
        if (useNewAPI == false && paymentType instanceof PaymentType.PGPayment) {
            if (((PaymentType.PGPayment) paymentType).isSingleHop()) { //this is place holder. In case if any one wants to use single Hop Payment
                useNewAPI = true;
            }
        }
        startCitrusActivity(paymentType, null, useNewAPI);
    }

    private <T> void registerReceiver(final Callback<T> callback, IntentFilter intentFilter) {
        paymentEventReceiver = new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                unregisterReceiver(this);
                // Reset netbank for otp
                resetNetBankForOTP();

                TransactionResponse transactionResponse = intent.getParcelableExtra(Constants.INTENT_EXTRA_TRANSACTION_RESPONSE);

                String action = intent.getAction();

                if (transactionResponse != null) {
                    TransactionResponse.TransactionStatus transactionStatus = transactionResponse.getTransactionStatus();
                    Status status = null;

                    if (transactionStatus != null) {
                        switch (transactionStatus) {

                            case SUCCESSFUL:
                                status = Status.SUCCESSFUL;
                                break;
                            case FAILED:
                                status = Status.FAILED;
                                persistPaymentOption = null;
                                break;
                            case CANCELLED:
                                status = Status.CANCELLED;
                                persistPaymentOption = null;
                                break;
                            case PG_REJECTED:
                                status = Status.PG_REJECTED;
                                persistPaymentOption = null;
                                break;
                        }
                    }
                    if (transactionStatus == TransactionResponse.TransactionStatus.SUCCESSFUL) {
                        if (transactionResponse.getBinCardType() != null) {
                            if (!transactionResponse.getBinCardType().contains("Credit")) { //auto works only for credit card. we are not allowing transaction if its done through debit card.
                                subscriptionRequest = null;
                                mAUTO_load_type = null;
                            }
                        }
                        if (persistPaymentOption != null) {
                            if (persistPaymentOption instanceof CardOption && Constants.COUNTRY_INDIA.equalsIgnoreCase(transactionResponse.getCountry())) { //do save one tap only for indian cards
                                CardOption cardOption = (CardOption) persistPaymentOption;
                                if (cardOption.isTokenizedPayment()) {
                                    if (cardOption.getCardCVV() != null && cardOption.getFingerPrint() != null) {
                                        // we are saving CVV for saved Card
                                        saveCVV(persistPaymentOption.getFingerPrint(), ((CardOption) persistPaymentOption).getCardCVV());//save token and CVV after successful transaction.
                                        persistPaymentOption = null;
                                    }
                                } else { // we are saving CVV for new card
                                    saveCard(persistPaymentOption, new Callback<AddCardResponse>() {
                                        @Override
                                        public void success(AddCardResponse addCardResponse) {
                                            saveCVV(addCardResponse.getFingerPrint(), ((CardOption) persistPaymentOption).getCardCVV());// we need to encrypt using
                                            persistPaymentOption = null;
                                        }

                                        @Override
                                        public void error(CitrusError error) {

                                        }
                                    });
                                }
                            }
                        }

                        sendResponse(callback, transactionResponse);
                    } else {
                        updateSubscriptionRequest = null;
                        subscriptionRequest = null;
                        mAUTO_load_type = null;
                        sendError(callback, new CitrusError(transactionResponse.getMessage(), transactionResponse.getJsonResponse(), status, transactionResponse));
                    }
                }
            }
        };

        LocalBroadcastManager.getInstance(mContext).registerReceiver(paymentEventReceiver, intentFilter);
    }


    private synchronized boolean validate() {
        if (!TextUtils.isEmpty(signinId) && !TextUtils.isEmpty(signinSecret)
                && !TextUtils.isEmpty(signupId) && !TextUtils.isEmpty(signupSecret)
                && !TextUtils.isEmpty(vanity)) {
            return true;
        } else {
            throw new IllegalArgumentException(ResponseMessages.ERROR_MESSAGE_BLANK_CONFIG_PARAMS);
        }
    }

    private <T> void sendResponse(Callback callback, T t) {
        if (callback != null) {
            callback.success(t);
        }
    }

    private void sendError(Callback callback, CitrusError error) {
        if (callback != null) {
            callback.error(error);
        }
    }

    /*
    NEW APIS for Cube starts here
     */

    /**
     * Reset the user password. The password reset link will be sent to the user. UM implementation
     *
     * @param emailId
     * @param callback
     */
    public synchronized void resetUserPassword(final String emailId, @NonNull final Callback<CitrusUMResponse> callback) {

        retrofitAPIWrapper.resetUserPassword(emailId, callback);
    }

    public synchronized void setDefaultPaymentOption(final PaymentOption defaultPaymentOption, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.setDefaultPaymentOption(defaultPaymentOption, callback);
    }


    public synchronized void signUpUser(final String email, final String mobile, final String password, final String firstName, final String lastName,
                                        final String sourceType, final boolean markMobileVerified, final boolean markEmailVerified,
                                        final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.signUpUser(email, mobile, password, firstName, lastName, sourceType, markMobileVerified, markEmailVerified, callback);
    }


    public synchronized void changePassword(final String oldPassword, final String newPassword, final Callback<CitrusUMResponse> changePasswordResponseCallback) {

        retrofitAPIWrapper.changePassword(oldPassword, newPassword, changePasswordResponseCallback);
    }


    public synchronized void updateProfileInfo(final String firstName, final String lastName, final Callback<CitrusUMResponse> citrusUMResponseCallback) {

        retrofitAPIWrapper.updateProfileInfo(firstName, lastName, citrusUMResponseCallback);
    }


    public synchronized void sendOneTimePassword(String source, String otpType, String identity, final Callback<CitrusUMResponse> umResponseCallback) {

        retrofitAPIWrapper.sendOneTimePassword(source, otpType, identity, umResponseCallback);
    }

    /**
     * @param first6Digits
     * @param cardDetailsCallback
     * @deprecated Please use {@link CitrusClient#getBINDetails(CardOption, Callback)} instead.
     */
    public synchronized void getCardType(final String first6Digits, final Callback<CardBinDetails> cardDetailsCallback) {

        retrofitAPIWrapper.getCardType(first6Digits, cardDetailsCallback);
    }

    public synchronized void getMerchantName(final Callback<String> callback) {
        retrofitAPIWrapper.getMerchantName(callback);
    }


    /**
     * This method is used to initiate the wallet pg transaction. This is used internally.
     *
     * @param walletPGPaymentJSON
     * @param callback
     */
    public synchronized void makeWalletPGPayment(String walletPGPaymentJSON, Callback<WalletPGPaymentResponse> callback) {
        retrofitAPIWrapper.makeWalletPGPayment(walletPGPaymentJSON, callback);
    }

    /**
     * This method is used to initiate the transaction. This is used internally.
     *
     * @param paymentJSON
     * @param callback
     */
    public synchronized void makeMOTOPayment(String paymentJSON, Callback<StructResponsePOJO> callback) {
        retrofitAPIWrapper.makeMOTOPayment(paymentJSON, callback);
    }

    /**
     * This method is used to initiate the transaction. This is used internally.
     *
     * @param paymentJSON
     * @param callback
     */
    synchronized void newMakePayment(String paymentJSON, Callback<String> callback) {

        retrofitAPIWrapper.newMakePayment(paymentJSON, callback);
    }

    /**
     * Get prepaid bill. This is used internally.
     *
     * @param amount
     * @param returnUrl
     * @param callback
     */
    public synchronized void getPrepaidBill(Amount amount, String returnUrl, Callback<CitrusPrepaidBill> callback) {

        retrofitAPIWrapper.getPrepaidBill(amount, returnUrl, callback);
    }

    // Getters and setters.
    public String getSigninId() {
        return signinId;
    }

    public void setSigninId(String signinId) {
        this.signinId = signinId;
    }

    public String getSigninSecret() {
        return signinSecret;
    }

    public void setSigninSecret(String signinSecret) {
        this.signinSecret = signinSecret;
    }

    public String getSignupId() {
        return signupId;
    }

    public void setSignupId(String signupId) {
        this.signupId = signupId;
    }

    public String getSignupSecret() {
        return signupSecret;
    }

    public void setSignupSecret(String signupSecret) {
        this.signupSecret = signupSecret;
    }

    public String getVanity() {
        return vanity;
    }

    public void setVanity(String vanity) {
        this.vanity = vanity;
    }

    public String getMerchantName() {
        return merchantName;
    }

    public Environment getEnvironment() {
        return environment;
    }

    public void setEnvironment(Environment environment) {
        this.environment = environment;
    }

    private String generateDeviceSpecificKeys() {
        String android_id =
                Settings.Secure.getString(mContext.getContentResolver(), Settings.Secure.ANDROID_ID);
        // String device_id = tManager.getDeviceId();
        String str1 = Build.BOARD + Build.BRAND + Build.CPU_ABI + Build.DEVICE +
                Build.DISPLAY + Build.FINGERPRINT + Build.HOST + Build.ID + Build.MANUFACTURER
                +
                Build.MODEL + Build.PRODUCT + Build.TAGS + Build.TYPE + Build.USER;
        String key2 = str1 + android_id;
        return key2;
    }

    private void initCrypto() {
        crypto = new Crypto(
                new SharedPrefsBackedKeyChain(mContext),
                new SystemNativeCryptoLibrary());
        if (crypto.isAvailable()) {
            isFacebookNativeLibraryLoaded = true;
        } else {
            isFacebookNativeLibraryLoaded = false;
            crypto = null;
        }
    }

    private void saveCVV(String fingerPrint, String CVV) {
        // Crypto library is not loaded on the device so do not proceed.
        if (!CitrusConfig.getInstance().isOneTapPaymentEnabled() || !isFacebookNativeLibraryLoaded || !isCitrusNativeLibraryLoaded) {
            return;
        }

        String seed = (generateDeviceSpecificKeys() + fingerPrint + getEncryptionKey());
        Entity myEntity = new Entity(seed);

        try {
            byte[] cipherTextToken = crypto.encrypt(fingerPrint.getBytes(), myEntity);
            //  String encryptedToken = new String(cipherText);
            byte[] cipherText = crypto.encrypt(CVV.getBytes(), myEntity);

            byte[] cipherText2 = crypto.encrypt(CVV.getBytes(), myEntity);
            //String encryptedCVV = new String(cipherText);
            oauthToken.saveCVV(cipherTextToken, cipherText);
        } catch (KeyChainException e) {
            // e.printStackTrace();
            Logger.d("KeyChainException ");
        } catch (CryptoInitializationException e) {
            Logger.d("CryptoInitializationException");
            // e.printStackTrace();
        } catch (IOException e) {
            Logger.d("IOException");
            // e.printStackTrace();
        }
    }

    private String getCVVOfFingerPrint(CardOption cardOption) {

        String cvv = null;

        try {
            // Check if crypto library is not loaded.
            if (!CitrusConfig.getInstance().isOneTapPaymentEnabled() || !isFacebookNativeLibraryLoaded || !isCitrusNativeLibraryLoaded) {
                return null;
            }

            String seed = (generateDeviceSpecificKeys() + cardOption.getFingerPrint() + getEncryptionKey());

            Entity myEntity = new Entity(seed);

            byte[] encryptedCVV = oauthToken.getCVV(cardOption.getFingerPrint(), seed);
            if (encryptedCVV != null) {

                byte[] decryptedCVV = crypto.decrypt(encryptedCVV, myEntity);
                cvv = new String(decryptedCVV);
            } else {
                cvv = null;

            }
        } catch (KeyChainException e) {
            // e.printStackTrace();
            Logger.d("KeyChainException ");
        } catch (CryptoInitializationException e) {
            Logger.d("CryptoInitializationException");
            // e.printStackTrace();
        } catch (IOException e) {
            Logger.d("IOException");
            // e.printStackTrace();
        }
        return cvv;
    }

    public synchronized boolean isOneTapPaymentEnabledForCard(CardOption cardOption) {
        if (cardOption != null) {
            String cvv = getCVVOfFingerPrint(cardOption);
            return cvv != null;
        }
        return false;
    }

    /**
     * AutoCVV may not be work on few devices. Use this method before prompting for AutoCVV feature.
     *
     * @return
     */
    public synchronized boolean isOneTapPaymentSupported() {
        return (isCitrusNativeLibraryLoaded && isFacebookNativeLibraryLoaded);
    }


    @Override
    public void onOneTapPaymentEnabled(boolean isEnabled) {
        if (isEnabled) {
            try {
                System.loadLibrary("citruslibrary");
                isCitrusNativeLibraryLoaded = true;
            } catch (UnsatisfiedLinkError error) {
                isCitrusNativeLibraryLoaded = false;
            }
            initCrypto();

            if (isOneTapPaymentSupported()) {
                Logger.d("This device supports One Tap Payment.");
            } else {
                Logger.d("This device does not support One Tap Payment.");
            }
        }


    }

    private synchronized boolean validatePaymentOptionForMerchant(PaymentOption paymentOption, MerchantPaymentOption merchantPaymentOption) {
        if (paymentOption != null) {
            if (merchantPaymentOption != null) {
                CardOption.CardScheme cardScheme = null;
                if (paymentOption instanceof CreditCardOption) {
                    cardScheme = ((CreditCardOption) paymentOption).getCardScheme();
                    Set<CardOption.CardScheme> cardSchemeSet = merchantPaymentOption.getCreditCardSchemeSet();
                    if (cardSchemeSet != null) {
                        return cardSchemeSet.contains(cardScheme);
                    } else {
                        return true; ////in case no Card enabled let moto handle
                    }
                } else if (paymentOption instanceof DebitCardOption) {
                    cardScheme = ((DebitCardOption) paymentOption).getCardScheme();
                    Set<CardOption.CardScheme> cardSchemeSet = merchantPaymentOption.getDebitCardSchemeSet();
                    if (cardSchemeSet != null) {
                        return cardSchemeSet.contains(cardScheme);
                    } else {
                        return true; //in case no Card enabled let moto handle
                    }
                } else if (paymentOption instanceof NetbankingOption) {
                    if (!paymentOption.isTokenizedPayment()) {
                        Set<String> netbankingCIDSet = merchantPaymentOption.getBankCIDSet();
                        return netbankingCIDSet == null || netbankingCIDSet.contains(((NetbankingOption) paymentOption).getBankCID());
                    } else {
                        return merchantPaymentOption.getNetbankingOptionList().contains(paymentOption);
                    }
                }
            }

            return true;
        } else {
            return false;
        }
    }


    void makeBlazeCardPayment(String paymentJSON, Callback<BCResponse> callback) {
        retrofitAPIWrapper.makeBlazeCardPayment(paymentJSON, callback);
    }


    void makeBlazeCardTokenizedPayment(String paymentJSON, Callback<BCResponse> callback) {
        retrofitAPIWrapper.makeBlazeCardTokenizedPayment(paymentJSON, callback);
    }

    void cancelBCTransaction(String cancelPaymentJSON, Callback<BCCancelResponse> callback) {
        retrofitAPIWrapper.cancelBCTransaction(cancelPaymentJSON, callback);
    }

    /**
     * This is single endpoint for all the payments
     *
     * @param paymentType
     * @param callback
     */
    public void simpliPay(PaymentType paymentType, final Callback<TransactionResponse> callback) {
        if (paymentType instanceof PaymentType.SplitPayment) {

            if (!preValidation(paymentType.getPaymentOption(), retrofitAPIWrapper.getPaymentOption(paymentType), callback)) //check login and payment options
                return;
            if (!updateOneTapPaymentOption(paymentType.getPaymentOption(), callback)) //update payment option with CVV
                return;
            createWalletPGPaymentOption(paymentType, callback);

        } else if (paymentType instanceof PaymentType.LoadMoney) {

            if (!preValidation(paymentType.getPaymentOption(), retrofitAPIWrapper.getPaymentOption(paymentType), callback)) //check login and payment options
                return;
            if (!updateOneTapPaymentOption(paymentType.getPaymentOption(), callback)) //update payment option with CVV
                return;

            loadMoney((PaymentType.LoadMoney) paymentType, callback);


        } else if (paymentType instanceof PaymentType.CitrusCash) {
            if (!validateLogin(callback)) {
                return;
            }
            try {
                PaymentType.SplitPayment splitPayment = null;
                if (paymentType.getPaymentBill() == null) {

                    splitPayment = new PaymentType.SplitPayment(paymentType.getAmount(), paymentType.getUrl(), null, true, false);

                } else {
                    splitPayment = new PaymentType.SplitPayment(paymentType.getPaymentBill(), null, true, false);
                }
                createWalletPGPaymentOption(splitPayment, callback);
            } catch (CitrusException e) {
                e.printStackTrace();
            }
        } else if (paymentType instanceof PaymentType.PGPayment) { // this is Guest Payment. Does not require token. Check Payment Options.
            PaymentType.PGPayment pgPayment = (PaymentType.PGPayment) paymentType;
            if (!validatePaymentOptions(pgPayment.getPaymentOption(), retrofitAPIWrapper.getPaymentOption(pgPayment), callback))
                return;
            if (!updateOneTapPaymentOption(pgPayment.getPaymentOption(), callback))
                return;

            if ((isUserLoggedIn && pgPayment.getPaymentOption().isTokenizedPayment() && !pgPayment.isDyanmicPricingRequest())) { // if user is fully logged in && tokenized payment redirect to wallet PG Payment
                PaymentType.SplitPayment splitPayment = new PaymentType.SplitPayment(pgPayment);
                createWalletPGPaymentOption(splitPayment, callback);
            } else if (pgPayment.isDyanmicPricingRequest()) {
                // If dynamic pricing payment request use another method
                pgPayment(pgPayment.getDynamicPricingResponse(), callback);
            } else {
                pgPayment(pgPayment, callback);
            }
        }
    }

    /**
     * this method will convert SplitPayment object to WalletPG Object
     *
     * @param paymentType
     * @param callback
     */
    private void createWalletPGPaymentOption(final PaymentType paymentType, final Callback<TransactionResponse> callback) {

        if (((PaymentType.SplitPayment) paymentType).isCitrusCashEnabled() || ((PaymentType.SplitPayment) paymentType).isMVCEnabled()) {  // fetch consumer profile
            getWallet(new Callback<List<PaymentOption>>() {
                @Override
                public void success(List<PaymentOption> paymentOptionList) {
                    List<PaymentOption> walletList = new ArrayList<>();
                    MVCOption mvcOption = null;
                    CitrusCash citrusCash = null;
                    Amount transactionAmount = paymentType.getAmount();
                    double transactionAmountInDouble = transactionAmount.getValueAsDouble();
                    boolean isAmountCompleted = false;
                    PaymentType.SplitPayment splitPayment = (PaymentType.SplitPayment) paymentType;
                    double mvcAmount = 0;
                    double citrusCashAmount = 0;
                    for (PaymentOption paymentOption : paymentOptionList) {
                        if (splitPayment.isMVCEnabled() && paymentOption instanceof MVCOption) {
                            mvcOption = (MVCOption) paymentOption;
                        }
                        if (splitPayment.isCitrusCashEnabled() && paymentOption instanceof CitrusCash) {
                            citrusCash = (CitrusCash) paymentOption;
                        }
                    }
                    if (splitPayment.isCitrusCashEnabled() && !splitPayment.isMVCEnabled() && splitPayment.getPaymentOption() == null) { //this is only pay using Citrus Cash
                        if (citrusCash.getMaxBalance().getValueAsDouble() < transactionAmount.getValueAsDouble()) { // if no sufficient balance...
                            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_INSUFFICIENT_BALANCE, CitrusResponse.Status.FAILED));
                            return;
                        }
                    }

                    if (splitPayment.isMVCEnabled() && mvcOption != null && mvcOption.getMaxBalance().getValueAsDouble() >= 1) {
                        if (transactionAmount.getValueAsDouble() <= mvcOption.getMaxBalance().getValueAsDouble()) {
                            mvcAmount = paymentType.getAmount().getValueAsDouble();
                            isAmountCompleted = true;  //only MVC is enough
                        } else {
                            mvcAmount = mvcOption.getMaxBalanceRounded().getValueAsDouble();
                        }

                        BigDecimal amount1 = BigDecimal.valueOf(transactionAmountInDouble); //this is required. ie. 88.2 - 88.0 = returns 0.2 now
                        BigDecimal amount2 = BigDecimal.valueOf(mvcAmount);//direct double substraction returns 0.200000008
                        BigDecimal difference = amount1.subtract(amount2);
                        transactionAmountInDouble = difference.doubleValue();
                        mvcOption.setTransactionAmount(new Amount(String.valueOf(mvcAmount)));
                        walletList.add(mvcOption);
                    }

                    if (splitPayment.isCitrusCashEnabled() && !isAmountCompleted && citrusCash.getMaxBalance().getValueAsDouble() >= 1) {
                        if (transactionAmountInDouble <= citrusCash.getMaxBalance().getValueAsDouble()) {
                            citrusCashAmount = transactionAmountInDouble;
                            isAmountCompleted = true; // only Citrus Cash is enough
                        } else {
                            citrusCashAmount = citrusCash.getMaxBalanceRounded().getValueAsDouble();
                        }
                        BigDecimal amount1 = BigDecimal.valueOf(transactionAmountInDouble);//this is required. ie. 88.2 - 88.0 = returns 0.2 now.Direct double substraction returns 0.200000008
                        BigDecimal amount2 = BigDecimal.valueOf(citrusCashAmount);
                        BigDecimal difference = amount1.subtract(amount2);
                        transactionAmountInDouble = difference.doubleValue();
                        citrusCash.setTransactionAmount(new Amount(String.valueOf(citrusCashAmount)));
                        walletList.add(citrusCash);
                    }

                    if (!isAmountCompleted) {  // we still need payment option for the payment
                        paymentType.getPaymentOption().setTransactionAmount(new Amount(String.valueOf(transactionAmountInDouble)));
                        walletList.add(paymentType.getPaymentOption());
                    }

                    try {
                        WalletPGPayment walletPGPayment = new WalletPGPayment((PaymentType.SplitPayment) paymentType, walletList);
                        walletPGCharge(walletPGPayment, callback);
                    } catch (CitrusException e) {
                        e.printStackTrace();
                    }
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error); // if we failed to get Consumer Profile return error
                }
            });
        } else { // no MVC, no Cash -  no need to fecth consumer profile.
            if (paymentType.getPaymentOption().isTokenizedPayment()) {
                List<PaymentOption> walletList = new ArrayList<>();
                if (paymentType.getPaymentOption().getTransactionAmount() == null) {
                    paymentType.getPaymentOption().setTransactionAmount(paymentType.getAmount());
                }
                walletList.add(paymentType.getPaymentOption());
                try {
                    WalletPGPayment walletPGPayment = new WalletPGPayment((PaymentType.SplitPayment) paymentType, walletList);
                    walletPGCharge(walletPGPayment, callback);
                } catch (CitrusException e) {
                    e.printStackTrace();
                }
            } else { //he is trying wallet PG with new Card..so redirect to Moto
                try {
                    pgPayment(new PaymentType.PGPayment((PaymentType.SplitPayment) paymentType), callback);
                } catch (CitrusException e) {
                    e.printStackTrace();
                }
            }


        }
    }

    boolean validateLogin(Callback callback) {
        if (!isUserLoggedIn) {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_SIGNIN_TOKEN_NOT_FOUND, CitrusResponse.Status.FAILED));
            return false;
        } else {
            return true;
        }
    }

    boolean validatePaymentOptions(PaymentOption paymentOption, MerchantPaymentOption merchantPaymentOption, Callback callback) {
        if (paymentOption != null && !validatePaymentOptionForMerchant(paymentOption, merchantPaymentOption)) {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_PAYMENT_OPTION_NOT_SUPPORTED, Status.FAILED));
            return false;
        } else {
            return true;
        }
    }

    boolean preValidation(PaymentOption paymentOption, MerchantPaymentOption merchantPaymentOption, Callback callback) {
        boolean isValid = false;
        isValid = validateLogin(callback);
        if (isValid) {
            isValid = validatePaymentOptions(paymentOption, merchantPaymentOption, callback);
            return isValid;
        } else {
            return false;
        }

    }

    /**
     * This method will update Payment Option with CVV if CVV not present
     *
     * @param paymentOption
     * @return
     */
    boolean updateOneTapPaymentOption(PaymentOption paymentOption, Callback callback) {
        if (!CitrusConfig.getInstance().isOneTapPaymentEnabled()) {  //if not one tap enabled return object as it is
            return true;
        }
        if (!(paymentOption instanceof CardOption)) { //if not card option return as it is
            return true;
        }
        if (((CardOption) paymentOption).getCardScheme() == CardOption.CardScheme.MAESTRO) { //maestro does not require CVV
            return true;
        }

        if (((CardOption) paymentOption).getCardCVV() != null) { // if CVV not null then return object as it is
            return true;
        }
        if (getCVVOfFingerPrint((CardOption) paymentOption) == null) {   // No saved CVV exists
            sendError(callback, new CitrusError(ResponseMessages.ERROR_NO_CVV_FOR_SAVED_CARD, Status.FAILED));
            return false;
        } else {
            ((CardOption) paymentOption).setCardCVV(getCVVOfFingerPrint((CardOption) paymentOption));
            return true;
        }
    }

    /**
     * Use this method to find the payment options using which payment can be done.
     *
     * @param callback
     */
    public void getPaymentDistribution(final Amount transactionAmount, final Callback<PaymentDistribution> callback) {
        getWallet(new Callback<List<PaymentOption>>() {
            @Override
            public void success(List<PaymentOption> paymentOptionList) {
                MVCOption mvcOption = null;
                CitrusCash citrusCash = null;
                PaymentDistribution paymentDistribution = new PaymentDistribution();
                double transactionAmountInDouble = transactionAmount.getValueAsDouble();
                boolean isAmountCompleted = false;
                double mvcAmount = 0;
                double citrusCashAmount = 0;
                for (PaymentOption paymentOption : paymentOptionList) {
                    if (paymentOption instanceof MVCOption) {
                        mvcOption = (MVCOption) paymentOption;
                    }
                    if (paymentOption instanceof CitrusCash) {
                        citrusCash = (CitrusCash) paymentOption;
                    }
                }

                if (mvcOption != null && mvcOption.getMaxBalance().getValueAsDouble() >= 1) {
                    if (transactionAmount.getValueAsDouble() <= mvcOption.getMaxBalance().getValueAsDouble()) {
                        mvcAmount = transactionAmount.getValueAsDouble();
                        isAmountCompleted = true;  //only MVC is enough

                    } else {
                        mvcAmount = mvcOption.getMaxBalanceRounded().getValueAsDouble();
                    }
                    BigDecimal amount1 = BigDecimal.valueOf(transactionAmountInDouble);//this is required. ie. 88.2 - 88.0 = returns 0.2 now.Direct double substraction returns 0.200000008
                    BigDecimal amount2 = BigDecimal.valueOf(mvcAmount);
                    BigDecimal difference = amount1.subtract(amount2);

                    transactionAmountInDouble = difference.doubleValue();


                    mvcOption.setTransactionAmount(new Amount(String.valueOf(mvcAmount)));
                    paymentDistribution.setMVCAvailable(true);
                    paymentDistribution.setMvcAmount(new Amount(String.valueOf(mvcAmount)));
                }

                if (!isAmountCompleted && citrusCash != null && citrusCash.getMaxBalance().getValueAsDouble() >= 1) {
                    if (transactionAmountInDouble <= citrusCash.getMaxBalance().getValueAsDouble()) {
                        citrusCashAmount = transactionAmountInDouble;
                        isAmountCompleted = true; // only Citrus Cash is enough
                    } else {
                        citrusCashAmount = citrusCash.getMaxBalanceRounded().getValueAsDouble();
                    }

                    BigDecimal amount1 = BigDecimal.valueOf(transactionAmountInDouble);//this is required. ie. 88.2 - 88.0 = returns 0.2 now.Direct double substraction returns 0.200000008
                    BigDecimal amount2 = BigDecimal.valueOf(citrusCashAmount);
                    BigDecimal difference = amount1.subtract(amount2);
                    transactionAmountInDouble = difference.doubleValue();
                    citrusCash.setTransactionAmount(new Amount(String.valueOf(citrusCashAmount)));
                    paymentDistribution.setCitrusCashAvailable(true);
                    paymentDistribution.setCitrusCashAmount(new Amount(String.valueOf(citrusCashAmount)));
                }

                if (!isAmountCompleted) {  // we still need payment option for the payment
                    paymentDistribution.setOthePaymentOptionAvailable(true);
                    paymentDistribution.setOtherPaymentOptionAmount(new Amount(String.valueOf(transactionAmountInDouble)));
                }

                sendResponse(callback, paymentDistribution);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error); // if we failed to get Consumer Profile return error
            }
        });
    }


    public void getActiveSubscriptions(final Callback<SubscriptionResponse> callback) {
       //  Callback<SubscriptionResponse> responseCallback =;
        retrofitAPIWrapper.getActiveSubscriptions( new Callback<SubscriptionResponse>() {
            @Override
            public void success(SubscriptionResponse subscriptionResponse) {
                activeSubscription = subscriptionResponse; //cache current active subscription for future use...
                sendResponse(callback, subscriptionResponse);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }


    /**
     * this method will deactivate current active subscription
     *
     * @param callback
     */

    public void deActivateSubscription(final Callback<SubscriptionResponse> callback) {

        //check first if active subscription exists then do deactivation
        isActiveSubscriptionPresent(new Callback<Boolean>() {
            @Override
            public void success(Boolean aBoolean) {
                if (aBoolean) { //active subscriptione exists
                    Callback<SubscriptionResponse> responseCallback = new Callback<SubscriptionResponse>() {
                        @Override
                        public void success(SubscriptionResponse subscriptionResponse) {
                            activeSubscription = null;
                            sendResponse(callback, subscriptionResponse);
                        }

                        @Override
                        public void error(CitrusError error) {
                            sendError(callback, error);
                        }
                    };
                    retrofitAPIWrapper.deActivateSubscription(activeSubscription.getSubscriptionId(), responseCallback);
                } else { //no active subscription exists
                    CitrusError citrusError = new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_NO_ACTIVE_SUBSCRIPTION, Status.FAILED);
                    sendError(callback, citrusError);
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error); // we failed to get active subscription
            }
        });


    }

    /**
     * this method will create subscription after successful load money called internally.
     *
     * @param callback
     * @param transactionResponse
     */
    private void createAutoLoadSubscription(final Callback callback, final TransactionResponse transactionResponse) {
        // retrofitAPIWrapper.createAutoLoadSubscription()
        Callback<SubscriptionResponse> subscriptionResponseCallback = new Callback<SubscriptionResponse>() {
            @Override
            public void success(SubscriptionResponse subscriptionResponse) {
                subscriptionResponse.setTransactionResponse(transactionResponse);
                subscriptionResponse.setSubscriptionResponseMessage(ResponseMessages.SUCCESS_MESSAGE_AUTO_LOAD);
                activeSubscription = subscriptionResponse;
                subscriptionRequest = null;
                mAUTO_load_type = null;
                sendResponse(callback, subscriptionResponse);//this wiil return callback in success
            }

            @Override
            public void error(CitrusError error) {
                subscriptionRequest = null;
                mAUTO_load_type = null;
                sendError(callback, error);//this will return callback in error
            }
        };

        retrofitAPIWrapper.createAutoLoadSubscription(subscriptionRequest, subscriptionResponseCallback);

    }


    /**
     * This method will update existing auto Load Subscription to higher value -- this is called internally
     *
     * @param callback
     * @param transactionResponse
     */
    private void updateAutoLoadSubscriptiontoHigherValue(final Callback callback, final TransactionResponse transactionResponse) {

        Callback<SubscriptionResponse> subscriptionResponseCallback = new Callback<SubscriptionResponse>() {
            @Override
            public void success(SubscriptionResponse subscriptionResponse) {
                activeSubscription = subscriptionResponse;
                updateSubscriptionRequest = null;
                mAUTO_load_type = null;
                subscriptionResponse.setTransactionResponse(transactionResponse);
                subscriptionResponse.setSubscriptionResponseMessage(ResponseMessages.SUCCESS_MESSAGE_UPDATE_SUBSCRIPTION);
                sendResponse(callback, subscriptionResponse);

            }

            @Override
            public void error(CitrusError error) {
                updateSubscriptionRequest = null;
                mAUTO_load_type = null;
                sendError(callback, error);
            }
        };
        retrofitAPIWrapper.updateSubScriptiontoHigherValue(updateSubscriptionRequest, subscriptionResponseCallback);
    }

    /**
     * This method will return true if active subscription exits else returns false
     *
     * @param callback
     */
    public void isActiveSubscriptionPresent(final Callback<Boolean> callback) {

        if (activeSubscription != null) { //
            callback.success(true);
        } else {
            retrofitAPIWrapper.getActiveSubscriptions(new Callback<SubscriptionResponse>() {
                @Override
                public void success(SubscriptionResponse subscriptionResponses) {
                    if (subscriptionResponses != null) {
                        activeSubscription = subscriptionResponses;
                        callback.success(true);
                    } else
                        callback.success(false);
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        }


    }

    /**
     * This method will create Auto Load transaction.Initiate as Auto Load transactions.
     *
     * @param loadMoney
     * @param callback
     */
    public synchronized void autoLoadMoney(final PaymentType.LoadMoney loadMoney, final Amount thresholdAmount, final Amount loadAmount, final Callback<SubscriptionResponse> callback) throws CitrusException {


        if (thresholdAmount == null)
            throw new CitrusException("ThresHold should not be null or empty.");
        if (loadAmount == null)
            throw new CitrusException("LoadAmount should not be null or empty.");
        if (loadAmount.compareTo(thresholdAmount) < 0) {
            throw new CitrusException("LoadAmount should not be less than ThresHold amount.");
        }

        if (thresholdAmount.getValueAsDouble() < new Double("500")) {
            throw new CitrusException("Threshold amount should not be less than 500");
        }

        if (loadAmount.getValueAsDouble() < new Double("500")) {
            throw new CitrusException("Threshold amount should not be less than 500");
        }
        final PaymentOption paymentOption = loadMoney.getPaymentOption();
        //Auto LOad available only for credit card payment or saved credit card
        if (paymentOption instanceof DebitCardOption || paymentOption instanceof NetbankingOption) {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_PAYMENT_OPTION, Status.FAILED));
            return;
        }

        //Auto Load is supported only for master and visa card
        if (!Arrays.asList(Constants.AUTO_LOAD_CARD_SCHEMS).contains(((CardOption) paymentOption).getCardScheme().toString())) {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_CARD_SCHEME_ERROR, Status.FAILED));
            return;
        }
        isActiveSubscriptionPresent(new Callback<Boolean>() {  //first check if there is active subscription before update
            @Override
            public void success(Boolean isAcitive) {
                if (!isAcitive) { //new auto load only if no active subscription exists...

                    getBINDetails((CardOption) paymentOption, new Callback<BinServiceResponse>() { //check bin response if its debit card return...
                        @Override
                        public void success(BinServiceResponse binServiceResponse) {
                            if (binServiceResponse.getCardType().contains("Credit")) {

                                mAUTO_load_type = AUTO_LOAD_TYPE.QUICK_AUTO_LOAD;
                                createAutoLoadRequest(paymentOption); //changed constructor so that it will be used from multiple locations
                                subscriptionRequest.setLoadAmount(loadAmount.getValue()); // we need to call this if some 1 wants auto load after transactions.
                                subscriptionRequest.setThresholdAmount(thresholdAmount.getValue());
                                Callback<TransactionResponse> responseCallback = new Callback<TransactionResponse>() {
                                    @Override
                                    public void success(TransactionResponse transactionResponse) {
                                        createAutoLoadSubscription(callback, transactionResponse);// create auto load...
                                    }

                                    @Override
                                    public void error(CitrusError error) {
                                        sendError(callback, error); //transaction failed..
                                    }
                                };
                                simpliPay(loadMoney, responseCallback);

                            } else {
                                sendError(callback, new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_CARD_SCHEME_ERROR, Status.FAILED));
                            }

                        }

                        @Override
                        public void error(CitrusError error) {

                            mAUTO_load_type = AUTO_LOAD_TYPE.QUICK_AUTO_LOAD;
                            createAutoLoadRequest(paymentOption); //changed constructor so that it will be used from multiple locations
                            subscriptionRequest.setLoadAmount(loadAmount.getValue()); // we need to call this if some 1 wants auto load after transactions.
                            subscriptionRequest.setThresholdAmount(thresholdAmount.getValue());
                            Callback<TransactionResponse> responseCallback = new Callback<TransactionResponse>() {
                                @Override
                                public void success(TransactionResponse transactionResponse) {
                                    createAutoLoadSubscription(callback, transactionResponse);// create auto load...
                                }

                                @Override
                                public void error(CitrusError error) {
                                    sendError(callback, error); //transaction failed..
                                }
                            };
                            simpliPay(loadMoney, responseCallback);

                        }
                    });

                } else {
                    CitrusError citrusError = new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_ACTIVE_SUBSCRIPTION, Status.FAILED);
                    sendError(callback, citrusError);
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error); //probably we failed to get active subscription from server...
            }
        });


    }

    /**
     * Call this method if u want to enable auto load after successful transaction.
     *
     * @param thresholdAmount
     * @param loadAmount
     * @param callback
     * @throws CitrusException
     */
    public synchronized void autoLoadMoney(final Amount thresholdAmount, final Amount loadAmount, final Callback<SubscriptionResponse> callback) throws CitrusException {
        if (thresholdAmount == null)
            throw new CitrusException("ThresHold should not be null or empty.");
        if (loadAmount == null)
            throw new CitrusException("LoadAmount should not be null or empty.");
        if (loadAmount.compareTo(thresholdAmount) < 0) {
            throw new CitrusException("LoadAmount should not be less than ThresHold amount.");
        }

        if (thresholdAmount.getValueAsDouble() < new Double("500")) {
            throw new CitrusException("Threshold amount should not be less than 500");
        }

        if (loadAmount.getValueAsDouble() < new Double("500")) {
            throw new CitrusException("Threshold amount should not be less than 500");
        }
        if (subscriptionRequest == null) {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_NO_ACTIVE_SUBSCRIPTION, Status.FAILED));
        } else {
            subscriptionRequest.setLoadAmount(loadAmount.getValue());
            subscriptionRequest.setThresholdAmount(thresholdAmount.getValue());
            createAutoLoadSubscription(callback, null);
        }
    }


    /**
     * This method will create auto load subscription request.
     *
     * @param paymentOption
     */
    private void createAutoLoadRequest(PaymentOption paymentOption) {
        //available only for Credit Card and Master and VISA scheme
        if (paymentOption instanceof CreditCardOption && Arrays.asList(Constants.AUTO_LOAD_CARD_SCHEMS).contains(((CardOption) paymentOption).getCardScheme().toString())) {
            if (subscriptionRequest == null) { // for auto load transaction amount will be set...
                if (paymentOption.isTokenizedPayment()) {
                    subscriptionRequest = new SubscriptionRequest(paymentOption.getToken());//subscription using saved card
                } else {
                    CardOption cardOption = (CardOption) paymentOption;
                    String holder = null;
                    if (getCitrusUser() != null) {
                        if (!TextUtils.isEmpty(getCitrusUser().getFirstName())) {
                            holder = getCitrusUser().getFirstName();
                        } else {
                            holder = Utils.getMaskedCardNumber(cardOption.getCardNumber());
                        }
                    } else {
                        holder = Utils.getMaskedCardNumber(cardOption.getCardNumber());
                    }
                    subscriptionRequest = new SubscriptionRequest(cardOption.getCardNumber(), cardOption.getCardExpiry(), holder);//subscription using new credit card
                }
            }
        } else {
            subscriptionRequest = null;//this is required for next transaction.
            mAUTO_load_type = null;
        }
    }


    /**
     * This method will return true if auto load can be initiated with existing transactions else returns false.
     *
     * @return
     */
    public void isAutoLoadAvailable(final Callback<Boolean> callback) {

        isActiveSubscriptionPresent(new Callback<Boolean>() {
            @Override
            public void success(Boolean aBoolean) {
                if (!aBoolean) {
                    if (mAUTO_load_type != null && mAUTO_load_type == AUTO_LOAD_TYPE.LAZY_AUTO_LOAD) {
                        callback.success(true);
                    } else {
                        callback.success(false);
                    }
                } else {
                    callback.success(false);
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /**
     * call this method when you are updating active subscription's threshold amount and loadamount to lower value than current amount
     *
     * @param thresholdAmount
     * @param autoloadAmount
     * @param callback
     */
    public void updateSubScriptiontoLoweValue(final Amount thresholdAmount, final Amount autoloadAmount, final Callback<SubscriptionResponse> callback) {


        isActiveSubscriptionPresent(new Callback<Boolean>() { //first check if there is active subscription before update
            @Override
            public void success(Boolean aBoolean) {
                if (aBoolean) { // active subscription exists
                    Callback<SubscriptionResponse> subscriptionResponseCallback = new Callback<SubscriptionResponse>() {
                        @Override
                        public void success(SubscriptionResponse subscriptionResponse) {
                            activeSubscription = subscriptionResponse;
                            sendResponse(callback, subscriptionResponse); //subscription updated..
                        }

                        @Override
                        public void error(CitrusError error) {
                            if (TextUtils.equals(error.getMessage(), ResponseMessages.ERROR_AUTO_LOAD_LOWER_AMOUNT_SERVER_RESPONSE)) {
                                CitrusError citrusError = new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_CUSTOMIZED_FOR_LOWER_AMOUNT, Status.FAILED);
                                sendError(callback, citrusError);
                            } else {
                                sendError(callback, error);
                            }
                        }
                    };

                    //request for update subscription
                    UpdateSubscriptionRequest updateSubscriptionRequest = null;
                    try {
                        updateSubscriptionRequest = new UpdateSubscriptionRequest(activeSubscription.getSubscriptionId(), autoloadAmount, thresholdAmount);
                        retrofitAPIWrapper.updateSubScriptiontoLowerValue(updateSubscriptionRequest, subscriptionResponseCallback);
                    } catch (CitrusException e) {
                        sendError(callback, new CitrusError(e.getMessage(), Status.FAILED));
                    }

                } else { // no active subscription to update...
                    CitrusError citrusError = new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_NO_ACTIVE_SUBSCRIPTION, Status.FAILED);
                    sendError(callback, citrusError);
                }
            }

            @Override
            public void error(CitrusError error) {
                callback.error(error);//probably we failed to get active subscription from server...
            }
        });
    }

    /**
     * This method will be used to update existing subscription to higher value.
     *
     * @param loadMoney
     * @param thresholdAmount
     * @param autoloadAmount
     * @param callback
     * @throws CitrusException
     */
    public void updateSubscriptiontoHigherValue(final PaymentType.LoadMoney loadMoney, final Amount thresholdAmount, final Amount autoloadAmount, final Callback<SubscriptionResponse> callback) throws CitrusException {

        //do validation first
        if (thresholdAmount == null) {
            throw new CitrusException("ThresHold should not be null or empty.");
        }
        if (autoloadAmount == null)
            throw new CitrusException("Auto Load amount should not be null or empty.");
        if (autoloadAmount.compareTo(thresholdAmount) < 0) {
            throw new CitrusException("Auto Load amount should not be less than ThresHold amount.");
        }

        if (thresholdAmount.getValueAsDouble() < new Double("500")) {
            throw new CitrusException("Threshold amount should not be less than 500");
        }

        if (autoloadAmount.getValueAsDouble() < new Double("500")) {
            throw new CitrusException("Threshold amount should not be less than 500");
        }

        PaymentOption paymentOption = loadMoney.getPaymentOption();
        //Auto LOad available only for credit card payment or saved credit card
        if (paymentOption instanceof DebitCardOption || paymentOption instanceof NetbankingOption) {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_PAYMENT_OPTION, Status.FAILED));
            return;
        }

        //Auto Load is supported only for master and visa card
        if (!Arrays.asList(Constants.AUTO_LOAD_CARD_SCHEMS).contains(((CardOption) paymentOption).getCardScheme().toString())) {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_CARD_SCHEME_ERROR, Status.FAILED));
            return;
        }
        //check if active subscription exists before update
        isActiveSubscriptionPresent(new Callback<Boolean>() {
            @Override
            public void success(Boolean aBoolean) {
                if (aBoolean) {
                    Amount currentThresholdAmount = new Amount(String.valueOf(activeSubscription.getThresholdAmount()));

                    if (thresholdAmount.compareTo(currentThresholdAmount) < 0) {  // for update to higher value should be equal or greater...
                        sendError(callback, new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_LOW_THRESHOLD_AMOUNT, Status.FAILED));
                        return;
                    }

                    Amount currentLoadAmount = new Amount(String.valueOf(activeSubscription.getLoadAmount()));
                    if (autoloadAmount.compareTo(currentLoadAmount) < 0) { // // for update to higher value should be equal or greater...
                        sendError(callback, new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_LOW_LOAD_AMOUNT, Status.FAILED));
                        return;
                    }

                    getBINDetails((CardOption) loadMoney.getPaymentOption(), new Callback<BinServiceResponse>() { //check if card is only credit
                        @Override
                        public void success(BinServiceResponse binServiceResponse) {
                            if (binServiceResponse.getCardType().contains("Credit")) {
                                try {
                                    updateSubscriptionRequest = new UpdateSubscriptionRequest(activeSubscription.getSubscriptionId(), autoloadAmount, thresholdAmount);
                                    mAUTO_load_type = AUTO_LOAD_TYPE.UPDATE_AUTO_LOAD;
                                    Callback<TransactionResponse> responseCallback = new Callback<TransactionResponse>() {
                                        @Override
                                        public void success(TransactionResponse transactionResponse) {
                                            updateAutoLoadSubscriptiontoHigherValue(callback, transactionResponse);//update the subscription...
                                        }

                                        @Override
                                        public void error(CitrusError error) {
                                            sendError(callback, error);
                                        }
                                    };
                                    simpliPay(loadMoney, responseCallback);
                                } catch (CitrusException e) {
                                    sendError(callback, new CitrusError(e.getMessage(), Status.FAILED));
                                }
                            } else {
                                sendError(callback, new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_CARD_SCHEME_ERROR, Status.FAILED));
                            }
                        }

                        @Override
                        public void error(CitrusError error) {
                            try {
                                updateSubscriptionRequest = new UpdateSubscriptionRequest(activeSubscription.getSubscriptionId(), autoloadAmount, thresholdAmount);
                                mAUTO_load_type = AUTO_LOAD_TYPE.UPDATE_AUTO_LOAD;
                                Callback<TransactionResponse> responseCallback = new Callback<TransactionResponse>() {
                                    @Override
                                    public void success(TransactionResponse transactionResponse) {
                                        updateAutoLoadSubscriptiontoHigherValue(callback, transactionResponse);//update the subscription...
                                    }

                                    @Override
                                    public void error(CitrusError error) {
                                        sendError(callback, error);
                                    }
                                };
                                simpliPay(loadMoney, responseCallback);
                            } catch (CitrusException e) {
                                sendError(callback, new CitrusError(e.getMessage(), Status.FAILED));
                            }
                        }
                    });

                } else {
                    CitrusError citrusError = new CitrusError(ResponseMessages.ERROR_AUTO_LOAD_NO_ACTIVE_SUBSCRIPTION, Status.FAILED);
                    sendError(callback, citrusError); //active subscription already exists..
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error); // we failed to get active subscription
            }
        });
    }
}