/*
   Copyright 2014 Citrus Payment Solutions Pvt. Ltd.
   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at
     http://www.apache.org/licenses/LICENSE-2.0
   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package com.citrus.sdk.payment;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Parcel;

import com.citrus.sdk.CitrusUser;
import com.citrus.sdk.classes.Amount;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by salil on 4/3/15.
 */
public final class CitrusCash extends PaymentOption {

    private CitrusUser citrusUser = null;
    private Amount maxBalance = null;

    public CitrusCash(String name, String token) {
        super(name, token);
    }

    public CitrusCash(Amount transactionAmount) {
        this.transactionAmount = transactionAmount;
        this.name = "Citrus Cash : \u20B9 " + transactionAmount.getValueAsDouble();
    }

    CitrusCash(Amount transactionAmount, Amount maxBalance) {
        this(transactionAmount);
        this.maxBalance = maxBalance;
        this.name = "Citrus Cash Balance : \u20B9 " + maxBalance.getValueAsDouble();
    }

    public CitrusCash(Amount transactionAmount, CitrusUser citrusUser) {
        this(transactionAmount);
        this.citrusUser = citrusUser;
    }

    public Amount getMaxBalance() {
        return maxBalance;
    }

    public Amount getMaxBalanceRounded() {
        int value = (int) Math.floor(maxBalance.getValueAsDouble());
        return new Amount(String.valueOf(value)); //this will convert 2.3 to 2.0 e.g
    }

    @Override
    public Drawable getOptionIcon(Context context) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            return context.getResources().getDrawable(context.getResources().getIdentifier("citrus_cash", "drawable", context.getPackageName()), null);
        } else {
            Drawable drawable;
            int resourceId = context.getResources().getIdentifier("citrus_cash", "drawable", context.getPackageName());
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                drawable = context.getResources().getDrawable(resourceId, null);
            } else {
                drawable = context.getResources().getDrawable(resourceId);
            }

            return drawable;
        }
    }

    @Override
    public String getSavePaymentOptionObject() {
        return null;
    }

    @Override
    public String getDynamicPricingPaymentMode() {
        return "PREPAID_CARD";
    }

    @Override
    public String getSaveDefaultPaymentOptionObject() {
        return null;
    }

    // @formatter:off

    /**
     * This will return the json object for moto or new make payment call.
     *
     * @return For Cards the format is
     * <p/>
     * {
     * "type":"paymentOptionToken",
     * "paymentMode":{
     * "cvv":"000",
     * "holder":"salilgodbole@gmail.com",
     * "number":"1234561234561234",
     * "scheme":"CPAY",
     * "type":"prepaid",
     * "expiry":"04\/2030"
     * }
     * }
     */
    // @formatter:on
    @Override
    public JSONObject getMOTOPaymentOptionObject() throws JSONException {
        JSONObject jsonObject = new JSONObject();

        jsonObject.put("type", "paymentOptionToken");
        // PaymentMode
        JSONObject paymentMode = new JSONObject();
        if (citrusUser != null) {
            paymentMode.put("holder", citrusUser.getEmailId());
        } else {
            paymentMode.put("holder", "");
        }
        paymentMode.put("number", "1234561234561234");
        paymentMode.put("scheme", "CPAY");
        paymentMode.put("type", "prepaid");
        paymentMode.put("cvv", "000");
        paymentMode.put("expiry", "04/2030");

        jsonObject.put("paymentMode", paymentMode);

        return jsonObject;
    }

    /**
     * This will return the json required for wallet charge api.
     *
     * @return The json in following format
     * <p/>
     * * {
     * "paymentMode": "PREPAID_CARD",
     * "amount": "3.0",
     * "currency": "INR"
     * }
     * @throws JSONException
     */

    @Override
    public JSONObject getWalletChargePaymentOptionObject() throws JSONException {
        JSONObject jsonObject = new JSONObject();

        if (transactionAmount != null) {
            jsonObject.put("paymentMode", "PREPAID_CARD");

            // Amount Details.
            jsonObject.put("amount", String.valueOf(transactionAmount.getValueAsDouble()));
            jsonObject.put("currency", transactionAmount.getCurrency());
        }
        return jsonObject;
    }

    @Override
    public String toString() {
        return super.toString() + "CitrusCash{" +
                "Balance amount='" + maxBalance.getValueAsDouble() + '\'' +
                '}';
    }

    public void setName(String name) {
        super.name = name;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        super.writeToParcel(dest, flags);
        dest.writeParcelable(this.citrusUser, 0);
    }

    protected CitrusCash(Parcel in) {
        super(in);
        this.citrusUser = in.readParcelable(CitrusUser.class.getClassLoader());
    }

    public static final Creator<CitrusCash> CREATOR = new Creator<CitrusCash>() {
        public CitrusCash createFromParcel(Parcel source) {
            return new CitrusCash(source);
        }

        public CitrusCash[] newArray(int size) {
            return new CitrusCash[size];
        }
    };
}
