package com.citrus.sdk.network.request;

import android.util.Log;

import com.android.volley.AuthFailureError;
import com.android.volley.NetworkResponse;
import com.android.volley.ParseError;
import com.android.volley.Request;
import com.android.volley.Response;
import com.android.volley.VolleyLog;
import com.android.volley.toolbox.HttpHeaderParser;
import com.citrus.sdk.network.NetworkConstants;
import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.Type;
import java.util.Map;

/**
 * Created by vinay on 5/4/16.
 */

public class GsonRequest<T> extends Request<T> {

    private static final String TAG = GsonRequest.class.getSimpleName();

    private final Gson gson = new Gson();
    private final Class<T> clazz;
    private final Type type ;
    private final Map<String, String> headers;
    private final Response.Listener<T> listener;

    private final Map<String, String> params;
    private final String bodyContentResponseType;

    private final String mRequestBody;

    public GsonRequest(final String url, final Class<T> clazz,
                       final int method, final Map<String, String> headers,
                       final Map<String, String> params, final String bodyContentResponseType,
                       final Response.Listener<T> listener, final Response.ErrorListener errorListener,
                       String body) {
        super(method, url, errorListener);

        Log.i(TAG, TAG + ".GsonRequest(): class = " + clazz + ", " + url);

        this.type                       = null ;
        this.clazz                      = clazz;
        this.headers                    = headers;
        this.listener                   = listener;
        this.params                     = params;
        this.bodyContentResponseType    = bodyContentResponseType;
        this.mRequestBody               = body;
    }

    public GsonRequest(final String url, final Type type,
                       final int method, final Map<String, String> headers,
                       final Map<String, String> params, final String bodyContentResponseType,
                       final Response.Listener<T> listener, final Response.ErrorListener errorListener,
                       String body) {
        super(method, url, errorListener);

        Log.i(TAG, TAG + ".GsonRequest(): type = " + type + ", " + url);

        this.type                       = type ;
        this.clazz                      = null;
        this.headers                    = headers;
        this.listener                   = listener;
        this.params                     = params;
        this.bodyContentResponseType    = bodyContentResponseType;
        this.mRequestBody               = body;
    }

    @Override
    public Map<String, String> getHeaders() throws AuthFailureError {
        Log.i(TAG, TAG + ".getHeaders(): headers = " + headers);
        return headers != null ? headers : super.getHeaders();
    }

    @Override
    protected void deliverResponse(T response) {
        Log.i(TAG, TAG + ".deliverResponse(): response = " + response);
        listener.onResponse(response);
    }

    @Override
    protected Response<T> parseNetworkResponse(NetworkResponse response) {
        Log.i(TAG, TAG + ".parseNetworkResponse(): for " + super.getUrl());
        Log.i(TAG, TAG + ".parseNetworkResponse(): response = " + response);

        final Map<String, String> responseHeaders = response.headers;
        if (responseHeaders != null) {
            Log.i(TAG, TAG + ".parseNetworkResponse(): total responseHeaders = " + responseHeaders.size() + " =>");
            for (String key : responseHeaders.keySet()) {
                Log.i(TAG, TAG + ".parseNetworkResponse(): Header = " + key + ':' + responseHeaders.get(key));
            }
        }
        try {

            final String json = new String(response.data, HttpHeaderParser.parseCharset(response.headers));
            Log.i(TAG, TAG + ".parseNetworkResponse(): json = " + json);

            if( clazz != null ){
                return Response.success(
                        gson.fromJson(json, clazz),
                        HttpHeaderParser.parseCacheHeaders(response));
            }else if( type != null ){
                 return (Response<T>) Response.success(
                        gson.fromJson(json, type ),
                        HttpHeaderParser.parseCacheHeaders(response));
            }
        } catch (UnsupportedEncodingException e) {
            return Response.error(new ParseError(e));
        } catch (JsonSyntaxException e) {
            return Response.error(new ParseError(e));
        } catch (Exception e) {
            return Response.error(new ParseError(e));
        }
        return null;
    }

    @Override
    protected Map<String, String> getParams() throws AuthFailureError {
        Log.i(TAG, TAG + ".getParams(): params = " + params);
        return params != null ? params : super.getParams();
    }


    @Override
    public String getBodyContentType() {
        Log.i(TAG, TAG + ".getBodyContentType(): bodyContentResponseType = " + bodyContentResponseType);
        return bodyContentResponseType != null ? bodyContentResponseType : super.getBodyContentType();
    }

    @Override
    public byte[] getBody() throws AuthFailureError {
        Log.i(TAG, TAG + ".getBody(): mRequestBody = " + mRequestBody + ", params = " + params);

        try {
            if (mRequestBody != null) {
                return mRequestBody.getBytes(NetworkConstants.PROTOCOL_CHARSET);
            } else if (params != null) {
                String body = "";
                for (String key : params.keySet()) {
                    body += "&" + key + "=" + params.get(key);
                }
                Log.i(TAG, TAG + ".getBody(): param body = " + body);
                return body.getBytes(NetworkConstants.PROTOCOL_CHARSET);
            }
        } catch (UnsupportedEncodingException uee) {
            VolleyLog.wtf("Unsupported Encoding while trying to get the bytes of %s using %s",
                    mRequestBody, NetworkConstants.PROTOCOL_CHARSET);
        }

        return null;
    }
}

