package com.citrus.sdk.network;

import android.content.Context;
import android.text.TextUtils;

import com.android.volley.NetworkResponse;
import com.android.volley.VolleyError;
import com.citrus.sdk.Callback;
import com.citrus.sdk.Environment;
import com.citrus.sdk.ResponseMessages;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusResponse;

import org.json.JSONException;
import org.json.JSONObject;

import java.net.HttpURLConnection;

/**
 * Created by vinay on 1/4/16.
 */
public abstract class BaseClient {
    protected final Context mContext;
    protected final Environment environment ;
    protected TokenUtils tokenUtils ;

    protected BaseClient(Context context, Environment environment) {
        this.mContext       = context;
        this.environment    = environment;
        tokenUtils          = TokenUtils.getInstance( mContext );
    }

    protected boolean validate() {
        return tokenUtils.validate() ;
    }

    public abstract String getBaseUrl( ) ;

    public <T> void sendResponse(Callback callback, T t) {
        if (callback != null) {
            callback.success(t);
        }
    }

    public <T> void sendError(Callback callback, VolleyError volleyError) {
        if (callback != null ) {
            CitrusError citrusError;
            final NetworkResponse networkResponse   = volleyError.networkResponse ;
            final String volleyErrorMessage         = volleyError.getMessage() ;

            if (volleyErrorMessage != null) {
                if( networkResponse != null ){

                    final int statusCode = networkResponse.statusCode ;

                    // If the response code is 500 then directly send message as INTERNAL ERROR
                    if ( statusCode == HttpURLConnection.HTTP_INTERNAL_ERROR) {
                        citrusError = new CitrusError(ResponseMessages.ERROR_INTERNAL_SERVER_ERROR, CitrusResponse.Status.FAILED);
                    } else {
                        String message = null;
                        try {
                            message = new String(networkResponse.data);
                            JSONObject jsonObject = new JSONObject(message);
                            // If the response does not contain error_description then look for errorMessage.
                            String errorMessage = null;
                            if (!TextUtils.isEmpty(jsonObject.optString("error_description"))) {
                                errorMessage = jsonObject.optString("error_description");
                            } else if (!TextUtils.isEmpty(jsonObject.optString("errorMessage"))) {
                                errorMessage = jsonObject.optString("errorMessage");
                            } else {
                                errorMessage = volleyErrorMessage;
                            }

                            citrusError = new CitrusError(errorMessage, CitrusResponse.Status.FAILED);
                        } catch (JSONException e) {
                            e.printStackTrace();
                            citrusError = new CitrusError(volleyErrorMessage, CitrusResponse.Status.FAILED);
                        } catch (Exception e) {
                            e.printStackTrace();
                            citrusError = new CitrusError(ResponseMessages.ERROR_SOMETHING_WENT_WRONG, CitrusResponse.Status.FAILED);
                        }
                    }

                    citrusError.setStatusCode( statusCode );
                }
                else if (volleyError.getCause() != null) {
                    citrusError = new CitrusError(volleyError.getCause().getMessage(), CitrusResponse.Status.FAILED);
                } else {
                    citrusError = new CitrusError(ResponseMessages.ERROR_SOMETHING_WENT_WRONG, CitrusResponse.Status.FAILED);
                }
            } else {
                citrusError = new CitrusError(ResponseMessages.ERROR_SOMETHING_WENT_WRONG, CitrusResponse.Status.FAILED);
            }

            sendError(callback, citrusError);
        }
    }

    protected void sendError(Callback callback, Throwable t) {
        if (callback != null) {
            sendError(callback, new CitrusError(t.getMessage(), CitrusResponse.Status.FAILED));
        }
    }

    protected void sendError(Callback callback, CitrusError citrusError) {
        if (callback != null) {
            callback.error(citrusError);
        }
    }

    protected void setVanity(final String vanity ){
        tokenUtils.setVanity( vanity );
    }

    protected String getVanity( ){
        return  tokenUtils.getVanity() ;
    }
}
