package com.citrus.sdk.walletpg;

import android.text.TextUtils;
import android.util.Log;

import com.citrus.sdk.CitrusUser;
import com.citrus.sdk.TransactionResponse;
import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.payment.CitrusCash;
import com.citrus.sdk.payment.MVCOption;
import com.citrus.sdk.payment.PaymentOption;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by salil on 5/2/16.
 */
public class WalletPGPaymentResponse {

    private String redirectUrl;
    private String responseCode;
    private TransactionResponse.TransactionStatus transactionStatus;
    private String description;
    private String transactionId;
    private CitrusUser citrusUser;
    private String signature;
    private Amount transactionAmount;
    private List<PaymentOption> listPaymentOptions;
    private String authIdCode;
    private String issuerRefNo;
    private String pgTxnId;
    private String pgRespCode;

    public WalletPGPaymentResponse(String redirectUrl, String responseCode, TransactionResponse.TransactionStatus transactionStatus,
                                   String description, String transactionId, CitrusUser citrusUser, String signature, Amount transactionAmount,
                                   List<PaymentOption> listPaymentOptions, String authIdCode, String issuerRefNo, String pgTxnId) {
        this.redirectUrl = redirectUrl;
        this.responseCode = responseCode;
        this.transactionStatus = transactionStatus;
        this.description = description;
        this.transactionId = transactionId;
        this.citrusUser = citrusUser;
        this.signature = signature;
        this.transactionAmount = transactionAmount;
        this.listPaymentOptions = listPaymentOptions;
        this.authIdCode = authIdCode;
        this.issuerRefNo = issuerRefNo;
        this.pgTxnId = pgTxnId;
    }

    public String getRedirectUrl() {
        return redirectUrl;
    }

    public String getResponseCode() {
        return responseCode;
    }

    public TransactionResponse.TransactionStatus getTransactionStatus() {
        return transactionStatus;
    }

    public String getDescription() {
        return description;
    }

    public String getTransactionId() {
        return transactionId;
    }

    public CitrusUser getCitrusUser() {
        return citrusUser;
    }

    public String getSignature() {
        return signature;
    }

    public Amount getTransactionAmount() {
        return transactionAmount;
    }

    public List<PaymentOption> getListPaymentOptions() {
        return listPaymentOptions;
    }

    /**
     * {
     * "redirectUrl": "https://salty-plateau-1529.herokuapp.com/redirectURL.sandbox.wallet.php",
     * "responseCode": "0",
     * "txnStatus": "Success",
     * "txnMsg": {
     * "respCode": "0",
     * "status": "Success",
     * "description": "Transaction successful",
     * "merchantTransactionId": "145466886075877",
     * "amount": 30,
     * "currency": "INR",
     * "signature": "0f596ef9ef425033952bbbd1c2147b580a9e4eb5",
     * "txnDateTime": 1454674010205,
     * "userDetails": {
     * "firstName": "Salil",
     * "lastName": "Godbole",
     * "email": "salilgodbole@gmail.com",
     * "mobile": "9970950374"
     * },
     * "paymentDetailsLst": [
     * {
     * "paymentMode": "PREPAID_CARD",
     * "status": "Success",
     * "description": "Transaction successful",
     * "amount": "30.0",
     * "currency": "INR"
     * }
     * ],
     * "paymentMode": "WALLET",
     * "walletType": "Citrus Wallet",
     * "citrusTransactionId": "WTX1602051206505375345"
     * }
     * }
     *
     * @param json
     * @return
     */
    public static WalletPGPaymentResponse fromJSON(String json) {
        WalletPGPaymentResponse walletPGPaymentResponse = null;

        if (!TextUtils.isEmpty(json)) {
            try {
                JSONObject jsonObject = new JSONObject(json);
                String redirectUrl = jsonObject.optString("redirectUrl");
                String responseCode = jsonObject.optString("responseCode");
                String txnStatus = jsonObject.optString("txnStatus");
                TransactionResponse.TransactionStatus transactionStatus = TransactionResponse.TransactionStatus.getTransactionStatus(txnStatus);

                JSONObject txnMsg = jsonObject.optJSONObject("txnMsg");
                String description = txnMsg.optString("description");
                String transactionId = txnMsg.optString("merchantTransactionId");
                String amount = txnMsg.optString("amount");
                String currency = txnMsg.optString("currency");
                Amount transactionAmount = new Amount(amount, currency);
                String signature = txnMsg.optString("signature");
                JSONObject userDetails = txnMsg.optJSONObject("userDetails");
                CitrusUser citrusUser = CitrusUser.fromJSONObject(userDetails, false);

                JSONArray paymentDetailsArray = jsonObject.optJSONArray("paymentDetailsLst");
                List<PaymentOption> paymentOptionList = null;
                if (paymentDetailsArray != null) {
                    for (int i = 0; i < paymentDetailsArray.length(); i++) {
                        JSONObject paymentOptionObject = paymentDetailsArray.optJSONObject(i);
                        String paymentMode = paymentOptionObject.optString("paymentMode");
                        String amountValue = paymentOptionObject.optString("amount");
                        String currencyValue = paymentOptionObject.optString("currency");
                        PaymentOption paymentOption = null;
                        if ("PREPAID_CARD".equalsIgnoreCase(paymentMode)) {
                            paymentOption = new CitrusCash(new Amount(amountValue, currencyValue));
                        } else if ("MVC".equalsIgnoreCase(paymentMode)) {
                            paymentOption = new MVCOption(new Amount(amountValue, currencyValue), null);
                        }

                        if (paymentOption != null) {
                            if (paymentOptionList == null) {
                                paymentOptionList = new ArrayList<>();
                            }

                            paymentOptionList.add(paymentOption);
                        }
                    }
                }

                String authIdCode = txnMsg.optString("authIdCode");
                String issuerRefNo = txnMsg.optString("issuerRefNo");
                String pgTxnId = txnMsg.optString("pgTxnId");

                walletPGPaymentResponse = new WalletPGPaymentResponse(redirectUrl, responseCode, transactionStatus,
                        description, transactionId, citrusUser, signature, transactionAmount,
                        paymentOptionList, authIdCode, issuerRefNo, pgTxnId);

            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        return walletPGPaymentResponse;
    }

    public final String getURLEncodedParams() {
        StringBuffer buffer = new StringBuffer();

        buffer.append("pgTxnNo=");
        buffer.append(pgTxnId);

        buffer.append("&issuerRefNo=");
        buffer.append(issuerRefNo);

        buffer.append("&authIdCode=");
        buffer.append(authIdCode);

        buffer.append("&TxId=");
        buffer.append(getTransactionId());

        if (transactionAmount != null) {
            buffer.append("&amount=");
            buffer.append(transactionAmount.getValueAsDouble());

            buffer.append("&currency=");
            buffer.append(transactionAmount.getCurrency());
        }

        if (transactionStatus != null) {
            String status = "";
            String description = "";
            String pgRespCode = "";
            switch (transactionStatus) {
                case SUCCESSFUL:
                    status = "Success";
                    description = "Transaction Successful";
                    pgRespCode = "0";
                    break;
                case FAILED:
                    status = "Fail";
                    description = "Transaction Failed";
                    pgRespCode = "1";
                    break;
                case CANCELLED:
                    status = "CANCELED";
                    description = "Transaction Cancelled By User";
                    pgRespCode = "3";
                    break;
            }

            buffer.append("&TxStatus=");
            buffer.append(status);

            buffer.append("&description=");
            buffer.append(description);

            buffer.append("&TxMsg=");
            buffer.append(description);

            buffer.append("&pgRespCode=");
            buffer.append(pgRespCode);
        }

        buffer.append("&signature=");
        buffer.append(signature);

        buffer.append("&paymentMode=WALLET");

        if (citrusUser != null) {
            buffer.append("&firstName=");
            buffer.append(citrusUser.getFirstName());

            buffer.append("&lastName=");
            buffer.append(citrusUser.getLastName());

            buffer.append("&email=");
            buffer.append(citrusUser.getEmailId());

            buffer.append("&mobileNo=");
            buffer.append(citrusUser.getMobileNo());

            CitrusUser.Address address = citrusUser.getAddress();
            if (address != null) {
                buffer.append("&addressStreet1=");
                buffer.append(address.getStreet1());

                buffer.append("&addressStreet2=");
                buffer.append(address.getStreet2());

                buffer.append("&mobileNo=");
                buffer.append(citrusUser.getMobileNo());

                buffer.append("&addressCity=");
                buffer.append(address.getCity());

                buffer.append("&addressCountry=");
                buffer.append(address.getCountry());

                buffer.append("&addressState=");
                buffer.append(address.getState());

                buffer.append("&addressZip=");
                buffer.append(address.getZip());
            }
        }

        return buffer.toString();
    }

}