package com.citrus.sdk;

import android.content.Context;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Log;

import com.citrus.mobile.OAuth2GrantType;
import com.citrus.sdk.classes.AccessToken;
import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.classes.BCCancelResponse;
import com.citrus.sdk.classes.BCResponse;
import com.citrus.sdk.classes.BinServiceResponse;
import com.citrus.sdk.classes.CardBinDetails;
import com.citrus.sdk.classes.CashoutInfo;
import com.citrus.sdk.classes.CitrusPrepaidBill;
import com.citrus.sdk.classes.CitrusUMResponse;
import com.citrus.sdk.classes.LinkUserExtendedResponse;
import com.citrus.sdk.classes.LinkUserResponse;
import com.citrus.sdk.classes.MemberInfo;
import com.citrus.sdk.classes.PGHealthResponse;
import com.citrus.sdk.classes.StructResponsePOJO;
import com.citrus.sdk.classes.UpdateSubscriptionRequest;
import com.citrus.sdk.classes.Utils;
import com.citrus.sdk.dynamicPricing.DynamicPricingRequestType;
import com.citrus.sdk.dynamicPricing.DynamicPricingResponse;
import com.citrus.sdk.network.BillingClient;
import com.citrus.sdk.network.BinServiceClient;
import com.citrus.sdk.network.CitrusBasePgClient;
import com.citrus.sdk.network.DynamicPricingPgClient;
import com.citrus.sdk.network.PgClient;
import com.citrus.sdk.network.PrepaidClient;
import com.citrus.sdk.network.TokenUtils;
import com.citrus.sdk.network.UmClient;
import com.citrus.sdk.network.WalletPgClient;
import com.citrus.sdk.network.request.ApiExecutor;
import com.citrus.sdk.network.request.ApiRequest;
import com.citrus.sdk.otp.NetBankForOTP;
import com.citrus.sdk.payment.CardOption;
import com.citrus.sdk.payment.MerchantPaymentOption;
import com.citrus.sdk.payment.PaymentBill;
import com.citrus.sdk.payment.PaymentOption;
import com.citrus.sdk.payment.PaymentType;
import com.citrus.sdk.response.BindUserResponse;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusResponse;
import com.citrus.sdk.response.PaymentResponse;
import com.citrus.sdk.response.SubscriptionResponse;
import com.citrus.sdk.walletpg.WalletConsumerProfile;
import com.citrus.sdk.walletpg.WalletPGPaymentResponse;
import com.orhanobut.logger.Logger;

import java.util.List;

/**
 * Created by salil on 27/11/15.
 */
public class RetrofitAPIWrapper {

    private static final String TAG = RetrofitAPIWrapper.class.getSimpleName();

    private static RetrofitAPIWrapper instance = null;

    private Context context = null;
    private boolean isPrepaymentTokenValid = false;

    private PgClient volleyPgApi;
    private CitrusBasePgClient citrusBasePgApi;
    private UmClient volleyUmApi;
    private WalletPgClient volleyWalletPgApi;
    private BinServiceClient volleyBinServiceApi;
    private DynamicPricingPgClient volleyDynamicPricingPgApi;
    private BillingClient volleyBillingApi;
    private PrepaidClient volleyPrePaidApi;

    private ApiExecutor volleyApiExecutor;

    private NetBankForOTP netBankForOTP = NetBankForOTP.UNKNOWN;

    public static RetrofitAPIWrapper getInstance(Context context) {
        if (instance == null) {
            synchronized (RetrofitAPIWrapper.class) {
                if (instance == null) {
                    instance = new RetrofitAPIWrapper(context);
                }
            }
        }

        return instance;
    }

    private RetrofitAPIWrapper(Context context) {
        this.context = context;
        // CitrusConfig.getInstance().setLogChangeListener(umClient);//base client will listen for enable disable logs...
    }

    public synchronized void init(final String signupId, final String signupSecret,final  String signinId,
                                  final String signinSecret, final String vanity, final Environment environment) {
        final TokenUtils tokenUtils = TokenUtils.getInstance(context);
        tokenUtils.init(signupId, signupSecret, signinId, signinSecret, vanity);

        volleyPgApi                 = PgClient.getInstance(this.context, environment);
        citrusBasePgApi             = CitrusBasePgClient.getInstance(context, environment);
        volleyWalletPgApi           = WalletPgClient.getInstance(context, environment);
        volleyUmApi                 = UmClient.getInstance(this.context, environment);
        volleyBinServiceApi         = BinServiceClient.getInstance(this.context, environment);
        volleyBillingApi            = BillingClient.getInstance(this.context, environment);
        volleyDynamicPricingPgApi   = DynamicPricingPgClient.getInstance(this.context, environment);
        volleyPrePaidApi            = PrepaidClient.getInstance(context, environment);
        volleyApiExecutor           = ApiExecutor.getInstance(context);
    }

    /*
    UM APIs.
     */
    public synchronized void getSignUpToken(final Callback<AccessToken> callback) {
        volleyUmApi.getSignUpToken(callback);
    }

    public synchronized void createUser(final String emailId, final String mobileNo, final Callback<String> callback) {
        volleyUmApi.createUser(emailId, mobileNo, callback);
    }

    public synchronized void getUserNameToken(String userName, Callback<AccessToken> callback) {
        volleyUmApi.getUserNameToken(userName, callback);
    }

    public void getUserNameToken(Callback<AccessToken> callback) {
        volleyUmApi.getUserNameToken(callback);
    }

    public void getPrepaidToken(Callback<AccessToken> callback) {
        volleyUmApi.getPrepaidToken(callback);
    }

    public void getPrepaidPayToken(Callback<AccessToken> callback) {
        volleyUmApi.getPrepaidPayToken(callback);
    }

    /**
     * This api will check whether the user is existing user or not. If the user is existing user,
     * then it will return the existing details, else it will create an account internally and
     * then call signUp to set the password and activate the account.
     *
     * @param emailId  - emailId of the user
     * @param mobileNo - mobileNo of the user
     * @param callback - callback
     */
    public synchronized void isCitrusMember(final String emailId, final String mobileNo, final Callback<Boolean> callback) {
        volleyUmApi.isCitrusMember(emailId, mobileNo, callback);
    }

    public synchronized void getMemberInfo(final String emailId, final String mobileNo, final Callback<MemberInfo> callback) {
        volleyUmApi.getMemberInfo(emailId, mobileNo, callback);
    }

    public synchronized void bindUserByMobile(String emailId, String mobileNo, com.citrus.sdk.Callback<BindUserResponse> callback) {
        volleyUmApi.bindUserByMobile(emailId, mobileNo, callback);
    }

    @Deprecated
    /**
     * use {@link RetrofitAPIWrapper#linkUserExtended(String, String, Callback)}
     */
    public synchronized void linkUserWithOTP(final String emailId, final String mobileNo,
                                             final boolean forceMobileVerification, final Callback<LinkUserResponse> callback) {
        volleyUmApi.linkUserWithOTP(emailId, mobileNo, forceMobileVerification, callback);
    }

    public synchronized void linkUserExtended(final String emailId, final String mobileNo, final Callback<LinkUserExtendedResponse> callback) {
        volleyUmApi.linkUserExtended(emailId, mobileNo, callback);
    }

    public void linkUserExtendedVerifyEOTPAndUpdateMobile(final String emailId, String signInGrantType, final String linkUserPassword, LinkUserExtendedResponse linkUserExtended, final Callback<CitrusResponse> callback) {
        volleyUmApi.linkUserExtendedVerifyEOTPAndUpdateMobile(emailId, signInGrantType, linkUserPassword, linkUserExtended, new Callback<CitrusResponse>() {
            @Override
            public void success(CitrusResponse citrusResponse) {
                isPrepaymentTokenValid = true; //since we are checking if this token present in ProceedtoPayCash method
                sendResponse(callback, new CitrusResponse(ResponseMessages.SUCCESS_MESSAGE_SIGNIN, CitrusResponse.Status.SUCCESSFUL));
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public void linkUserExtendedVerifyMobileAndSignIn(final LinkUserExtendedResponse linkUserExtendedResponse, final String verificationCode, final Callback<CitrusResponse> callback) {
        volleyUmApi.linkUserExtendedVerifyMobileAndSignIn(linkUserExtendedResponse, verificationCode, new Callback<CitrusResponse>() {
            @Override
            public void success(CitrusResponse citrusResponse) {
                isPrepaymentTokenValid = true; //since we are checking if this token present in ProceedtoPayCash method
                sendResponse(callback, new CitrusResponse(ResponseMessages.SUCCESS_MESSAGE_SIGNIN, CitrusResponse.Status.SUCCESSFUL));

            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    @Deprecated
    /**
     * use {@link RetrofitAPIWrapper#linkUserExtended(String, String, Callback)}
     */
    public synchronized void signUp(final String emailId, final String mobileNo, final String password, final com.citrus.sdk.Callback<CitrusResponse> callback) {

        volleyUmApi.signUp(emailId, mobileNo, password, new Callback<CitrusResponse>() {
            @Override
            public void success(CitrusResponse citrusResponse) {
                signIn(emailId, password, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /**
     * This method will sign in into prepaid acccount. This method retrieves and saves grant type username and password token.
     * <p/>
     * Default grantType is <b> password </b>.
     * Use {@link RetrofitAPIWrapper#signIn(String, String, OAuth2GrantType, Callback)} for sending other granttypes.
     *
     * @param emailId
     * @param password
     * @param callback
     */

    @Deprecated
    /**
     * use {@link RetrofitAPIWrapper#linkUserExtended(String, String, Callback)}
     */
    public synchronized void signIn(final String emailId, final String password, final com.citrus.sdk.Callback<CitrusResponse> callback) {
        signIn(emailId, password, OAuth2GrantType.password, callback);
    }

    /**
     * This method will sign in into prepaid acccount. This method retrieves and saves grant type username and password token.
     *
     * @param emailId
     * @param password
     * @param grantType
     * @param callback
     */
    private synchronized void signIn(final String emailId, final String password,
                                     final OAuth2GrantType grantType, final com.citrus.sdk.Callback<CitrusResponse> callback) {
        volleyUmApi.getUsernameAndPrepaidToken(emailId, password, grantType, new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                activatePrepaidUser(new Callback<Amount>() {
                    @Override
                    public void success(Amount amount) {
                        sendResponse(callback, new CitrusResponse(ResponseMessages.SUCCESS_MESSAGE_SIGNIN, CitrusResponse.Status.SUCCESSFUL));
                        isPrepaymentTokenValid = true;
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void signInWithMobileNo(String mobileNo, String password, Callback<CitrusResponse> callback) {
        signIn(mobileNo, password, callback);
    }

    // TODO: Check for load money flow
    public synchronized void signInWithOTP(String emailIdOrMobileNo, String otp, Callback<CitrusResponse> callback) {
        signIn(emailIdOrMobileNo, otp, OAuth2GrantType.onetimepass, callback);
    }

    public synchronized void resetPassword(String emailId, @NonNull Callback<CitrusResponse> callback) {
        volleyUmApi.resetPassword(emailId, callback);
    }

    public synchronized void updateMobile(String mobileNo, Callback<String> callback) {
        volleyUmApi.updateMobile(mobileNo, callback);
    }

    public synchronized void verifyMobile(String verificationCode, Callback<String> callback) {
        volleyUmApi.verifyMobile(verificationCode, callback);
    }

    public synchronized void getProfileInfo(Callback<CitrusUser> callback) {
        volleyUmApi.getProfileInfo(callback);
    }

    public synchronized void changePassword(String oldPassword, String newPassword, Callback<CitrusUMResponse> changePasswordResponseCallback) {
        volleyUmApi.changePassword(oldPassword, newPassword, changePasswordResponseCallback);
    }

    public synchronized void updateProfileInfo(String firstName, String lastName, Callback<CitrusUMResponse> citrusUMResponseCallback) {
        volleyUmApi.updateProfileInfo(firstName, lastName, citrusUMResponseCallback);
    }

    public synchronized void sendOneTimePassword(String source, String otpType, String identity, Callback<CitrusUMResponse> umResponseCallback) {
        volleyUmApi.sendOneTimePassword(source, otpType, identity, umResponseCallback);
    }

    public synchronized void resetUserPassword(String emailId, @NonNull Callback<CitrusUMResponse> callback) {
        volleyUmApi.resetUserPassword(emailId, callback);
    }

    public synchronized void signUpUser(final String email, String mobile, final String password,
                                        final String firstName, final String lastName, final String sourceType,
                                        final boolean markMobileVerified, final boolean markEmailVerified,
                                        final Callback<CitrusResponse> callback) {
        volleyUmApi.signUpUser(email, mobile, password, firstName, lastName, sourceType, markMobileVerified, markEmailVerified, new Callback<CitrusResponse>() {
            @Override
            public void success(CitrusResponse citrusResponse) {
                signIn(email, password, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void signOut(final Callback<CitrusResponse> callback) {

        volleyUmApi.signOut(new Callback<Boolean>() {
            @Override
            public void success(Boolean aBoolean) {
                if (aBoolean) {
                    resetPrepaymentTokenValidity();
                    CitrusResponse citrusResponse = new CitrusResponse("User Logged Out Successfully.", CitrusResponse.Status.SUCCESSFUL);
                    sendResponse(callback, citrusResponse);

                } else {
                    CitrusError citrusError = new CitrusError("Failed to logout.", CitrusResponse.Status.FAILED);
                    callback.error(citrusError);
                }

            }

            @Override
            public void error(CitrusError error) {
                CitrusError citrusError = new CitrusError("Failed to logout.", CitrusResponse.Status.FAILED);
                callback.error(citrusError);

            }
        });
    }


    /*********************
     * Prepaid API Start *
     ********************/

    public synchronized void resetPrepaymentTokenValidity() {

        isPrepaymentTokenValid = false;
    }

    public synchronized void checkPrepaymentTokenValidity(final Callback<Boolean> callback) {
        getSignUpToken(new Callback<AccessToken>() {
            @Override
            public void success(final AccessToken signUpToken) {
                // Get the prepaidToken whose validity needs to be checked.
                getPrepaidToken(new Callback<AccessToken>() {
                    @Override
                    public void success(AccessToken prepaidToken) {
                        if (prepaidToken.hasPrepaidPayToken()) {

                            // Check the validity of the prepayment token.
                            volleyPrePaidApi.getPrepaymentTokenValidity(signUpToken, prepaidToken, Constants.SCOPE_PREPAID_MERCHANT_PAY, new Callback<Boolean>() {
                                @Override
                                public void success(Boolean valid) {
                                    isPrepaymentTokenValid = true;

                                    sendResponse(callback, valid);
                                }

                                @Override
                                public void error(CitrusError error) {
                                    isPrepaymentTokenValid = false;

                                    sendError(callback, error);
                                }
                            });
                        } else {
                            // Since the payment token is not present, make it true;
                            isPrepaymentTokenValid = true;

                        }
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void getPrepaidBill(final Amount amount, final String returnUrl, final Callback<CitrusPrepaidBill> callback) {
        volleyUmApi.getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPrePaidApi.getPrepaidBill(accessToken, amount, returnUrl, new Callback<CitrusPrepaidBill>() {
                    @Override
                    public void success(CitrusPrepaidBill citrusPrepaidBill) {
                        sendResponse(callback, citrusPrepaidBill);
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void getBalance(final Callback<Amount> callback) {
        // First fetch the granttype username token
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPrePaidApi.getBalance(accessToken, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void activatePrepaidUser(final Callback<Amount> callback) {
        // First fetch the prepaid token.
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPrePaidApi.activatePrepaidUser(accessToken, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void newPrepaidPay(final PaymentType.CitrusCash citrusCash, final Callback<PaymentResponse> callback) {
        if (isPrepaymentTokenValid) {
            Amount amountToBePaid = null;
            if (citrusCash.getPaymentBill() != null) {
                amountToBePaid = citrusCash.getPaymentBill().getAmount();
            } else {
                amountToBePaid = citrusCash.getAmount();
            }

            isEnoughPrepaidBalance(amountToBePaid, new Callback<Boolean>() {
                @Override
                public void success(Boolean sufficientBalance) {
                    if (sufficientBalance) {
                        if (citrusCash.getPaymentBill() != null) {
                            // Merchant has Bill Ready, so directly proceed for the payment.
                            proceedToPayUsingCitrusCash(citrusCash, callback);
                        } else {
                            // Fetch the bill from the server and then proceed to payment.
                            final String billUrl;
                            billUrl = citrusCash.getUrl();
                            getBill(billUrl, citrusCash.getAmount(), null, new Callback<PaymentBill>() {
                                @Override
                                public void success(PaymentBill paymentBill) {
                                    citrusCash.setPaymentBill(paymentBill);
                                    proceedToPayUsingCitrusCash(citrusCash, callback);
                                }

                                @Override
                                public void error(CitrusError error) {
                                    sendError(callback, error);
                                }
                            });
                        }
                    } else {
                        //dont have enough balance in users account
                        sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_INSUFFICIENT_BALANCE, CitrusResponse.Status.FAILED));
                    }
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        } else {
            Logger.d("User's cookie has expired. Please signin");
            sendError(callback, new CitrusError("User's cookie has expired. Please signin.", CitrusResponse.Status.FAILED));
        }
    }

    private void proceedToPayUsingCitrusCash(final PaymentType.CitrusCash citrusCash, final Callback<PaymentResponse> callback) {
        getPrepaidPayToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken prepaidToken) {

                // Use the user details sent by the merchant, else use the user details from the token.
                if (citrusCash.getCitrusUser() == null) {
                    CitrusUser citrusUser = getCitrusUser();
                    citrusCash.setCitrusUser(citrusUser);
                }
                if (prepaidToken.hasPrepaidPayToken()) {
                    volleyPrePaidApi.newPrepaidPay(prepaidToken, citrusCash, new Callback<PaymentResponse>() {
                        @Override
                        public void success(PaymentResponse paymentResponse) {
                            sendResponse(callback, paymentResponse);

                            // Send the response on the return url asynchronously, so as to keep the integration same.
                            Utils.sendResponseToReturnUrlAsync(context, citrusCash.getPaymentBill().getReturnUrl(), paymentResponse);
                        }

                        @Override
                        public void error(CitrusError error) {
                            sendError(callback, error);
                        }
                    });
                } else {
                    Logger.d("Please contact our dev support and configure your signup & signin keys to access this API");
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_PREPAID_INNER_TOKEN_MISSING, CitrusResponse.Status.FAILED));
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /**
     * Check the prepaid balance is sufficient for transaction or not.
     *
     * @param transactionAmount
     * @param callback
     */
    private void isEnoughPrepaidBalance(final Amount transactionAmount, final Callback<Boolean> callback) {

        getBalance(new Callback<Amount>() {
            @Override
            public void success(Amount userBalance) {
                if (userBalance.getValueAsDouble() >= transactionAmount.getValueAsDouble()) {
                    sendResponse(callback, true);
                } else {
                    sendResponse(callback, false);
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /**
     * Send money to your friend.
     *
     * @param amount   - Amount to be sent
     * @param toUser   - The user detalis. Enter emailId if send by email or mobileNo if send by mobile.
     * @param message  - Optional message
     * @param callback - Callback
     * @deprecated Use {@link RetrofitAPIWrapper#sendMoneyToMoblieNo(Amount, String, String, Callback)} instead.
     */
    public synchronized void sendMoney(final Amount amount, final CitrusUser toUser, final String message, final Callback<PaymentResponse> callback) {

        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPrePaidApi.sendMoney(accessToken, amount, toUser, message, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /**
     * @param amount
     * @param mobileNo
     * @param message
     * @param callback
     */
    public synchronized void sendMoneyToMoblieNo(final Amount amount, final String mobileNo, final String message, final Callback<PaymentResponse> callback) {
        // Do the validation for amount.
        if (amount == null || TextUtils.isEmpty(amount.getValue())) {
            sendError(callback, new CitrusError("Amount should be not null or blank.", CitrusResponse.Status.FAILED));
            return;
        } else if (!(amount.getValueAsDouble() > 0)) {
            sendError(callback, new CitrusError("Amount should be greater than 0", CitrusResponse.Status.FAILED));
            return;
        }
        // Now the validation is done, proceed with sending the money.
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPrePaidApi.sendMoneyToMoblieNo(accessToken, amount, mobileNo, message, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /**
     * @param cashoutInfo
     * @param callback
     */
    public synchronized void cashout(@NonNull final CashoutInfo cashoutInfo, final Callback<PaymentResponse> callback) {
        // Do the validation for amount.
        if (cashoutInfo != null && cashoutInfo.validate()) {
            Amount amount = cashoutInfo.getAmount();
            if (amount == null || TextUtils.isEmpty(amount.getValue())) {
                sendError(callback, new CitrusError("Amount should be not null or blank.", CitrusResponse.Status.FAILED));
                return;
            } else if (!(amount.getValueAsDouble() > 0)) {
                sendError(callback, new CitrusError("Amount should be greater than 0", CitrusResponse.Status.FAILED));
                return;
            }

            // Now the validation is done, proceed with cashout/withdraw.
            getPrepaidToken(new Callback<AccessToken>() {
                @Override
                public void success(AccessToken accessToken) {
                    volleyPrePaidApi.cashout(accessToken, cashoutInfo, callback);
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_INVALID_CASHOUT_INFO, CitrusResponse.Status.FAILED));
        }
    }

    public synchronized void getCashoutInfo(final Callback<CashoutInfo> callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPrePaidApi.getCashoutInfo(accessToken, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void saveCashoutInfo(final CashoutInfo cashoutInfo, final Callback<CitrusResponse> callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPrePaidApi.saveCashoutInfo(accessToken, cashoutInfo, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /*********************
     * Prepaid API End *
     ********************/

    /*********************
     * PG API Start *
     ********************/
    public synchronized void getMerchantPaymentOptions(final Callback<MerchantPaymentOption> callback) {
        Log.v(TAG, TAG + ".getMerchantPaymentOptions() ");
        volleyPgApi.getMerchantPaymentOptions(callback);
    }

    public synchronized void getLoadMoneyPaymentOptions(final Callback<MerchantPaymentOption> callback) {
        volleyPgApi.getLoadMoneyPaymentOptions(callback);
    }

    public synchronized void getWallet(final Callback<List<PaymentOption>> callback) {
        Log.v(TAG, TAG + ".getWallet() ");
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(final AccessToken accessToken) {
                volleyPgApi.getMerchantPaymentOptions(new Callback<MerchantPaymentOption>() {

                    @Override
                    public void success(MerchantPaymentOption merchantPaymentOption) {
                        fetchConsumerProfile(callback);
                    }

                    @Override
                    public void error(CitrusError error) {
                        fetchConsumerProfile(callback);
                    }
                });

            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void getWalletWithDefaultBank(final Callback<List<PaymentOption>> callback, final BankCID bankCID) {
        getWallet(callback);
    }

    public synchronized void fetchConsumerProfile(final Callback<List<PaymentOption>> callback) {
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                // Create dummy amount if the input amount is null. Input amount may be null if call to this api comes in the begining.
                Amount amount = new Amount("1");
                volleyWalletPgApi.fetchConsumerProfile(amount, accessToken, new Callback<WalletConsumerProfile>() {
                    @Override
                    public void success(WalletConsumerProfile walletConsumerProfile) {
                        if (walletConsumerProfile != null) {
                            setCitrusUser(walletConsumerProfile.getCitrusUser());
                            sendResponse(callback, walletConsumerProfile.getPaymentOptionList());
                        } else {
                            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_GET_WALLET, CitrusResponse.Status.FAILED));
                        }
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void savePaymentOption(final PaymentOption paymentOption, final Callback<CitrusResponse> callback) {
        if (paymentOption != null) {

            // If the CardOption is invalid, check what is incorrect and respond with proper message.
            if (paymentOption instanceof CardOption && !((CardOption) paymentOption).validateForSaveCard()) {

                sendError(callback, new CitrusError(((CardOption) paymentOption).getCardValidityFailureReasonsForSaveCard(), CitrusResponse.Status.FAILED));
                return;
            }

            getUserNameToken(new Callback<AccessToken>() {
                @Override
                public void success(AccessToken accessToken) {
                    volleyPgApi.savePaymentOption(accessToken, paymentOption, callback);
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_NULL_PAYMENT_OPTION, CitrusResponse.Status.FAILED));
        }
    }

    public synchronized void saveCard(final PaymentOption paymentOption, final Callback<AddCardResponse> callback) {
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPgApi.saveCard(accessToken, paymentOption, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });

    }

    public synchronized void deletePaymentOption(final PaymentOption paymentOption, final Callback<CitrusResponse> callback) {
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPgApi.deletePaymentOption(accessToken, paymentOption, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void setDefaultPaymentOption(final PaymentOption defaultPaymentOption, final Callback<CitrusResponse> callback) {
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPgApi.setDefaultPaymentOption(accessToken, defaultPaymentOption, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void getBINDetails(final CardOption cardOption, final Callback<BinServiceResponse> callback) {
        getBinServiceResponse(cardOption, callback);
    }

    private void getBinServiceResponse(final CardOption cardOption, final Callback<BinServiceResponse> callback) {
        if (cardOption != null) {
            String cardNumber = cardOption.getCardNumber();
            String first6Digits = "";
            String token = "";
            if (!TextUtils.isEmpty(cardOption.getToken())) {
                token = cardOption.getToken();
            } else {
                first6Digits = cardNumber.length() > 6 ? cardNumber.substring(0, 6) : "";
            }

            final Callback<BinServiceResponse> callbackBinService = new Callback<BinServiceResponse>() {
                @Override
                public void success(final BinServiceResponse binServiceResponse) {
                    if (binServiceResponse != null) {
                        netBankForOTP = binServiceResponse.getNetBankForOTP();
                    }
                    sendResponse(callback, binServiceResponse);
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, new CitrusError("Unable to get BIN Details", CitrusResponse.Status.FAILED));
                }
            };

            // If card is new card use bin service api, else use token api
            if (!TextUtils.isEmpty(first6Digits)) {
                volleyBinServiceApi.getBinInfo(first6Digits, callbackBinService);
            } else if (!TextUtils.isEmpty(token)) {
                volleyPgApi.getBinInfoUsingToken(token, callbackBinService);
            } else {
                sendError(callback, new CitrusError("Unable to get BIN Details", CitrusResponse.Status.FAILED));
            }
        } else {
            sendError(callback, new CitrusError("Unable to get BIN Details", CitrusResponse.Status.FAILED));
        }
    }


    /**
     * Use {@link RetrofitAPIWrapper#getBINDetails} instead. This method will be used in Cube.
     *
     * @param first6Digits
     * @param cardDetailsCallback
     */
    public synchronized void getCardType(final String first6Digits, final Callback<CardBinDetails> cardDetailsCallback) {
        volleyBinServiceApi.getCardType(first6Digits, cardDetailsCallback);
    }

    public synchronized void getBill(final String billUrl, final Amount amount, String format, final Callback<PaymentBill> callback) {

        volleyBillingApi.getBill(billUrl, amount, format, callback);
    }

    public synchronized void getPGHealth(final PaymentOption paymentOption, final Callback<PGHealthResponse> callback) {
        citrusBasePgApi.getPGHealth(paymentOption, callback);
    }

    void fetchPGHealthForAllBanks() {
        citrusBasePgApi.fetchPGHealthForAllBanks();
    }

    /**
     * Perform Dynamic Pricing. You can specify one the dynamicPricingRequestType to perform Dynamic Pricing.
     *
     * @param dynamicPricingRequestType - One of the dynamicPricingRequestType from {@link DynamicPricingRequestType}
     * @param billUrl                   - billUrl from where we will fetch the bill.
     * @param callback                  - callback
     */
    public synchronized void performDynamicPricing(@NonNull final DynamicPricingRequestType dynamicPricingRequestType,
                                                   @NonNull final String billUrl,
                                                   @NonNull final Callback<DynamicPricingResponse> callback) {
        volleyDynamicPricingPgApi.performDynamicPricing(volleyBillingApi, dynamicPricingRequestType, billUrl, callback);
    }


    /**
     * Perform Dynamic Pricing. You can specify one the dynamicPricingRequestType to perform Dynamic Pricing.
     *
     * @param dynamicPricingRequestType - One of the dynamicPricingRequestType from {@link DynamicPricingRequestType}
     * @param paymentBill               - PaymentBill in case you are fetching bill response from your server.
     * @param callback                  - callback
     */
    public synchronized void performDynamicPricing(@NonNull final DynamicPricingRequestType dynamicPricingRequestType,
                                                   @NonNull final PaymentBill paymentBill,
                                                   @NonNull final Callback<DynamicPricingResponse> callback) {
        volleyDynamicPricingPgApi.performDynamicPricing(dynamicPricingRequestType, paymentBill, callback);
    }

    public synchronized void makeMOTOPayment(String paymentJSON, Callback<StructResponsePOJO> callback) {

        Log.i(TAG, TAG + ".makeMOTOPayment(): paymentJSON = " + paymentJSON);
        final ApiRequest motoApiRequest = volleyPgApi.getMotoRequestApi(paymentJSON);
        volleyApiExecutor.executeCustomObjectApi(volleyPgApi, motoApiRequest, callback);
    }

    public synchronized void newMakePayment(String paymentJSON, Callback<String> callback) {
        citrusBasePgApi.newMakePayment(paymentJSON, callback);
    }

    public synchronized void makeWalletPGPayment(final String walletPGPaymentJSON, final Callback<WalletPGPaymentResponse> callback) {

        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyWalletPgApi.makeWalletPGPayment(accessToken, walletPGPaymentJSON, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void getMerchantName(@NonNull final Callback<String> callback) {
        volleyPgApi.getMerchantName(callback);
    }

    public synchronized MerchantPaymentOption getPaymentOption(PaymentType paymentType) {
        if (paymentType instanceof PaymentType.LoadMoney) {
            return volleyPgApi.getLoadMoneyPaymentOption();
        } else {
            return volleyPgApi.getMerchantPaymentOption();
        }
    }

    /*********************
     * PG API END *
     ********************/

    public NetBankForOTP getNetBankForOTP() {
        return netBankForOTP;
    }

    public CitrusUser getCitrusUser() {
        return volleyUmApi.getCitrusUser();
    }

    private void setCitrusUser(CitrusUser citrusUser) {
        volleyUmApi.setCitrusUser(citrusUser);
    }

    public void resetNetBankForOTP() {
        this.netBankForOTP = NetBankForOTP.UNKNOWN;
    }

    private <T> void sendResponse(Callback<T> callback, T t) {
        if (callback != null) {
            callback.success(t);
        }
    }

    private <T> void sendError(Callback<T> callback, CitrusError citrusError) {
        if (callback != null) {
            callback.error(citrusError);
        }
    }

    public void cancelAllRequests() {
        ApiExecutor.getInstance(context).cancelAllApiRequests();
    }

    public void makeBlazeCardPayment(String paymentJSON, Callback<BCResponse> callback) {
        // TODO
//        blazeCardClient.makeBlazeCardPayment(paymentJSON, callback);
    }

    public void makeBlazeCardTokenizedPayment(String paymentJSON, Callback<BCResponse> callback) {
        // TODO
//        blazeCardClient.makeBlazeCardTokenizedPayment(paymentJSON, callback);
    }

    public void getBlazeCardPaymentOptions(String vanity, Callback<List<String>> callback) {
        // TODO
//        blazeCardClient.getMerchantCardScheme(vanity, callback);
    }

    public void cancelBCTransaction(String cancelPaymentJSON, Callback<BCCancelResponse> callback) {
        // TODO
//        blazeCardClient.cancelTransaction(cancelPaymentJSON, callback);
    }


    public void getActiveSubscriptions(final Callback<SubscriptionResponse> callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPrePaidApi.getActiveSubscriptions( accessToken, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }


    public void deActivateSubscription(final String subscriptionID, final Callback<SubscriptionResponse> callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPrePaidApi.deActivateSubscription( accessToken, subscriptionID, callback );
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public void createAutoLoadSubscription(final SubscriptionRequest subscriptionRequest, final Callback<SubscriptionResponse> callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPrePaidApi.createSubscription(accessToken, subscriptionRequest, callback );
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /**
     * call this method when you are updating active subscription's threshold amount and loadamount to lower value than current amount
     *
     * @param subscriptionRequest
     * @param callback
     */
    public void updateSubScriptiontoLowerValue(final UpdateSubscriptionRequest subscriptionRequest, final Callback<SubscriptionResponse> callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPrePaidApi.updateSubscriptiontoLowerValue(accessToken, subscriptionRequest, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }


    /**
     * call this method when you are updating active subscription's threshold amount and loadamount to higher value than current amount
     *
     * @param subscriptionRequest
     * @param callback
     */
    public void updateSubScriptiontoHigherValue(final UpdateSubscriptionRequest subscriptionRequest, final Callback<SubscriptionResponse> callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                volleyPrePaidApi.updateSubscriptiontoHigherValue(accessToken, subscriptionRequest, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }
}
