package com.citrus.sdk.network;

import android.content.Context;
import android.content.SharedPreferences;
import android.text.TextUtils;
import android.util.Base64;

import com.citrus.mobile.OauthToken;
import com.citrus.sdk.Constants;
import com.citrus.sdk.ResponseMessages;
import com.citrus.sdk.UMCallback;
import com.citrus.sdk.classes.AccessToken;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusResponse;
import com.facebook.android.crypto.keychain.SharedPrefsBackedKeyChain;
import com.facebook.crypto.Crypto;
import com.facebook.crypto.Entity;
import com.facebook.crypto.exception.CryptoInitializationException;
import com.facebook.crypto.exception.KeyChainException;
import com.facebook.crypto.util.SystemNativeCryptoLibrary;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.orhanobut.logger.Logger;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.Date;

/**
 * Created by vinay on 10/4/16.
 */
public class TokenUtils {

    private Context mContext;
    private String signupId;

    private String signupSecret;
    private String signinId;
    private String signinSecret;


    private String vanity;

    private TokenUtils(Context mContext) {
        this.mContext = mContext;
    }

    private static TokenUtils instance = null;

    public static TokenUtils getInstance(Context context ) {
        if (instance == null) {
            synchronized ( TokenUtils.class) {
                if (instance == null) {
                    instance = new TokenUtils(context);
                }
            }
        }
        return instance;
    }

    public void init(String signupId, String signupSecret, String signinId, String signinSecret , final String vanity){
        this.signupId       = signupId ;
        this.signupSecret   = signupSecret ;
        this.signinId       = signinId ;
        this.signinSecret   = signinSecret ;
        this.vanity         = vanity ;
    }

    public boolean validate() {
        if (!TextUtils.isEmpty(signinId) && !TextUtils.isEmpty(signinSecret)
                && !TextUtils.isEmpty(signupId) && !TextUtils.isEmpty(signupSecret)
                && !TextUtils.isEmpty(vanity) ) {
            return true;
        } else {
            throw new IllegalArgumentException(ResponseMessages.ERROR_MESSAGE_BLANK_CONFIG_PARAMS);
        }
    }

    public void getToken(boolean isPrepaidPayRequest, Constants.TokenType tokenType, UMCallback<AccessToken> callback) {
        JSONObject token        = null;
        String originalToken    = null;

        try {
            SharedPreferences tokenPrefs = mContext.getSharedPreferences(Constants.STORED_VALUES, 0);
            if (tokenPrefs.contains(tokenType.toString())) { //token exists in shared pref
                if (Constants.isEncryptedAccessToken) {
                    String encryptedToken = tokenPrefs.getString(tokenType.toString(), null);
                    originalToken = getDecryptedToken(tokenType.toString(), encryptedToken);
                } else {
                    originalToken = tokenPrefs.getString(tokenType.toString(), null);
                }
                if (originalToken != null) {
                    token = new JSONObject(originalToken); //read token from shared preferences
                    if (token.has("refresh_token")) { //check if token contains refresh token element
                        if (hasExpired(token) || (!token.has("Prepaid_Pay_Token") && tokenType == Constants.TokenType.PREPAID_TOKEN && isPrepaidPayRequest && !tokenPrefs.getBoolean(Constants.IS_TOKEN_REFRESHED, false))) {
                            callback.expired(token.optString("refresh_token"));
                        } else { //saved token is not expired - this is for grantType username and password token
                            Gson gson = new GsonBuilder().create();
                            AccessToken accessToken = gson.fromJson(token.toString(), AccessToken.class);
                            callback.success(accessToken);
                        }
                    } else { //return AccessToken Object - this is usually for granttype implicit token
                        Gson gson = new GsonBuilder().create();
                        AccessToken accessToken = gson.fromJson(token.toString(), AccessToken.class);
                        callback.success(accessToken);
                    }
                } else {
                    String errorMessage = tokenType.toString().equalsIgnoreCase(Constants.SIGNUP_TOKEN) ? ResponseMessages.ERROR_SIGNUP_TOKEN_NOT_FOUND : ResponseMessages.ERROR_SIGNIN_TOKEN_NOT_FOUND;
                    CitrusError error = new CitrusError(errorMessage, CitrusResponse.Status.FAILED);//token not found in shared preferences!!!
                    callback.error(error);
                }
            } else { //no token saved in shared pref
                String errorMessage = tokenType.toString().equalsIgnoreCase(Constants.SIGNUP_TOKEN) ? ResponseMessages.ERROR_SIGNUP_TOKEN_NOT_FOUND : ResponseMessages.ERROR_SIGNIN_TOKEN_NOT_FOUND;
                CitrusError error = new CitrusError(errorMessage, CitrusResponse.Status.FAILED);//token not found in shared preferences!!!
                callback.error(error);
            }

        } catch (JSONException e) { //this is rarent scenario
            CitrusError error = new CitrusError("Failed to get Access Token", CitrusResponse.Status.FAILED);
            callback.error(error);
        }

    }


    private String getDecryptedToken(String token_type, String encryptedToken) {
        String decryptedToken = null;

        String seed = (token_type + signinId + signinSecret);
        try {
            Crypto crypto = new Crypto(
                    new SharedPrefsBackedKeyChain(mContext),
                    new SystemNativeCryptoLibrary());

            if (!crypto.isAvailable()) {
                // If the library is not loaded then the token will be plain text and not encrypted.
                return encryptedToken;
            }

            Entity myEntity = new Entity(seed);

            if (encryptedToken != null) {
                byte[] array = Base64.decode(encryptedToken, Base64.DEFAULT);
                byte[] decryptedtoken = crypto.decrypt(array, myEntity);
                decryptedToken = new String(decryptedtoken);
            } else {
                decryptedToken = null;
            }
        } catch (KeyChainException e) {
            // e.printStackTrace();
            Logger.d("KeyChainException ");
        } catch (CryptoInitializationException e) {
            Logger.d("CryptoInitializationException");
            // e.printStackTrace();
        } catch (IOException e) {
            Logger.d("IOException");
            // e.printStackTrace();
        }

        return decryptedToken;
    }

    private boolean hasExpired(JSONObject token) {
        try {
            return token.getLong("expiry") <= (new Date().getTime() / 1000l);
        } catch (JSONException e) {
            return true;
        }
    }


    public void saveToken(final Constants.TokenType tokenType, final AccessToken accessToken) {
        OauthToken oauthToken = new OauthToken(mContext, tokenType.toString(), signinId, signinSecret);
        oauthToken.createToken(accessToken.getJSON());
    }

    public String getSignupSecret() {
        return signupSecret;
    }

    public String getSignupId() {
        return signupId;
    }

    public String getSigninId() {
        return signinId;
    }

    public String getSigninSecret() {
        return signinSecret;
    }

    public String getVanity() {
        return vanity;
    }

    public void setVanity(String vanity) {
        this.vanity = vanity;
    }
}

