package com.citrus.sdk.network;

import android.content.Context;
import android.util.Log;

import com.citrus.sdk.AddCardResponse;
import com.citrus.sdk.Callback;
import com.citrus.sdk.Constants;
import com.citrus.sdk.Environment;
import com.citrus.sdk.ResponseMessages;
import com.citrus.sdk.classes.AccessToken;
import com.citrus.sdk.classes.BinServiceResponse;
import com.citrus.sdk.network.request.ApiExecutor;
import com.citrus.sdk.network.request.ApiRequest;
import com.citrus.sdk.network.request.RequestBody;
import com.citrus.sdk.network.request.RequestBodyType;
import com.citrus.sdk.payment.MerchantPaymentOption;
import com.citrus.sdk.payment.PaymentOption;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusResponse;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;


public class PgClient extends BaseClient {

    private static final String TAG = PgClient.class.getSimpleName();

    private MerchantPaymentOption merchantPaymentOption;
    private MerchantPaymentOption loadMoneyPaymentOption;

    private static PgClient instance = null;
    private ApiExecutor executor;

    public static PgClient getInstance(Context context, Environment environment) {
        if (instance == null) {
            synchronized (PgClient.class) {
                if (instance == null) {
                    instance = new PgClient(context, environment);
                }
            }
        }
        return instance;
    }

    private PgClient(Context context, Environment environment) {
        super(context, environment);
        executor = ApiExecutor.getInstance(context);
    }

    @Override
    public String getBaseUrl() {
        return environment.getBaseUrl();
    }

    public ApiRequest getMotoRequestApi(final String jsonBody) {
        final RequestBody requestBody = new RequestBody(RequestBodyType.JSON, jsonBody);
        final ApiRequest motoRequestApi = new ApiRequest.ApiRequestBuilder(Api.PG_MOTO_REQUEST)
                .requestBody(requestBody)
                .build();

        return motoRequestApi;
    }


    public void saveCard(AccessToken accessToken, PaymentOption paymentOption, final Callback<AddCardResponse> callback) {
        if (validate()) {
            if (paymentOption != null) {
                final ApiRequest saveCardApi = getSaveCardApi(accessToken.getHeaderAccessToken(), paymentOption.getSavePaymentOptionObject());

                executor.executeCustomObjectApi(this, saveCardApi, callback);
            } else {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_NULL_PAYMENT_OPTION, CitrusResponse.Status.FAILED));
            }
        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_CONFIG_PARAMS, CitrusResponse.Status.FAILED));
        }

    }


    public ApiRequest getSaveCardApi(final String accessToken, final String jsonBody) {
        final RequestBody requestBody = new RequestBody(RequestBodyType.JSON, jsonBody);
        final ApiRequest saveCardRequestApi = new ApiRequest.ApiRequestBuilder(Api.PG_SAVE_CARD)
                .authorizationToken(accessToken)
                .requestBody(requestBody)
                .build();

        return saveCardRequestApi;
    }

    private void getMerchantPaymentOptions(final String vanity, final Callback<MerchantPaymentOption> callback) {
        if (validate()) {
            final ApiRequest merchantPaymentOptionsApi = getMerchantPaymentOptionsApi(vanity);
            executor.executeCustomJsonApi(this, merchantPaymentOptionsApi, new Callback<JSONObject>() {

                @Override
                public void success(final JSONObject paymentOptionObj) {
                    if (paymentOptionObj != null) {
                        MerchantPaymentOption merchantPaymentOption = MerchantPaymentOption.getMerchantPaymentOptions(paymentOptionObj);
                        sendResponse(callback, merchantPaymentOption);
                    } else {
                        sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_FAILED_MERCHANT_PAYMENT_OPTIONS, CitrusResponse.Status.FAILED));
                    }
                }

                @Override
                public void error(CitrusError error) {
                    // NOOP
                }
            });

        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_CONFIG_PARAMS, CitrusResponse.Status.FAILED));
        }
    }

    public synchronized void getMerchantPaymentOptions(final Callback<MerchantPaymentOption> callback) {
        Log.v(TAG, TAG + ".getMerchantPaymentOptions() " + merchantPaymentOption);

        if (merchantPaymentOption == null) {
            getMerchantPaymentOptions(getVanity(), new Callback<MerchantPaymentOption>() {
                @Override
                public void success(MerchantPaymentOption merchantPaymentOption) {
                    Log.v(TAG, TAG + ".getMerchantPaymentOptions(): SUCCESS ");
                    PgClient.this.merchantPaymentOption = merchantPaymentOption;
                    sendResponse(callback, merchantPaymentOption);
                }

                @Override
                public void error(CitrusError error) {
                    Log.e(TAG, TAG + ".getMerchantPaymentOptions().error():  " + error);
                    sendError(callback, error);
                }
            });
        } else {
            sendResponse(callback, merchantPaymentOption);
        }
    }

    public synchronized void getLoadMoneyPaymentOptions(final Callback<MerchantPaymentOption> callback) {
        Log.v(TAG, TAG + ".getLoadMoneyPaymentOptions() ");

        if (loadMoneyPaymentOption == null) {
            getMerchantPaymentOptions(Constants.PREPAID_VANITY, new Callback<MerchantPaymentOption>() {
                @Override
                public void success(MerchantPaymentOption merchantPaymentOption) {
                    PgClient.this.loadMoneyPaymentOption = merchantPaymentOption;
                    sendResponse(callback, merchantPaymentOption);
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        } else {
            sendResponse(callback, loadMoneyPaymentOption);
        }
    }


    private ApiRequest getMerchantPaymentOptionsApi(final String vanity) {
        final Map<String, String> params = new HashMap<>(1);
        params.put("vanity", vanity);
        final ApiRequest merchantPaymentApi = new ApiRequest.ApiRequestBuilder(Api.PG_MERCHANT_PAYMENT)
                .params(params)
                .build();

        return merchantPaymentApi;
    }

    public MerchantPaymentOption getMerchantPaymentOption() {
        return merchantPaymentOption;
    }

    public MerchantPaymentOption getLoadMoneyPaymentOption() {
        return loadMoneyPaymentOption;
    }

    public synchronized void setDefaultPaymentOption(final AccessToken accessToken,
                                                     final PaymentOption defaultPaymentOption,
                                                     final Callback<CitrusResponse> callback) {
        if (validate()) {
            if (defaultPaymentOption != null) {
                final ApiRequest defaultPaymentApiRequest = getSetDefaultPaymentOptionsApi(accessToken.getHeaderAccessToken(),
                        defaultPaymentOption.getSaveDefaultPaymentOptionObject());

                executor.executeCustomObjectApi(this, defaultPaymentApiRequest, new Callback<JSONObject>() {

                    @Override
                    public void success(JSONObject jsonObject) {
                        sendResponse(callback, new CitrusResponse(ResponseMessages.SUCCESS_MESSAGE_SAVED_PAYMENT_OPTIONS, CitrusResponse.Status.SUCCESSFUL));
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            } else {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_NULL_PAYMENT_OPTION, CitrusResponse.Status.FAILED));
            }
        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_CONFIG_PARAMS, CitrusResponse.Status.FAILED));
        }
    }

    private ApiRequest getSetDefaultPaymentOptionsApi(final String headerAccessToken, final String jsonBody) {
        final RequestBody requestBody = new RequestBody(RequestBodyType.JSON, jsonBody);
        final ApiRequest setDefaultPaymentApi = new ApiRequest.ApiRequestBuilder(Api.PG_SET_DEFAULT_PAYMENT_OPTION)
                .authorizationToken(headerAccessToken)
                .requestBody(requestBody)
                .build();

        return setDefaultPaymentApi;
    }

    public synchronized void savePaymentOption(final AccessToken accessToken, final PaymentOption paymentOption, final Callback<CitrusResponse> callback) {
        if (validate()) {
            if (paymentOption != null) {
                final ApiRequest savePaymentApiRequest = getSavePaymentOptionsApi(accessToken.getHeaderAccessToken(),
                        paymentOption.getSaveDefaultPaymentOptionObject());

                executor.executeCustomObjectApi(this, savePaymentApiRequest, callback);
            } else {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_NULL_PAYMENT_OPTION, CitrusResponse.Status.FAILED));
            }
        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_CONFIG_PARAMS, CitrusResponse.Status.FAILED));
        }
    }

    private ApiRequest getSavePaymentOptionsApi(final String headerAccessToken, final String jsonBody) {
        final RequestBody requestBody = new RequestBody(RequestBodyType.JSON, jsonBody);
        final ApiRequest savePaymentOptionApi = new ApiRequest.ApiRequestBuilder(Api.PG_SAVE_PAYMENT_OPTION)
                .authorizationToken(headerAccessToken)
                .requestBody(requestBody)
                .build();

        return savePaymentOptionApi;
    }


    public synchronized void deletePaymentOption(final AccessToken accessToken, final PaymentOption paymentOption,
                                                 final Callback<CitrusResponse> callback) {
        if (validate()) {

            if (paymentOption != null) {
                final ApiRequest deletePaymentApiRequest = getDeletePaymentOptionsApi(accessToken.getHeaderAccessToken(),
                        paymentOption.getToken());
                executor.executeCustomObjectApi(this, deletePaymentApiRequest, callback);
            } else {
                sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_NULL_PAYMENT_OPTION, CitrusResponse.Status.FAILED));
            }
        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_BLANK_CONFIG_PARAMS, CitrusResponse.Status.FAILED));
        }
    }

    private ApiRequest getDeletePaymentOptionsApi(final String headerAccessToken, final String token) {
        final ApiRequest deletePaymentOptionApi = new ApiRequest.ApiRequestBuilder(Api.PG_DELETE_PAYMENT_OPTION)
                .authorizationToken(headerAccessToken)
                .pathParams(token)
                .build();

        return deletePaymentOptionApi;
    }


    public synchronized void getBinInfoUsingToken(final String token, final Callback<BinServiceResponse> callback) {
        final ApiRequest binInfoUsingTokenApi = getBinInfoUsingTokenApiRequest(token);
        executor.executeCustomJsonApi(this, binInfoUsingTokenApi, new Callback<JSONObject>() {
            @Override
            public void success(JSONObject jsonObject) {
                if (jsonObject != null) {
                    final String jsonString = jsonObject.toString();
                    final BinServiceResponse binServiceResponse = BinServiceResponse.fromJSON(jsonString);
                    sendResponse(callback, binServiceResponse);
                } else {
                    sendError(callback, new CitrusError("Unable to get bin service response", CitrusResponse.Status.FAILED));
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }


    private ApiRequest getBinInfoUsingTokenApiRequest(final String token) {
        final ApiRequest apiRequest = new ApiRequest.ApiRequestBuilder(Api.PG_GET_BIN_INFO_USING_TOKEN)
                .pathParams(token)
                .build();

        return apiRequest;
    }

    public void getMerchantName(final Callback<String> callback) {

        final ApiRequest merchantNameApi = getMerchantNameApiRequest(super.getVanity());
        executor.executeStringApi(this, merchantNameApi, callback);
    }

    private ApiRequest getMerchantNameApiRequest(final String vanity) {
        final ApiRequest apiRequest = new ApiRequest.ApiRequestBuilder(Api.PG_GET_MERCHANT_NAME)
                .pathParams(vanity)
                .build();

        return apiRequest;
    }
}