package com.citrus.sdk.network;

import android.content.Context;

import com.citrus.sdk.Callback;
import com.citrus.sdk.Environment;
import com.citrus.sdk.classes.AllBanksHealth;
import com.citrus.sdk.classes.PGHealth;
import com.citrus.sdk.classes.PGHealthResponse;
import com.citrus.sdk.network.request.ApiExecutor;
import com.citrus.sdk.network.request.ApiRequest;
import com.citrus.sdk.network.request.RequestBody;
import com.citrus.sdk.network.request.RequestBodyType;
import com.citrus.sdk.payment.NetbankingOption;
import com.citrus.sdk.payment.PaymentOption;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusResponse;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;


/**
 * Created by vinay on 10/4/16.
 */
public class CitrusBasePgClient extends BaseClient {

    private static CitrusBasePgClient instance = null;
    final ApiExecutor executor = ApiExecutor.getInstance(mContext);

    public static CitrusBasePgClient getInstance(Context context, Environment environment) {
        if (instance == null) {
            synchronized (PgClient.class) {
                if (instance == null) {
                    instance = new CitrusBasePgClient(context, environment);
                }
            }
        }
        return instance;
    }

    private CitrusBasePgClient(Context context, Environment environment) {
        super(context, environment);
    }

    @Override
    public String getBaseUrl() {
        return environment.getBaseCitrusUrl();
    }

    public void newMakePayment(final String paymentJSON, final Callback<String> callback) {
        final ApiRequest makeNewPaymentApi = getNewMakePaymentApi(paymentJSON);

        executor.executeStringApi(this, makeNewPaymentApi, new Callback<String>() {
            @Override
            public void success(String response) {
                try {
                    String bankHTML = response;
                    if (bankHTML.contains("Access is denied")) {
                        String errorString = "";
                        int end = -1;
                        String[] errorCodes = bankHTML.split("class=\\\"transDetails\\\">(.*)\\s");
                        if (errorCodes != null && errorCodes.length == 2 && ((end = errorCodes[1].indexOf("</div>"))) != -1) {
                            String input = errorCodes[1];
                            String output = input.substring(0, end);
                            errorString = output.replaceAll("\n", "").replaceAll("\t", "");
                        } else {
                            errorString = "Some Error Occurred";
                        }

                        sendError(callback, new CitrusError(errorString, CitrusResponse.Status.FAILED));
                    } else {
                        sendResponse(callback, bankHTML);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    sendError(callback, new CitrusError(e.getMessage(), CitrusResponse.Status.FAILED));
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }


    private ApiRequest getNewMakePaymentApi(final String jsonBody) {
        final RequestBody requestBody = new RequestBody(RequestBodyType.JSON, jsonBody);
        final ApiRequest newMakePaymentApi = new ApiRequest.ApiRequestBuilder(Api.CITRUSBASE_NEW_MAKE_PAYMENT)
                .requestBody(requestBody)
                .build();

        return newMakePaymentApi;

    }

    public synchronized void getPGHealth(final PaymentOption paymentOption, final Callback<PGHealthResponse> callback) {
        // Currently PG health supports netbanking only. So in case of any other payment Options it will return GOOD by default.
        if (!(paymentOption instanceof NetbankingOption)) {
            sendResponse(callback, new PGHealthResponse(PGHealth.GOOD, "All Good"));
        } else {

            final ApiRequest pgHealthApi = getPGHealthApi(super.getVanity(), ((NetbankingOption) paymentOption).getBankCID());
            executor.executeCustomObjectApi(this, pgHealthApi, callback);
        }
    }


    private ApiRequest getPGHealthApi(final String path, final String bankCode) {
        final Map<String, String> params = new HashMap<>(1);
        params.put("bankCode", bankCode);

        final ApiRequest pgHealthApi = new ApiRequest.ApiRequestBuilder(Api.CITRUSBASE_GET_PG_HEALTH)
                .params(params)
                .pathParams(path)
                .build();

        return pgHealthApi;
    }


    public void fetchPGHealthForAllBanks() {

        final ApiRequest pgHealthApi = getPGHealthApi(super.getVanity(), "ALLBANKS");
        executor.executeCustomJsonApi(this, pgHealthApi, new Callback<JSONObject>() {

            @Override
            public void success(JSONObject jsonObject) {
                Iterator<String> keys = jsonObject.keys();
                while (keys.hasNext()) {
                    final AllBanksHealth allBanksHealth = AllBanksHealth.getInstance();
                    /*if (allBanksHealth.getPgHealthMap() == null) {
                        allBanksHealth.setPgHealthMap( new HashMap<String, PGHealth>() );
                    }*/
                    final String key = keys.next();
                    final String health = jsonObject.optString(key);

                    allBanksHealth.addHealthForBank(key, PGHealth.getPGHealth(health));
                }
            }

            @Override
            public void error(CitrusError error) {
                // NOOP
                // No need to send any error, since it will be kept locally.
            }
        });
    }
}
