/*
 *
 *    Copyright 2014 Citrus Payment Solutions Pvt. Ltd.
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 * /
 */

package com.citrus.sdk;

import android.annotation.TargetApi;
import android.app.AlertDialog;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.net.http.SslError;
import android.os.Build;
import android.os.Bundle;
import android.os.CountDownTimer;
import android.os.Handler;
import android.support.v4.content.LocalBroadcastManager;
import android.support.v7.app.ActionBar;
import android.support.v7.app.AppCompatActivity;
import android.text.Html;
import android.text.TextUtils;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.webkit.CookieManager;
import android.webkit.JavascriptInterface;
import android.webkit.SslErrorHandler;
import android.webkit.WebChromeClient;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.ImageView;

import com.citrus.cash.PersistentConfig;
import com.citrus.library.R;
import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.classes.BCCancelResponse;
import com.citrus.sdk.classes.BCCancelTransactionDetails;
import com.citrus.sdk.classes.BCResponse;
import com.citrus.sdk.classes.BinServiceResponse;
import com.citrus.sdk.classes.CitrusConfig;
import com.citrus.sdk.classes.CitrusPrepaidBill;
import com.citrus.sdk.classes.StructResponsePOJO;
import com.citrus.sdk.classes.Utils;
import com.citrus.sdk.dynamicPricing.DynamicPricingResponse;
import com.citrus.sdk.otp.NetBankForOTP;
import com.citrus.sdk.otp.OTPPopupView;
import com.citrus.sdk.otp.OTPViewListener;
import com.citrus.sdk.otp.SMSReceiver;
import com.citrus.sdk.payment.CardOption;
import com.citrus.sdk.payment.CitrusCash;
import com.citrus.sdk.payment.NetbankingOption;
import com.citrus.sdk.payment.PaymentBill;
import com.citrus.sdk.payment.PaymentData;
import com.citrus.sdk.payment.PaymentOption;
import com.citrus.sdk.payment.PaymentType;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.views.CitrusProgressBar;
import com.citrus.sdk.walletpg.WalletPGPaymentResponse;
import com.orhanobut.logger.Logger;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

public class CitrusActivity extends AppCompatActivity implements OTPViewListener {

    private final int WAIT_TIME = 300;
    private final String WAIT_MESSAGE = "Processing Payment. Please Wait...";
    private final String CANCEL_MESSAGE = "Cancelling Transaction. Please Wait...";

    private WebView mPaymentWebview = null;
    private Context mContext = this;
    private PaymentType mPaymentType = null;
    private PaymentOption mPaymentOption = null;
    private List<PaymentOption> mPaymentOptionList = null;
    private String mTransactionId = null;
    private ActionBar mActionBar = null;
    private CitrusProgressBar mProgressBar = null;
    private String mColorPrimary = null;
    private String mColorPrimaryDark = null;
    private String mTextColorPrimary = null;
    private CitrusConfig mCitrusConfig = null;
    private CitrusUser mCitrusUser = null;
    private String sessionCookie;
    private CookieManager cookieManager;
    private String mpiServletUrl = null;
    private Map<String, String> customParametersOriginalMap = null;
    private CitrusClient mCitrusClient = null;
    private String mActivityTitle = null;
    private int mRequestCode = -1;
    private CountDownTimer mTimer = null;
    private boolean mLoading = false;
    private boolean mShowingDialog = false;
    private boolean isBackKeyPressedByUser = false;
    private DynamicPricingResponse mDynamicPricingResponse = null;
    private PaymentBill mPaymentBill = null;

    // Auto OTP
    private SMSReceiver mSMSReceiver = null;
    private BroadcastReceiver mAutoOtpSMSReceiveListener = null;
    private OTPPopupView mOTPPopupView = null;
    private String otpProcessTransactionJS = null;
    private ImageView otpPopupCancelImgView = null;
    private boolean autoOTPEnabled = false;
    private NetBankForOTP netBankForOTP = NetBankForOTP.UNKNOWN;
    private String country = null; // required for One Tap Payment - it should work only for Indian Cards
    private String binCardType = null;
    private String otp;
    private static final long OTP_READ_TIMEOUT = 45000;
    private boolean transactionProcessed = false;
    private boolean mMultipartSendOTPJS = false;
    private boolean mMultipartEnterPasswordJS = false;
    private boolean useNewAPI = false;
    private boolean otpPopupDismissed = false;
    private boolean autoReadOTP = false;
    private boolean isCardUnRegistered = false;
    private Environment mEnvironment;
    private boolean mSMSReceiverRegistered = false;

    private enum BlazePayments {
        BlazeCard,
        BlazeNet
    }


    private BlazePayments mBlazePayments = null;

    private BCCancelTransactionDetails mBCCancelTransactionDetails = null;

    @Override
    protected void onCreate(Bundle savedInstanceState) {

        mPaymentType = getIntent().getParcelableExtra(Constants.INTENT_EXTRA_PAYMENT_TYPE);
        mRequestCode = getIntent().getIntExtra(Constants.INTENT_EXTRA_REQUEST_CODE_PAYMENT, -1);
        useNewAPI = getIntent().getBooleanExtra(Constants.INTENT_EXTRA_USE_NEW_API, false);

        // Initialize CitrusClient.
        mCitrusClient = CitrusClient.getInstance(mContext);

        mEnvironment = mCitrusClient.getEnvironment();

        // Initialize things.
        autoOTPEnabled = mCitrusClient.isAutoOtpReading();

//        if (!(mPaymentType instanceof PaymentType.CitrusCash)) {
//            setTheme(R.style.Base_Theme_AppCompat_Light_DarkActionBar);
//        }

        if (mPaymentType instanceof PaymentType.PGPayment || mPaymentType instanceof PaymentType.LoadMoney) {
            setTheme(R.style.Base_Theme_AppCompat_Light_DarkActionBar);
        } else if (mPaymentType instanceof WalletPGPayment) {
            if (!((WalletPGPayment) mPaymentType).isWalletOnlyPayment()) {
                setTheme(R.style.Base_Theme_AppCompat_Light_DarkActionBar);
            }
        }

        super.onCreate(savedInstanceState);

        setContentView(R.layout.activity_citrus);

        mOTPPopupView = (OTPPopupView) findViewById(R.id.otpPopupViewId);
        otpPopupCancelImgView = (ImageView) findViewById(R.id.otpPopupCancelImgViewId);
        otpPopupCancelImgView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mOTPPopupView.getOtpViewToggleStatus()) {
                    mOTPPopupView.setVisibility(View.VISIBLE);
                    mOTPPopupView.setOtpViewToggleStatus(false);
                    otpPopupCancelImgView.setBackgroundResource(R.drawable.arrow_down_icon);
                    findViewById(R.id.otpPopupSeparatorId).setVisibility(View.VISIBLE);
                    // Show the OTP Popup Overlay i.e. dark grey portion on the back
                    findViewById(R.id.otpPopupOverlayId).setVisibility(View.VISIBLE);

                } else {
                    mOTPPopupView.setVisibility(View.GONE);
                    mOTPPopupView.setOtpViewToggleStatus(true);
                    otpPopupCancelImgView.setBackgroundResource(R.drawable.arrow_up_icon);
                    findViewById(R.id.otpPopupSeparatorId).setVisibility(View.GONE);
                    // Hide the OTP Popup Overlay i.e. dark grey portion on the back
                    findViewById(R.id.otpPopupOverlayId).setVisibility(View.GONE);
                }

            }
        });
        mOTPPopupView.setListener(this);
        mSMSReceiver = new SMSReceiver();
        mAutoOtpSMSReceiveListener = new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                autoOtpReceived(intent);
            }
        };

        mDynamicPricingResponse = getIntent().getParcelableExtra(Constants.INTENT_EXTRA_DYNAMIC_PRICING_RESPONSE);
        mCitrusConfig = CitrusConfig.getInstance();
        mActivityTitle = mCitrusConfig.getCitrusActivityTitle();

        initializeTimer();

        if (mPaymentType != null) {
            mPaymentOption = mPaymentType.getPaymentOption();
            mCitrusUser = mPaymentType.getCitrusUser();

            mColorPrimary = mCitrusConfig.getColorPrimary();
            mColorPrimaryDark = mCitrusConfig.getColorPrimaryDark();
            mTextColorPrimary = mCitrusConfig.getTextColorPrimary();

            // In case of wallet pg transaction, there can be multiple payment options involved.
            if (mPaymentType instanceof WalletPGPayment) {
                mPaymentOptionList = mPaymentType.getPaymentOptionList();
                // Since the paymentOption will be null in case of wallet pg.
                // We will assign other payment option to payment option, since paymentOption is required to display auto-otp popup.
                mPaymentOption = ((WalletPGPayment) mPaymentType).getOtherPaymentOption();
            }
        } else {
            throw new IllegalArgumentException("Payment Type Should not be null");
        }

        registerSMSReceivers();

        final CitrusUser citrusUser = mCitrusClient.getCitrusUser();

        // Set the citrusUser.
        // Use details from the token in case of load money
        if (mPaymentType instanceof PaymentType.LoadMoney || mPaymentType instanceof PaymentType.CitrusCash || mPaymentType instanceof WalletPGPayment) {
            if (citrusUser != null) {
                mCitrusUser = citrusUser;
            } else if (mCitrusUser == null) {
                mCitrusUser = CitrusUser.DEFAULT_USER;
            }
        } else if (mPaymentType instanceof PaymentType.PGPayment) {
            // In case of PG Payment, send the merchant values.
            if (mCitrusUser == null) {
                // If the user details from token are available use those, put the details sent by the merchant while bind.
                if (citrusUser != null) {
                    mCitrusUser = citrusUser;
                } else {
                    mCitrusUser = CitrusUser.DEFAULT_USER;
                }
            }
        }

        //4280940002506772
        mActionBar = getSupportActionBar();
        mProgressBar = new CitrusProgressBar(mContext);
        mPaymentWebview = (WebView) findViewById(R.id.payment_webview);
        mPaymentWebview.getSettings().setLayoutAlgorithm(WebSettings.LayoutAlgorithm.NORMAL);
        mPaymentWebview.getSettings().setJavaScriptEnabled(true);

        // This is done to have horizontal scroll for 2 banks whose page renders improperly in the webview
        if (mPaymentOption instanceof NetbankingOption) {

            // This is done to enable old pg payment for these banks.
            useNewAPI = false;

            NetbankingOption netbankingOption = (NetbankingOption) mPaymentOption;

            if ("CID032".equalsIgnoreCase(netbankingOption.getBankCID()) // Karur Vyasa
                    || "CID029".equalsIgnoreCase(netbankingOption.getBankCID())//ING VYSYA
                    || "CID066".equalsIgnoreCase(netbankingOption.getBankCID())//SARASWAT
                    || "CID051".equalsIgnoreCase(netbankingOption.getBankCID())) // Canara Bank
            {
                mPaymentWebview.getSettings().setUseWideViewPort(true);
            }

//            // Following 6 banks are facing issues in case of new pg payment api. So reverting to the older payment api in this case.
//            if ("CID002".equalsIgnoreCase(netbankingOption.getBankCID()) // Axis
//                    || "CID019".equalsIgnoreCase(netbankingOption.getBankCID()) // BOI
//                    || "CID041".equalsIgnoreCase(netbankingOption.getBankCID()) // United BOI
//                    || "CID046".equalsIgnoreCase(netbankingOption.getBankCID()) // BOB
//                    || "CID016".equalsIgnoreCase(netbankingOption.getBankCID()) // Andhra
//                    || "CID026".equalsIgnoreCase(netbankingOption.getBankCID()) // DCB
//                    || "CID070".equalsIgnoreCase(netbankingOption.getBankCID()) // UCO
//                    || "CID031".equalsIgnoreCase(netbankingOption.getBankCID()) // Karnataka Bank
//                )
//            {
            // This is done to enable old pg payment for these banks.
//                useNewAPI = false;
//            }
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            /*
            This setting is required to enable redirection of urls from https to http or vice-versa.
            This redirection is blocked by default from Lollipop (Android 21).
             */
            mPaymentWebview.getSettings().setMixedContentMode(WebSettings.MIXED_CONTENT_ALWAYS_ALLOW);
        }
        mPaymentWebview.addJavascriptInterface(new JsInterface(), Constants.JS_INTERFACE_NAME);

        mPaymentWebview.setWebChromeClient(new WebChromeClient());
        mPaymentWebview.setWebViewClient(new CitrusWebClient());

        // Make the webview visible only in case of PGPayment or LoadMoney.
        if (mPaymentType instanceof PaymentType.CitrusCash || (mPaymentType instanceof WalletPGPayment && ((WalletPGPayment) mPaymentType).isWalletOnlyPayment())) {
            mPaymentWebview.setVisibility(View.GONE);
        }

        // Get BIN Details required for autoOTP only when the environment is PRODUCTION
        if ((autoOTPEnabled || CitrusConfig.getInstance().isOneTapPaymentEnabled()) && mPaymentOption instanceof CardOption) {
            fetchBinRequestData((CardOption) mPaymentOption);
        }

        /*
         * Validations and Process payments
         */
        // Check whether the request is coming directly for payment without validation. Do validation.
        if (mRequestCode == Constants.REQUEST_CODE_PAYMENT) {
            if (mPaymentOption instanceof CardOption && !((CardOption) mPaymentOption).validateCard()) {
                sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, ((CardOption) mPaymentOption).getCardValidityFailureReasons(), null));

                return;
            }
        }

        // For PG Payment or Pay Using Citrus Cash
        if (mPaymentType instanceof PaymentType.PGPayment || mPaymentType instanceof PaymentType.CitrusCash || mPaymentType instanceof WalletPGPayment) {
            if (mPaymentType.getPaymentBill() != null) {
                mPaymentBill = mPaymentType.getPaymentBill();
                proceedToPayment(mPaymentBill);
            } else {
                // Show text while processing payments
                if (mCitrusClient.isShowDummyScreenWhilePayments()) {
                    mPaymentWebview.loadData("<html><body><h5><center>Processing, please wait...<center></h5></body></html>", "text/html", "utf-8");
                }

                showDialog(WAIT_MESSAGE, true);
                fetchBill();
            }
        } else {
            //load cash does not requires Bill Generator
            Amount amount = mPaymentType.getAmount();

            showDialog(WAIT_MESSAGE, true);


            mCitrusClient.getPrepaidBill(amount, mPaymentType.getUrl(), new com.citrus.sdk.Callback<CitrusPrepaidBill>() {
                @Override
                public void success(CitrusPrepaidBill citrusPrepaidBill) {
                    PaymentData paymentData = new PaymentData(citrusPrepaidBill, mPaymentOption, citrusUser);

                    if ((mCitrusClient.getAUTO_load_type() == CitrusClient.AUTO_LOAD_TYPE.QUICK_AUTO_LOAD) || (mCitrusClient.getAUTO_load_type() == CitrusClient.AUTO_LOAD_TYPE.LAZY_AUTO_LOAD)) { // we need load money transaction ID in case auto Load is enabled
                        SubscriptionRequest subscriptionRequest = mCitrusClient.getSubscriptionRequest();
                        if (subscriptionRequest != null)
                            mCitrusClient.getSubscriptionRequest().setAuthRefId(citrusPrepaidBill.getMerchantTransactionId());
                    }
                    if (mCitrusClient.getAUTO_load_type() == CitrusClient.AUTO_LOAD_TYPE.UPDATE_AUTO_LOAD) { // this is required to update existing auto load subscription to higher value
                        if (mCitrusClient.getUpdateSubscriptionRequest() != null) {
                            mCitrusClient.getUpdateSubscriptionRequest().setAuthRefId(citrusPrepaidBill.getMerchantTransactionId());
                        }
                    }

                    if (isBlazePaymentAvailable()) {
                        if (mBlazePayments == BlazePayments.BlazeCard)
                            makeBlazeCardLoadMoneyPayment(paymentData, mPaymentOption.isTokenizedPayment());
                        else {
                            //do nothing... this was supposed to be BN
                        }

                    } else if (isSingleHopAvailable()) { //priority is single hop. As of now it works only for Card Payments
                        useNewAPI = true;
                        processNewMakePaymentTransaction(paymentData);
                    } else { // if its Bank Payment redirect it to Moto
                        mCitrusClient.makeMOTOPayment(paymentData.getPaymentJSON(), new com.citrus.sdk.Callback<StructResponsePOJO>() {
                            @Override
                            public void success(StructResponsePOJO structResponsePOJO) {
                                openPaymentUrl(structResponsePOJO);
                            }

                            @Override
                            public void error(CitrusError error) {
                                sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, error.getMessage(), null));
                            }
                        });
                    }
                }

                @Override
                public void error(CitrusError error) {
                    sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, error.getMessage(), null));
                }
            });
        }

        // Set the title for the activity
        if (TextUtils.isEmpty(mActivityTitle)) {
            mActivityTitle = "Processing...";
        }

        setTitle(Html.fromHtml("<font color=\"" + mTextColorPrimary + "\">" + mActivityTitle + "</font>"));
        setActionBarBackground();

        // Enable webContentDebugging, only for apps in debug mode.
        enableWebContentDebugging();
    }

    @Override
    protected void onResume() {
        super.onResume();

        registerSMSReceivers();
    }

    @Override
    protected void onPause() {
        super.onPause();
        if (mAutoOtpSMSReceiveListener != null)
            unregisterSMSReceivers();
    }

    private void initializeTimer() {
        // Timer to dismiss dialog after specific time once the url loading is complete.
        mTimer = new CountDownTimer(WAIT_TIME, 100) {
            @Override
            public void onTick(long millisUntilFinished) {
            }

            @Override
            public void onFinish() {
                if (!mLoading) {
                    dismissDialog();
                    if (!transactionProcessed && !mMultipartEnterPasswordJS && !isBackKeyPressedByUser && !isCardUnRegistered) {
                        displayOtpPopup();
                    }

                    // If the sendOTP js is multipart, load the js.
                    if (mMultipartSendOTPJS) {
                        mPaymentWebview.loadUrl(netBankForOTP.getMultiPartSendOTPJS());
                        mMultipartSendOTPJS = false;
                    }

                    // If the sendOTP js is multipart, load the js.
                    if (mMultipartEnterPasswordJS) {
                        mPaymentWebview.loadUrl(netBankForOTP.getMultiPartEnterPasswordJS());
                        mMultipartEnterPasswordJS = false;
                    }

                    // Set the title since the transaction is done.
                    setTitle(Html.fromHtml("<font color=\"" + mTextColorPrimary + "\"> 3D Secure </font>"));
                }
            }
        };
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    private void setActionBarBackground() {
        // Set primary color
        if (mColorPrimary != null && mActionBar != null) {
            mActionBar.setBackgroundDrawable(new ColorDrawable(Color.parseColor(mColorPrimary)));
        }

        // Set action bar color. Available only on android version Lollipop or higher.
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP && mColorPrimaryDark != null) {
            Window window = getWindow();
            window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS);
            window.setStatusBarColor(Color.parseColor(mColorPrimaryDark));
        }
    }

    private void fetchBill() {
        String billUrl = mPaymentType.getUrl();

        String format = null;
        if (mPaymentType instanceof WalletPGPayment) {
            format = Constants.AMOUNT_DOUBLE_PRECISION_FORMAT;
        }
      /*  if (CitrusConfig.getInstance().isBlazeCardEnabled()) {
            format = "#.00";
        }*/

        mCitrusClient.getBill(billUrl, mPaymentType.getAmount(), format, new com.citrus.sdk.Callback<PaymentBill>() {
            @Override
            public void success(PaymentBill paymentBill) {
                // PaymentBill required for the later use.
                mPaymentBill = paymentBill;
                if (paymentBill != null) {
                    customParametersOriginalMap = paymentBill.getCustomParametersMap();
                    proceedToPayment(paymentBill);
                } else {
                    TransactionResponse transactionResponse = new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, ResponseMessages.ERROR_MESSAGE_INVALID_BILL, mTransactionId);
                    sendResult(transactionResponse);
                }
            }

            @Override
            public void error(CitrusError error) {
                TransactionResponse transactionResponse = new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, error.getMessage(), mTransactionId);
                sendResult(transactionResponse);
            }
        });
    }

    private void openPaymentUrl(StructResponsePOJO structResponsePOJO) {

        String redirectUrl = structResponsePOJO.getRedirectUrl();
        if (!TextUtils.isEmpty(redirectUrl)) {
            mpiServletUrl = redirectUrl;
            mPaymentWebview.loadUrl(redirectUrl);
        } else {
            sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, structResponsePOJO.getTxMsg(), mTransactionId));
        }
    }

    /**
     * Open payment url for payment processing for wallet pg payment.
     *
     * @param walletPGPaymentResponse
     */
    private void openPaymentUrl(WalletPGPaymentResponse walletPGPaymentResponse) {

        String redirectUrl = walletPGPaymentResponse.getRedirectUrl();
        if (!TextUtils.isEmpty(redirectUrl)) {
            mpiServletUrl = redirectUrl;
            mPaymentWebview.loadUrl(redirectUrl);
        } else {
            sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, walletPGPaymentResponse.getDescription(), mTransactionId));
        }
    }

    private void proceedToPayment(PaymentBill paymentBill) {
        // Proceed to payment depending upon the payment type
        if (mPaymentType instanceof PaymentType.CitrusCash) {
            processPayUsingCitrusCashPayment(paymentBill);
        } else if (mPaymentType instanceof PaymentType.PGPayment) {

            if (useNewAPI) {
                processNewMakePaymentTransaction(paymentBill);
            } else {
                //uncomment this if you want BC or BN for PG Payment
                /*if (isBlazePaymentAvailable()) {
                    if (mBlazePayments == BlazePayments.BlazeCard) {
                        makeBlazeCardPayment(paymentBill);
                    } else {
                        makeBlazeNetPayment(paymentBill);
                    }
                } else*/
                {
                    processMOTOTransaction(paymentBill);
                }
            }
        } else if (mPaymentType instanceof WalletPGPayment) {
            processWalletPGTransaction(paymentBill);
        }
    }

    private boolean isBlazePaymentAvailable() {


        if (mPaymentOption instanceof CardOption) // Card Payment
        {
            if (CitrusConfig.getInstance().isBlazeCardEnabled() && Arrays.asList(BlazeWrapper.cardSchemes).contains(((CardOption) mPaymentOption).getCardScheme().toString())) {
                mBlazePayments = BlazePayments.BlazeCard;
                return true;
            } else {
                return false;
            }

        } else { //Net Banking Payment
           return false;
        }
    }

    private boolean isSingleHopAvailable() {
        return false;//turn on below code when you want to enable single hop. Waiting for Arindam to fix concurrency issue. Date - 11-04-16
       /* if (mPaymentOption instanceof CardOption) {
            return true;
        } else {
            return false;
        }*/
    }

    private void processWalletPGTransaction(PaymentBill paymentBill) {
        PaymentData paymentData = new PaymentData(paymentBill, mPaymentOptionList, mCitrusUser, mDynamicPricingResponse);
        mCitrusClient.makeWalletPGPayment(paymentData.getWalletPGPaymentJSON(), new Callback<WalletPGPaymentResponse>() {
            @Override
            public void success(WalletPGPaymentResponse walletPGPaymentResponse) {
                TransactionResponse.TransactionStatus transactionStatus = walletPGPaymentResponse.getTransactionStatus();

                if (transactionStatus == TransactionResponse.TransactionStatus.PG_FORWARD_REQUESTED) {
                    // Process the transaction.
                    openPaymentUrl(walletPGPaymentResponse);
                } else if (transactionStatus == TransactionResponse.TransactionStatus.SUCCESSFUL) {
                    // Process The Successful transaction in case wallet pg.
                    // The callback will come here in case of Payment Using MVC and/or Citrus Cash Payment Mode.

                    Utils.sendResponseToReturnUrlAsync(mContext, walletPGPaymentResponse.getRedirectUrl(), walletPGPaymentResponse.getURLEncodedParams());
                    sendResult(new TransactionResponse(transactionStatus, ResponseMessages.SUCCESS_TRANSACTION, walletPGPaymentResponse.getTransactionId()));
                } else if (transactionStatus == TransactionResponse.TransactionStatus.FAILED) {
                    sendResult(new TransactionResponse(transactionStatus, walletPGPaymentResponse.getDescription(), walletPGPaymentResponse.getTransactionId()));
                }
            }

            @Override
            public void error(CitrusError error) {
                sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, error.getMessage(), mTransactionId));
            }
        });
    }

    private void processMOTOTransaction(PaymentBill paymentBill) {
        PaymentData paymentData = new PaymentData(paymentBill, mPaymentOption, mCitrusUser, mDynamicPricingResponse);
        mCitrusClient.makeMOTOPayment(paymentData.getPaymentJSON(), new com.citrus.sdk.Callback<StructResponsePOJO>() {
            @Override
            public void success(StructResponsePOJO structResponsePOJO) {
                openPaymentUrl(structResponsePOJO);
            }

            @Override
            public void error(CitrusError error) {
                sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, error.getMessage(), mTransactionId));
            }
        });
    }


    private void makeBlazeCardPayment(final PaymentBill paymentBill) {
        final PaymentData paymentData = new PaymentData(paymentBill, mPaymentOption, mCitrusUser, mDynamicPricingResponse);
        mCitrusClient.makeBlazeCardPayment(paymentData.getPaymentJSON(), new Callback<BCResponse>() {
            @Override
            public void success(BCResponse bcResponse) {
                Logger.d("RESPONSE **** ***" + bcResponse.toString());
                int errorCode = bcResponse.getErrorCode();
                mBCCancelTransactionDetails = new BCCancelTransactionDetails(bcResponse, paymentData);
                if (errorCode == 0) {
                    String html = String.format("<html><head><title>Redirecting to Bank</title></head><body>%s</body><script type=\"text/javascript\">document.citrusSubmitForm.submit();</script></html>", bcResponse.getInfo().getCitrusSubmitForm());
                    mPaymentWebview.loadData(html, "text/html", "utf-8");
                } else {
                    sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, bcResponse.getErrorMessage(), mTransactionId));
                }

            }

            @Override
            public void error(CitrusError error) {
                sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, error.getMessage(), mTransactionId));
            }
        });
    }


    private void makeBlazeCardLoadMoneyPayment(final PaymentData paymentData, boolean isTokenizedPayment) {
        if (isTokenizedPayment) {
            mCitrusClient.makeBlazeCardTokenizedPayment(paymentData.getPaymentJSON(), new Callback<BCResponse>() {
                @Override
                public void success(BCResponse bcResponse) {
                    Logger.d("RESPONSE **** ***" + bcResponse.toString());
                    int errorCode = bcResponse.getErrorCode();
                    mBCCancelTransactionDetails = new BCCancelTransactionDetails(bcResponse, paymentData);
                    if (errorCode == 0) {
                        String html = String.format("<html><head><title>Redirecting to Bank</title></head><body>%s</body><script type=\"text/javascript\">document.citrusSubmitForm.submit();</script></html>", bcResponse.getInfo().getCitrusSubmitForm());
                        mPaymentWebview.loadData(html, "text/html", "utf-8");
                    } else {
                        sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, bcResponse.getErrorMessage(), mTransactionId));
                    }
                }

                @Override
                public void error(CitrusError error) {
                    sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, error.getMessage(), mTransactionId));
                }
            });
        } else {
            mCitrusClient.makeBlazeCardPayment(paymentData.getPaymentJSON(), new Callback<BCResponse>() {
                @Override
                public void success(BCResponse bcResponse) {
                    Logger.d("RESPONSE **** ***" + bcResponse.toString());
                    int errorCode = bcResponse.getErrorCode();
                    mBCCancelTransactionDetails = new BCCancelTransactionDetails(bcResponse, paymentData);
                    if (errorCode == 0) {
                        String html = String.format("<html><head><title>Redirecting to Bank</title></head><body>%s</body><script type=\"text/javascript\">document.citrusSubmitForm.submit();</script></html>", bcResponse.getInfo().getCitrusSubmitForm());
                        mPaymentWebview.loadData(html, "text/html", "utf-8");
                    } else {
                        sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, bcResponse.getErrorMessage(), mTransactionId));
                    }

                }

                @Override
                public void error(CitrusError error) {
                    sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, error.getMessage(), mTransactionId));
                }
            });
        }
    }



    private void processNewMakePaymentTransaction(PaymentBill paymentBill) {
        PaymentData paymentData = new PaymentData(paymentBill, mPaymentOption, mCitrusUser, mDynamicPricingResponse);
        mCitrusClient.newMakePayment(paymentData.getPaymentJSON(), new com.citrus.sdk.Callback<String>() {
            @Override
            public void success(String bankHTML) {
                // Loading html directly
                mPaymentWebview.loadDataWithBaseURL(mCitrusClient.getEnvironment().getBaseCitrusUrl(), bankHTML, "text/html", "utf-8", null);
            }

            @Override
            public void error(CitrusError error) {
                sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, error.getMessage(), mTransactionId));
            }
        });
    }

    /**
     * THis method will be used for Load Money transaction using new Make Payment. We have already Payment Data ready.
     *
     * @param paymentData
     */
    private void processNewMakePaymentTransaction(PaymentData paymentData) {
        mCitrusClient.newMakePayment(paymentData.getPaymentJSON(), new com.citrus.sdk.Callback<String>() {
            @Override
            public void success(String bankHTML) {
                // Loading html directly
                mPaymentWebview.loadDataWithBaseURL(mCitrusClient.getEnvironment().getBaseCitrusUrl(), bankHTML, "text/html", "utf-8", null);
            }

            @Override
            public void error(CitrusError error) {
                sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, error.getMessage(), mTransactionId));
            }
        });
    }

    private void processPayUsingCitrusCashPayment(PaymentBill paymentBill) {
        Amount amount = paymentBill.getAmount();
        CitrusCash citrusCashOption = new CitrusCash(amount, mCitrusUser);
        PaymentData paymentData = new PaymentData(paymentBill, citrusCashOption, mCitrusUser, mDynamicPricingResponse);

        mCitrusClient.makeMOTOPayment(paymentData.getPaymentJSON(), new com.citrus.sdk.Callback<StructResponsePOJO>() {
            @Override
            public void success(StructResponsePOJO structResponsePOJO) {
                setCookie();
                openPaymentUrl(structResponsePOJO);
            }

            @Override
            public void error(CitrusError error) {
                sendResult(new TransactionResponse(TransactionResponse.TransactionStatus.FAILED, error.getMessage(), null));
            }
        });
    }


    private void showDialog(String message, boolean cancelable) {
        if (mProgressBar != null) {
            mProgressBar.setMessage(message);
            mProgressBar.show();
            mProgressBar.setCanceledOnTouchOutside(false);
            mProgressBar.setCancelable(cancelable);

            mShowingDialog = true;
        }
    }

    private void dismissDialog() {
        if (mProgressBar != null) {
            mProgressBar.dismiss();
            mShowingDialog = false;
        }
    }

    private void registerSMSReceivers() {
        // Register receivers only if the autoOTP is enabled and payment mode is Credit/Debit Card.
        if (autoOTPEnabled && mPaymentOption instanceof CardOption && mEnvironment == Environment.PRODUCTION) {
            Logger.d("Registering SMS receivers");

            mSMSReceiverRegistered = true;

            if (mSMSReceiver == null) {
                mSMSReceiver = new SMSReceiver();
            }

            if (mAutoOtpSMSReceiveListener == null) {
                mAutoOtpSMSReceiveListener = new BroadcastReceiver() {
                    @Override
                    public void onReceive(Context context, Intent intent) {
                        autoOtpReceived(intent);
                    }
                };
            }

            IntentFilter intentFilter = new IntentFilter(Constants.ACTION_SMS_RECEIVED);
            intentFilter.setPriority(Constants.SMS_RECEIVER_PRIORITY);
            registerReceiver(mSMSReceiver, intentFilter);
            LocalBroadcastManager.getInstance(this).registerReceiver(mAutoOtpSMSReceiveListener, new IntentFilter(Constants.ACTION_AUTO_READ_OTP));
        }
    }

    private void unregisterSMSReceivers() {

        // Unregister receivers only if the autoOTP is enabled and payment mode is Credit/Debit Card.
        if (autoOTPEnabled && mPaymentOption instanceof CardOption && mSMSReceiverRegistered) {

            Logger.d("Unregistering SMS receivers");

            if (mSMSReceiver != null) {
                unregisterReceiver(mSMSReceiver);
                mSMSReceiverRegistered = false;
                mSMSReceiver = null;
            }

            if (mAutoOtpSMSReceiveListener != null) {
                LocalBroadcastManager.getInstance(this).unregisterReceiver(mAutoOtpSMSReceiveListener);
                mAutoOtpSMSReceiveListener = null;
            }
        }
    }

    private void autoOtpReceived(Intent intent) {
        otp = intent.getStringExtra(Constants.INTENT_EXTRA_AUTO_OTP);
        // Check whether the otp is not empty
        if (!TextUtils.isEmpty(otp)) {
            otpProcessTransactionJS = String.format(netBankForOTP.getTransactionJS(), otp);

            unregisterSMSReceivers();
            autoReadOTP = true;

            // This is done to avoid,
            // when user cancels transaction, so OTP Dialog is dismissed.
            // After this, OTP is received and trying to set on a Null reference field.
            if (!otpPopupDismissed) {
                mOTPPopupView.setOTP(otp);
            }
        }
    }

    private void displayOtpPopup() {
        // Display popup only if the autoOTP is enabled and payment mode is Credit/Debit Card.
        if (autoOTPEnabled && mPaymentOption instanceof CardOption && netBankForOTP != NetBankForOTP.UNKNOWN && mEnvironment == Environment.PRODUCTION) {

            // Prevent the activity from sleeping.
            getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
            // Show the Overlay on the otp popup i.e. dark gray screen
            findViewById(R.id.otpPopupOverlayId).setVisibility(View.VISIBLE);

            // Show OTP Popup Layout
            findViewById(R.id.otp_popup_layout).setVisibility(View.VISIBLE);

            // If in case of few banks there is no option to enter password, only otp is directly triggered, so hide the enter password button.
            if (netBankForOTP.isBypassEnterPasswordButton()) {
                mOTPPopupView.enableEnterPasswordButton(false);
            }

            // If in case of few banks the otp is directly triggered, so hide the send OTP button.
            if (netBankForOTP.isBypassSendOTPButton()) {
                mOTPPopupView.displayOtpAutoDetectPopup();
                startOtpReadTimer();
            }
        }
    }

    private void dismissOtpPopup() {
        // Hide the Overlay on the otp popup i.e. dark gray screen
        findViewById(R.id.otpPopupOverlayId).setVisibility(View.GONE);

        // Hide the OTP Popup.
        findViewById(R.id.otp_popup_layout).setVisibility(View.GONE);

        otpPopupDismissed = true;
    }

    private void fetchBinRequestData(CardOption cardOption) {
        mCitrusClient.getBINDetails(cardOption, new com.citrus.sdk.Callback<BinServiceResponse>() {
            @Override
            public void success(BinServiceResponse binServiceResponse) {
                netBankForOTP = binServiceResponse.getNetBankForOTP();
                country = binServiceResponse.getCountry();
                binCardType = binServiceResponse.getCardType();
                Logger.d("netbankForOTP : " + netBankForOTP);

                mOTPPopupView.setNetBankForOTP(netBankForOTP);
            }

            @Override
            public void error(CitrusError error) {
                // NOOP
            }
        });
    }

    private void enableWebContentDebugging() {
        mPaymentWebview.setEnabled(true); // Required for the automation testing.

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            if (0 != (getApplicationInfo().flags &= ApplicationInfo.FLAG_DEBUGGABLE)) {
                WebView.setWebContentsDebuggingEnabled(true);
            }
        }
    }

    @Override
    public void onBackPressed() {
        handleCancelTransaction();
    }

    private void handleCancelTransaction() {
        AlertDialog.Builder builder = new AlertDialog.Builder(this);
        // Add the buttons
        builder.setPositiveButton("Yes", new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialog, int id) {
                dialog.dismiss();
                isBackKeyPressedByUser = true;

                // Set the title since the transaction is done.
                setTitle(Html.fromHtml("<font color=\"" + mTextColorPrimary + "\"> Cancelling... </font>"));

                if (useNewAPI || mPaymentType instanceof WalletPGPayment) {
                    String vanity = mCitrusClient.getVanity();
                    String postData = Utils.getURLEncodedParamsForCancelTransaction(mCitrusUser, mPaymentBill, mPaymentOption, mDynamicPricingResponse, vanity);
                    Environment environment = mCitrusClient.getEnvironment();
                    mPaymentWebview.postUrl(environment.getCancelUrl(vanity), postData.getBytes());

                    dismissOtpPopup();
                } else {

                    if (mBlazePayments != null) {
                        if (mBlazePayments == BlazePayments.BlazeCard) {
                            //mCitrusClient.cancelBCTransaction();
                            // mBCCancelTransactionDetails.ge
                            String cancelTransactionJSON = mBCCancelTransactionDetails.getPaymentData().getBCCancelJSON(mBCCancelTransactionDetails.getBCResponse().getCitrusTransactionId());

                            mCitrusClient.cancelBCTransaction(cancelTransactionJSON, new Callback<BCCancelResponse>() {
                                @Override
                                public void success(BCCancelResponse bcCancelResponse) {
                                    Logger.d("BC CANCEL SUCCESS RESPONSE ***", bcCancelResponse.toString());

                                    String html = String.format("<html><head><title>Redirecting to Bank</title></head><body>%s</body><script type=\"text/javascript\">document.citrusReturnUrlForm.submit();</script></html>", bcCancelResponse.getInfo().getHtmlData());
                                    mPaymentWebview.loadData(html, "text/html", "utf-8");
                                }

                                @Override
                                public void error(CitrusError error) {
                                    Logger.d("BC CANCEL RESPONSE ***", error.getMessage());
                                }
                            });

                        }
                    }
                    // If the PaymentType is CitrusCash or network is not available, finish the activity and mark the status as cancelled.
                    // else load the url again so that Citrus can cancel the transaction and return the control to app normal way.
                    else if (mPaymentType instanceof PaymentType.CitrusCash || !Utils.isNetworkConnected(mContext)) {
                        TransactionResponse transactionResponse = new TransactionResponse(TransactionResponse.TransactionStatus.CANCELLED, "Cancelled By User", mTransactionId);
                        sendResult(transactionResponse);
                    } else {
                        mPaymentWebview.loadUrl(mpiServletUrl);

                        dismissOtpPopup();
                    }
                }
            }
        });
        builder.setNegativeButton("No", new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialog, int id) {

            }
        });
        // Set other dialog properties
        builder.setMessage("Do you want to cancel the transaction?")
                .setTitle("Cancel Transaction?");
        // Create the AlertDialog
        AlertDialog dialog = builder.create();
        dialog.setCanceledOnTouchOutside(false);
        dialog.show();
    }

    private void setCookie() {
        String baseUrl = mCitrusClient.getEnvironment().getBaseUrl();
        cookieManager = CookieManager.getInstance();
        sessionCookie = new PersistentConfig(CitrusActivity.this).getCookieString();
        cookieManager.setCookie(baseUrl, sessionCookie);
    }

    private void removeCookies() {
        String baseUrl = mCitrusClient.getEnvironment().getBaseUrl();
        CookieManager.getInstance().setCookie(baseUrl, Constants.CITRUS_PREPAID_COOKIE);
    }

    private void sendResult(TransactionResponse transactionResponse) {

        if (!isFinishing()) {
            transactionResponse.setCountry(country);
            transactionResponse.setBinCardType(binCardType);
            // When the activity is still in foreground,
            // Send the response to the caller.
            Intent intent = new Intent();
            intent.putExtra(Constants.INTENT_EXTRA_TRANSACTION_RESPONSE, transactionResponse);

            // According new implementation, finish the activity and post the event to citrusClient.
            intent.setAction(mPaymentType.getIntentAction());
            // Send the broadcast for normal requets.
            if (mRequestCode != Constants.REQUEST_CODE_PAYMENT) {
                LocalBroadcastManager.getInstance(mContext).sendBroadcast(intent);
            }

            setResult(RESULT_OK, intent);
            finish();

        }
    }

    @Override
    protected void onStop() {
        super.onStop();

        dismissDialog();
    }

    @Override
    public void onDestroy() {
        super.onDestroy();

        if (mPaymentWebview != null) {
            mPaymentWebview.stopLoading();
            mPaymentWebview.destroy();
        }
        mPaymentBill = null;
        mPaymentWebview = null;
        mPaymentType = null;
        mCitrusConfig = null;
        mCitrusUser = null;
        mTransactionId = null;

        dismissDialog();
        netBankForOTP = NetBankForOTP.UNKNOWN;
        mProgressBar = null;
        mPaymentOption = null;
        mActivityTitle = null;
        transactionProcessed = false;
        mMultipartSendOTPJS = false;
    }

    @Override
    public void onSendOtpClicked() {
        mMultipartSendOTPJS = netBankForOTP.isMultipartSendOTPJS();

        mPaymentWebview.loadUrl(netBankForOTP.getSendOTPJS());
        mOTPPopupView.displayOtpAutoDetectPopup();
        startOtpReadTimer();
    }

    @Override
    public void onEnterPasswordClicked() {

        mMultipartEnterPasswordJS = netBankForOTP.isMultipartEnterPasswordJS();

        String enterPwdJS = netBankForOTP.getEnterPasswordJS();
        mPaymentWebview.loadUrl(enterPwdJS);

        // Hide the OTP PopUp View.
        dismissOtpPopup();
    }

    @Override
    public void onCancelClicked() {
        handleCancelTransaction();
    }

    @Override
    public void onProcessTransactionClicked(String otp) {
        // Set OTP on bank's page.
        mPaymentWebview.loadUrl(netBankForOTP.getSetOTPJS(otp));

        String js = String.format(netBankForOTP.getTransactionJS(), otp);
        mPaymentWebview.loadUrl(js);

        transactionProcessed = true;

        // Hide the popup since proceeding with transaction.
        dismissOtpPopup();
    }

    @Override
    public void onResendOTPClicked() {
        mPaymentWebview.loadUrl(netBankForOTP.getReSendOTPJS());

        // Register sms receivers
        registerSMSReceivers();
        startOtpReadTimer();

        autoReadOTP = false;
    }

    @Override
    public void startOtpReadTimer() {
        mOTPPopupView.handleResendOTP();

        final Handler handler = new Handler();
        handler.postDelayed(new Runnable() {
            @Override
            public void run() {

                unregisterSMSReceivers();
                if (!autoReadOTP) {
                    mOTPPopupView.otpReadTimeout();
                }
            }
        }, OTP_READ_TIMEOUT);
    }

    /**
     * Handle all the Webview loading in custom webview client.
     */
    private class CitrusWebClient extends WebViewClient {

        @Override
        public boolean shouldOverrideUrlLoading(WebView view, String url) {
            // Let this webview handle all the urls loaded inside. Return false to denote that.
            view.loadUrl(url);

            return true;
        }

        @Override
        public void onPageStarted(WebView view, String url, Bitmap favicon) {
            super.onPageStarted(view, url, favicon);
            // Display the message in case of cancelled transaction.
            if (isBackKeyPressedByUser) {
                // Show cancel message.
                showDialog(CANCEL_MESSAGE, true);
            } else if (!mShowingDialog) {
                // Show dialog is not already shown. Applies when the user clicks on 3DS page.
                showDialog(WAIT_MESSAGE, true);
            }

            mTimer.cancel();
        }

        @Override
        public void onPageFinished(WebView view, String url) {
            super.onPageFinished(view, url);
            // If the data is loaded for the dummy screen, then do not dismiss the dialog, i.e. do not start the timer to dismiss the dialog.
            // This case will occur only once, and if citrusClient.isShowDummyScreenWhilePayments is true.
            // Not inverting the condition for code readability and understanding.
            if (mCitrusClient.isShowDummyScreenWhilePayments() && url.startsWith("data:text/html")) {
                // Do nothing i.e. do not start the timer.
            } else {
                mTimer.start();
            }

            view.loadUrl("javascript:window.CitrusResponse.printHTML('<html>'+document.getElementsByTagName('html')[0].innerHTML+'</html>');");

        }

        @Override
        public void onReceivedSslError(WebView view, SslErrorHandler handler, SslError error) {
            // Dismiss Dialog
            dismissDialog();
            // Cancelling loading of the page.
            handler.cancel();
        }
    }


    /**
     * This class will be loaded as JSInterface and the methods of this class will be called from
     * the javascript loaded inside webview.
     * <p/>
     * Handle the payment response and take actions accordingly.
     */
    private class JsInterface {

        @JavascriptInterface
        public void pgResponse(String response) {

            Logger.d("PG Response :: " + response);

            if (mPaymentType instanceof PaymentType.CitrusCash) {
                removeCookies();
            }
            TransactionResponse transactionResponse = TransactionResponse.fromJSON(response, customParametersOriginalMap);
            sendResult(transactionResponse);
        }

        /**
         * This method will be called by returnURL when Cash is loaded in user's account
         *
         * @param response post parameters sent by Citrus
         */
        @JavascriptInterface
        public void loadWalletResponse(String response) {

            Logger.d("Wallet response :: " + response);

            TransactionResponse transactionResponse = TransactionResponse.parseLoadMoneyResponse(response);
            sendResult(transactionResponse);
        }

        @JavascriptInterface
        public void rawPGResponse(String response) {

            Logger.d("rawPGResponse :: " + response);

            TransactionResponse transactionResponse = new TransactionResponse(TransactionResponse.TransactionStatus.SUCCESSFUL, "", null);
            transactionResponse.setJsonResponse(response);
            sendResult(transactionResponse);
        }

        @JavascriptInterface
        public void printHTML(String response) {
            if (!TextUtils.isEmpty(response)) {
                if (response.contains("Un-Registered") || response.contains("Registration") || response.contains("Step 1")) {
                    isCardUnRegistered = true;
                }
            }
        }
    }
}
