/*
   Copyright 2014 Citrus Payment Solutions Pvt. Ltd.
   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at
     http://www.apache.org/licenses/LICENSE-2.0
   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package com.citrus.sdk.payment;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.classes.PGHealth;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by salil on 13/2/15.
 */
public final class NetbankingOption extends PaymentOption implements Parcelable {

    private String bankName = null;
    private String bankCID = null;

    private NetbankingOption() {

    }

    public NetbankingOption(String bankName, String bankCID) {
        this.bankName = bankName;
        this.bankCID = bankCID;
    }

    /**
     * Use this constructor for wallet PG payment processing.
     *
     * @param transactionAmount
     * @param bankName
     * @param bankCID
     */
    public NetbankingOption(Amount transactionAmount, String bankName, String bankCID) {
        this.transactionAmount = transactionAmount;
        this.bankName = bankName;
        this.bankCID = bankCID;
    }

    /**
     * Use this constructor for tokenized payments.
     *
     * @param token Token of the bank
     */
    public NetbankingOption(String token) {
        super(null, token);
    }

    /**
     * @param name     - Human readable names for banks. e.g. Net Banking - AXIS BANK
     * @param token    - Token for netbanking payment.
     * @param bankName - Bank's name - ICICI, AXIS.
     */
    NetbankingOption(String name, String token, String bankName) {
        super(name, token);
        this.bankName = bankName;
    }

    public String getBankName() {
        return bankName;
    }

    public void setName(String bankName) {
        super.name = bankName;
    }

    public String getBankCID() {
        return bankCID;
    }

    @Override
    public void setPgHealth(PGHealth pgHealth) {
        super.setPgHealth(pgHealth);
    }

    @Override
    public String getDynamicPricingPaymentMode() {
        // In case of tokenized payments. The payment mode is CITRUS_WALLET
        if (!TextUtils.isEmpty(token)) {
            return "CITRUS_WALLET";
        }

        return "NET_BANKING";
    }

    @Override
    public Drawable getOptionIcon(Context context) {
        // Return the icon depending upon the scheme of the card.
        Drawable drawable = null;

        int resourceId = 0;
        if ("AXIS Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("axis_bank", "drawable", context.getPackageName());
        } else if ("Andhra Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("andhra_bank", "drawable", context.getPackageName());
        } else if ("Bank of India".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("bank_of_india", "drawable", context.getPackageName());
        } else if ("Bank Of Baroda".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("bank_of_baroda", "drawable", context.getPackageName());
        } else if ("Bank of Maharashtra".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("bank_of_maharashtra", "drawable", context.getPackageName());
        } else if ("Catholic Syrian Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("catholic_syrian", "drawable", context.getPackageName());
        } else if ("Central Bank of India".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("central_bank_of_india", "drawable", context.getPackageName());
        } else if ("Citibank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("citi_bank", "drawable", context.getPackageName());
        } else if ("CITI Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("citi_bank", "drawable", context.getPackageName());
        } else if ("Corporation Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("corporation_bank", "drawable", context.getPackageName());
        } else if ("City Union Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("city_union_bank", "drawable", context.getPackageName());
        } else if ("Canara Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("canara_bank", "drawable", context.getPackageName());
        } else if ("Cosmos Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("cosmos_bank", "drawable", context.getPackageName());
        } else if ("DEUTSCHE Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("deutsche_bank", "drawable", context.getPackageName());
        } else if ("DCB Bank Personal".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("dcb_bank", "drawable", context.getPackageName());
        } else if ("Federal Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("federal_bank", "drawable", context.getPackageName());
        } else if ("HDFC Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("hdfc_bank", "drawable", context.getPackageName());
        } else if ("ICICI Bank".equalsIgnoreCase(bankName) || "ICICI Corporate Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("icici_bank", "drawable", context.getPackageName());
        } else if ("IDBI Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("idbi_bank", "drawable", context.getPackageName());
        } else if ("Indian Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("indian_bank", "drawable", context.getPackageName());
        } else if ("Indian Overseas Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("indian_overseas_bank", "drawable", context.getPackageName());
        } else if ("IndusInd Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("indusind_bank", "drawable", context.getPackageName());
        } else if ("ING VYSA".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("ing_vysa", "drawable", context.getPackageName());
        } else if ("Kotak Mahindra Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("kotak_mahindra_bank", "drawable", context.getPackageName());
        } else if ("ING Vysya Bank (now Kotak)".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("kotak_mahindra_bank", "drawable", context.getPackageName());
        } else if ("Karur Vysya Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("karur_vyasa_bank", "drawable", context.getPackageName());
        } else if ("Karnataka Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("karnataka_bank", "drawable", context.getPackageName());
        } else if ("PNB Retail".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("punjab_national", "drawable", context.getPackageName());
        } else if ("PNB Corporate".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("punjab_national", "drawable", context.getPackageName());
        } else if ("SBI Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("sbi_bank", "drawable", context.getPackageName());
        } else if ("State Bank of Bikaner and Jaipur".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("state_bank_of_bikaner_and_jaipur", "drawable", context.getPackageName());
        } else if ("State Bank of Hyderabad".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("state_bank_of_hyderabad", "drawable", context.getPackageName());
        } else if ("State Bank of Mysore".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("state_bank_of_mysore", "drawable", context.getPackageName());
        } else if ("State Bank of Travancore".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("state_bank_of_travancore", "drawable", context.getPackageName());
        } else if ("State Bank of Patiala".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("state_bank_of_patiala", "drawable", context.getPackageName());
        } else if ("South Indian Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("south_indian_bank", "drawable", context.getPackageName());
        } else if ("Union Bank Of India".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("union_bank", "drawable", context.getPackageName());
        } else if ("Union Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("union_bank", "drawable", context.getPackageName());
        } else if ("UCO Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("uco", "drawable", context.getPackageName());
        } else if ("United Bank of India".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("united_bank_of_india", "drawable", context.getPackageName());
        } else if ("Vijaya Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("vijaya_bank", "drawable", context.getPackageName());
        } else if ("YES Bank".equalsIgnoreCase(bankName)) {
            resourceId = context.getResources().getIdentifier("yes_bank", "drawable", context.getPackageName());
        } else {
            resourceId = context.getResources().getIdentifier("default_bank", "drawable", context.getPackageName());
        }

        if (resourceId == 0) {
            resourceId = context.getResources().getIdentifier("default_bank", "drawable", context.getPackageName());
        }

        if (resourceId != 0) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                drawable = context.getResources().getDrawable(resourceId, null);
            } else {
                drawable = context.getResources().getDrawable(resourceId);
            }
        }

        return drawable;
    }

    @Override
    public String getSavePaymentOptionObject() {
        JSONObject object = null;
        try {
            object = new JSONObject();
            JSONArray paymentOptions = new JSONArray();

            JSONObject option = new JSONObject();
            option.put("owner", "");
            option.put("bank", bankName);
            option.put("type", "netbanking");
            paymentOptions.put(option);

            object.put("paymentOptions", paymentOptions);
            object.put("type", "payment");
        } catch (JSONException e) {
            e.printStackTrace();
        }

        return object.toString();
    }

    @Override
    public String getSaveDefaultPaymentOptionObject() {

        JSONObject object = null;
        try {
            object = new JSONObject();
            JSONArray paymentOptions = new JSONArray();

            JSONObject option = new JSONObject();
            option.put("owner", "");
            option.put("bank", bankName);
            option.put("type", "netbanking");
            option.put("number", "null");
            option.put("scheme", "null");
            option.put("expiryDate", "null");
            option.put("name", super.getName());
            paymentOptions.put(option);

            object.put("paymentOptions", paymentOptions);
            object.put("type", "payment");
            object.put("defaultOption", super.getName());
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return object.toString();
    }

    // @formatter:off

    /**
     * This will return the json object for moto or new make payment call.
     *
     * @return For Netbanking the format is
     * <p/>
     * {
     * "type":"paymentOptionToken",
     * "paymentMode":{
     * "type":"netbanking",
     * "code":"CID001"
     * }
     * }
     * <p/>
     * OR
     * {
     * "id":"4e361d717bf26359f1b5ac6f33edda37",
     * "type":"paymentOptionIdToken"
     * }
     */
    // @formatter:on
    @Override
    public JSONObject getMOTOPaymentOptionObject() throws JSONException {
        JSONObject jsonObject = new JSONObject();

        if (isTokenizedPayment()) {
            jsonObject.put("type", "paymentOptionIdToken");
            jsonObject.put("id", token);
        } else {
            jsonObject.put("type", "paymentOptionToken");
            // PaymentMode
            JSONObject paymentMode = new JSONObject();
            paymentMode.put("type", "netbanking");
            paymentMode.put("code", bankCID);

            jsonObject.put("paymentMode", paymentMode);
        }

        return jsonObject;
    }

    /**
     * This will return the json required for wallet charge api.
     * The json format is
     * <p/>
     * {
     * "paymentMode": "NET_BANKING",
     * "name": "",
     * "cardNumber": "",
     * "cardExpiryDate": "",
     * "cardScheme": "",
     * "amount": "",
     * "currency": "",
     * "cvv": "",
     * "issuerCode" : "CID001",
     * "bank" : "ICICI Bank"
     * }
     */
    public JSONObject getWalletChargePaymentOptionObject() throws JSONException {
        JSONObject jsonObject = null;
        if (transactionAmount != null) {

            jsonObject = new JSONObject();
            jsonObject.put("paymentMode", "NET_BANKING");

            // These are mandatory parameters as of now for netbanking as well.
            // Hence passing these values as blank.
            if (isTokenizedPayment()) {
                jsonObject.put("savedCardToken", token);
            } else {
                // Bank Details
                jsonObject.put("issuerCode", bankCID);
                jsonObject.put("bank", bankName);
            }

            jsonObject.put("name", "");
            jsonObject.put("cardNumber", "");
            jsonObject.put("cardExpiryDate", "");
            jsonObject.put("cardScheme", "");
            jsonObject.put("cvv", "");

            jsonObject.put("amount", String.valueOf(transactionAmount.getValueAsDouble()));
            jsonObject.put("currency", transactionAmount.getCurrency());

        }
        return jsonObject;
    }

    @Override
    public String toString() {
        return bankName;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        NetbankingOption that = (NetbankingOption) o;

        return bankName.equals(that.bankName);

    }

    @Override
    public int hashCode() {
        return bankName.hashCode();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        super.writeToParcel(dest, flags);
        dest.writeString(this.bankName);
        dest.writeString(this.bankCID);
    }

    protected NetbankingOption(Parcel in) {
        super(in);
        this.bankName = in.readString();
        this.bankCID = in.readString();
    }

    public static final Creator<NetbankingOption> CREATOR = new Creator<NetbankingOption>() {
        public NetbankingOption createFromParcel(Parcel source) {
            return new NetbankingOption(source);
        }

        public NetbankingOption[] newArray(int size) {
            return new NetbankingOption[size];
        }
    };
}
