package com.citrus.sdk;

import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.classes.CitrusException;
import com.citrus.sdk.payment.CardOption;
import com.citrus.sdk.payment.MVCOption;
import com.citrus.sdk.payment.NetbankingOption;
import com.citrus.sdk.payment.PaymentBill;
import com.citrus.sdk.payment.PaymentOption;
import com.citrus.sdk.payment.PaymentType;

import java.math.BigDecimal;
import java.util.List;

/**
 * Created by mangesh on 3/5/16.
 */
class WalletPGPayment extends PaymentType implements Parcelable {

    private PaymentOption otherPaymentOption = null;
    private com.citrus.sdk.payment.CitrusCash mCitrusCashOption = null;
    private MVCOption mMVCOption = null;

    protected WalletPGPayment() {
    }

    /**
     * @param amount            - Amount to be Payed
     * @param billUrl           - url of the billGenerator
     * @param paymentOptionList - List of PaymentOptions to be used e.g. Card details, netbanking or Wallet etc selected.
     * @throws CitrusException - if Amount or returnUrl, or paymentOption is null.
     */
    public WalletPGPayment(Amount amount, String billUrl, List<PaymentOption> paymentOptionList) throws CitrusException {
        super(amount, billUrl, paymentOptionList);

        if (amount == null || TextUtils.isEmpty(amount.getValue())) {
            throw new CitrusException("Amount should be not null or blank.");
        } else if (!(amount.getValueAsDouble() > 0)) {
            throw new CitrusException("Amount should be greater than 0");
        } else if (billUrl == null) {
            throw new CitrusException("billUrl should be not null.");
        } else if (paymentOptionList == null && paymentOptionList.size() == 0) {
            throw new CitrusException("PaymentOptions List should be not null or empty");
        }

        separatePaymentOptions();
    }


    public WalletPGPayment(PaymentType.SplitPayment splitPayment, List<PaymentOption> paymentOptionList) throws CitrusException {

        if (splitPayment.getPaymentBill() != null) {
            this.paymentBill = splitPayment.getPaymentBill();
        } else {
            this.amount = splitPayment.getAmount();
            this.url = splitPayment.getUrl();
        }

        this.paymentOptionList = paymentOptionList;

        if (amount == null || TextUtils.isEmpty(amount.getValue())) {
            throw new CitrusException("Amount should be not null or blank.");
        } else if (!(amount.getValueAsDouble() > 0)) {
            throw new CitrusException("Amount should be greater than 0");
        } else if (this.url == null) {
            throw new CitrusException("billUrl should be not null.");
        } else if (paymentOptionList == null && paymentOptionList.size() == 0) {
            throw new CitrusException("PaymentOptions List should be not null or empty");
        }

        separatePaymentOptions();

    }

    /**
     * Pay using the response of the billGenerator. If you are fetching your bill, please use this constructor.
     *
     * @param paymentBill
     * @param paymentOptionList - List of payment options to be used e.g. Debit/Credit Card and CitrusCash or
     * @throws CitrusException if the paymentBill or paymentOption is null.
     */
    public WalletPGPayment(PaymentBill paymentBill, List<PaymentOption> paymentOptionList) throws CitrusException {
        super(paymentBill);
        this.paymentOptionList = paymentOptionList;

        if (paymentBill == null) {
            throw new CitrusException("PaymentBill should not be null.");
        }

        if (paymentOptionList == null && paymentOptionList.size() == 0) {
            throw new CitrusException("PaymentOptions List should be not null or empty");
        }

        separatePaymentOptions();
    }

    private void separatePaymentOptions() {
        if (paymentOptionList != null) {
            for (PaymentOption paymentOption : paymentOptionList) {
                if (paymentOption instanceof NetbankingOption || paymentOption instanceof CardOption) {
                    otherPaymentOption = paymentOption;
                } else if (paymentOption instanceof com.citrus.sdk.payment.CitrusCash) {
                    mCitrusCashOption = (com.citrus.sdk.payment.CitrusCash) paymentOption;
                } else if (paymentOption instanceof MVCOption) {
                    mMVCOption = (MVCOption) paymentOption;
                }
            }
        }
    }

    /**
     * Validate the total transaction amount of all the payment options should not be greater than the actual transaction amount.
     *
     * @return
     */
    public boolean validateTotalTransactionAmount() {
        // No need to check null for amount since it is already done.
        BigDecimal transactionAmount = new BigDecimal(amount.getValue());
        BigDecimal totalAmount = BigDecimal.ZERO;

        if (otherPaymentOption != null && otherPaymentOption.getTransactionAmount() != null) {
            BigDecimal amount = new BigDecimal(otherPaymentOption.getTransactionAmount().getValue());
            totalAmount = totalAmount.add(amount);
        }

        if (mMVCOption != null && mMVCOption.getTransactionAmount() != null) {
            BigDecimal amount = new BigDecimal(mMVCOption.getTransactionAmount().getValue());
            totalAmount = totalAmount.add(amount);
        }

        if (mCitrusCashOption != null && mCitrusCashOption.getTransactionAmount() != null) {
            BigDecimal amount = new BigDecimal(mCitrusCashOption.getTransactionAmount().getValue());
            totalAmount = totalAmount.add(amount);
        }

        return transactionAmount.compareTo(totalAmount) == 0;
    }

    public PaymentOption getOtherPaymentOption() {
        return otherPaymentOption;
    }

    public com.citrus.sdk.payment.CitrusCash getCitrusCashOption() {
        return mCitrusCashOption;
    }

    public MVCOption getMVCOption() {
        return mMVCOption;
    }

    /**
     * Check whether the current transaction is only with wallet.
     *
     * @return true if the payment options are only MVC and Citrus Cash. False if other payment option such as Netbanking or Cards are also used.
     */
    public boolean isWalletOnlyPayment() {
        return otherPaymentOption == null;
    }

    @Override
    public final String getIntentAction() {
        return Constants.ACTION_WALLET_PG_PAYMENT;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        super.writeToParcel(dest, flags);
        dest.writeParcelable(this.otherPaymentOption, 0);
        dest.writeParcelable(this.mCitrusCashOption, 0);
        dest.writeParcelable(this.mMVCOption, 0);
    }

    protected WalletPGPayment(Parcel in) {
        super(in);
        this.otherPaymentOption = in.readParcelable(PaymentOption.class.getClassLoader());
        this.mCitrusCashOption = in.readParcelable(com.citrus.sdk.payment.CitrusCash.class.getClassLoader());
        this.mMVCOption = in.readParcelable(MVCOption.class.getClassLoader());
    }

    public static final Creator<WalletPGPayment> CREATOR = new Creator<WalletPGPayment>() {
        public WalletPGPayment createFromParcel(Parcel source) {
            return new WalletPGPayment(source);
        }

        public WalletPGPayment[] newArray(int size) {
            return new WalletPGPayment[size];
        }
    };
}