package com.citrus.sdk.payment;

import com.citrus.sdk.CitrusUser;
import com.citrus.sdk.classes.CitrusPrepaidBill;
import com.citrus.sdk.dynamicPricing.DynamicPricingResponse;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

/**
 * Created by salil on 3/2/16.
 */
public class PaymentData {

    private PaymentBill paymentBill = null;
    private CitrusPrepaidBill prepaidBill = null;
    private PaymentOption paymentOption = null;
    private List<PaymentOption> listPaymentOptions = null;
    private CitrusUser citrusUser = null;
    private DynamicPricingResponse dynamicPricingResponse = null;

    /**
     * Use this constructor for PG Payment Or New PG Payment with Dynamic Pricing Response Processing.
     *
     * @param paymentBill
     * @param paymentOption
     * @param citrusUser
     * @param dynamicPricingResponse
     */
    public PaymentData(PaymentBill paymentBill, PaymentOption paymentOption, CitrusUser citrusUser, DynamicPricingResponse dynamicPricingResponse) {
        this.paymentBill = paymentBill;
        this.paymentOption = paymentOption;
        this.citrusUser = citrusUser;
        this.dynamicPricingResponse = dynamicPricingResponse;
    }

    /**
     * Use this constructor for Wallet PG with Dynamic Pricing Response Processing.
     *
     * @param paymentBill
     * @param listPaymentOptions
     * @param citrusUser
     * @param dynamicPricingResponse
     */
    public PaymentData(PaymentBill paymentBill, List<PaymentOption> listPaymentOptions, CitrusUser citrusUser, DynamicPricingResponse dynamicPricingResponse) {
        this.paymentBill = paymentBill;
        this.listPaymentOptions = listPaymentOptions;
        this.citrusUser = citrusUser;
        this.dynamicPricingResponse = dynamicPricingResponse;
    }

    /**
     * Use this constructor for processing Load Money Payment.
     *
     * @param prepaidBill
     * @param paymentOption
     * @param citrusUser
     */
    public PaymentData(CitrusPrepaidBill prepaidBill, PaymentOption paymentOption, CitrusUser citrusUser) {
        this.prepaidBill = prepaidBill;
        this.paymentOption = paymentOption;
        this.citrusUser = citrusUser;
    }
/*
    {
   "returnUrl":"https:\/\/salty-plateau-1529.herokuapp.com\/redirectURL.production.php",
   "notifyUrl":"https:\/\/salty-plateau-1529.herokuapp.com\/notifyUrl.production.php",
   "amount":{
      "value":"2",
      "currency":"INR"
   },
   "merchantAccessKey":"06SLEEBYLVZELISZ5ECU",
   "customParameters":{
      "param1":"1000",
      "param2":"CitrusTestSDK"
   },
   "paymentToken":{
      "type":"paymentOptionToken",
      "paymentMode":{
         "type":"netbanking",
         "code":"CID002"
      }
   },
   "merchantTxnId":"145372279750468",
   "requestSignature":"40632e40763042b96c4a932a6e5fde3e1cbb346c",
   "requestOrigin":"MSDKG",
   "userDetails":{
      "email":"salilgodbole@gmail.com",
      "mobileNo":"9970950374",
      "firstName":"Salil",
      "lastName":"Godbole",
      "address":{
         "state":"Maharashtra",
         "street1":"streetone",
         "street2":"streettwo",
         "city":"Mumbai",
         "country":"India",
         "zip":"400052"
      }
   }
}
     */

    public String getPaymentJSON() {
        String json = null;
        JSONObject jsonObject;

        try {
            if (paymentBill != null) {
                jsonObject = PaymentBill.toJSONObject(paymentBill);
            } else if (prepaidBill != null) {
                jsonObject = CitrusPrepaidBill.toJSONObject(prepaidBill);
            } else {
                jsonObject = new JSONObject();
            }

            // Put the dynamic pricing offer token.
            if (dynamicPricingResponse != null) {
                jsonObject.put("offerToken", dynamicPricingResponse.getOfferToken());
            }

            // Put the userdetails
            jsonObject.put("userDetails", CitrusUser.toJSONObject(citrusUser, false));

            if (paymentOption != null) {
                JSONObject paymentToken = paymentOption.getMOTOPaymentOptionObject();
                jsonObject.put("paymentToken", paymentToken);

                // RequestOrigin
                if (paymentOption.isTokenizedPayment()) {
                    jsonObject.put("requestOrigin", "MSDKW");
                } else {
                    jsonObject.put("requestOrigin", "MSDKG");
                }
            }

            json = jsonObject.toString();
        } catch (JSONException e) {
            e.printStackTrace();
        }

        return json;
    }

    /*
        {
        "merchantAccessKey": "AHQDTIEYS32HBHD5O74I",
        "email": "testmail1@testmail.com",
        "mobile": 9890793617,
        "currency": "INR",
        "amount": 30,
        "signature": "sckmsncsnc",
        "merchantTransactionId": "ExtMtx222",
        "returnUrl": "http://google.com",
        "sdkSigner", "true",
        "paymentOptions": [
        {
                "paymentMode": "CREDIT_CARD",
                "name": "Test",
                "cardNumber": "4028530052708001",
                "cardExpiryDate": "02/2017",
                "cardScheme": "VISA",
                "amount": "10.00",
                "currency": "INR",
                "cvv": "018"
            },
            {
                "paymentMode": "PREPAID_CARD",
                "amount": "10",
                "currency": "INR"
            },
            {
                "paymentMode": "MVC",
                "amount": "10",
                "currency": "INR",
                "campaignCode": "VCASH"
            }
            ]
         }
     */

    public String getWalletPGPaymentJSON() {
        String json = null;
        JSONObject jsonObject = null;

        try {
            if (paymentBill != null) {
                jsonObject = PaymentBill.toJSONObject(paymentBill, true);

                if (citrusUser != null) {
                    jsonObject.put("email", citrusUser.getEmailId());
                    jsonObject.put("mobile", citrusUser.getMobileNo());
                }

                // Put the dynamic pricing offer token.
                if (dynamicPricingResponse != null) {
                    jsonObject.put("offerToken", dynamicPricingResponse.getOfferToken());
                }

                // Required to bypass the signature matching logic of wallet and use the moto signature logic.
                jsonObject.put("moto", "true");

                if (listPaymentOptions != null) {
                    JSONArray paymentOptionsArray = new JSONArray();
                    for (PaymentOption paymentOption : listPaymentOptions) {
                        JSONObject paymentMode = paymentOption.getWalletChargePaymentOptionObject();
                        paymentOptionsArray.put(paymentMode);
                    }

                    jsonObject.put("paymentOptions", paymentOptionsArray);
                }

                json = jsonObject.toString();
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }

        return json;
    }


    /**
     * this method will be used for BC Cancel Transaction API
     * @param citrusTransacionID
     * @return
     */
    public String getBCCancelJSON(String citrusTransacionID) {
        String json = getPaymentJSON();
        JSONObject jsonObject = null;
        try {
            jsonObject  = new JSONObject(json);
            jsonObject.put("citrusTransactionId", citrusTransacionID);
            json = jsonObject.toString();
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return json;
    }
}
