/*
   Copyright 2014 Citrus Payment Solutions Pvt. Ltd.
   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at
     http://www.apache.org/licenses/LICENSE-2.0
   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package com.citrus.sdk.payment;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.classes.Month;
import com.citrus.sdk.classes.PGHealth;
import com.citrus.sdk.classes.Year;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by salil on 13/2/15.
 */
public abstract class PaymentOption implements Parcelable {

    /**
     * Following variables will be used in case of tokenized payments and mostly internally.
     * Hence no public constructor with these variables is required. If required create a constructor
     * with default access modifier so as to avoid confusion for the merchant developer.
     */
    protected String name = null; // Denotes the friendly name for the payment option.
    protected String token = null; // Denotes the token for the payment option.
    protected boolean savePaymentOption = false;
    protected Amount transactionAmount = null;
    protected PGHealth pgHealth = PGHealth.UNKNOWN;
    protected boolean defaultPaymentMode = false;
    protected boolean selected = false;
    protected boolean shown = false;
    protected String fingerPrint = null;
    private int amount;
    private String cvvText = "";

    PaymentOption() {
    }

    /**
     * @param name  - User friendly name of the payment option. e.g. Debit Card (4242) or Net Banking - ICICI Bank
     * @param token - Token for payment option, used for tokenized payment.
     */
    PaymentOption(String name, String token) {
        this.name = name;
        this.token = token;
    }

    /**
     * Use this method for walletPGPayment
     *
     * @param transactionAmount - TransactionAmount for this payment option.
     * @param name              - User friendly name of the payment option. e.g. Debit Card (4242) or Net Banking - ICICI Bank
     * @param token             - Token for payment option, used for tokenized payment.
     */
    PaymentOption(Amount transactionAmount, String name, String token) {
        this.transactionAmount = transactionAmount;
        this.name = name;
        this.token = token;
    }

    public static PaymentOption fromJSONObject(JSONObject walletObject) {
        return fromJSONObject(walletObject, false);
    }

    public static PaymentOption fromJSONObject(JSONObject walletObject, boolean walletPGParsing) {

        PaymentOption paymentOption = null;

        String type;
        String name;
        String token;
        String cardHolderName;
        String expiry;
        String cardNumber;
        String cardSchemeStr;
        CardOption.CardScheme cardScheme;
        String fingerPrint;
        String bankName;

        String expiryMonth = "";
        String expiryYear = "";

        String amount = "";
        String campaignCode = "";
        String maxBalance = "";
        boolean defaultPaymentMode = false;

        if (walletPGParsing) {
            type = walletObject.optString("paymentMode");
            name = walletObject.optString("name");
            token = walletObject.optString("savedCardToken");
            expiry = walletObject.optString("cardExpiryDate");
            cardNumber = walletObject.optString("cardNumber");
            cardSchemeStr = walletObject.optString("cardScheme");
            cardScheme = CardOption.CardScheme.getCardScheme(cardSchemeStr);
            fingerPrint = walletObject.optString("fingerPrint");
            bankName = walletObject.optString("bank");
            cardHolderName = "";

            amount = walletObject.optString("amount");
            campaignCode = walletObject.optString("campaignCode");
            maxBalance = walletObject.optString("maxBalance");
            defaultPaymentMode = walletObject.optBoolean("defaultPaymentMode", false);
        } else {

            type = walletObject.optString("type");
            name = walletObject.optString("name");
            token = walletObject.optString("token");
            cardHolderName = walletObject.optString("owner");
            expiry = walletObject.optString("expiryDate");
            cardNumber = walletObject.optString("number");
            cardSchemeStr = walletObject.optString("scheme");
            cardScheme = CardOption.CardScheme.getCardScheme(cardSchemeStr);
            fingerPrint = walletObject.optString("fingerPrint");
            bankName = walletObject.optString("bank");
        }

        // The received expiry date is in MMYYYY format so take out expiry month and year which will be required for display purpose.
        if (!TextUtils.isEmpty(expiry) && expiry.length() > 2) {
            expiryMonth = TextUtils.substring(expiry, 0, 2);
            expiryYear = TextUtils.substring(expiry, 2, expiry.length());
        }

        if (TextUtils.equals("credit", type) || TextUtils.equals("CREDIT_CARD", type)) {
            paymentOption = new CreditCardOption(cardHolderName, cardNumber, null, Month.getMonth(expiryMonth), Year.getYear(expiryYear));
            paymentOption.setName(name);
            paymentOption.setToken(token);
            ((CreditCardOption) paymentOption).setCardScheme(cardScheme);
        } else if (TextUtils.equals("debit", type) || TextUtils.equals("DEBIT_CARD", type)) {
            paymentOption = new DebitCardOption(cardHolderName, cardNumber, null, Month.getMonth(expiryMonth), Year.getYear(expiryYear));
            paymentOption.setName(name);
            paymentOption.setToken(token);
            ((DebitCardOption) paymentOption).setCardScheme(cardScheme);
        } else if (TextUtils.equals("netbanking", type) || TextUtils.equals("NET_BANKING", type)) {
            paymentOption = new NetbankingOption(name, token, bankName);
        } else if (TextUtils.equals("MVC", type)) {
            Amount maxBalanceAmount = new Amount(maxBalance);
            Amount transactionAmount = new Amount(amount);
            paymentOption = new MVCOption(transactionAmount, campaignCode, maxBalanceAmount);
        } else if (TextUtils.equals("PREPAID_CARD", type)) {
            Amount maxBalanceAmount = new Amount(maxBalance);
            Amount transactionAmount = new Amount(amount);
            paymentOption = new CitrusCash(transactionAmount, maxBalanceAmount);
        }

        // Set if it is default payment mode.
        paymentOption.setDefaultPaymentMode(defaultPaymentMode);

        if (!TextUtils.isEmpty(fingerPrint)) {
            paymentOption.setFingerPrint(fingerPrint);
        }

        return paymentOption;
    }

    public Amount getTransactionAmount() {
        return transactionAmount;
    }

    public void setTransactionAmount(Amount transactionAmount) {
        this.transactionAmount = transactionAmount;
    }

    public String getName() {
        return name;
    }

    private void setName(String name) {
        this.name = name;
    }

    public String getToken() {
        return token;
    }

    private void setToken(String token) {
        this.token = token;
    }

    public boolean isSavePaymentOption() {
        return savePaymentOption;
    }

    public void setSavePaymentOption(boolean savePaymentOption) {
        this.savePaymentOption = savePaymentOption;
    }

    public PGHealth getPgHealth() {
        return pgHealth;
    }

    protected void setPgHealth(PGHealth pgHealth) {
        this.pgHealth = pgHealth;
    }

    public void setFingerPrint(String fingerPrint) {
        this.fingerPrint = fingerPrint;
    }

    public String getFingerPrint() {
        return fingerPrint;
    }

    public boolean isDefaultPaymentMode() {
        return defaultPaymentMode;
    }

    void setDefaultPaymentMode(boolean defaultPaymentMode) {
        this.defaultPaymentMode = defaultPaymentMode;
    }

    public boolean isSelected() {
        return selected;
    }

    public boolean isShown() {
        return shown;
    }

    public int getCashAmount() {
        return amount;
    }

    public void setCashAmount(int amount) {
        this.amount = amount;
    }

    public String getCvvText() {
        return cvvText;
    }

    public void setCvvText(String cvvText) {
        this.cvvText = cvvText;
    }

    public abstract String getSavePaymentOptionObject();

    public abstract String getSaveDefaultPaymentOptionObject();

    /**
     * This will return the json object for moto or new make payment call.
     *
     * @return
     */
    public abstract JSONObject getMOTOPaymentOptionObject() throws JSONException;

    /**
     * This will return the json object for wallet charge api.
     *
     * @return
     */
    public abstract JSONObject getWalletChargePaymentOptionObject() throws JSONException;

    public abstract Drawable getOptionIcon(Context context);

    public abstract com.citrus.analytics.PaymentType getAnalyticsPaymentType();

    public abstract String getDynamicPricingPaymentMode();

    @Override
    public String toString() {
        return "PaymentOption{" +
                "name='" + name + '\'' +
                ", token='" + token + '\'' +
                ", defaultPaymentMode=" + defaultPaymentMode +
                ", selected=" + selected +
                '}';
    }

    @Override
    public boolean equals(Object object) {
        boolean result = false;
        if (object == null || object.getClass() != getClass()) {
            result = false;
        } else {
            PaymentOption paymentOption = (PaymentOption) object;
            if (this.name.equalsIgnoreCase(paymentOption.getName())) {
                result = true;
            }
        }
        return result;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.name);
        dest.writeString(this.token);
        dest.writeByte(savePaymentOption ? (byte) 1 : (byte) 0);
        dest.writeParcelable(this.transactionAmount, 0);
        dest.writeInt(this.pgHealth == null ? -1 : this.pgHealth.ordinal());
        dest.writeByte(defaultPaymentMode ? (byte) 1 : (byte) 0);
        dest.writeByte(selected ? (byte) 1 : (byte) 0);
        dest.writeByte(shown ? (byte) 1 : (byte) 0);
        dest.writeString(this.fingerPrint);
        dest.writeInt(this.amount);
        dest.writeString(this.cvvText);
    }

    protected PaymentOption(Parcel in) {
        this.name = in.readString();
        this.token = in.readString();
        this.savePaymentOption = in.readByte() != 0;
        this.transactionAmount = in.readParcelable(Amount.class.getClassLoader());
        int tmpPgHealth = in.readInt();
        this.pgHealth = tmpPgHealth == -1 ? null : PGHealth.values()[tmpPgHealth];
        this.defaultPaymentMode = in.readByte() != 0;
        this.selected = in.readByte() != 0;
        this.shown = in.readByte() != 0;
        this.fingerPrint = in.readString();
        this.amount = in.readInt();
        this.cvvText = in.readString();
    }

    /**
     * Helper method to check whether the payment option contains token or not.
     *
     * @return
     */
    public final boolean isTokenizedPayment() {
        return !TextUtils.isEmpty(token);
    }
}
