/*
   Copyright 2014 Citrus Payment Solutions Pvt. Ltd.
   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at
     http://www.apache.org/licenses/LICENSE-2.0
   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package com.citrus.sdk.payment;

import android.os.Parcel;
import android.text.TextUtils;

import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.classes.Month;
import com.citrus.sdk.classes.Year;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by salil on 13/2/15.
 */
public final class CreditCardOption extends CardOption implements android.os.Parcelable {

    public CreditCardOption() {
    }

    /**
     * @param token
     * @param cardCVV
     * @deprecated use {@link CardOption#setCardCVV(String)} instead
     */
    public CreditCardOption(String token, String cardCVV) {
        super(token, cardCVV);
    }

    /**
     * @param cardHolderName  - Name of the card holder.
     * @param cardNumber      - Card number.
     * @param cardCVV         - CVV of the card. We do not store CVV at our end.
     * @param cardExpiryMonth - Card Expiry Month 01 to 12 e.g. 01 for January.
     * @param cardExpiryYear  - Card Expiry Year in the form of YYYY e.g. 2015.
     */
    public CreditCardOption(String cardHolderName, String cardNumber, String cardCVV, Month cardExpiryMonth, Year cardExpiryYear) {
        super(cardHolderName, cardNumber, cardCVV, cardExpiryMonth, cardExpiryYear);
    }

    /**
     * Use this for wallet PG Payment
     *
     * @param transactionAmount
     * @param cardHolderName
     * @param cardNumber
     * @param cardCVV
     * @param cardExpiryMonth
     * @param cardExpiryYear
     */
    public CreditCardOption(Amount transactionAmount, String cardHolderName, String cardNumber, String cardCVV, Month cardExpiryMonth, Year cardExpiryYear) {
        super(transactionAmount, cardHolderName, cardNumber, cardCVV, cardExpiryMonth, cardExpiryYear);
    }

    /**
     * @param cardNumber
     * @param cardScheme
     * @deprecated use {@link CardOption#setCardScheme(CardScheme)} instead.
     */
    public CreditCardOption(String cardNumber, CardScheme cardScheme) {
        super(cardNumber, cardScheme);
    }

    @Override
    /**
     * Returns the type of the card i.e. CREDIT OR DEBIT
     */
    public String getCardType() {
        // In case of Maestro card the card type is always debit, hence hardcoding the value.
        if (cardScheme == CardScheme.MAESTRO || cardScheme == CardScheme.RPAY) {
            return CardType.DEBIT.getCardType();
        }

        return CardType.CREDIT.getCardType();
    }

    @Override
    public com.citrus.analytics.PaymentType getAnalyticsPaymentType() {
        return com.citrus.analytics.PaymentType.CREDIT_CARD;
    }

    @Override
    public String getDynamicPricingPaymentMode() {
        // In case of tokenized payments. The payment mode is CITRUS_WALLET
        if (!TextUtils.isEmpty(token)) {
            return "CITRUS_WALLET";
        }

        return "CREDIT_CARD";
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.cardHolderName);
        dest.writeString(this.cardNumber);
        dest.writeString(this.cardCVV);
        dest.writeString(this.cardExpiry);
        dest.writeString(this.cardExpiryMonth);
        dest.writeString(this.cardExpiryYear);
        dest.writeSerializable(this.cardScheme);
        dest.writeString(this.name);
        dest.writeString(this.token);
        dest.writeByte(savePaymentOption ? (byte) 1 : (byte) 0);
        dest.writeParcelable(this.transactionAmount, 0);
    }

    private CreditCardOption(Parcel in) {
        this.cardHolderName = in.readString();
        this.cardNumber = in.readString();
        this.cardCVV = in.readString();
        this.cardExpiry = in.readString();
        this.cardExpiryMonth = in.readString();
        this.cardExpiryYear = in.readString();
        this.cardScheme = (CardScheme) in.readSerializable();
        this.name = in.readString();
        this.token = in.readString();
        this.savePaymentOption = in.readByte() != 0;
        this.transactionAmount = in.readParcelable(Amount.class.getClassLoader());
    }

    public static final Creator<CreditCardOption> CREATOR = new Creator<CreditCardOption>() {
        public CreditCardOption createFromParcel(Parcel source) {
            return new CreditCardOption(source);
        }

        public CreditCardOption[] newArray(int size) {
            return new CreditCardOption[size];
        }
    };

    @Override
    public String getSaveDefaultPaymentOptionObject() {

        JSONObject object = null;
        try {
            object = new JSONObject();
            JSONArray paymentOptions = new JSONArray();

            JSONObject option = new JSONObject();
            option.put("owner", cardHolderName);
            option.put("bank", "null");
            option.put("type", "credit");
            option.put("number", cardNumber);
            option.put("scheme", cardScheme);
            option.put("expiryDate", cardExpiry);
            option.put("name", super.getName());
            paymentOptions.put(option);

            object.put("paymentOptions", paymentOptions);
            object.put("type", "payment");
            object.put("defaultOption", super.getName());
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return object.toString();
    }
}
