package com.citrus.sdk.login;

import android.app.Activity;
import android.content.Context;

import com.citrus.sdk.Callback;
import com.citrus.sdk.CitrusClient;
import com.citrus.sdk.classes.LinkBindUserResponse;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusResponse;

/**
 * Created by salil on 9/5/16.
 */
public class CitrusLoginClient {

    public enum AccessType {
        FULL, LIMITED
    }

    public enum PasswordType {
        OTP, PASSWORD, NONE
    }

    private Activity activity = null;
    private final String email;
    private final String mobile;
    private final AccessType accessType;
    private CitrusLoginClientListener listener = null;
    private CitrusClient citrusClient = null;
    private LoginProcessor loginProcessor = null;

    public CitrusLoginClient(Activity activity, String email, String mobile, AccessType accessType) {
        this.activity = activity;
        this.email = email;
        this.mobile = mobile;
        this.accessType = accessType;
        this.citrusClient = CitrusClient.getInstance(this.activity);

        loginProcessor = new CitrusLoginProcessor(this.activity);
    }

    public interface CitrusLoginClientListener {
        void onLoginSuccess();

        void onError(CitrusError error);

        void onLoginCancelled();
    }

    public static abstract class LoginProcessor {
        protected Context context;

        public LoginProcessor(Context context) {
            this.context = context;
        }

        public abstract void onShowLoginScreen(LoginInfo loginInfo, CitrusLoginClient citrusLoginClient);

        public abstract PasswordType getPasswordType();

        public abstract String getPasswordOrOTP();

        public abstract void onLoginError(PasswordType passwordType, CitrusError error);

        public void onHideLoginScreen() {

        }
    }

    public String getEmail() {
        return email;
    }

    public String getMobile() {
        return mobile;
    }

    public AccessType getAccessType() {
        return accessType;
    }

    public void setListener(CitrusLoginClientListener listener) {
        this.listener = listener;
    }

    public void setLoginProcessor(LoginProcessor loginProcessor) {
        if (loginProcessor != null) {
            this.loginProcessor = loginProcessor;
        }
    }

    public void login() {
        citrusClient.linkBindUser(email, mobile, accessType, new Callback<LinkBindUserResponse>() {
            @Override
            public void success(LinkBindUserResponse linkBindUserResponse) {
                // Since the requested access type is limited and we have already got the access token in the response,
                // respond to client.
                // Else proceed with further steps.
                if (accessType == AccessType.LIMITED) {
                    sendResponse();
                } else {
                    // Since the requested access type is full, proceed with the login of the user.
                    AvailableLoginType availableLoginType = AvailableLoginType.getAvailableLoginType(linkBindUserResponse.getLinkUserSignInType());
                    LoginInfo loginInfo = new LoginInfo(linkBindUserResponse.getResponseCode(), linkBindUserResponse.getCitrusUser(), availableLoginType, linkBindUserResponse.getResponseMessage());

                    // Check if the caller wants to override the Citrus Login Screen.
                    // For merchant in order to user their own login screen, they need to set the listener and also use loginProcessor.
                    loginProcessor.onShowLoginScreen(loginInfo, CitrusLoginClient.this);
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(error);
            }
        });
    }

    public void proceed(final LoginInfo loginInfo) {
        /*
         * Take the password and type from the loginProcessor and do the login.
         */

        String passwordOrOTP = loginProcessor.getPasswordOrOTP();
        PasswordType passwordType = loginProcessor.getPasswordType();

        doLogin(loginInfo, passwordOrOTP, passwordType);
    }

    private void doLogin(final LoginInfo loginInfo, String passwordOrOTP, final PasswordType passwordType) {
        citrusClient.linkBindUserSignIn(loginInfo, passwordType, passwordOrOTP, new Callback<CitrusResponse>() {
            @Override
            public void success(CitrusResponse citrusResponse) {

                loginProcessor.onHideLoginScreen();
                sendResponse();
            }

            @Override
            public void error(CitrusError error) {
                loginProcessor.onLoginError(passwordType, error);
            }
        });
    }

    void onLoginCancelled() {
        if (listener != null) {
            listener.onLoginCancelled();
        }
    }

    public static class Builder {
        private Activity activity;
        private String email = "";
        private String mobile = "";
        private AccessType accessType = AccessType.FULL;

        public Builder(Activity activity) {
            this.activity = activity;
        }

        public Builder email(String email) {
            this.email = email;
            return this;
        }

        public Builder mobile(String mobile) {
            this.mobile = mobile;
            return this;
        }

        public Builder accessType(AccessType accessType) {
            if (accessType != null) {
                this.accessType = accessType;
            }

            return this;
        }

        public CitrusLoginClient build() {
            CitrusLoginClient citrusLoginClient = new CitrusLoginClient(activity, email, mobile, accessType);

            return citrusLoginClient;
        }
    }

    protected <T> void sendResponse() {
        if (listener != null) {
            listener.onLoginSuccess();
        }
    }

    protected void sendError(CitrusError error) {
        if (listener != null) {
            listener.onError(error);
        }
    }
}

