/*
 *
 *    Copyright 2014 Citrus Payment Solutions Pvt. Ltd.
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 * /
 */

package com.citrus.sdk;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Build;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.v4.content.LocalBroadcastManager;
import android.text.TextUtils;
import android.widget.Toast;

import com.citrus.cash.PersistentConfig;
import com.citrus.mobile.OAuth2GrantType;
import com.citrus.mobile.OauthToken;
import com.citrus.sdk.classes.AccessToken;
import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.classes.BinServiceResponse;
import com.citrus.sdk.classes.CardBinDetails;
import com.citrus.sdk.classes.CashoutInfo;
import com.citrus.sdk.classes.CitrusConfig;
import com.citrus.sdk.classes.CitrusException;
import com.citrus.sdk.classes.CitrusPrepaidBill;
import com.citrus.sdk.classes.CitrusUMResponse;
import com.citrus.sdk.classes.LinkUserExtendedResponse;
import com.citrus.sdk.classes.LinkUserPasswordType;
import com.citrus.sdk.classes.LinkUserResponse;
import com.citrus.sdk.classes.MemberInfo;
import com.citrus.sdk.classes.PGHealthResponse;
import com.citrus.sdk.classes.StructResponsePOJO;
import com.citrus.sdk.dynamicPricing.DynamicPricingRequestType;
import com.citrus.sdk.dynamicPricing.DynamicPricingResponse;
import com.citrus.sdk.otp.NetBankForOTP;
import com.citrus.sdk.payment.CardOption;
import com.citrus.sdk.payment.MerchantPaymentOption;
import com.citrus.sdk.payment.PaymentBill;
import com.citrus.sdk.payment.PaymentOption;
import com.citrus.sdk.payment.PaymentType;
import com.citrus.sdk.response.BindUserResponse;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusLogger;
import com.citrus.sdk.response.CitrusResponse;
import com.citrus.sdk.response.PaymentResponse;
import com.facebook.android.crypto.keychain.SharedPrefsBackedKeyChain;
import com.facebook.crypto.Crypto;
import com.facebook.crypto.Entity;
import com.facebook.crypto.exception.CryptoInitializationException;
import com.facebook.crypto.exception.KeyChainException;
import com.facebook.crypto.util.SystemNativeCryptoLibrary;
import com.orhanobut.logger.Logger;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import static com.citrus.sdk.classes.Utils.isValidEmail;
import static com.citrus.sdk.classes.Utils.isValidMobile;
import static com.citrus.sdk.response.CitrusResponse.Status;

/**
 * Created by salil on 11/5/15.
 */
public class CitrusClient implements ConfigChangeListener {


    private String signinId;
    private String signinSecret;
    private String signupId;
    private String signupSecret;
    private String vanity;

    private String merchantName;
    private Environment environment = Environment.SANDBOX;
    private Amount balanceAmount;
    private static CitrusClient instance;
    private final Context mContext;
    private MerchantPaymentOption merchantPaymentOption = null;

    private OauthToken oauthToken = null;
    private BroadcastReceiver paymentEventReceiver = null;
    private boolean initialized = false;
    private boolean showDummyScreen = false;
    private boolean autoOtpReading = false;

    private RetrofitAPIWrapper retrofitAPIWrapper = null;

    private PaymentOption persistPaymentOption = null;

    public Crypto getCrypto() {
        return crypto;
    }

    private Crypto crypto = null;

    private CitrusClient(Context context) {
        mContext = context;
        retrofitAPIWrapper = RetrofitAPIWrapper.getInstance(context);
        CitrusConfig.getInstance().setConfigChangeListener(this);
    }

    public void enableLog(boolean enable) {
        if (enable) {
            CitrusLogger.enableLogs();
        } else {
            CitrusLogger.disableLogs();
        }
    }

    public void showDummyScreenWhilePayments(boolean showDummyScreen) {
        this.showDummyScreen = showDummyScreen;
    }

    public boolean isShowDummyScreenWhilePayments() {
        return showDummyScreen;
    }

    public boolean isAutoOtpReading() {
        return autoOtpReading;
    }

    public void enableAutoOtpReading(boolean enable) {
        this.autoOtpReading = enable;
    }

    public native String getEncryptionKey();

    public static boolean isCitrusNativeLibraryLoaded = false;

    public static boolean isFacebookNativeLibraryLoaded = false;


    public void destroy() {
        initialized = false;
        signinId = null;
        signinSecret = null;
        signupId = null;
        signupSecret = null;
        vanity = null;
        environment = null;
        oauthToken = null;
    }

    public void init(@NonNull String signupId, @NonNull String signupSecret, @NonNull String signinId, @NonNull String signinSecret, @NonNull String vanity, @NonNull Environment environment) {
        if (!initialized) {

            oauthToken = new OauthToken(mContext);
            this.signupId = signupId;
            this.signupSecret = signupSecret;
            this.signinId = signinId;
            this.signinSecret = signinSecret;
            this.vanity = vanity;

            if (validate()) {
                // Initialize the wrapper for retrofit clients.
                retrofitAPIWrapper.init(signupId, signupSecret, signinId, signinSecret, vanity, environment);
            }

            if (!CitrusLogger.isEnableLogs()) {
                CitrusLogger.disableLogs();
            }

            if (environment == null) {
                this.environment = Environment.SANDBOX;
            }
            this.environment = environment;
            saveSDKEnvironment();

//            fetchPGHealthForAllBanks();

            getMerchantPaymentOptions(null);

            // Fetch profile info if the user is signed in.
            // If not signed in the information will be fetched once the user signs in.
            isUserSignedIn(new Callback<Boolean>() {
                @Override
                public void success(Boolean signedIn) {
                    if (signedIn) {
                        getProfileInfo(null);

                        // Check whether the prepaid token is valid or not.
                        checkPrepaymentTokenValidity(null);
                    }
                }

                @Override
                public void error(CitrusError error) {
                    // Not required to handle the error.
                }
            });

            initialized = true;

        }
        if (environment != Environment.PRODUCTION) {
            Toast.makeText(mContext, "Environment is *** " + environment.toString(), Toast.LENGTH_LONG).show();
        }
    }

    private void fetchPGHealthForAllBanks() {
        // Fetch the pg health for all the banks which will be used later.
        retrofitAPIWrapper.fetchPGHealthForAllBanks();
    }

    private void saveSDKEnvironment() {
        if (oauthToken.getCurrentEnvironment() == Environment.NONE) { //no environment saved till now
            oauthToken.saveEnvironment(environment);
            Logger.d("NO ENVIRONMENT EXISTS earlier");
        } else if (oauthToken.getCurrentEnvironment() == environment) {
            //dont save new enviroment
            Logger.d("PREVIOUS AND CURRENT ENVIRONMENT IS SAME");
        } else { //environment changed-  logout user, save new environment
            signOut(new Callback<CitrusResponse>() {
                @Override
                public void success(CitrusResponse citrusResponse) {
                    oauthToken.saveEnvironment(environment);
                    Logger.d("ENVIRONMMENT MISMATCH ***" + "user Logging out");
                }

                @Override
                public void error(CitrusError error) {
                    oauthToken.saveEnvironment(environment);
                }
            });
        }
    }

    public static CitrusClient getInstance(Context context) {
        if (instance == null) {
            synchronized (CitrusClient.class) {
                if (instance == null) {
                    instance = new CitrusClient(context);
                }
            }
        }

        return instance;
    }

    private void validateLinkUserCredentials(String email, String mobileNo, final Callback callback) {
        if (email == null) {
            sendError(callback, new CitrusError("Invalid Email ID is null.", Status.FAILED));
        } else if (mobileNo == null) {
            sendError(callback, new CitrusError("Invalid Mobile number null.", Status.FAILED));
        } else if (!isValidEmail(email) && !isValidMobile(mobileNo)) {
            sendError(callback, new CitrusError("Invalid Credentials entered.", Status.FAILED));
        } else if (!isValidEmail(email)) {
            sendError(callback, new CitrusError("Invalid Email Id entered.", Status.FAILED));
        } else if (!isValidMobile(mobileNo)) {
            sendError(callback, new CitrusError("Invalid Mobile Number entered.", Status.FAILED));
        } else {
            sendResponse(callback, true);
        }
    }

    // Public APIS start

    /**
     * Link User Extended Api call
     *
     * @param emailId
     * @param mobileNo
     * @param callback
     */
    public synchronized void linkUserExtended(final String emailId, final String mobileNo, final Callback<LinkUserExtendedResponse> callback) {

        validateLinkUserCredentials(emailId, mobileNo, new Callback() {
            @Override
            public void success(Object o) {
                retrofitAPIWrapper.linkUserExtended(emailId, mobileNo, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public void linkUserExtendedSignIn(LinkUserExtendedResponse linkUserExtended, LinkUserPasswordType linkUserPasswordType, String linkUserPassword, final Callback<CitrusResponse> callback) {

        String inputEmail = linkUserExtended.getInputEmail();
        if (linkUserPassword == null || linkUserPassword.equalsIgnoreCase("")) {
            sendError(callback, new CitrusError(linkUserExtended.getLinkUserMessage(), Status.FAILED));
        } else if (linkUserPasswordType == LinkUserPasswordType.None) {
            sendError(callback, new CitrusError(linkUserExtended.getLinkUserMessage(), Status.FAILED));
        } else {
            int linkUserCase = linkUserExtended.formatResponseCode();
            switch (linkUserPasswordType) {
                case Otp:
                    if (linkUserCase == 1 || linkUserCase == 6 || linkUserCase == 2 || linkUserCase == 7) {
                        Logger.d("old sigin, otp");
                        final String linkUserMobile = linkUserExtended.getLinkUserMobile();
                        retrofitAPIWrapper.signInWithOTP(linkUserMobile, linkUserPassword, callback);
//                        signInWithOTP(linkUserExtended, linkUserPassword, callback);
                    } else if (linkUserCase == 3 || linkUserCase == 4 || linkUserCase == 5 || linkUserCase == 10 || linkUserCase == 11 || linkUserCase == 12) {
                        Logger.d("verify and sigin, otp");
                        synchronized (this) {
                            retrofitAPIWrapper.linkUserExtendedVerifyMobileAndSignIn(linkUserExtended, linkUserPassword, callback);
                        }
//                        linkUserExtendedVerifyMobileAndSignIn(linkUserExtended, linkUserPassword, callback);

                    } else if (linkUserCase == 8 || linkUserCase == 9) {
                        Logger.d("Update mobile sigin, e-otp");
                        String signInGrantType = OAuth2GrantType.onetimepass.toString();
                        synchronized (this) {
                            retrofitAPIWrapper.linkUserExtendedVerifyEOTPAndUpdateMobile(inputEmail, signInGrantType, linkUserPassword, linkUserExtended, callback);
                        }
//                        linkUserExtendedVerifyEOTPAndUpdateMobile(inputEmail, signInGrantType, linkUserPassword, linkUserExtended, callback);
                    }
                    break;

                case Password:
                    if (linkUserCase == 1 || linkUserCase == 6 || linkUserCase == 2 || linkUserCase == 5) {
                        Logger.d("old sigin, password");
                        signIn(inputEmail, linkUserPassword, callback);
                    } else if (linkUserCase == 8) {
                        Logger.d("Update mobile sigin, password");
                        String signInGrantType = OAuth2GrantType.password.toString();
                        synchronized (this) {
                            retrofitAPIWrapper.linkUserExtendedVerifyEOTPAndUpdateMobile(inputEmail, signInGrantType, linkUserPassword, linkUserExtended, callback);
                        }
//                        linkUserExtendedVerifyEOTPAndUpdateMobile(inputEmail, signInGrantType, linkUserPassword, linkUserExtended, callback);
                    }
                    break;
            }
        }
    }

    /**
     * LinkUserExtended VerifyEOTPAndUpdateMobile API call
     *
     * @param emailId
     * @param linkUserPassword
     * @param linkUserExtended
     * @param callback
     */
    public void linkUserExtendedVerifyEOTPAndUpdateMobile(final String emailId, final String signInGrantType, final String linkUserPassword, final LinkUserExtendedResponse linkUserExtended, final Callback<CitrusResponse> callback) {
        retrofitAPIWrapper.linkUserExtendedVerifyEOTPAndUpdateMobile(emailId, signInGrantType, linkUserPassword, linkUserExtended, callback);
    }

    /**
     * * LinkUserExtended VerifyMobileAndSignIn API call
     *
     * @param linkUserExtendedResponse
     * @param verificationCode
     * @param callback
     */
    public synchronized void linkUserExtendedVerifyMobileAndSignIn(final LinkUserExtendedResponse linkUserExtendedResponse, final String verificationCode, final Callback<CitrusResponse> callback) {
        retrofitAPIWrapper.linkUserExtendedVerifyMobileAndSignIn(linkUserExtendedResponse, verificationCode, callback);
    }


    /**
     * @param emailId  - emailId of the user
     * @param mobileNo - mobileNo of the user
     * @param callback - callback
     *                 <p/>
     * @deprecated Use {@link CitrusClient#linkUserExtended(String, String, Callback)} instead.
     * <p/>
     * This api will check whether the user is existing user or not. If the user is existing user,
     * then it will return the existing details, else it will create an account internally and
     * then call signUp to set the password and activate the account.
     */
    public synchronized void isCitrusMember(final String emailId, final String mobileNo, final Callback<Boolean> callback) {

        retrofitAPIWrapper.isCitrusMember(emailId, mobileNo, callback);
    }

    public synchronized void getMemberInfo(final String emailId, final String mobileNo, final Callback<MemberInfo> callback) {

        retrofitAPIWrapper.getMemberInfo(emailId, mobileNo, callback);
    }

    /**
     * This method can be used by non prepaid merchants - where email and mobile is enough for save/get Cards.
     * This method will create Citrus Account of the user with email. All the cards will be saved to emailID.
     *
     * @param emailId  - emailId of the user
     * @param mobileNo - mobileNo of the user
     * @param callback - callback
     */
    public synchronized void bindUser(final String emailId, final String mobileNo, final Callback<String> callback) {

        retrofitAPIWrapper.createUser(emailId, mobileNo, callback);

    }

    public synchronized void bindUserByMobile(final String emailId, final String mobileNo, final Callback<BindUserResponse> callback) {

        retrofitAPIWrapper.bindUserByMobile(emailId, mobileNo, callback);
    }

    /**
     * This method will enable the user to Sign in with OTP.
     * <li>
     * <ol>If user's mobileNo no is verified then OTP will be sent on the mobileNo and user will be able to sign in using mOTP. </ol>
     * <ol>If the mobileNo no is not found/unverified and email is found then OTP is sent on the emailId and user will be able to sign in using eOTP. </ol>
     * <ol>If the emailId and mobileNo no are not found then the user's account is created.</ol>
     * </li>
     *
     * @param emailId
     * @param mobileNo
     * @param callback
     * @deprecated Use {@link CitrusClient#linkUserExtended(String, String, Callback)} instead.
     */
    public synchronized void linkUserWithOTP(final String emailId, final String mobileNo, final boolean forceMobileVerification, final Callback<LinkUserResponse> callback) {

        retrofitAPIWrapper.linkUserWithOTP(emailId, mobileNo, forceMobileVerification, callback);
    }

    /**
     * @param emailId
     * @param password
     * @param callback
     * @deprecated Use {@link CitrusClient#linkUserExtended(String, String, Callback) and respective signin } instead.
     */
    public synchronized void signIn(final String emailId, final String password, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.signIn(emailId, password, callback);
    }

    /**
     * Signin with mobile no and password.
     * <p/>
     *
     * @param mobileNo
     * @param password
     * @param callback
     * @deprecated Use {@link CitrusClient#linkUserExtended(String, String, Callback) and respective signin } instead.
     */
    public synchronized void signInWithMobileNo(final String mobileNo, final String password, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.signInWithMobileNo(mobileNo, password, callback);
    }

    /**
     * Signin with mobile no and password.
     *
     * @param linkUserExtended
     * @param otp
     * @param callback
     * @deprecated Use {@link CitrusClient#linkUserExtended(String, String, Callback) and respective signin } instead.
     */
    public synchronized void signInWithOTP(final LinkUserExtendedResponse linkUserExtended, final String otp, final Callback<CitrusResponse> callback) {
        final String linkUserMobile = linkUserExtended.getLinkUserMobile();
        retrofitAPIWrapper.signInWithOTP(linkUserMobile, otp, callback);
    }

    public synchronized void getCookie(String email, String password, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.getCookie(email, password, callback);
    }

    /**
     * Signout the existing logged in user.
     */
    public synchronized void signOut(Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.signOut(callback);
    }

    /**
     * Set the user password.
     *
     * @param emailId
     * @param mobileNo
     * @param password
     * @param callback
     * @deprecated Use {@link CitrusClient#linkUserExtended(String, String, Callback) } instead.
     */
    public synchronized void signUp(final String emailId, String mobileNo, final String password, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.signUp(emailId, mobileNo, password, callback);
    }

    /**
     * Reset the user password. The password reset link will be sent to the user.
     *
     * @param emailId
     * @param callback
     */
    public synchronized void resetPassword(final String emailId, @NonNull final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.resetPassword(emailId, callback);
    }

    /**
     * Update or Change or Verify mobile no.
     *
     * @param mobileNo
     * @param callback
     */
    public synchronized void updateMobile(final String mobileNo, final Callback<String> callback) {

        retrofitAPIWrapper.updateMobile(mobileNo, callback);
    }

    /**
     * Verify mobile no.
     *
     * @param verificationCode
     * @param callback
     */
    public synchronized void verifyMobile(final String verificationCode, final Callback<String> callback) {

        retrofitAPIWrapper.verifyMobile(verificationCode, callback);
    }

    /**
     * Get the user saved payment options.
     *
     * @param callback - callback
     */
    public synchronized void getWallet(final Callback<List<PaymentOption>> callback) {

        retrofitAPIWrapper.getWallet(callback);
    }


    /**
     * Get the user saved payment options. If you want any specific bank at 0th position pass BankCID
     *
     * @param callback - callback
     */
    public synchronized void getWalletWithDefaultBank(final Callback<List<PaymentOption>> callback, final BankCID bankCID) {

        retrofitAPIWrapper.getWalletWithDefaultBank(callback, bankCID);
    }

    /**
     * Activate the prepaid user.
     *
     * @param callback
     */
    public synchronized void activatePrepaidUser(final Callback<Amount> callback) {

        retrofitAPIWrapper.activatePrepaidUser(callback);
    }

    /**
     * This will retrun profile of the signed in user such as first name, last name, email and mobile.
     *
     * @param callback
     */
    public synchronized void getProfileInfo(final Callback<CitrusUser> callback) {

        retrofitAPIWrapper.getProfileInfo(callback);
    }

    /**
     * Get the balance of the user.
     *
     * @param callback
     */
    public synchronized void getBalance(final Callback<Amount> callback) {

        retrofitAPIWrapper.getBalance(callback);
    }

    /**
     * Save the paymentOption.
     *
     * @param paymentOption - PaymentOption to be saved.
     * @param callback
     */

    public synchronized void savePaymentOption(final PaymentOption paymentOption, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.savePaymentOption(paymentOption, callback);
    }

    /**
     * Save the card - this is required for auto CVV feature
     *
     * @param paymentOption - PaymentOption to be saved.
     * @param callback
     */

    public synchronized void saveCard(final PaymentOption paymentOption, final Callback<AddCardResponse> callback) {

        // we will not save the card if conceal or citruslibrary is not loaded - this will be called only for auto CVV feature
        if (CitrusConfig.getInstance().isOneTapPaymentEnabled() && isOneTapPaymentSupported() ) {
            retrofitAPIWrapper.saveCard(paymentOption, callback);
        }
    }


    /**
     * Deletes the saved Payment Option
     *
     * @param paymentOption
     * @param callback
     */
    public synchronized void deletePaymentOption(final PaymentOption paymentOption, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.deletePaymentOption(paymentOption, callback);
    }

    /**
     * Get the payment bill for the transaction.
     *
     * @param amount   - Transaction amount
     * @param callback
     */
    public synchronized void getBill(String billUrl, Amount amount, final Callback<PaymentBill> callback) {

        retrofitAPIWrapper.getBill(billUrl, amount, callback);
    }

    // Dynamic Pricing.

    /**
     * Perform Dynamic Pricing. You can specify one the dynamicPricingRequestType to perform Dynamic Pricing.
     *
     * @param dynamicPricingRequestType - One of the dynamicPricingRequestType from {@link DynamicPricingRequestType}
     * @param billUrl                   - billUrl from where we will fetch the bill.
     * @param callback                  - callback
     */
    public synchronized void performDynamicPricing(@NonNull final DynamicPricingRequestType dynamicPricingRequestType, @NonNull final String billUrl, @NonNull final Callback<DynamicPricingResponse> callback) {

        retrofitAPIWrapper.performDynamicPricing(dynamicPricingRequestType, billUrl, callback);
    }

    /**
     * Perform Dynamic Pricing. You can specify one the dynamicPricingRequestType to perform Dynamic Pricing.
     *
     * @param dynamicPricingRequestType - One of the dynamicPricingRequestType from {@link DynamicPricingRequestType}
     * @param paymentBill               - PaymentBill in case you are fetching bill response from your server.
     * @param callback                  - callback
     */
    public synchronized void performDynamicPricing(@NonNull final DynamicPricingRequestType dynamicPricingRequestType, @NonNull final PaymentBill paymentBill, @NonNull final Callback<DynamicPricingResponse> callback) {

        retrofitAPIWrapper.performDynamicPricing(dynamicPricingRequestType, paymentBill, callback);
    }

    /**
     * Send money to your friend.
     *
     * @param amount   - Amount to be sent
     * @param toUser   - The user detalis. Enter emailId if send by email or mobileNo if send by mobile.
     * @param message  - Optional message
     * @param callback - Callback
     * @deprecated Use {@link CitrusClient#sendMoneyToMoblieNo(Amount, String, String, Callback)} instead.
     */
    public synchronized void sendMoney(final Amount amount, final CitrusUser toUser, final String message, final Callback<PaymentResponse> callback) {

        retrofitAPIWrapper.sendMoney(amount, toUser, message, callback);
    }

    /**
     * @param amount
     * @param mobileNo
     * @param message
     * @param callback
     */
    public synchronized void sendMoneyToMoblieNo(final Amount amount, final String mobileNo, final String message, final Callback<PaymentResponse> callback) {

        retrofitAPIWrapper.sendMoneyToMoblieNo(amount, mobileNo, message, callback);
    }

    /**
     * Returns the access token of the currently logged in user.
     */
    public void getPrepaidToken(final Callback<AccessToken> callback) {

        retrofitAPIWrapper.getPrepaidToken(callback);
    }


    /**
     * Get the merchant available payment options. You need to show the user available payment option in your app.
     *
     * @param callback
     */
    public synchronized void getMerchantPaymentOptions(final Callback<MerchantPaymentOption> callback) {

        retrofitAPIWrapper.getMerchantPaymentOptions(callback);
    }


    /**
     * Get the payment options available for load money. You need to show the user available payment option in your app.
     *
     * @param callback
     */
    public synchronized void getLoadMoneyPaymentOptions(final Callback<MerchantPaymentOption> callback) {

        retrofitAPIWrapper.getLoadMoneyPaymentOptions(callback);
    }

    public synchronized void isUserSignedIn(final Callback<Boolean>
                                                    callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                sendResponse(callback, true);
            }

            @Override
            public void error(CitrusError error) {
                sendResponse(callback, false);
            }
        });
    }

    public synchronized void loadMoney(final PaymentType.LoadMoney loadMoney, final Callback<TransactionResponse> callback) {

        // Validate the card details before forwarding transaction.
        if (loadMoney != null) {
            PaymentOption paymentOption = loadMoney.getPaymentOption();
            persistPaymentOption = paymentOption;
            // If the CardOption is invalid, check what is incorrect and respond with proper message.
            if (paymentOption instanceof CardOption && !((CardOption) paymentOption).validateCard()) {

                sendError(callback, new CitrusError(((CardOption) paymentOption).getCardValidityFailureReasons(), Status.FAILED));
                return;
            }
        }

        registerReceiver(callback, new IntentFilter(loadMoney.getIntentAction()));

        startCitrusActivity(loadMoney, false);
    }


    public synchronized void loadMoneyWithOneTap(final PaymentType.LoadMoney loadMoney, final Callback<TransactionResponse> callback) {

        // Validate the card details before forwarding transaction.
        if (loadMoney != null) {
            if (loadMoney.getPaymentOption() instanceof CardOption) {
                if (loadMoney.getPaymentOption().getFingerPrint() == null) {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_WITHOUT_SAVED_CARD_CVV_PAYMENT, Status.FAILED)); //token does not exist
                    return;
                }
                if (getCVVOfFingerPrint((CardOption) loadMoney.getPaymentOption()) == null) {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_NO_CVV_FOR_SAVED_CARD, Status.FAILED));
                    return;
                }
                ((CardOption) loadMoney.getPaymentOption()).setCardCVV(getCVVOfFingerPrint((CardOption) loadMoney.getPaymentOption())); // read the saved CVV and update Payment Option

                persistPaymentOption = loadMoney.getPaymentOption();
                loadMoney(loadMoney, callback);
            } else { //One tap should work only for Card Payment
                sendError(callback, new CitrusError(ResponseMessages.ERROR_ONE_TAP_ONLY_CARD, Status.FAILED));
            }

        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_NULL_PAYMENT_OPTION, Status.FAILED));
        }
    }

    public synchronized void pgPayment(final PaymentType.PGPayment pgPayment, final Callback<TransactionResponse> callback) {

        // Validate the card details before forwarding transaction.
        if (pgPayment != null) {
            PaymentOption paymentOption = pgPayment.getPaymentOption();
            persistPaymentOption = paymentOption;
            // If the CardOption is invalid, check what is incorrect and respond with proper message.
            if (paymentOption instanceof CardOption && !((CardOption) paymentOption).validateCard()) {

                sendError(callback, new CitrusError(((CardOption) paymentOption).getCardValidityFailureReasons(), Status.FAILED));
                return;
            }
        }

        registerReceiver(callback, new IntentFilter(pgPayment.getIntentAction()));

        startCitrusActivity(pgPayment, false);
    }

    /**
     * This method will be called only for SAVED CVV
     *
     * @param pgPayment
     * @param callback
     */
    public synchronized void pgPaymentWithOneTap(final PaymentType.PGPayment pgPayment, final Callback<TransactionResponse> callback) {

        // Validate the card details before forwarding transaction.
        if (pgPayment != null) {

            if (pgPayment.getPaymentOption() instanceof CardOption) {
                if (pgPayment.getPaymentOption().getFingerPrint() == null) {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_WITHOUT_SAVED_CARD_CVV_PAYMENT, Status.FAILED)); //token does not exist
                    return;
                }
                if (getCVVOfFingerPrint((CardOption) pgPayment.getPaymentOption()) == null) {
                    //ERROR_NO_CVV_FOR_SAVED_CARD
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_NO_CVV_FOR_SAVED_CARD, Status.FAILED));
                    return;
                }
                ((CardOption) pgPayment.getPaymentOption()).setCardCVV(getCVVOfFingerPrint((CardOption) pgPayment.getPaymentOption())); // read the saved CVV and update Payment Option

                persistPaymentOption = pgPayment.getPaymentOption();
                pgPayment(pgPayment, callback);
            } else { //One tap should work only for Card Payment
                sendError(callback, new CitrusError(ResponseMessages.ERROR_ONE_TAP_ONLY_CARD, Status.FAILED));

            }

        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_NULL_PAYMENT_OPTION, Status.FAILED));
        }

    }

    public synchronized void pgPayment(final DynamicPricingResponse dynamicPricingResponse, final Callback<TransactionResponse> callback) {

        if (dynamicPricingResponse != null) {
            PaymentBill paymentBill = dynamicPricingResponse.getPaymentBill();

            PaymentType.PGPayment pgPayment;
            try {
                pgPayment = new PaymentType.PGPayment(paymentBill, dynamicPricingResponse.getPaymentOption(), dynamicPricingResponse.getCitrusUser());

                registerReceiver(callback, new IntentFilter(pgPayment.getIntentAction()));

                startCitrusActivity(pgPayment, dynamicPricingResponse, false);
            } catch (CitrusException e) {
                e.printStackTrace();
                sendError(callback, new CitrusError(e.getMessage(), Status.FAILED));
            }
        } else {
            sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_NULL_DYNAMIC_RESPONSE, Status.FAILED));
        }
    }

    public synchronized void makePayment(final PaymentType.PGPayment pgPayment, final Callback<TransactionResponse> callback) {

        // Validate the card details before forwarding transaction.
        if (pgPayment != null) {
            PaymentOption paymentOption = pgPayment.getPaymentOption();
            persistPaymentOption = paymentOption;
            // If the CardOption is invalid, check what is incorrect and respond with proper message.
            if (paymentOption instanceof CardOption && !((CardOption) paymentOption).validateCard()) {

                sendError(callback, new CitrusError(((CardOption) paymentOption).getCardValidityFailureReasons(), Status.FAILED));
                return;
            }
        }

        registerReceiver(callback, new IntentFilter(pgPayment.getIntentAction()));

        startCitrusActivity(pgPayment, true);
    }

    public synchronized void makePaymentWithOneTap(final PaymentType.PGPayment pgPayment, final Callback<TransactionResponse> callback) {

        // Validate the card details before forwarding transaction.
        if (pgPayment != null) {
            PaymentOption paymentOption = pgPayment.getPaymentOption();

            if (paymentOption instanceof CardOption) {
                CardOption savedCardOption = ((CardOption) paymentOption);
                if (savedCardOption.getFingerPrint() == null) {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_WITHOUT_SAVED_CARD_CVV_PAYMENT, Status.FAILED)); //token does not exist
                    return;
                }
                if (getCVVOfFingerPrint(savedCardOption) == null) {
                    //ERROR_NO_CVV_FOR_SAVED_CARD
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_NO_CVV_FOR_SAVED_CARD, Status.FAILED));
                    return;
                }
                ((CardOption) paymentOption).setCardCVV(getCVVOfFingerPrint(savedCardOption)); // read the saved CVV and update Payment Option

                persistPaymentOption = paymentOption;
            }

            // If the CardOption is invalid, check what is incorrect and respond with proper message.
            if (paymentOption instanceof CardOption && !((CardOption) paymentOption).validateCard()) {

                sendError(callback, new CitrusError(((CardOption) paymentOption).getCardValidityFailureReasons(), Status.FAILED));
                return;
            }
        }

        registerReceiver(callback, new IntentFilter(pgPayment.getIntentAction()));

        startCitrusActivity(pgPayment, true);
    }

    /**
     * @param citrusCash
     * @param callback
     */
    public synchronized void payUsingCitrusCash(final PaymentType.CitrusCash citrusCash, final Callback<TransactionResponse> callback) {

        String cookie = "";
        String cookieExpiryDate = "";
        PersistentConfig persistentConfig = new PersistentConfig(mContext);
        String sessionCookie = persistentConfig.getCookieString();
        String[] cookieParts = sessionCookie.split(";");
        if (cookieParts != null && cookieParts.length > 1) {
            cookie = cookieParts[0];
        }

        // If cookie length > 22 then there is cookie string, and proceed for the expiry validity check.
        if (cookie.length() > 22) {
            // Extract the cookie expiry date
            int start = sessionCookie.indexOf("Expires=");
            int end = sessionCookie.indexOf("GMT;");
            if (start != -1 && end != -1 && sessionCookie.length() > start + 13 && sessionCookie.length() > end) {
                cookieExpiryDate = sessionCookie.substring(start + 13, end);
            }

            SimpleDateFormat dateFormat = new SimpleDateFormat("dd-MMM-yyyy hh:mm:ss");
            Date expiryDate = new Date();
            Date currentDate = new Date(System.currentTimeMillis());
            try {
                expiryDate = dateFormat.parse(cookieExpiryDate);

                Logger.d("Expiry date : %s, Current Date : %s", expiryDate, currentDate);

                if (currentDate.before(expiryDate)) {

                    // Check whether the balance in the wallet is greater than the transaction amount.
                    payCash(citrusCash, callback);
                } else {
                    Logger.d("User's cookie has expired. Please signin");
                    sendError(callback, new CitrusError("User's cookie has expired. Please signin.", Status.FAILED));
                }
            } catch (ParseException e) {
                e.printStackTrace();

                // In the worst case, it will try to redirect user to the Citrus Page.
                // Check whether the balance in the wallet is greater than the transaction amount.
                payCash(citrusCash, callback);
            }
        } else {
            // Since the cookie string is blank. Directly proceed for the payment and SDK will ask for the password in that case to complete the transaction.
            payCash(citrusCash, callback);
        }
    }

    private void payCash(final PaymentType.CitrusCash citrusCash, final Callback<TransactionResponse> callback) {

        isEnoughPrepaidBalance(citrusCash.getAmount(), new Callback<Boolean>() {
            @Override
            public void success(Boolean sufficientBalance) {
                if (sufficientBalance) {
                    registerReceiver(callback, new IntentFilter(citrusCash.getIntentAction()));

                    startCitrusActivity(citrusCash, false);
                } else {
                    sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_INSUFFICIENT_BALANCE, Status.FAILED));
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    private void isEnoughPrepaidBalance(final Amount amount, final Callback<Boolean> callback) {

        getBalance(new Callback<Amount>() {
            @Override
            public void success(Amount userBalance) {
                if (userBalance.getValueAsDouble() >= amount.getValueAsDouble()) {
                    sendResponse(callback, true);
                } else {
                    sendResponse(callback, false);
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });

    }

    //this is new implementation of Pay Using Cash API - without hitting PG
    public synchronized void prepaidPay(final PaymentType.CitrusCash citrusCash, final Callback<PaymentResponse> callback) {

        retrofitAPIWrapper.newPrepaidPay(citrusCash, callback);
    }

    private void checkPrepaymentTokenValidity(final Callback<Boolean> callback) {

        retrofitAPIWrapper.checkPrepaymentTokenValidity(callback);
    }

    // Cashout Related APIs
    public synchronized void cashout(@NonNull final CashoutInfo cashoutInfo, final Callback<PaymentResponse> callback) {

        retrofitAPIWrapper.cashout(cashoutInfo, callback);
    }

    public synchronized void getCashoutInfo(final Callback<CashoutInfo> callback) {

        retrofitAPIWrapper.getCashoutInfo(callback);
    }

    public synchronized void saveCashoutInfo(final CashoutInfo cashoutInfo, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.saveCashoutInfo(cashoutInfo, callback);
    }

    // PG Health.

    /**
     * It returns {@link com.citrus.sdk.classes.PGHealth} which denotes the health of the PG for in
     * If the health is bad merchants can warn user to use another payment method.
     *
     * @param paymentOption
     * @param callback
     */
    public synchronized void getPGHealth(PaymentOption paymentOption, final Callback<PGHealthResponse> callback) {

        retrofitAPIWrapper.getPGHealth(paymentOption, callback);
    }

    /**
     * Internal.
     * Returns the bank details using the card number.
     *
     * @param cardOption
     */
    public void getBINDetails(CardOption cardOption, final Callback<BinServiceResponse> callback) {

        retrofitAPIWrapper.getBINDetails(cardOption, callback);
    }

    /**
     * Internal.
     * Returns the netbank for OTP which will be used to detect the bank for which auto OTP is being processed.
     */
    public NetBankForOTP getNetBankForOTP() {
        return retrofitAPIWrapper.getNetBankForOTP();
    }

    /**
     * Internal.
     * Set the netbank
     */
    public void resetNetBankForOTP() {
        retrofitAPIWrapper.resetNetBankForOTP();
    }

    public synchronized String getUserEmailId() {
        CitrusUser citrusUser = getCitrusUser();
        if (citrusUser != null) {
            return citrusUser.getEmailId();
        } else {
            return null;
        }
    }

    public synchronized String getUserMobileNumber() {
        CitrusUser citrusUser = getCitrusUser();
        if (citrusUser != null) {
            return citrusUser.getMobileNo();
        } else {
            return null;
        }
    }

    public synchronized CitrusUser getCitrusUser() {
        return retrofitAPIWrapper.getCitrusUser();
    }

    // Public APIS end

    private void unregisterReceiver(BroadcastReceiver receiver) {
        LocalBroadcastManager.getInstance(mContext).unregisterReceiver(receiver);
    }

    private void startCitrusActivity(PaymentType paymentType, DynamicPricingResponse dynamicPricingResponse, boolean useNewAPI) {
        Intent intent = new Intent(mContext, CitrusActivity.class);
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        intent.putExtra(Constants.INTENT_EXTRA_PAYMENT_TYPE, paymentType);
        intent.putExtra(Constants.INTENT_EXTRA_DYNAMIC_PRICING_RESPONSE, dynamicPricingResponse);
        intent.putExtra(Constants.INTENT_EXTRA_USE_NEW_API, useNewAPI);

        mContext.startActivity(intent);
    }

    private void startCitrusActivity(PaymentType paymentType, boolean useNewAPI) {
        startCitrusActivity(paymentType, null, useNewAPI);
    }

    private <T> void registerReceiver(final Callback<T> callback, IntentFilter intentFilter) {
        paymentEventReceiver = new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                unregisterReceiver(this);
                // Reset netbank for otp
                resetNetBankForOTP();

                TransactionResponse transactionResponse = intent.getParcelableExtra(Constants.INTENT_EXTRA_TRANSACTION_RESPONSE);
                if (transactionResponse != null) {
                    TransactionResponse.TransactionStatus transactionStatus = transactionResponse.getTransactionStatus();
                    Status status = null;

                    if (transactionStatus != null) {
                        switch (transactionStatus) {

                            case SUCCESSFUL:
                                status = Status.SUCCESSFUL;
                                break;
                            case FAILED:
                                status = Status.FAILED;
                                break;
                            case CANCELLED:
                                status = Status.CANCELLED;
                                break;
                            case PG_REJECTED:
                                status = Status.PG_REJECTED;
                                break;
                        }
                    }
                    if (transactionStatus == TransactionResponse.TransactionStatus.SUCCESSFUL) {
                        if (persistPaymentOption != null) {
                            if (persistPaymentOption instanceof CardOption) {
                                CardOption cardOption = (CardOption) persistPaymentOption;
                                if (cardOption.isTokenizedPayment()) {
                                    if (cardOption.getCardCVV() != null && cardOption.getFingerPrint() != null) {
                                        // we are saving CVV for saved Card
                                        saveSecureCVV(persistPaymentOption.getFingerPrint(), ((CardOption) persistPaymentOption).getCardCVV());//save token and CVV after successful transaction.
                                        persistPaymentOption = null;
                                    }
                                } else { // we are saving CVV for new card
                                    saveCard(persistPaymentOption, new Callback<AddCardResponse>() {
                                        @Override
                                        public void success(AddCardResponse addCardResponse) {
                                            saveSecureCVV(addCardResponse.getFingerPrint(), ((CardOption) persistPaymentOption).getCardCVV());// we need to encrypt using
                                            persistPaymentOption = null;
                                        }

                                        @Override
                                        public void error(CitrusError error) {

                                        }
                                    });
                                }
                            }
                        }
                        sendResponse(callback, transactionResponse);
                    } else {
                        sendError(callback, new CitrusError(transactionResponse.getMessage(), transactionResponse.getJsonResponse(), status, transactionResponse));
                    }
                }
            }
        };

        LocalBroadcastManager.getInstance(mContext).registerReceiver(paymentEventReceiver, intentFilter);
    }


    private synchronized boolean validate() {
        if (!TextUtils.isEmpty(signinId) && !TextUtils.isEmpty(signinSecret)
                && !TextUtils.isEmpty(signupId) && !TextUtils.isEmpty(signupSecret)
                && !TextUtils.isEmpty(vanity)) {
            return true;
        } else {
            throw new IllegalArgumentException(ResponseMessages.ERROR_MESSAGE_BLANK_CONFIG_PARAMS);
        }
    }

    private <T> void sendResponse(Callback callback, T t) {
        if (callback != null) {
            callback.success(t);
        }
    }

    private void sendError(Callback callback, CitrusError error) {
        if (callback != null) {
            callback.error(error);
        }
    }

    /*
    NEW APIS for Cube starts here
     */

    /**
     * Reset the user password. The password reset link will be sent to the user. UM implementation
     *
     * @param emailId
     * @param callback
     */
    public synchronized void resetUserPassword(final String emailId, @NonNull final Callback<CitrusUMResponse> callback) {

        retrofitAPIWrapper.resetUserPassword(emailId, callback);
    }

    public synchronized void setDefaultPaymentOption(final PaymentOption defaultPaymentOption, final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.setDefaultPaymentOption(defaultPaymentOption, callback);
    }


    public synchronized void signUpUser(final String email, final String mobile, final String password, final String firstName, final String lastName,
                                        final String sourceType, final boolean markMobileVerified, final boolean markEmailVerified,
                                        final Callback<CitrusResponse> callback) {

        retrofitAPIWrapper.signUpUser(email, mobile, password, firstName, lastName, sourceType, markMobileVerified, markEmailVerified, callback);
    }


    public synchronized void changePassword(final String oldPassword, final String newPassword, final Callback<CitrusUMResponse> changePasswordResponseCallback) {

        retrofitAPIWrapper.changePassword(oldPassword, newPassword, changePasswordResponseCallback);
    }


    public synchronized void updateProfileInfo(final String firstName, final String lastName, final Callback<CitrusUMResponse> citrusUMResponseCallback) {

        retrofitAPIWrapper.updateProfileInfo(firstName, lastName, citrusUMResponseCallback);
    }


    public synchronized void sendOneTimePassword(String source, String otpType, String identity, final Callback<CitrusUMResponse> umResponseCallback) {

        retrofitAPIWrapper.sendOneTimePassword(source, otpType, identity, umResponseCallback);
    }


    public synchronized void getCardType(final String first6Digits, final Callback<CardBinDetails> cardDetailsCallback) {

        retrofitAPIWrapper.getCardType(first6Digits, cardDetailsCallback);
    }

    public synchronized void getMerchantName(final Callback<String> callback) {

    }

    /**
     * This method is used to initiate the transaction. This is used internally.
     *
     * @param paymentJSON
     * @param callback
     */
    public synchronized void makeMOTOPayment(String paymentJSON, Callback<StructResponsePOJO> callback) {
        retrofitAPIWrapper.makeMOTOPayment(paymentJSON, callback);
    }

    /**
     * This method is used to initiate the transaction. This is used internally.
     *
     * @param paymentJSON
     * @param callback
     */
    public synchronized void newMakePayment(String paymentJSON, Callback<String> callback) {
        retrofitAPIWrapper.newMakePayment(paymentJSON, callback);
    }

    /**
     * Get prepaid bill. This is used internally.
     *
     * @param amount
     * @param returnUrl
     * @param callback
     */
    public synchronized void getPrepaidBill(Amount amount, String returnUrl, Callback<CitrusPrepaidBill> callback) {

        retrofitAPIWrapper.getPrepaidBill(amount, returnUrl, callback);
    }

    // Getters and setters.
    public String getSigninId() {
        return signinId;
    }

    public void setSigninId(String signinId) {
        this.signinId = signinId;
    }

    public String getSigninSecret() {
        return signinSecret;
    }

    public void setSigninSecret(String signinSecret) {
        this.signinSecret = signinSecret;
    }

    public String getSignupId() {
        return signupId;
    }

    public void setSignupId(String signupId) {
        this.signupId = signupId;
    }

    public String getSignupSecret() {
        return signupSecret;
    }

    public void setSignupSecret(String signupSecret) {
        this.signupSecret = signupSecret;
    }

    public String getVanity() {
        return vanity;
    }

    public void setVanity(String vanity) {
        this.vanity = vanity;
    }

    public String getMerchantName() {
        return merchantName;
    }

    public Environment getEnvironment() {
        return environment;
    }

    public void setEnvironment(Environment environment) {
        this.environment = environment;
    }

    private String generateDeviceSpecificKeys() {
        String android_id =
                Settings.Secure.getString(mContext.getContentResolver(), Settings.Secure.ANDROID_ID);
        // String device_id = tManager.getDeviceId();
        String str1 = Build.BOARD + Build.BRAND + Build.CPU_ABI + Build.DEVICE +
                Build.DISPLAY + Build.FINGERPRINT + Build.HOST + Build.ID + Build.MANUFACTURER
                +
                Build.MODEL + Build.PRODUCT + Build.TAGS + Build.TYPE + Build.USER;
        String key2 = str1 + android_id;
        return key2;
    }

    private void initCrypto() {
        crypto = new Crypto(
                new SharedPrefsBackedKeyChain(mContext),
                new SystemNativeCryptoLibrary());
        if (crypto.isAvailable()) {
            isFacebookNativeLibraryLoaded = true;
        } else {
            isFacebookNativeLibraryLoaded = false;
            crypto = null;
        }
    }

    private void saveSecureCVV(String fingerPrint, String CVV) {
        // Crypto library is not loaded on the device so do not proceed.
        if (!CitrusConfig.getInstance().isOneTapPaymentEnabled() || !isFacebookNativeLibraryLoaded || !isCitrusNativeLibraryLoaded) {
            return;
        }

        String seed = (generateDeviceSpecificKeys() + fingerPrint + getEncryptionKey());
        Entity myEntity = new Entity(seed);

        try {
            byte[] cipherTextToken = crypto.encrypt(fingerPrint.getBytes(), myEntity);
            //  String encryptedToken = new String(cipherText);
            byte[] cipherText = crypto.encrypt(CVV.getBytes(), myEntity);

            byte[] cipherText2 = crypto.encrypt(CVV.getBytes(), myEntity);
            //String encryptedCVV = new String(cipherText);
            oauthToken.saveCVV(cipherTextToken, cipherText);
        } catch (KeyChainException e) {
            // e.printStackTrace();
            Logger.d("KeyChainException ");
        } catch (CryptoInitializationException e) {
            Logger.d("CryptoInitializationException");
            // e.printStackTrace();
        } catch (IOException e) {
            Logger.d("IOException");
            // e.printStackTrace();
        }
    }

    private String getCVVOfFingerPrint(CardOption cardOption) {

        String cvv = null;

        try {
            // Check if crypto library is not loaded.
            if (!CitrusConfig.getInstance().isOneTapPaymentEnabled() || !isFacebookNativeLibraryLoaded || !isCitrusNativeLibraryLoaded) {
                return null;
            }

            String seed = (generateDeviceSpecificKeys() + cardOption.getFingerPrint() + getEncryptionKey());

            Entity myEntity = new Entity(seed);

            byte[] encryptedCVV = oauthToken.getCVV(cardOption.getFingerPrint(), seed);
            if (encryptedCVV != null) {

                byte[] decryptedCVV = crypto.decrypt(encryptedCVV, myEntity);
                cvv = new String(decryptedCVV);
            } else {
                cvv = null;

            }
        } catch (KeyChainException e) {
            // e.printStackTrace();
            Logger.d("KeyChainException ");
        } catch (CryptoInitializationException e) {
            Logger.d("CryptoInitializationException");
            // e.printStackTrace();
        } catch (IOException e) {
            Logger.d("IOException");
            // e.printStackTrace();
        }
        return cvv;
    }

    public synchronized boolean isOneTapPaymentEnabledForCard(CardOption cardOption) {
        if (cardOption != null) {
            String cvv = getCVVOfFingerPrint(cardOption);
            return cvv != null;
        }
        return false;
    }

    /**
     * AutoCVV may not be work on few devices. Use this method before prompting for AutoCVV feature.
     *
     * @return
     */
    public synchronized boolean isOneTapPaymentSupported() {
        return (isCitrusNativeLibraryLoaded && isFacebookNativeLibraryLoaded);
    }


    @Override
    public void onOneTapPaymentEnabled(boolean isEnabled) {
        if (isEnabled) {
            try {
                System.loadLibrary("citruslibrary");
                isCitrusNativeLibraryLoaded = true;
            } catch (UnsatisfiedLinkError error) {
                isCitrusNativeLibraryLoaded = false;
            }
            initCrypto();

            if (isOneTapPaymentSupported()) {
                Logger.d("This device supports auto CVV feature");
            } else {
                Logger.d("This device does not support auto CVV feature");
            }
        }


    }

}