package com.citrus.sdk;

import android.content.Context;
import android.support.annotation.NonNull;

import com.citrus.mobile.OAuth2GrantType;
import com.citrus.sdk.apis.UMClientAPI;
import com.citrus.sdk.classes.AccessToken;
import com.citrus.sdk.classes.Amount;
import com.citrus.sdk.classes.BinServiceResponse;
import com.citrus.sdk.classes.CardBinDetails;
import com.citrus.sdk.classes.CashoutInfo;
import com.citrus.sdk.classes.CitrusPrepaidBill;
import com.citrus.sdk.classes.CitrusUMResponse;
import com.citrus.sdk.classes.LinkUserExtendedResponse;
import com.citrus.sdk.classes.LinkUserResponse;
import com.citrus.sdk.classes.MemberInfo;
import com.citrus.sdk.classes.PGHealthResponse;
import com.citrus.sdk.classes.StructResponsePOJO;
import com.citrus.sdk.classes.Utils;
import com.citrus.sdk.dynamicPricing.DynamicPricingRequestType;
import com.citrus.sdk.dynamicPricing.DynamicPricingResponse;
import com.citrus.sdk.otp.NetBankForOTP;
import com.citrus.sdk.payment.CardOption;
import com.citrus.sdk.payment.MerchantPaymentOption;
import com.citrus.sdk.payment.PaymentBill;
import com.citrus.sdk.payment.PaymentOption;
import com.citrus.sdk.payment.PaymentType;
import com.citrus.sdk.response.BindUserResponse;
import com.citrus.sdk.response.CitrusError;
import com.citrus.sdk.response.CitrusResponse;
import com.citrus.sdk.response.PaymentResponse;
import com.orhanobut.logger.Logger;

import java.util.List;

import client.PGClient;
import client.PrepaidClient;
import client.UMClient;

/**
 * Created by salil on 27/11/15.
 */
public class RetrofitAPIWrapper implements UMClientAPI {
    private static RetrofitAPIWrapper instance = null;
    private UMClient umClient = null;
    private PGClient pgClient = null;
    private PrepaidClient prepaidClient = null;
    private Context context = null;
    private boolean isPrepaymentTokenValid = false;

    public static RetrofitAPIWrapper getInstance(Context context) {
        if (instance == null) {
            synchronized (RetrofitAPIWrapper.class) {
                if (instance == null) {
                    instance = new RetrofitAPIWrapper(context);
                }
            }
        }

        return instance;
    }

    private RetrofitAPIWrapper(Context context) {
        this.context = context;

        umClient = UMClient.getInstance(context);
        pgClient = PGClient.getInstance(context);
        prepaidClient = PrepaidClient.getInstance(context);
    }

    public synchronized void init(String signupId, String signupSecret, String signinId, String signinSecret, String vanity, Environment environment) {
        umClient.init(signupId, signupSecret, signinId, signinSecret, vanity, environment);
        pgClient.init(signupId, signupSecret, signinId, signinSecret, vanity, environment);
        prepaidClient.init(signupId, signupSecret, signinId, signinSecret, vanity, environment);
    }

    /*
    UM APIs.
     */
    public synchronized void getSignUpToken(final Callback<AccessToken> callback) {
        umClient.getSignUpToken(callback);
    }

    public synchronized void createUser(final String emailId, final String mobileNo, final Callback<String> callback) {
        umClient.createUser(emailId, mobileNo, callback);
    }

    @Override
    public synchronized void getUserNameToken(String userName, Callback<AccessToken> callback) {
        umClient.getUserNameToken(userName, callback);
    }

    @Override
    public void getUserNameToken(Callback<AccessToken> callback) {
        umClient.getUserNameToken(callback);
    }

    @Override
    public void getPrepaidToken(Callback<AccessToken> callback) {
        umClient.getPrepaidToken(callback);
    }

    public void getPrepaidPayToken(Callback<AccessToken> callback) {
        umClient.getPrepaidPayToken(callback);
    }

    /**
     * This api will check whether the user is existing user or not. If the user is existing user,
     * then it will return the existing details, else it will create an account internally and
     * then call signUp to set the password and activate the account.
     *
     * @param emailId  - emailId of the user
     * @param mobileNo - mobileNo of the user
     * @param callback - callback
     */
    public synchronized void isCitrusMember(final String emailId, final String mobileNo, final Callback<Boolean> callback) {
        umClient.isCitrusMember(emailId, mobileNo, callback);
    }

    public synchronized void getMemberInfo(final String emailId, final String mobileNo, final Callback<MemberInfo> callback) {
        umClient.getMemberInfo(emailId, mobileNo, callback);
    }

    public synchronized void bindUserByMobile(String emailId, String mobileNo, com.citrus.sdk.Callback<BindUserResponse> callback) {
        umClient.bindUserByMobile(emailId, mobileNo, callback);
    }

    public synchronized void linkUserWithOTP(final String emailId, final String mobileNo, final boolean forceMobileVerification, final Callback<LinkUserResponse> callback) {
        umClient.linkUserWithOTP(emailId, mobileNo, forceMobileVerification, callback);
    }

    public synchronized void linkUserExtended(final String emailId, final String mobileNo, final Callback<LinkUserExtendedResponse> callback) {
        umClient.linkUserExtended(emailId, mobileNo, callback);
    }

    public void linkUserExtendedVerifyEOTPAndUpdateMobile(final String emailId, String signInGrantType, final String linkUserPassword, LinkUserExtendedResponse linkUserExtended, final Callback<CitrusResponse> callback) {
        umClient.linkUserExtendedVerifyEOTPAndUpdateMobile(emailId, signInGrantType, linkUserPassword, linkUserExtended, new Callback<CitrusResponse>() {
            @Override
            public void success(CitrusResponse citrusResponse) {
                sendResponse(callback, new CitrusResponse(ResponseMessages.SUCCESS_MESSAGE_SIGNIN, CitrusResponse.Status.SUCCESSFUL));
//                signIn(emailId, linkUserPassword, new Callback<CitrusResponse>() {
//                    @Override
//                    public void success(CitrusResponse citrusResponse) {
//                        sendResponse(callback, citrusResponse);
//                    }
//
//                    @Override
//                    public void error(CitrusError error) {
//                        sendError(callback, error);
//                    }
//                });
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public void linkUserExtendedVerifyMobileAndSignIn(final LinkUserExtendedResponse linkUserExtendedResponse, final String verificationCode, final Callback<CitrusResponse> callback) {
        umClient.linkUserExtendedVerifyMobileAndSignIn(linkUserExtendedResponse, verificationCode, new Callback<CitrusResponse>() {
            @Override
            public void success(CitrusResponse citrusResponse) {
                sendResponse(callback, new CitrusResponse(ResponseMessages.SUCCESS_MESSAGE_SIGNIN, CitrusResponse.Status.SUCCESSFUL));
//                signIn(linkUserExtendedResponse.getInputMobile(), verificationCode, OAuth2GrantType.onetimepass, new Callback<CitrusResponse>() {
//                    @Override
//                    public void success(CitrusResponse citrusResponse) {
//                        sendResponse(callback, citrusResponse);
//                    }
//
//                    @Override
//                    public void error(CitrusError error) {
//                        sendError(callback, error);
//                    }
//                });
            }

            @Override
            public void error(CitrusError error) {

            }
        });
    }

    @Override
    public synchronized void signUp(final String emailId, final String mobileNo, final String password, final com.citrus.sdk.Callback<CitrusResponse> callback) {

        umClient.signUp(emailId, mobileNo, password, new Callback<CitrusResponse>() {
            @Override
            public void success(CitrusResponse citrusResponse) {
                signIn(emailId, password, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /**
     * This method will sign in into prepaid acccount. This method retrieves and saves grant type username and password token.
     * <p/>
     * Default grantType is <b> password </b>.
     * Use {@link RetrofitAPIWrapper#signIn(String, String, OAuth2GrantType, Callback)} for sending other granttypes.
     *
     * @param emailId
     * @param password
     * @param callback
     */
    public synchronized void signIn(final String emailId, final String password, final com.citrus.sdk.Callback<CitrusResponse> callback) {
        signIn(emailId, password, OAuth2GrantType.password, callback);
    }

    /**
     * This method will sign in into prepaid acccount. This method retrieves and saves grant type username and password token.
     *
     * @param emailId
     * @param password
     * @param grantType
     * @param callback
     */
    private synchronized void signIn(final String emailId, final String password, final OAuth2GrantType grantType, final com.citrus.sdk.Callback<CitrusResponse> callback) {
        umClient.getUsernameAndPrepaidToken(emailId, password, grantType, new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                activatePrepaidUser(new Callback<Amount>() {
                    @Override
                    public void success(Amount amount) {
                        sendResponse(callback, new CitrusResponse(ResponseMessages.SUCCESS_MESSAGE_SIGNIN, CitrusResponse.Status.SUCCESSFUL));
                        getCookie(emailId, password, null);
                        checkPrepaymentTokenValidity(null);
                            }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void signInWithMobileNo(String mobileNo, String password, Callback<CitrusResponse> callback) {
        signIn(mobileNo, password, callback);
    }

    // TODO: Check for load money flow
    public synchronized void signInWithOTP(String emailIdOrMobileNo, String otp, Callback<CitrusResponse> callback) {
        signIn(emailIdOrMobileNo, otp, OAuth2GrantType.onetimepass, callback);
    }

    @Override
    public synchronized void resetPassword(String emailId, @NonNull Callback<CitrusResponse> callback) {
        umClient.resetPassword(emailId, callback);
    }

    @Override
    public synchronized void updateMobile(String mobileNo, Callback<String> callback) {
        umClient.updateMobile(mobileNo, callback);
    }

    @Override
    public synchronized void verifyMobile(String verificationCode, Callback<String> callback) {
        umClient.verifyMobile(verificationCode, callback);
    }

    @Override
    public synchronized void getProfileInfo(Callback<CitrusUser> callback) {
        umClient.getProfileInfo(callback);
    }

    @Override
    public synchronized void changePassword(String oldPassword, String newPassword, Callback<CitrusUMResponse> changePasswordResponseCallback) {
        umClient.changePassword(oldPassword, newPassword, changePasswordResponseCallback);
    }

    @Override
    public synchronized void updateProfileInfo(String firstName, String lastName, Callback<CitrusUMResponse> citrusUMResponseCallback) {
        umClient.updateProfileInfo(firstName, lastName, citrusUMResponseCallback);
    }

    @Override
    public synchronized void sendOneTimePassword(String source, String otpType, String identity, Callback<CitrusUMResponse> umResponseCallback) {
        umClient.sendOneTimePassword(source, otpType, identity, umResponseCallback);
    }

    @Override
    public synchronized void resetUserPassword(String emailId, @NonNull Callback<CitrusUMResponse> callback) {
        umClient.resetUserPassword(emailId, callback);
    }

    @Override
    public synchronized void signUpUser(final String email, String mobile, final String password, String firstName, String lastName, String sourceType, boolean markMobileVerified, boolean markEmailVerified, final Callback<CitrusResponse> callback) {
        umClient.signUpUser(email, mobile, password, firstName, lastName, sourceType, markMobileVerified, markEmailVerified, new Callback<CitrusResponse>() {
            @Override
            public void success(CitrusResponse citrusResponse) {
                signIn(email, password, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void signOut(final Callback<CitrusResponse> callback) {

        umClient.signOut(new Callback<Boolean>() {
            @Override
            public void success(Boolean aBoolean) {
                if (aBoolean) {
                    resetPrepaymentTokenValidity();
                    CitrusResponse citrusResponse = new CitrusResponse("User Logged Out Successfully.", CitrusResponse.Status.SUCCESSFUL);
                    sendResponse(callback, citrusResponse);

                } else {
                    CitrusError citrusError = new CitrusError("Failed to logout.", CitrusResponse.Status.FAILED);
                    callback.error(citrusError);
                }

            }

            @Override
            public void error(CitrusError error) {
                CitrusError citrusError = new CitrusError("Failed to logout.", CitrusResponse.Status.FAILED);
                callback.error(citrusError);

            }
        });

    }


    /*********************
     * Prepaid API Start *
     ********************/

    public synchronized void resetPrepaymentTokenValidity() {
        isPrepaymentTokenValid = false;
    }

    public synchronized void checkPrepaymentTokenValidity(final Callback<Boolean> callback) {
        getSignUpToken(new Callback<AccessToken>() {
            @Override
            public void success(final AccessToken signUpToken) {
                // Get the prepaidToken whose validity needs to be checked.
                getPrepaidToken(new Callback<AccessToken>() {
                    @Override
                    public void success(AccessToken prepaidToken) {
                        if (prepaidToken.hasPrepaidPayToken()) {

                            // Check the validity of the prepayment token.
                            prepaidClient.getPrepaymentTokenValidity(signUpToken, prepaidToken, Constants.SCOPE_PREPAID_MERCHANT_PAY, new Callback<Boolean>() {
                                @Override
                                public void success(Boolean valid) {
                                    isPrepaymentTokenValid = true;
                                    sendResponse(callback, valid);
                                }

                                @Override
                                public void error(CitrusError error) {
                                    isPrepaymentTokenValid = false;
                                    sendError(callback, error);
                                }
                            });
                        } else {
                            // Since the payment token is not present, make it true;
                            isPrepaymentTokenValid = true;
                        }
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void getPrepaidBill(final Amount amount, final String returnUrl, final Callback<CitrusPrepaidBill> callback) {
        umClient.getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                prepaidClient.getPrepaidBill(accessToken, amount, returnUrl, new Callback<CitrusPrepaidBill>() {
                    @Override
                    public void success(CitrusPrepaidBill citrusPrepaidBill) {
                        sendResponse(callback, citrusPrepaidBill);
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void getBalance(final Callback<Amount> callback) {
        // First fetch the granttype username token
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                prepaidClient.getBalance(accessToken, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void activatePrepaidUser(final Callback<Amount> callback) {
        // First fetch the prepaid token.
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                prepaidClient.activatePrepaidUser(accessToken, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void getCookie(String email, String password, final Callback<CitrusResponse> callback) {

        prepaidClient.getCookie(email, password, callback);
    }

    public synchronized void newPrepaidPay(final PaymentType.CitrusCash citrusCash, final Callback<PaymentResponse> callback) {
        if (isPrepaymentTokenValid) {
            Amount amountToBePaid = null;
            if (citrusCash.getPaymentBill() != null) {
                amountToBePaid = citrusCash.getPaymentBill().getAmount();
            } else {
                amountToBePaid = citrusCash.getAmount();
            }

            isEnoughPrepaidBalance(amountToBePaid, new Callback<Boolean>() {
                @Override
                public void success(Boolean sufficientBalance) {
                    if (sufficientBalance) {
                        if (citrusCash.getPaymentBill() != null) {
                            // Merchant has Bill Ready, so directly proceed for the payment.
                            proceedToPayUsingCitrusCash(citrusCash, callback);
                        } else {
                            // Fetch the bill from the server and then proceed to payment.
                            final String billUrl;
                            billUrl = Utils.getUrlFormatted(citrusCash.getUrl(), citrusCash.getAmount());
                            getBill(billUrl, citrusCash.getAmount(), new Callback<PaymentBill>() {
                                @Override
                                public void success(PaymentBill paymentBill) {
                                    citrusCash.setPaymentBill(paymentBill);
                                    proceedToPayUsingCitrusCash(citrusCash, callback);
                                }

                                @Override
                                public void error(CitrusError error) {
                                    sendError(callback, error);
                                }
                            });
                        }
                    } else {
                        //dont have enough balance in users account
                        sendError(callback, new CitrusError(ResponseMessages.ERROR_MESSAGE_INSUFFICIENT_BALANCE, CitrusResponse.Status.FAILED));
                    }
                }

                @Override
                public void error(CitrusError error) {
                    sendError(callback, error);
                }
            });
        } else {
            Logger.d("User's cookie has expired. Please signin");
            sendError(callback, new CitrusError("User's cookie has expired. Please signin.", CitrusResponse.Status.FAILED));
        }
    }

    private void proceedToPayUsingCitrusCash(final PaymentType.CitrusCash citrusCash, final Callback<PaymentResponse> callback) {
        getPrepaidPayToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken prepaidToken) {

                // Use the user details sent by the merchant, else use the user details from the token.
                if (citrusCash.getCitrusUser() == null) {
                    CitrusUser citrusUser = getCitrusUser();
                    citrusCash.setCitrusUser(citrusUser);
                }

                prepaidClient.newPrepaidPay(prepaidToken, citrusCash, new Callback<PaymentResponse>() {
                    @Override
                    public void success(PaymentResponse paymentResponse) {
                        sendResponse(callback, paymentResponse);

                        // Send the response on the return url asynchronously, so as to keep the integration same.
                        Utils.sendResponseToReturnUrlAsync(context, citrusCash.getPaymentBill().getReturnUrl(), paymentResponse);
                    }

                    @Override
                    public void error(CitrusError error) {
                        sendError(callback, error);
                    }
                });
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /**
     * Check the prepaid balance is sufficient for transaction or not.
     *
     * @param transactionAmount
     * @param callback
     */
    private void isEnoughPrepaidBalance(final Amount transactionAmount, final Callback<Boolean> callback) {

        getBalance(new Callback<Amount>() {
            @Override
            public void success(Amount userBalance) {
                if (userBalance.getValueAsDouble() >= transactionAmount.getValueAsDouble()) {
                    sendResponse(callback, true);
                } else {
                    sendResponse(callback, false);
                }
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /**
     * Send money to your friend.
     *
     * @param amount   - Amount to be sent
     * @param toUser   - The user detalis. Enter emailId if send by email or mobileNo if send by mobile.
     * @param message  - Optional message
     * @param callback - Callback
     * @deprecated Use {@link RetrofitAPIWrapper#sendMoneyToMoblieNo(Amount, String, String, Callback)} instead.
     */
    public synchronized void sendMoney(final Amount amount, final CitrusUser toUser, final String message, final Callback<PaymentResponse> callback) {

        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                prepaidClient.sendMoney(accessToken, amount, toUser, message, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /**
     * @param amount
     * @param mobileNo
     * @param message
     * @param callback
     */
    public synchronized void sendMoneyToMoblieNo(final Amount amount, final String mobileNo, final String message, final Callback<PaymentResponse> callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                prepaidClient.sendMoneyToMoblieNo(accessToken, amount, mobileNo, message, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /**
     * @param cashoutInfo
     * @param callback
     */
    public synchronized void cashout(@NonNull final CashoutInfo cashoutInfo, final Callback<PaymentResponse> callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                prepaidClient.cashout(accessToken, cashoutInfo, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void getCashoutInfo(final Callback<CashoutInfo> callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                prepaidClient.getCashoutInfo(accessToken, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void saveCashoutInfo(final CashoutInfo cashoutInfo, final Callback<CitrusResponse> callback) {
        getPrepaidToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                prepaidClient.saveCashoutInfo(accessToken, cashoutInfo, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    /*********************
     * Prepaid API End *
     ********************/

    /*********************
     * PG API Start *
     ********************/
    public synchronized void getMerchantPaymentOptions(final Callback<MerchantPaymentOption> callback) {

        pgClient.getMerchantPaymentOptions(callback);
    }

    public synchronized void getLoadMoneyPaymentOptions(final Callback<MerchantPaymentOption> callback) {

        pgClient.getLoadMoneyPaymentOptions(callback);
    }

    public synchronized void getWallet(final Callback<List<PaymentOption>> callback) {
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                pgClient.getWallet(accessToken, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void getWalletWithDefaultBank(final Callback<List<PaymentOption>> callback, final BankCID bankCID) {
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                pgClient.getWalletWithDefaultBank(accessToken, callback, bankCID);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void savePaymentOption(final PaymentOption paymentOption, final Callback<CitrusResponse> callback) {
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                pgClient.savePaymentOption(accessToken, paymentOption, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });

    }

    public synchronized void saveCard(final PaymentOption paymentOption, final Callback<AddCardResponse> callback) {
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                pgClient.saveCard(accessToken, paymentOption, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });

    }

    public synchronized void deletePaymentOption(final PaymentOption paymentOption, final Callback<CitrusResponse> callback) {
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                pgClient.deletePaymentOption(accessToken, paymentOption, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void setDefaultPaymentOption(final PaymentOption defaultPaymentOption, final Callback<CitrusResponse> callback) {
        getUserNameToken(new Callback<AccessToken>() {
            @Override
            public void success(AccessToken accessToken) {
                pgClient.setDefaultPaymentOption(accessToken, defaultPaymentOption, callback);
            }

            @Override
            public void error(CitrusError error) {
                sendError(callback, error);
            }
        });
    }

    public synchronized void getBINDetails(final CardOption cardOption, final Callback<BinServiceResponse> callback) {

        pgClient.getBINDetails(cardOption, callback);
    }

    /**
     * Use {@link RetrofitAPIWrapper#getBINDetails} instead. This method will be used in Cube.
     *
     * @param first6Digits
     * @param cardDetailsCallback
     */
    public synchronized void getCardType(final String first6Digits, final Callback<CardBinDetails> cardDetailsCallback) {

        pgClient.getCardType(first6Digits, cardDetailsCallback);
    }

    public synchronized void getBill(final String billUrl, final Amount amount, final Callback<PaymentBill> callback) {

        pgClient.getBill(billUrl, amount, callback);
    }

    public synchronized void getPGHealth(final PaymentOption paymentOption, final Callback<PGHealthResponse> callback) {

        pgClient.getPGHealth(paymentOption, callback);
    }

    void fetchPGHealthForAllBanks() {

        pgClient.fetchPGHealthForAllBanks();
    }

    /**
     * Perform Dynamic Pricing. You can specify one the dynamicPricingRequestType to perform Dynamic Pricing.
     *
     * @param dynamicPricingRequestType - One of the dynamicPricingRequestType from {@link DynamicPricingRequestType}
     * @param billUrl                   - billUrl from where we will fetch the bill.
     * @param callback                  - callback
     */
    public synchronized void performDynamicPricing(@NonNull final DynamicPricingRequestType dynamicPricingRequestType, @NonNull final String billUrl, @NonNull final Callback<DynamicPricingResponse> callback) {

        pgClient.performDynamicPricing(dynamicPricingRequestType, billUrl, callback);
    }

    /**
     * Perform Dynamic Pricing. You can specify one the dynamicPricingRequestType to perform Dynamic Pricing.
     *
     * @param dynamicPricingRequestType - One of the dynamicPricingRequestType from {@link DynamicPricingRequestType}
     * @param paymentBill               - PaymentBill in case you are fetching bill response from your server.
     * @param callback                  - callback
     */
    public synchronized void performDynamicPricing(@NonNull final DynamicPricingRequestType dynamicPricingRequestType, @NonNull final PaymentBill paymentBill, @NonNull final Callback<DynamicPricingResponse> callback) {

        pgClient.performDynamicPricing(dynamicPricingRequestType, paymentBill, callback);
    }

    public synchronized void makeMOTOPayment(String paymentJSON, Callback<StructResponsePOJO> callback) {

        pgClient.makeMOTOPayment(paymentJSON, callback);
    }

    public synchronized void newMakePayment(String paymentJSON, Callback<String> callback) {

        pgClient.newMakePayment(paymentJSON, callback);
    }

    public synchronized void getMerchantName(@NonNull final Callback<String> callback) {

        pgClient.getMerchantName(callback);
    }

    /*********************
     * PG API END *
     ********************/

    public NetBankForOTP getNetBankForOTP() {
        return pgClient.getNetBankForOTP();
    }

    public CitrusUser getCitrusUser() {
        return umClient.getCitrusUser();
    }

    public void resetNetBankForOTP() {
        pgClient.resetNetBankForOTP();
    }

    private <T> void sendResponse(Callback<T> callback, T t) {
        if (callback != null) {
            callback.success(t);
        }
    }

    private <T> void sendError(Callback<T> callback, CitrusError citrusError) {
        if (callback != null) {
            callback.error(citrusError);
        }
    }

}
