/*
 *
 *    Copyright 2014 Citrus Payment Solutions Pvt. Ltd.
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 * /
 */
package com.citrus.mobile;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.util.Base64;

import com.citrus.sdk.CitrusClient;
import com.citrus.sdk.Constants;
import com.citrus.sdk.Environment;
import com.citrus.sdk.classes.AccessToken;
import com.facebook.android.crypto.keychain.SharedPrefsBackedKeyChain;
import com.facebook.crypto.Crypto;
import com.facebook.crypto.Entity;
import com.facebook.crypto.exception.CryptoInitializationException;
import com.facebook.crypto.exception.KeyChainException;
import com.facebook.crypto.util.SystemNativeCryptoLibrary;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.Date;
import java.util.Iterator;
import java.util.Map;

public class OauthToken {

    //emailID and MobileNo is saved after SignIn token is received


    private Context context;

    private Activity activity;

    private JSONObject jsontoken;

    private SharedPreferences tokenPrefs;

    private String base_url, token_type;

    private com.citrus.sdk.Callback callback;
    private CitrusClient citrusClient;

    private String signinId = null;

    private String signinSecret = null;

    public OauthToken(Activity activity, String token_type) {
        this.activity = activity;
        tokenPrefs = this.activity.getSharedPreferences(Constants.STORED_VALUES, 0);
        base_url = Config.getEnv();
        this.token_type = token_type;
    }


    public OauthToken(Context context, String token_type) {
        this.context = context;
        tokenPrefs = this.context.getSharedPreferences(Constants.STORED_VALUES, 0);
        base_url = Config.getEnv();
        this.token_type = token_type;
    }

    public OauthToken(Context context, String token_type, String signinId, String signinSecret) {
        this.context = context;
        tokenPrefs = this.context.getSharedPreferences(Constants.STORED_VALUES, 0);
        base_url = Config.getEnv();
        this.token_type = token_type;
        this.signinId = signinId;
        this.signinSecret = signinSecret;
    }

    public OauthToken(Context context) {
        this.context = context;
        tokenPrefs = this.context.getSharedPreferences(Constants.STORED_VALUES, 0);
        base_url = Config.getEnv();
    }


    public boolean createToken(JSONObject usertoken) {

        jsontoken = new JSONObject();

        long expiry = new Date().getTime() / 1000L;

        try {
            expiry += usertoken.getLong("expires_in");
            jsontoken.put("expiry", expiry);
        } catch (JSONException e) {
            e.printStackTrace();
        }

        for (Iterator<?> keys = usertoken.keys(); keys.hasNext(); ) {
            String key = (String) keys.next();

            try {
                jsontoken.put(key, usertoken.get(key));
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        return storeToken();

    }

    private boolean storeToken() {
        SharedPreferences.Editor editor = tokenPrefs.edit();
        if(Constants.isEncryptedAccessToken) {
            editor.putString(token_type, getEncryptedToken());
        }
        else {
            editor.putString(token_type, jsontoken.toString());
        }
        return editor.commit();
    }

    private String getEncryptedToken() {
        String seed = (token_type + signinId + signinSecret);

        Crypto crypto = new Crypto(
                new SharedPrefsBackedKeyChain(context),
                new SystemNativeCryptoLibrary());

        // If library is not loaded do not proceed.
        if (!crypto.isAvailable()) {
            return jsontoken.toString();
        }

        Entity myEntity = new Entity(seed);

        String encryptedToken = null;
        try {
            byte[] cipherTextToken = crypto.encrypt(jsontoken.toString().getBytes(), myEntity);
            encryptedToken = Base64.encodeToString(cipherTextToken, Base64.DEFAULT);

        } catch (KeyChainException e) {
            e.printStackTrace();
        } catch (CryptoInitializationException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        return encryptedToken;
    }


    public synchronized void getSignUpToken(com.citrus.sdk.Callback<AccessToken> callback) {
        this.token_type = Constants.SIGNUP_TOKEN;
    }

    public synchronized void getSignInToken(com.citrus.sdk.Callback<AccessToken> callback) {
        this.token_type = Constants.SIGNIN_TOKEN;
    }


    public synchronized void getPrepaidToken(com.citrus.sdk.Callback<AccessToken> callback) {
        this.token_type = Constants.PREPAID_TOKEN;
    }

    public synchronized void getPayUsingCitrusCashToken(com.citrus.sdk.Callback<AccessToken> callback) {
        this.token_type = Constants.PAY_USING_CITRUS_CASH_TOKEN;
    }


    public boolean clearToken() {
        SharedPreferences.Editor editor = tokenPrefs.edit();
        editor.clear();
        return editor.commit();
    }


    public boolean saveUserDetails(String emailId, String mobileNo) {

        SharedPreferences.Editor editor = tokenPrefs.edit();
        editor.putString(Constants.EMAIL_ID, emailId);
        editor.putString(Constants.MOBILE_NO, mobileNo);
        return editor.commit();
    }


    public String getEmailId() {
        return tokenPrefs.getString(Constants.EMAIL_ID, null);
    }

    public String getMobileNumber() {
        return tokenPrefs.getString(Constants.MOBILE_NO, null);
    }

    public boolean saveEnvironment(Environment environment) {
        SharedPreferences.Editor editor = tokenPrefs.edit();
        editor.putString(Constants.ENVIRONMENT, environment.toString());
        return editor.commit();
    }

    public Environment getCurrentEnvironment() {
        String environment = tokenPrefs.getString(Constants.ENVIRONMENT, null);
        return (environment == null) ? Environment.NONE : Environment.valueOf(environment);
    }


    /**
     * This function will save encrypted token and CVV
     *
     * @param encryptedToken encrypted token from concel
     * @param encryptedCVV   encrypted CVV from concel
     * @return
     */
    public boolean saveToken(byte[] encryptedToken, byte[] encryptedCVV) {

        String token = Base64.encodeToString(encryptedToken, Base64.DEFAULT);
        String CVV = Base64.encodeToString(encryptedCVV, Base64.DEFAULT);

        SharedPreferences.Editor editor = tokenPrefs.edit();
        editor.putString(token, CVV);
        return editor.commit();
    }


    /**
     * This function will return byte[] of encrypted CVV
     *
     * @param token
     * @param seed
     * @return
     */
    public byte[] getCVV(String token, String seed) {

        byte[] array = null;

        Map<String, ?> prefsMap = tokenPrefs.getAll();
        for (Map.Entry<String, ?> entry : prefsMap.entrySet()) {
            String strEntry = entry.getKey();
            if (strEntry != Constants.ENVIRONMENT && strEntry != Constants.EMAIL_ID && strEntry != Constants.MOBILE_NO && strEntry != Constants.SIGNUP_TOKEN && strEntry != Constants.SIGNIN_TOKEN &&
                    strEntry != Constants.PREPAID_TOKEN) {
                if (isTokenPresent(token, strEntry, seed)) {
                    array = Base64.decode(entry.getValue().toString(), Base64.DEFAULT);
                    break;//given token exists return encrypted CVV against it
                }
            }
        }
        return array;
    }


    /**
     * This function will decrypt token and check with token
     *
     * @param originalToken  - token from saved card
     * @param encryptedToken - encrypted token from sharedpref
     * @param seed           -  seed used to encrypt the token
     * @return
     */
    public boolean isTokenPresent(String originalToken, String encryptedToken, String seed) {
        boolean isTokenPresent = false;
        try {
            Crypto crypto = new Crypto(
                    new SharedPrefsBackedKeyChain(context),
                    new SystemNativeCryptoLibrary());

            // If library is not loaded do not proceed.
            if (!crypto.isAvailable()) {
                return false;
            }

            Entity myEntity = new Entity(seed);
            byte[] decryptedBytes = crypto.decrypt(Base64.decode(encryptedToken, Base64.DEFAULT), myEntity);
            String decryptedOriginalToekn = new String(decryptedBytes);
            if (decryptedOriginalToekn.equalsIgnoreCase(originalToken)) {
                isTokenPresent = true;
            }
        } catch (KeyChainException e) {
            e.printStackTrace();
        } catch (CryptoInitializationException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        try {

        } catch (Exception e) {
            e.printStackTrace();
        }
        return isTokenPresent;
    }
}
