package org.opencv.chinaredstar.service;

import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.PixelFormat;
import android.hardware.Camera;
import android.os.Binder;
import android.os.Build;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.util.Log;
import android.view.Gravity;
import android.view.Surface;
import android.view.WindowManager;
import android.widget.ImageView;
import android.widget.Toast;

import org.opencv.R;
import org.opencv.android.BaseLoaderCallback;
import org.opencv.android.CameraBridgeViewBase;
import org.opencv.android.JavaCameraView;
import org.opencv.android.LoaderCallbackInterface;
import org.opencv.android.Utils;
import org.opencv.chinaredstar.HxFaceLibrary;
import org.opencv.chinaredstar.bean.HxRecommendBean;
import org.opencv.chinaredstar.service.mvp.IsearchFaceView;
import org.opencv.chinaredstar.service.mvp.SearchFacePresenter;
import org.opencv.chinaredstar.utils.Base64BitmapUtil;
import org.opencv.chinaredstar.utils.DeviceInfo;
import org.opencv.core.Core;
import org.opencv.core.Mat;
import org.opencv.core.MatOfRect;
import org.opencv.core.Rect;
import org.opencv.core.Scalar;
import org.opencv.core.Size;
import org.opencv.imgproc.Imgproc;
import org.opencv.objdetect.CascadeClassifier;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Timer;
import java.util.TimerTask;


public class HxFaceCaptureService extends Service implements CameraBridgeViewBase.CvCameraViewListener2 {

    private static final String TAG = "HxFaceCaptureService";
    private static final Scalar FACE_RECT_COLOR = new Scalar(0, 255, 0, 255);
    public static final int JAVA_DETECTOR = 0;
    public static final int NATIVE_DETECTOR = 1;

    private Mat mRgba;
    private Mat mGray;
    private Mat outPut;
    private Bitmap mBitmap;
    private File mCascadeFile;
    private CascadeClassifier mJavaDetector;
    private DetectionBasedTracker mNativeDetector;
    private JavaCameraView mOpenCvCameraView;
    private ImageView ivTest;

    private int mDetectorType = JAVA_DETECTOR;
//    private String[] mDetectorName;

    private float mRelativeFaceSize = 0.2f;
    private int mAbsoluteFaceSize = 0;

    private WindowManager windowManager;

    private OnHxFaceDataCallBack mOnHxFaceDataCallBack;
    private Timer mTimer;
    private TimerTask mTask;

    private boolean mHasFace;
    private boolean mCanSearch = true;
    private Handler mHandler;
    private SearchFacePresenter mSarchFacePresenter;
    private WindowManager.LayoutParams mLayoutParams;
    private Long time; //锁定检测时间超过5秒，可以开启检测。

    private BaseLoaderCallback mLoaderCallback = new BaseLoaderCallback(this) {
        @Override
        public void onManagerConnected(int status) {
            switch (status) {
                case LoaderCallbackInterface.SUCCESS: {
                    Log.i(TAG, "OpenCV loaded successfully");

                    // Load native library after(!) OpenCV initialization
                    System.loadLibrary("detection_based_tracker");

                    try {
                        // load cascade file from application resources
                        InputStream is = getResources().openRawResource(R.raw.lbpcascade_frontalface);
                        File cascadeDir = getDir("cascade", Context.MODE_PRIVATE);
                        mCascadeFile = new File(cascadeDir, "lbpcascade_frontalface.xml");
                        FileOutputStream os = new FileOutputStream(mCascadeFile);

                        byte[] buffer = new byte[4096];
                        int bytesRead;
                        while ((bytesRead = is.read(buffer)) != -1) {
                            os.write(buffer, 0, bytesRead);
                        }
                        is.close();
                        os.close();

                        mJavaDetector = new CascadeClassifier(mCascadeFile.getAbsolutePath());
                        if (mJavaDetector.empty()) {
                            Log.e(TAG, "Failed to load cascade classifier");
                            mJavaDetector = null;
                        } else
                            Log.i(TAG, "Loaded cascade classifier from " + mCascadeFile.getAbsolutePath());

                        mNativeDetector = new DetectionBasedTracker(mCascadeFile.getAbsolutePath(), 0);

                        cascadeDir.delete();

                    } catch (IOException e) {
                        e.printStackTrace();
                        Log.e(TAG, "Failed to load cascade. Exception thrown: " + e);
                    }

                    mOpenCvCameraView.enableView();
                }
                break;
                default: {
                    super.onManagerConnected(status);
                }
                break;
            }
        }
    };


    public HxFaceCaptureService() {
        Log.i(TAG, "Instantiated new " + this.getClass());
    }

    @Override
    public IBinder onBind(Intent intent) {
        Log.e("HxFaceCaptureService", "onBind");
        return new HxFaceCaptureBinder();
    }

    @Override
    public void onCreate() {
        super.onCreate();
        windowManager = (WindowManager) getSystemService(Context.WINDOW_SERVICE);
        mOpenCvCameraView = new JavaCameraView(this, JavaCameraView.CAMERA_ID_ANY);
        ivTest = new ImageView(this);
        if (Build.VERSION.SDK_INT >= 26) {
            mLayoutParams = new WindowManager.LayoutParams(
                    DeviceInfo.SCREEN_WIDTH, DeviceInfo.SCREEN_WIDTH,
                    WindowManager.LayoutParams.TYPE_APPLICATION_OVERLAY,
                    WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE,
                    PixelFormat.TRANSLUCENT);
        } else {
            mLayoutParams = new WindowManager.LayoutParams(
                    DeviceInfo.SCREEN_WIDTH, DeviceInfo.SCREEN_WIDTH,
                    WindowManager.LayoutParams.TYPE_SYSTEM_ALERT,
                    WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE,
                    PixelFormat.TRANSLUCENT);
        }

        mLayoutParams.gravity = Gravity.LEFT | Gravity.TOP;
        windowManager.addView(mOpenCvCameraView, mLayoutParams);
        mOpenCvCameraView.setVisibility(CameraBridgeViewBase.VISIBLE);
        mOpenCvCameraView.setPreviewSize(DeviceInfo.SCREEN_WIDTH, DeviceInfo.SCREEN_HEIGHT);
        mOpenCvCameraView.setCvCameraViewListener(this);
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        initOpenCV();
        initTask();
        mSarchFacePresenter = new SearchFacePresenter(this, new IsearchFaceView() {
            @Override
            public void onSearchSuccess(Bitmap bitmap, HxRecommendBean hxRecommendBean) {
                if (hxRecommendBean != null && mOnHxFaceDataCallBack != null) {
                    mOnHxFaceDataCallBack.HxFaceData(bitmap, hxRecommendBean);
                }
            }

            @Override
            public void onCanSearch(boolean canSearch) {
                mCanSearch = canSearch;
            }
        });
        mHandler = new Handler(new Handler.Callback() {
            @Override
            public boolean handleMessage(Message message) {
                mSarchFacePresenter.searchFace(mBitmap, Base64BitmapUtil.bitmapToBase64(mBitmap));
                return true;
            }
        });
        Log.e("HxFaceCaptureService", "onStartCommand");
        return super.onStartCommand(intent, flags, startId);

    }

    private void initTask() {
        mTimer = new Timer();
        mTask = new TimerTask() {
            @Override
            public void run() {
                if (mHasFace && mBitmap != null && mCanSearch && mHandler != null) {
                    mHandler.sendEmptyMessage(1);
                    mCanSearch = false;
                    mHasFace = false;
                    time = System.currentTimeMillis();
                } else {
                    if (!mCanSearch && System.currentTimeMillis() - time > 1000 * 5) {
                        mCanSearch = true;
                    }
                }
            }
        };
        mTimer.schedule(mTask, 0, 2000);
    }

    private void stopTimer() {

        if (mTimer != null) {
            mTimer.cancel();
            mTimer = null;
        }

        if (mTask != null) {
            mTask.cancel();
            mTask = null;
        }
    }

    /*初始化OpenCv*/
    private void initOpenCV() {
        try {
            Log.d(TAG, "OpenCV library found inside package. Using it!");
            mLoaderCallback.onManagerConnected(LoaderCallbackInterface.SUCCESS);
        } catch (Exception e) {
            e.printStackTrace();
            Toast.makeText(HxFaceCaptureService.this, "人脸识别初始化失败！", Toast.LENGTH_SHORT).show();
        }
    }

    @Override
    public boolean onUnbind(Intent intent) {
        if (mHandler != null) {
            mHandler.removeCallbacksAndMessages(this);
            mHandler = null;
        }
        stopTimer();
        if (windowManager != null && mOpenCvCameraView != null) {
            mOpenCvCameraView.disableView();
            windowManager.removeViewImmediate(mOpenCvCameraView);
        }
        stopSelf();
        return super.onUnbind(intent);

    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (mOpenCvCameraView != null) {
            mOpenCvCameraView.disableView();
        }
    }

    @Override
    public void onCameraViewStarted(int width, int height) {
        mGray = new Mat();
        mRgba = new Mat();
    }

    @Override
    public void onCameraViewStopped() {
        mGray.release();
        mRgba.release();
        if (outPut != null) {
            outPut.release();
        }
    }

    @Override
    public Mat onCameraFrame(CameraBridgeViewBase.CvCameraViewFrame inputFrame) {
        try {
            synchronized (HxFaceCaptureService.this) {
                mRgba = inputFrame.rgba();
                mGray = inputFrame.gray();
                if (mAbsoluteFaceSize == 0) {
                    int height = mGray.rows();
                    if (Math.round(height * mRelativeFaceSize) > 0) {
                        mAbsoluteFaceSize = Math.round(height * mRelativeFaceSize);
                    }
                    mNativeDetector.setMinFaceSize(mAbsoluteFaceSize);
                }

                MatOfRect faces = new MatOfRect();
//                mRotateMat = Imgproc.getRotationMatrix2D(new Point(mGray.rows() / 2, mGray.cols() / 2), getPreviewRotateDegree(), 1);
//                mRotateMat2 = Imgproc.getRotationMatrix2D(new Point(mRgba.rows() / 2, mRgba.cols() / 2), getPreviewRotateDegree(), 1);
//                Imgproc.warpAffine(mGray, mGray, mRotateMat, mGray.size());
//                Imgproc.warpAffine(mRgba, mRgba, mRotateMat2, mRgba.size());
                if (mDetectorType == JAVA_DETECTOR) {
                    if (mJavaDetector != null)
                        mJavaDetector.detectMultiScale(mGray, faces, 1.1, 6, 2, // TODO: objdetect.CV_HAAR_SCALE_IMAGE
                                new Size(mAbsoluteFaceSize, mAbsoluteFaceSize), new Size());
                } else if (mDetectorType == NATIVE_DETECTOR) {
                    if (mNativeDetector != null)
                        mNativeDetector.detect(mGray, faces);
                }
                Rect[] facesArray = faces.toArray();
                if (facesArray.length > 0) {
                    if (HxFaceLibrary.isDebug) {
                        Imgproc.rectangle(mRgba, facesArray[0].tl(), facesArray[0].br(), FACE_RECT_COLOR, 3);
                    }
                    if (mCanSearch) {
                        outPut = new Mat(mRgba, facesArray[0]);
                        mBitmap = Bitmap.createBitmap(outPut.cols(), outPut.rows(), Bitmap.Config.ARGB_8888);
                        Utils.matToBitmap(outPut, mBitmap);
                        mHasFace = true;
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            Log.e("HxFaceCaptureService", e.getMessage());
        }

        return mRgba;
    }

    private int getPreviewRotateDegree() {
        int phoneDegree = 0;
        int result = 0;
        //获得手机方向
        int phoneRotate = windowManager.getDefaultDisplay().getOrientation();
        //得到手机的角度
        switch (phoneRotate) {
            case Surface.ROTATION_0:
                phoneDegree = 0;
                break;        //0
            case Surface.ROTATION_90:
                phoneDegree = 90;
                break;      //90
            case Surface.ROTATION_180:
                phoneDegree = 180;
                break;    //180
            case Surface.ROTATION_270:
                phoneDegree = 270;
                break;    //270
        }
        //分别计算前后置摄像头需要旋转的角度
        Camera.CameraInfo cameraInfo = new Camera.CameraInfo();
        if (mOpenCvCameraView != null && !mOpenCvCameraView.IsBackCamera()) {
            Core.flip(mRgba, mRgba, 0);
            Core.flip(mGray, mGray, 0);
            Camera.getCameraInfo(Camera.CameraInfo.CAMERA_FACING_FRONT, cameraInfo);
            result = (cameraInfo.orientation + phoneDegree) % 360;
            result = (360 - result) % 360;
            Log.d(TAG, "front_degree" + result);
        } else {

            Camera.getCameraInfo(Camera.CameraInfo.CAMERA_FACING_BACK, cameraInfo);
            result = 360 - (cameraInfo.orientation - phoneDegree + 360) % 360;
            Log.d(TAG, "back_degree" + result);
        }

        return result;
    }


    public void setOnHxFaceDataCallBack(OnHxFaceDataCallBack callBack) {
        this.mOnHxFaceDataCallBack = callBack;
    }


    public interface OnHxFaceDataCallBack {
        void HxFaceData(Bitmap bitmap, HxRecommendBean recommendBean);
    }


    public class HxFaceCaptureBinder extends Binder {
        /**
         * 获取当前Service的实例
         */
        public HxFaceCaptureService getService() {
            return HxFaceCaptureService.this;
        }
    }

}
