package com.chinapex.android.monitor.view.statistics;

import com.chinapex.android.monitor.R;
import com.chinapex.android.monitor.bean.ContrastItem;
import com.chinapex.android.monitor.bean.StatisticsBean;
import com.chinapex.android.monitor.bean.request.ContrastDataRequest;
import com.chinapex.android.monitor.bean.response.ContrastDataResponse;
import com.chinapex.android.monitor.changelistener.MonitorListenerController;
import com.chinapex.android.monitor.executor.TaskController;
import com.chinapex.android.monitor.executor.runnable.IUpdateUI;
import com.chinapex.android.monitor.executor.runnable.PullContrastData;
import com.chinapex.android.monitor.global.Constant;
import com.chinapex.android.monitor.global.MonitorCache;
import com.chinapex.android.monitor.utils.ColorUtils;
import com.chinapex.android.monitor.utils.GsonUtils;
import com.chinapex.android.monitor.utils.MLog;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;


/**
 * @author wyhusky
 * @date 2019/1/17
 */
public class StatisticsViewPresenter implements StatisticsViewContract.Presenter, IUpdateUI {

    private static final String TAG = StatisticsViewPresenter.class.getSimpleName();

    private static final int FULL_PROPORTION = 100;

    private StatisticsViewContract.View mStatisticsView;

    public StatisticsViewPresenter(StatisticsViewContract.View view) {
        mStatisticsView = view;
    }

    @Override
    public void init() {

    }

    @Override
    public void loadContrastData() {
        Set<ContrastItem> contrastItems = MonitorCache.getInstance().getContrastItemList();
        if (null == contrastItems || contrastItems.isEmpty()) {
            MLog.e(TAG, "loadContrastData()-> contrastViewPathMD5List is null or empty");
            return;
        }
        Set<String> idSet = new HashSet<>();
        Set<String> pageSet = new HashSet<>();
        for (ContrastItem item : contrastItems) {
            idSet.add(item.getViewPathMD5());
            pageSet.add(item.getPageClassName());
        }
        String[] ids = idSet.toArray(new String[0]);
        String[] pages = pageSet.toArray(new String[0]);

        ContrastDataRequest contrastDataRequest = new ContrastDataRequest();
        contrastDataRequest.setIds(ids);
        contrastDataRequest.setPages(pages);
        contrastDataRequest.setPeriod(0);
        contrastDataRequest.setTerminal(Constant.TERMINAL);
        TaskController.getInstance().submit(new PullContrastData(contrastDataRequest, this));
    }

    @Override
    public void clearContrastItems() {
        List<StatisticsBean> statisticsBeans = MonitorCache.getInstance().getStatisticsBeans();
        if (null == statisticsBeans) {
            MLog.e(TAG, "clearContrastItems() -> StatisticsBeans is null !");
            return;
        }

        statisticsBeans.clear();
        Set<ContrastItem> contrastItems = MonitorCache.getInstance().getContrastItemList();
        if (null == contrastItems) {
            MLog.e(TAG, "loadContrastData()-> contrastViewPathMD5List is null or empty");
            return;
        }

        contrastItems.clear();
        MonitorListenerController.getInstance().notifyComparisonElementsChange();

        mStatisticsView.showEmptyView();
    }

    @Override
    public void deleteContrastItem(int position) {
        Set<ContrastItem> contrastItems = MonitorCache.getInstance().getContrastItemList();
        if (null == contrastItems || contrastItems.isEmpty()) {
            MLog.e(TAG, "loadContrastData()-> contrastViewPathMD5List is null or empty");
            return;
        }

        List<StatisticsBean> statisticsBeans = MonitorCache.getInstance().getStatisticsBeans();
        if (null == statisticsBeans || statisticsBeans.isEmpty()) {
            MLog.e(TAG, "deleteContrastItem() ->  statisticsBeans is null or isEmpty !!!");
            return;
        }

        if (position < 0 || position > statisticsBeans.size() - 1) {
            MLog.e(TAG, "deleteContrastItem()-> argument error, will throw IndexOutOfBoundsException!");
            return;
        }

        StatisticsBean statisticsBean = statisticsBeans.remove(position);
        for (Iterator<ContrastItem> it = contrastItems.iterator(); it.hasNext(); ) {
            ContrastItem item = it.next();
            if (item.getViewPathMD5().equals(statisticsBean.getViewPathMD5())) {
                it.remove();
            }
        }
        MonitorListenerController.getInstance().notifyComparisonElementsChange();

        if (statisticsBeans.isEmpty()) {
            mStatisticsView.showEmptyView();
            return;
        }

        if (Constant.OTHERS_VIEW_PATH_MD5.equals(statisticsBean.getViewPathMD5())) {
            long sum = 0;
            for (StatisticsBean bean : statisticsBeans) {
                sum += bean.getClickCount();
            }
            for (StatisticsBean bean : statisticsBeans) {
                bean.setClickProportion(bean.getClickCount() / (float) sum * FULL_PROPORTION);
            }
        } else {
            boolean containsOthers = false;
            for (StatisticsBean bean : statisticsBeans) {
                if (Constant.OTHERS_VIEW_PATH_MD5.equals(bean.getViewPathMD5())) {
                    bean.setClickCount(bean.getClickCount() + statisticsBean.getClickCount());
                    bean.setClickProportion(bean.getClickProportion() + statisticsBean.getClickProportion());
                    containsOthers = true;
                }
            }
            if (!containsOthers) {
                long sum = 0;
                for (StatisticsBean bean : statisticsBeans) {
                    sum += bean.getClickCount();
                }
                for (StatisticsBean bean : statisticsBeans) {
                    bean.setClickProportion(bean.getClickCount() / (float) sum * FULL_PROPORTION);
                }
            }
        }
        Collections.sort(statisticsBeans, new Comparator<StatisticsBean>() {
            @Override
            public int compare(StatisticsBean statisticsBean, StatisticsBean t1) {
                return statisticsBean.getClickProportion() > t1.getClickProportion() ? -1 :
                        statisticsBean.getClickCount() == t1.getClickCount() ? 0 : 1;
            }
        });

        mStatisticsView.refreshViews();
    }

    @Override
    public void updateUI(boolean isSuccess, String result) {
        if (!isSuccess) {
            MLog.e(TAG, "updateUI() -> false!");
            return;
        }

        ContrastDataResponse contrastDataResponse = GsonUtils.json2Bean(result, ContrastDataResponse.class);
        if (null == contrastDataResponse) {
            MLog.e(TAG, "updateUI()-> contrastDataResponse is null!");
            return;
        }

        int status = contrastDataResponse.getStatus();
        if (Constant.RESPONSE_OK != status) {
            MLog.e(TAG, "updateUI()-> error,status=" + status);
            return;
        }

        ContrastDataResponse.Data data = contrastDataResponse.getData();
        if (null == data) {
            MLog.e(TAG, "updateUI()-> data is null!");
            return;
        }

        ContrastDataResponse.Data.Element[] elements = data.getElements();
        if (null == elements || elements.length == 0) {
            MLog.e(TAG, "updateUI()-> elements is null or empty");
            return;
        }

        List<StatisticsBean> beans = MonitorCache.getInstance().getStatisticsBeans();
        if (null == beans) {
            MLog.e(TAG, "updateUI()-> beans is null");
            return;
        }
        beans.clear();

        long sum = 0;
        float sumProportion = 0;
        long total = data.getTotal();
        for (ContrastDataResponse.Data.Element element : elements) {
            if (null == element) {
                MLog.e(TAG, "testData()-> element is null!");
                continue;
            }

            long count = element.getCount();
            float clickProportion = (float) count / total * FULL_PROPORTION;
            if (total == 0) {
                clickProportion = 0f;
            }
            sum += count;
            sumProportion += clickProportion;

            StatisticsBean statisticsBean = new StatisticsBean();
            statisticsBean.setClickCount(count);
            statisticsBean.setViewPathMD5(element.getId());
            statisticsBean.setEventLabel(element.getAlias());
            statisticsBean.setColor(ColorUtils.generateColor());
            statisticsBean.setClickProportion(clickProportion);
            statisticsBean.setSnapshotUrl(element.getSnapshot());
            beans.add(statisticsBean);
        }
        if (sum < total) {
            StatisticsBean statisticsBean = new StatisticsBean();
            statisticsBean.setViewPathMD5(Constant.OTHERS_VIEW_PATH_MD5);
            statisticsBean.setClickCount(total - sum);
            statisticsBean.setColor(ColorUtils.generateColor());
            statisticsBean.setEventLabel(MonitorCache.getInstance().getContext().getResources().getString(R.string.others));
            statisticsBean.setClickProportion(FULL_PROPORTION - sumProportion);
            beans.add(statisticsBean);
        }
        ColorUtils.reset();

        Collections.sort(beans, new Comparator<StatisticsBean>() {
            @Override
            public int compare(StatisticsBean statisticsBean, StatisticsBean t1) {
                return statisticsBean.getClickProportion() > t1.getClickProportion() ? -1 :
                        statisticsBean.getClickCount() == t1.getClickCount() ? 0 : 1;
            }
        });

        TaskController.getInstance().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                mStatisticsView.refreshViews();
            }
        });

    }
}
