package com.chinapex.android.monitor.view.statistics;

import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Color;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.view.View;
import android.widget.AdapterView;
import android.widget.Button;
import android.widget.ExpandableListView;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.ScrollView;

import com.chinapex.android.monitor.R;
import com.chinapex.android.monitor.bean.StatisticsBean;
import com.chinapex.android.monitor.global.MonitorCache;
import com.chinapex.android.monitor.utils.MLog;
import com.chinapex.android.monitor.view.BaseFloatingView;
import com.chinapex.android.monitor.view.FloatingViewManager;
import com.chinapex.android.monitor.view.IFloatingView;
import com.chinapex.android.monitor.view.adapter.ColumnListAdapter;
import com.chinapex.android.monitor.view.adapter.StatisticsListAdapter;
import com.chinapex.android.monitor.view.charts.PieChart;
import com.chinapex.android.monitor.view.charts.PieForm;

import java.util.List;

/**
 * @author wyhusky
 * @date 2019/1/2
 */
public class StatisticsView extends BaseFloatingView implements StatisticsViewContract.View, View.OnClickListener,
        StatisticsListAdapter.OnItemClickListener {

    private static final String TAG = StatisticsView.class.getSimpleName();
    private StatisticsViewContract.Presenter mPresenter;
    private PieChart mPieChart;
    private PieForm mPieForm;
    private ScrollView mChartContainer;
    private LinearLayout mListContainer;
    private ExpandableListView mStatisticsListView;
    private ListView mColumnListView;

    private StatisticsListAdapter mListAdapter;
    private ColumnListAdapter mColumnListAdapter;
    private Button mBtnShowList;
    private Button mBtnShowChart;
    private Context mContext;

    public StatisticsView(@NonNull Context context, int windowType) {
        super(context, windowType);
        mContext = context;
        inflate(context, R.layout.layout_window_contrast_data, this);
        setPresenter(new StatisticsViewPresenter(this));
        initView();
    }

    private void initView() {
        mPieChart = findViewById(R.id.pie_chart);
        mPieForm = findViewById(R.id.pie_form);

        mColumnListView = findViewById(R.id.lv_count_rank);
        mStatisticsListView = findViewById(R.id.statistics_list);
        mListContainer = findViewById(R.id.statistics_list_container);
        mChartContainer = findViewById(R.id.statistics_chart_container);
        mBtnShowList = findViewById(R.id.btn_show_list);
        mBtnShowChart = findViewById(R.id.btn_show_chart);


        mListAdapter = new StatisticsListAdapter(MonitorCache.getInstance().getStatisticsBeans());
        mStatisticsListView.setAdapter(mListAdapter);
        mListAdapter.setOnItemClickListener(this);
        mStatisticsListView.setOnGroupExpandListener(new ExpandableListView.OnGroupExpandListener() {
            @Override
            public void onGroupExpand(int groupPosition) {
                int count = mListAdapter.getGroupCount();
                for (int i = 0; i < count; i++) {
                    if (i != groupPosition) {
                        mStatisticsListView.collapseGroup(i);
                    }
                }
            }
        });

        mColumnListAdapter = new ColumnListAdapter(MonitorCache.getInstance().getStatisticsBeans());
        mColumnListView.setAdapter(mColumnListAdapter);
        mColumnListView.setOnItemClickListener(new AdapterView.OnItemClickListener() {
            @Override
            public void onItemClick(AdapterView<?> adapterView, View view, int i, long l) {
                mPieChart.popChosenPie(i);
                mPieForm.setEventColor(MonitorCache.getInstance().getStatisticsBeans().get(i).getColor());
                mPieForm.setTvClickCount(MonitorCache.getInstance().getStatisticsBeans().get(i).getClickCount());
                mPieForm.setTvClickPercent(MonitorCache.getInstance().getStatisticsBeans().get(i).getClickProportion());
                String eventLabel = MonitorCache.getInstance().getStatisticsBeans().get(i).getEventLabel();
                mPieForm.setTvEventName(TextUtils.isEmpty(eventLabel) ?
                        MonitorCache.getInstance().getStatisticsBeans().get(i).getViewPathMD5() : eventLabel);
            }
        });

        mBtnShowChart.setOnClickListener(this);
        mBtnShowList.setOnClickListener(this);
        findViewById(R.id.btn_clear_list).setOnClickListener(this);
        findViewById(R.id.statistics_back).setOnClickListener(this);

        mBtnShowChart.setTextColor(Color.BLACK);
        mBtnShowList.setTextColor(getResources().getColor(R.color.c_548FA8));
        mChartContainer.setVisibility(View.GONE);
        mListContainer.setVisibility(View.VISIBLE);

        loadData();
    }

    private void loadData() {
        mPresenter.loadContrastData();
    }

    @Override
    protected void onConfigurationChanged(Configuration newConfig) {
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE
                && null != mChartContainer && mChartContainer.getVisibility() == VISIBLE) {
            postDelayed(new Runnable() {
                @Override
                public void run() {
                    mChartContainer.smoothScrollTo(0, 0);
                }
            }, 200);
        }
        super.onConfigurationChanged(newConfig);
    }

    @Override
    public void onClick(View v) {
        int viewId = v.getId();

        if (viewId == R.id.btn_clear_list) {
            mPresenter.clearContrastItems();
        } else if (viewId == R.id.btn_show_chart) {
            if (mChartContainer.getVisibility() == View.VISIBLE && mListContainer.getVisibility() == View.GONE) {
                MLog.i(TAG, "onClick() -> statistics chart is already showing  !!!");
                return;
            }
            mChartContainer.smoothScrollTo(0, 0);
            mBtnShowChart.setTextColor(getResources().getColor(R.color.c_548FA8));
            mBtnShowList.setTextColor(Color.BLACK);
            mChartContainer.setVisibility(View.VISIBLE);
            mListContainer.setVisibility(View.GONE);
        } else if (viewId == R.id.btn_show_list) {
            if (mChartContainer.getVisibility() == View.GONE && mListContainer.getVisibility() == View.VISIBLE) {
                MLog.i(TAG, "onClick() -> statistics list is already showing  !!!");
                return;
            }
            mBtnShowChart.setTextColor(Color.BLACK);
            mBtnShowList.setTextColor(getResources().getColor(R.color.c_548FA8));
            mChartContainer.setVisibility(View.GONE);
            mListContainer.setVisibility(View.VISIBLE);
        } else if (viewId == R.id.statistics_back) {
            FloatingViewManager.getInstance().dismissFloatingView(IFloatingView.WINDOW_CONTRAST_DATA);
        }

    }

    @Override
    public void onItemClick(int position) {
        mPresenter.deleteContrastItem(position);
    }

    @Override
    public void refreshViews() {
        mListAdapter.notifyDataSetChanged();
        mColumnListAdapter.notifyDataSetChanged();
        updatePieChart();
    }

    @Override
    public void showEmptyView() {
        mListAdapter.notifyDataSetChanged();
        mColumnListAdapter.notifyDataSetChanged();
        mPieForm.setEventColor(Color.LTGRAY);
        mPieForm.setTvEventName("");
        mPieForm.setTvClickPercent(0);
        mPieForm.setTvClickCount(0);
        updatePieChart();
    }

    private void updatePieChart() {
        List<StatisticsBean> statisticsBeans = MonitorCache.getInstance().getStatisticsBeans();
        if (null == statisticsBeans) {
            MLog.e(TAG, "setPieData() -> statisticsBeans is null !");
            return;
        }

        long[] data = new long[statisticsBeans.size()];
        int[] colorArray = new int[statisticsBeans.size()];
        for (int i = 0; i < data.length; i++) {
            data[i] = statisticsBeans.get(i).getClickCount();
            colorArray[i] = statisticsBeans.get(i).getColor();
        }
        mPieChart.setData(data, colorArray);
    }


    @Override
    public void setPresenter(Object presenter) {
        this.mPresenter = (StatisticsViewContract.Presenter) presenter;
    }
}
