package com.chinapex.android.monitor.view.event;


import android.app.Activity;
import android.graphics.Bitmap;
import android.text.TextUtils;
import android.view.View;

import com.chinapex.android.monitor.R;
import com.chinapex.android.monitor.bean.ConfigBean;
import com.chinapex.android.monitor.bean.request.EventDataRequest;
import com.chinapex.android.monitor.bean.response.EventDataResponse;
import com.chinapex.android.monitor.executor.TaskController;
import com.chinapex.android.monitor.executor.runnable.IUpdateUI;
import com.chinapex.android.monitor.executor.runnable.PullEventData;
import com.chinapex.android.monitor.executor.runnable.SaveViewSnapshot;
import com.chinapex.android.monitor.executor.runnable.WriteClickToSp;
import com.chinapex.android.monitor.executor.runnable.WriteElemPvToSp;
import com.chinapex.android.monitor.executor.runnable.WriteExpandablePvToSp;
import com.chinapex.android.monitor.executor.runnable.WriteListPvToSp;
import com.chinapex.android.monitor.executor.runnable.WriteListToSp;
import com.chinapex.android.monitor.executor.runnable.WritePvToSp;
import com.chinapex.android.monitor.executor.runnable.WriteRecyclerPvToSp;
import com.chinapex.android.monitor.global.Constant;
import com.chinapex.android.monitor.global.MonitorCache;
import com.chinapex.android.monitor.utils.GsonUtils;
import com.chinapex.android.monitor.utils.MD5Utils;
import com.chinapex.android.monitor.utils.MLog;
import com.chinapex.android.monitor.utils.SnapshotUtil;
import com.chinapex.android.monitor.utils.SystemUtils;
import com.chinapex.android.monitor.view.FloatingViewManager;
import com.chinapex.android.monitor.view.IFloatingView;
import com.chinapex.android.monitor.view.snapshot.SnapshotView;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;

/**
 * @author wyhusky
 * @date 2018/12/28
 */
public class DefineEventPresenter implements DefineEventContract.Presenter, IUpdateUI {

    private static final String TAG = DefineEventPresenter.class.getSimpleName();
    private DefineEventContract.View mDefineEventView;
    private ConfigBean.PvBean mPvBean;
    private ConfigBean.ClickBean mClickBean;
    private ConfigBean.ListBean mListBean;
    private Bitmap mViewSnapshot;

    public DefineEventPresenter(DefineEventContract.View view) {
        this.mDefineEventView = view;
        mPvBean = new ConfigBean.PvBean();
        mClickBean = new ConfigBean.ClickBean();
        mListBean = new ConfigBean.ListBean();
    }

    @Override
    public void loadPvData() {
        Map<Integer, Stack<Map<String, Boolean>>> tasks = MonitorCache.getInstance().getTasks();
        Map<Integer, Integer> top = MonitorCache.getInstance().getTop();
        int[] foregroundTask = MonitorCache.getInstance().getForegroundTask();
        if (null == tasks) {
            MLog.e(TAG, "loadPvData()-> tasks is null");
            return;
        }
        if (null == top) {
            MLog.e(TAG, "loadPvData()-> top is null");
            return;
        }
        if (null == foregroundTask) {
            MLog.e(TAG, "loadPvData()-> foregroundTask is null");
            return;
        }
        int taskId = foregroundTask[0];
        if (!tasks.containsKey(taskId)) {
            MLog.e(TAG, "loadPvData()-> tasks do not contain task[" + taskId + "]");
            return;
        }
        Stack<Map<String, Boolean>> task = tasks.get(taskId);
        if (null == task || task.isEmpty()) {
            MLog.e(TAG, "loadPvData()-> task[" + taskId + "] is empty");
            return;
        }

        List<String> pages = new ArrayList<>();
        String reference = null;
        if (!top.containsKey(taskId)) {
            MLog.e(TAG, "loadPvData()-> top does not contain top[" + taskId + "]");
            return;
        }
        Integer topActivity = top.get(taskId);
        if (null == topActivity) {
            MLog.e(TAG, "loadPvData()-> top has null element");
            return;
        }
        for (int i = task.size() - 1; i >= topActivity; i--) {
            Map<String, Boolean> map = task.get(i);
            if (null == map) {
                MLog.e(TAG, "loadPvData()-> map should not be null");
                return;
            }
            String[] key = map.keySet().toArray(new String[1]);
            if (key.length != 1) {
                MLog.e(TAG, "loadPvData()-> keyset size can only be 1");
                return;
            }
            pages.add(key[0]);
            Boolean isActivity = map.get(key[0]);
            if (null == isActivity) {
                MLog.e(TAG, "loadPvData()-> map has null element");
                return;
            }
            if (isActivity) {
                reference = key[0];
            }
        }
        if (pages.isEmpty()) {
            MLog.e(TAG, "loadPvData()-> pages is empty!");
            return;
        }
        if (null == reference) {
            MLog.e(TAG, "loadPvData()-> reference is null!");
            return;
        }

        mDefineEventView.loadPagePath(pages);
        mDefineEventView.showReference(reference);

        Activity activity = SystemUtils.getCurrentActivity();
        if (null == activity) {
            MLog.e(TAG, "loadPvData()-> fail to get top activity");
            return;
        }
        View decorView = activity.getWindow().getDecorView();
        mViewSnapshot = SnapshotUtil.takeSnapshot(decorView);
    }

    @Override
    public void updatePvData(String viewPath) {
        requestEventData(MD5Utils.getMD5(viewPath), Constant.TYPE_PAGE, Constant.X_AXIS_COUNTS);
    }

    @Override
    public void loadClickData(View view, String viewPath, String pageClass, String findPath) {
        if (null == viewPath) {
            MLog.e(TAG, "loadClickData()-> viewPath is null!");
            return;
        }

        if (null == pageClass) {
            MLog.e(TAG, "loadClickData()-> pageClass is null!");
            return;
        }

        if (null == findPath) {
            MLog.e(TAG, "loadClickData()-> findPath is null!");
            return;
        }

        Map<String, ConfigBean.PvBean> pvBeans = MonitorCache.getInstance().getPvBeans();
        if (null == pvBeans) {
            MLog.e(TAG, "loadClickData()-> pvBeans is null!");
            return;
        }
        String definedPage = null;
        for (ConfigBean.PvBean pvBean : pvBeans.values()) {
            if (null == pvBean) {
                MLog.e(TAG, "loadClickData()-> pvBean is null!");
                continue;
            }
            if (null == pvBean.getViewPath()) {
                MLog.e(TAG, "loadClickData()-> viewPath of pvBean is null!");
                continue;
            }
            if (pageClass.equals(pvBean.getViewPath())) {
                definedPage = pvBean.getViewPath();
                break;
            }
        }
        if (TextUtils.isEmpty(definedPage)) {
            MLog.e(TAG, "loadClickData()-> no defined page!");
            mDefineEventView.showToast(R.string.defined_page_hint);
            dismiss();
            return;
        }

        Map<String, ConfigBean.ClickBean> clickBeans = MonitorCache.getInstance().getClickBeans();
        if (null == clickBeans) {
            MLog.e(TAG, "loadClickData()-> clickBeans is null!");
            return;
        }

        String viewPathMd5 = MD5Utils.getMD5(viewPath);
        ConfigBean.ClickBean clickBean = clickBeans.get(viewPathMd5);
        if (null != clickBean) {
            mDefineEventView.showAlias(clickBean.getAlias());
        }

        mViewSnapshot = SnapshotUtil.takeSnapshot(view);

        mClickBean.setDefinedPage(definedPage);
        mClickBean.setPageClassName(pageClass);
        mClickBean.setViewPath(viewPath);
        mClickBean.setFindPath(findPath);
        mDefineEventView.showViewPath(viewPath);
        mDefineEventView.showReference(definedPage);
        requestEventData(viewPathMd5, Constant.TYPE_ELEMENT, Constant.X_AXIS_COUNTS);
    }

    @Override
    public void loadListItemClickData(View view, String viewPath, String pageClass, String itemPath, String findPath, int listType) {
        if (null == viewPath) {
            MLog.e(TAG, "loadListItemClickData()-> viewPath is null!");
            return;
        }
        if (null == pageClass) {
            MLog.e(TAG, "loadListItemClickData()-> pageClass is null!");
            return;
        }
        if (null == itemPath) {
            MLog.e(TAG, "loadListItemClickData()-> itemPath is null!");
            return;
        }
        Map<String, ConfigBean.PvBean> pvMap = MonitorCache.getInstance().getPvBeans();
        if (null == pvMap) {
            MLog.e(TAG, "loadListItemClickData()-> pvMap is null!");
            return;
        }
        String definedPage = null;
        for (ConfigBean.PvBean pvBean : pvMap.values()) {
            if (null == pvBean) {
                MLog.e(TAG, "loadListItemClickData()-> pvBean is null!");
                continue;
            }
            if (null == pvBean.getViewPath()) {
                MLog.e(TAG, "loadListItemClickData()-> viewPath of pvBean is null!");
                continue;
            }
            if (pageClass.equals(pvBean.getViewPath())) {
                definedPage = pvBean.getViewPath();
                break;
            }
        }
        if (TextUtils.isEmpty(definedPage)) {
            MLog.e(TAG, "loadListItemClickData()-> no defined page!");
            mDefineEventView.showToast(R.string.defined_page_hint);
            dismiss();
            return;
        }

        Map<String, ConfigBean.ListBean> listBeans = MonitorCache.getInstance().getListBeans();
        if (null == listBeans) {
            MLog.e(TAG, "loadListItemClickData()-> listBeans is null!");
            return;
        }

        String itemMd5 = MD5Utils.getMD5(itemPath);
        ConfigBean.ListBean listBean = listBeans.get(MD5Utils.getMD5(viewPath));
        if (null != listBean) {
            Map<String, String> itemAliases = listBean.getItemAliases();
            if (null != itemAliases) {
                mDefineEventView.showAlias(itemAliases.get(itemMd5));
            }
        }

        mViewSnapshot = SnapshotUtil.takeSnapshot(view);

        mListBean.setDefinedPage(definedPage);
        mListBean.setPageClassName(pageClass);
        mListBean.setViewPath(viewPath);
        mListBean.setFindPath(findPath);
        mListBean.setListType(listType);

        mDefineEventView.showViewPath(itemPath);
        mDefineEventView.showReference(definedPage);
        requestEventData(itemMd5, Constant.TYPE_ELEMENT, Constant.X_AXIS_COUNTS);
    }

    @Override
    public void saveClickEvent(String alias) {
        mClickBean.setAlias(alias);
        if (TextUtils.isEmpty(mClickBean.getViewPath())) {
            MLog.e(TAG, "saveClickEvent()-> viewPath is empty, save failed");
            return;
        }

        if (TextUtils.isEmpty(mClickBean.getDefinedPage())) {
            MLog.e(TAG, "saveClickEvent()-> defined page is empty, save failed");
            return;
        }

        String mClickPathMD5 = MD5Utils.getMD5(mClickBean.getViewPath());
        MonitorCache.getInstance().putClickBeans(mClickPathMD5, mClickBean);

        String pageClassNameMD5 = MD5Utils.getMD5(mClickBean.getPageClassName());
        MonitorCache.getInstance().putElemPvBeans(pageClassNameMD5, mClickPathMD5, mClickBean.getFindPath());

        TaskController.getInstance().submit(new WriteClickToSp());
        TaskController.getInstance().submit(new WriteElemPvToSp());
        TaskController.getInstance().submit(new SaveViewSnapshot(mClickPathMD5, mViewSnapshot));

        dismiss();
    }

    @Override
    public void saveListItemClickEvent(String alias, String itemPath) {
        String itemPathMD5 = MD5Utils.getMD5(itemPath);
        boolean isListExist = false;
        Map<String, ConfigBean.ListBean> listMap = MonitorCache.getInstance().getListBeans();
        if (null == listMap) {
            MLog.e(TAG, "saveListItemClickEvent()-> listMap is null");
            return;
        }
        for (ConfigBean.ListBean bean : listMap.values()) {
            if (mListBean.getViewPath().equals(bean.getViewPath())) {
                isListExist = true;
                break;
            }
        }

        if (isListExist) {
            String listBeanKey = MD5Utils.getMD5(mListBean.getViewPath());
            ConfigBean.ListBean bean = listMap.get(listBeanKey);
            if (null == bean) {
                MLog.e(TAG, "saveListItemClickEvent()-> bean is null!");
                return;
            }

            Map<String, String> itemMap = bean.getItemAliases();
            if (null == itemMap) {
                itemMap = new HashMap<>();
            }
            itemMap.put(itemPathMD5, alias);
        } else {
            if (TextUtils.isEmpty(mListBean.getDefinedPage())) {
                MLog.e(TAG, "saveListItemClickEvent()-> defined page is empty, save failed");
                return;
            }
            if (TextUtils.isEmpty(mListBean.getViewPath())) {
                MLog.e(TAG, "saveListItemClickEvent()-> viewPath is empty, save failed");
            }

            Map<String, String> itemMap = new HashMap<>();
            itemMap.put(itemPathMD5, alias);
            mListBean.setItemAliases(itemMap);

            String viewPathMD5 = MD5Utils.getMD5(mListBean.getViewPath());
            MonitorCache.getInstance().putListBeans(viewPathMD5, mListBean);

            String findPath = mListBean.getFindPath();
            String pageClassNameMD5 = MD5Utils.getMD5(mListBean.getPageClassName());

            switch (mListBean.getListType()) {
                case Constant.LIST_CLICK_TYPE_LIST:
                    MonitorCache.getInstance().putListPvBeans(pageClassNameMD5, viewPathMD5, findPath);
                    TaskController.getInstance().submit(new WriteListPvToSp());
                    break;
                case Constant.LIST_CLICK_TYPE_RECYCLER:
                    MonitorCache.getInstance().putRecyclerPvBeans(pageClassNameMD5, viewPathMD5, findPath);
                    TaskController.getInstance().submit(new WriteRecyclerPvToSp());
                    break;
                case Constant.LIST_CLICK_TYPE_EXPANDABLE:
                    MonitorCache.getInstance().putExpandablePvBeans(pageClassNameMD5, viewPathMD5, findPath);
                    TaskController.getInstance().submit(new WriteExpandablePvToSp());
                    break;
                default:
                    break;
            }
        }

        TaskController.getInstance().submit(new WriteListToSp());
        TaskController.getInstance().submit(new SaveViewSnapshot(itemPathMD5, mViewSnapshot));

        dismiss();
    }

    @Override
    public void savePageViewEvent(String alias, String definedPage) {
        mPvBean.setAlias(alias);
        mPvBean.setViewPath(definedPage);

        if (TextUtils.isEmpty(mPvBean.getViewPath())) {
            MLog.e(TAG, "savePageViewEvent()-> viewPath is empty, save failed");
            return;
        }

        String mPvMD5 = MD5Utils.getMD5(definedPage);
        MonitorCache.getInstance().putPvBeans(mPvMD5, mPvBean);
        TaskController.getInstance().submit(new WritePvToSp());
        TaskController.getInstance().submit(new SaveViewSnapshot(mPvMD5, mViewSnapshot));
        dismiss();
    }

    @Override
    public void dismiss() {
        mListBean = null;
        mClickBean = null;
        mPvBean = null;
        mViewSnapshot = null;
        FloatingViewManager.getInstance().dismissFloatingView(mDefineEventView.getViewType());
    }

    @Override
    public void init() {

    }

    private void requestEventData(String id, String type, int counts) {
        EventDataRequest eventDataRequest = new EventDataRequest();
        eventDataRequest.setId(id);
        eventDataRequest.setType(type);
        eventDataRequest.setCounts(counts);
        eventDataRequest.setTerminal(Constant.TERMINAL);
        eventDataRequest.setUnit(Constant.TIME_UNIT);
        TaskController.getInstance().submit(new PullEventData(eventDataRequest, this));
    }


    @Override
    public void updateUI(boolean isSuccess, String result) {
        if (isSuccess) {
            EventDataResponse eventDataResponse = GsonUtils.json2Bean(result, EventDataResponse.class);
            if (null == eventDataResponse) {
                MLog.e(TAG, "updateUI()-> eventDataResponse");
                return;
            }

            int status = eventDataResponse.getStatus();
            if (Constant.RESPONSE_OK != status) {
                MLog.e(TAG, "updateUI()-> error,status=" + status);
                return;
            }

            EventDataResponse.Data[] datas = eventDataResponse.getData();
            if (null == datas || datas.length == 0) {
                MLog.e(TAG, "updateUI()-> datas is null or empty");
                return;
            }

            int dateLength = datas.length;
            final long[] date = new long[dateLength];
            final long[] click = new long[dateLength];
            final long[] pv = new long[dateLength];
            for (int i = 0; i < dateLength; i++) {
                date[i] = datas[i].getDate();
                click[i] = datas[i].getCount();
                pv[i] = datas[i].getPv();
            }

            TaskController.getInstance().runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    mDefineEventView.showLineChart(date, click, pv);
                }
            });
        }
    }

    @Override
    public void showSnapShot() {
        FloatingViewManager.getInstance().showFloatingView(IFloatingView.WINDOW_SNAPSHOT);
        SnapshotView snapshotView =
                (SnapshotView) FloatingViewManager.getInstance().getFloatingView(IFloatingView.WINDOW_SNAPSHOT);
        if (null == snapshotView) {
            MLog.e(TAG, "showSnapshot()-> snapshotView is null");
            return;
        }
        snapshotView.loadImage(mViewSnapshot);
    }
}
