package com.chinapex.android.monitor.view.upload;

import android.content.Context;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.view.View;
import android.widget.ExpandableListView;

import com.chinapex.android.monitor.R;
import com.chinapex.android.monitor.bean.ConfigBean;
import com.chinapex.android.monitor.bean.StatisticsBean;
import com.chinapex.android.monitor.bean.UploadViewBean;
import com.chinapex.android.monitor.changelistener.IDownloadConfigCallback;
import com.chinapex.android.monitor.changelistener.MonitorListenerController;
import com.chinapex.android.monitor.executor.TaskController;
import com.chinapex.android.monitor.executor.runnable.WriteClickToSp;
import com.chinapex.android.monitor.executor.runnable.WriteElemPvToSp;
import com.chinapex.android.monitor.executor.runnable.WriteExpandablePvToSp;
import com.chinapex.android.monitor.executor.runnable.WriteListPvToSp;
import com.chinapex.android.monitor.executor.runnable.WriteListToSp;
import com.chinapex.android.monitor.executor.runnable.WritePvToSp;
import com.chinapex.android.monitor.executor.runnable.WriteRecyclerPvToSp;
import com.chinapex.android.monitor.global.Constant;
import com.chinapex.android.monitor.global.MonitorCache;
import com.chinapex.android.monitor.utils.MD5Utils;
import com.chinapex.android.monitor.utils.MLog;
import com.chinapex.android.monitor.view.BaseFloatingView;
import com.chinapex.android.monitor.view.FloatingViewManager;
import com.chinapex.android.monitor.view.IFloatingView;
import com.chinapex.android.monitor.view.adapter.UploadListAdapter;
import com.chinapex.android.monitor.view.loading.LoadingView;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author : Seven Qiu
 * @date : 2019/1/8
 */
public class UploadView extends BaseFloatingView implements UploadViewContract.View, View.OnClickListener,
        UploadListAdapter.OnItemClickListener, IDownloadConfigCallback {

    private static final String TAG = UploadView.class.getSimpleName();

    private ExpandableListView mExpandableListView;

    private UploadListAdapter mListAdapter;
    private List<UploadViewBean> mUploadViewBeans;
    private Map<String, ConfigBean.ClickBean> mClickBeans;
    private Map<String, Map<String, String>> mElemPvBeans;
    private Map<String, Map<String, String>> mRecyclerPvBeans;
    private Map<String, Map<String, String>> mListPvBeans;
    private Map<String, Map<String, String>> mExpandablePvBeans;
    private Map<String, ConfigBean.PvBean> mPvBeans;
    private Map<String, ConfigBean.ListBean> mListBeans;
    private UploadViewContract.Presenter mUploadViewPresenter;

    public UploadView(@NonNull Context context, int windowType) {
        super(context, windowType);
        inflate(context, R.layout.upload_view_layout, this);
        setPresenter(new UploadViewPresenter(this));

        initView();
        initData();
    }

    private void initView() {
        mExpandableListView = findViewById(R.id.upload_event_list);

        mUploadViewBeans = getUploadViewBeans();
        mListAdapter = new UploadListAdapter(mUploadViewBeans);
        mExpandableListView.setAdapter(mListAdapter);
        mListAdapter.setOnItemClickListener(this);

        mExpandableListView.setOnGroupExpandListener(new ExpandableListView.OnGroupExpandListener() {
            @Override
            public void onGroupExpand(int groupPosition) {
                int count = mListAdapter.getGroupCount();
                for (int i = 0; i < count; i++) {
                    if (i != groupPosition) {
                        mExpandableListView.collapseGroup(i);
                    }
                }
            }
        });

        findViewById(R.id.btn_upload).setOnClickListener(this);
        findViewById(R.id.btn_cancel_upload).setOnClickListener(this);
        findViewById(R.id.btn_import_config).setOnClickListener(this);
    }


    private void initData() {
        List<StatisticsBean> statisticsBeans = MonitorCache.getInstance().getStatisticsBeans();

        if (null == statisticsBeans) {
            MonitorCache.getInstance().setStatisticsBeans(new ArrayList<StatisticsBean>());
        }
        mListAdapter.notifyDataSetChanged();
    }

    @Override
    public void show() {
        super.show();
        MonitorListenerController.getInstance().addIDownloadConfigCallback(this);
    }

    @Override
    public void dismiss() {
        MonitorListenerController.getInstance().removeIDownloadConfigCallback(this);
        super.dismiss();
    }

    @Override
    public void onClick(View v) {
        int viewId = v.getId();
        if (R.id.btn_upload == viewId) {
            mUploadViewPresenter.upload(getUploadViewBeans());
        } else if (R.id.btn_cancel_upload == viewId) {
            FloatingViewManager.getInstance().dismissFloatingView(IFloatingView.WINDOW_UPLOAD);
        } else if (R.id.btn_import_config == viewId) {
            FloatingViewManager.getInstance().showFloatingView(IFloatingView.WINDOW_IMPORT_CONFIG);
        }
    }

    @Override
    public void onItemClick(int position) {
        if (null == mUploadViewBeans) {
            MLog.e(TAG, "onItemClick() -> mUploadViewBeans is null!");
            return;
        }

        UploadViewBean uploadViewBean = mUploadViewBeans.get(position);
        if (null == uploadViewBean) {
            MLog.e(TAG, "onItemClick() -> uploadViewBean is null!");
            return;
        }

        String md5 = uploadViewBean.getMd5();
        if (TextUtils.isEmpty(md5)) {
            MLog.e(TAG, "onItemClick() -> md5 is null or empty!");
            return;
        }

        int eventType = uploadViewBean.getEventType();
        String pageClassName = uploadViewBean.getPageClassName();
        String pageClassNameMd5 = MD5Utils.getMD5(pageClassName);
        switch (eventType) {
            case Constant.DEFINE_EVENT_TYPE_CLICK:
                if (null == mClickBeans) {
                    MLog.e(TAG, "onItemClick() -> mClickBeans is null!");
                    break;
                }

                mClickBeans.remove(md5);
                mUploadViewBeans.remove(position);
                TaskController.getInstance().submit(new WriteClickToSp());

                if (null == mElemPvBeans) {
                    MLog.e(TAG, "onItemClick() -> mElemPvBeans is null!");
                    break;
                }

                if (TextUtils.isEmpty(pageClassNameMd5)) {
                    MLog.e(TAG, "onItemClick() -> pageClassNameMd5 is null or empty!");
                    break;
                }

                Map<String, String> elemPvBeans = mElemPvBeans.get(pageClassNameMd5);
                if (null == elemPvBeans) {
                    MLog.e(TAG, "onItemClick() -> elemPvBeans is null!");
                    break;
                }

                elemPvBeans.remove(md5);
                TaskController.getInstance().submit(new WriteElemPvToSp());
                break;
            case Constant.DEFINE_EVENT_TYPE_PV:
                if (null == mPvBeans) {
                    MLog.e(TAG, "onItemClick() -> mPvBeans is null!");
                    break;
                }

                mPvBeans.remove(md5);
                mUploadViewBeans.remove(position);
                TaskController.getInstance().submit(new WritePvToSp());
                break;
            case Constant.DEFINE_EVENT_TYPE_LIST:
                if (null == mListBeans) {
                    MLog.e(TAG, "onItemClick() -> mListBeans is null!");
                    break;
                }

                String listId = uploadViewBean.getListId();
                if (TextUtils.isEmpty(listId)) {
                    MLog.e(TAG, "onItemClick() -> listId is null or empty!");
                    break;
                }

                ConfigBean.ListBean listBean = mListBeans.get(listId);
                if (null == listBean) {
                    MLog.e(TAG, "onItemClick() -> listBean is null!");
                    break;
                }

                Map<String, String> itemAliases = listBean.getItemAliases();
                if (null == itemAliases || itemAliases.isEmpty()) {
                    MLog.e(TAG, "onItemClick() -> itemAliases is null or empty!");
                    break;
                }

                itemAliases.remove(md5);
                mUploadViewBeans.remove(position);
                TaskController.getInstance().submit(new WriteListToSp());

                if (TextUtils.isEmpty(pageClassNameMd5)) {
                    MLog.e(TAG, "onItemClick() -> pageClassNameMd5 is null or empty!");
                    return;
                }

                deleteRecyclerPv(pageClassNameMd5, listId);
                deleteListPv(pageClassNameMd5, listId);
                deleteExpandablePv(pageClassNameMd5, listId);
                break;
            default:
                break;
        }
    }

    private void deleteExpandablePv(String pageClassNameMd5, String listId) {
        if (null == mExpandablePvBeans) {
            MLog.e(TAG, "deleteExpandablePv() -> mExpandablePvBeans is null!");
            return;
        }

        Map<String, String> expandablePvBeans = mExpandablePvBeans.get(pageClassNameMd5);
        if (null == expandablePvBeans) {
            return;
        }

        expandablePvBeans.remove(listId);
        TaskController.getInstance().submit(new WriteExpandablePvToSp());
    }

    private void deleteListPv(String pageClassNameMd5, String listId) {
        if (null == mListPvBeans) {
            MLog.e(TAG, "deleteListPv() -> mListPvBeans is null!");
            return;
        }

        Map<String, String> listPvBeans = mListPvBeans.get(pageClassNameMd5);
        if (null == listPvBeans) {
            return;
        }

        listPvBeans.remove(listId);
        TaskController.getInstance().submit(new WriteListPvToSp());
    }

    private void deleteRecyclerPv(String pageClassNameMd5, String listId) {
        if (null == mRecyclerPvBeans) {
            MLog.e(TAG, "deleteRecyclerPv() -> mRecyclerPvBeans is null!");
            return;
        }

        Map<String, String> recyclerPvBeans = mRecyclerPvBeans.get(pageClassNameMd5);
        if (null == recyclerPvBeans) {
            return;
        }

        recyclerPvBeans.remove(listId);
        TaskController.getInstance().submit(new WriteRecyclerPvToSp());
    }

    @Override
    public void setPresenter(Object presenter) {
        mUploadViewPresenter = (UploadViewContract.Presenter) presenter;
    }

    private List<UploadViewBean> getUploadViewBeans() {
        List<UploadViewBean> uploadViewBeans = new ArrayList<>();
        mPvBeans = MonitorCache.getInstance().getPvBeans();
        mClickBeans = MonitorCache.getInstance().getClickBeans();
        mElemPvBeans = MonitorCache.getInstance().getElemPvBeans();
        mListBeans = MonitorCache.getInstance().getListBeans();
        mRecyclerPvBeans = MonitorCache.getInstance().getRecyclerPvBeans();
        mListPvBeans = MonitorCache.getInstance().getListPvBeans();
        mExpandablePvBeans = MonitorCache.getInstance().getExpandablePvBeans();

        if (null == mClickBeans || null == mElemPvBeans || null == mPvBeans || null == mListBeans || null == mRecyclerPvBeans || null == mListPvBeans || null == mExpandablePvBeans) {
            MLog.e(TAG, "getUploadViewBeans() -> mClickBeans or mElemPvBeans or mPvBeans or mListBeans or mRecyclerPvBeans or mListPvBeans or mExpandablePvBeans is null!");
            return uploadViewBeans;
        }

        for (Map.Entry<String, ConfigBean.ClickBean> clickBeanEntry : mClickBeans.entrySet()) {
            if (null == clickBeanEntry) {
                MLog.e(TAG, "getUploadViewBeans() -> clickBeanEntry is null!");
                continue;
            }

            String md5 = clickBeanEntry.getKey();
            ConfigBean.ClickBean clickBean = clickBeanEntry.getValue();
            if (TextUtils.isEmpty(md5) || null == clickBean) {
                MLog.e(TAG, "getUploadViewBeans() -> md5 or clickBean is null or empty!");
                continue;
            }

            UploadViewBean uploadViewBean = new UploadViewBean();
            uploadViewBean.setEventType(Constant.DEFINE_EVENT_TYPE_CLICK);
            uploadViewBean.setMd5(md5);
            uploadViewBean.setAlias(clickBean.getAlias());
            uploadViewBean.setPageClassName(clickBean.getPageClassName());

            uploadViewBeans.add(uploadViewBean);
        }

        for (Map.Entry<String, ConfigBean.PvBean> pvBeanEntry : mPvBeans.entrySet()) {
            if (null == pvBeanEntry) {
                MLog.e(TAG, "getUploadViewBeans() -> pvBeanEntry is null!");
                continue;
            }

            String md5 = pvBeanEntry.getKey();
            ConfigBean.PvBean pvBean = pvBeanEntry.getValue();
            if (TextUtils.isEmpty(md5) || null == pvBean) {
                MLog.e(TAG, "getUploadViewBeans() -> md5 or pvBean is null or empty!");
                continue;
            }

            UploadViewBean uploadViewBean = new UploadViewBean();
            uploadViewBean.setEventType(Constant.DEFINE_EVENT_TYPE_PV);
            uploadViewBean.setMd5(md5);
            uploadViewBean.setAlias(pvBean.getAlias());
            uploadViewBean.setPageClassName(pvBean.getViewPath());

            uploadViewBeans.add(uploadViewBean);
        }

        for (Map.Entry<String, ConfigBean.ListBean> listBeanEntry : mListBeans.entrySet()) {
            if (null == listBeanEntry) {
                MLog.e(TAG, "getUploadViewBeans() -> listBeanEntry is null!");
                continue;
            }

            String listId = listBeanEntry.getKey();
            ConfigBean.ListBean listBean = listBeanEntry.getValue();
            if (TextUtils.isEmpty(listId) || null == listBean) {
                MLog.e(TAG, "getUploadViewBeans() -> listId or listBean is null or empty!");
                continue;
            }

            Map<String, String> itemAliases = listBean.getItemAliases();
            if (null == itemAliases || itemAliases.isEmpty()) {
                MLog.e(TAG, "getUploadViewBeans() -> itemAliases is null or empty!");
                continue;
            }

            for (Map.Entry<String, String> itemAlias : itemAliases.entrySet()) {
                if (null == itemAlias) {
                    MLog.e(TAG, "getUploadViewBeans() -> itemAlias is null!");
                    continue;
                }

                String md5 = itemAlias.getKey();
                String alias = itemAlias.getValue();
                if (TextUtils.isEmpty(md5) || TextUtils.isEmpty(alias)) {
                    MLog.e(TAG, "getUploadViewBeans() -> md5 or alias is null!");
                    continue;
                }

                UploadViewBean uploadViewBean = new UploadViewBean();
                uploadViewBean.setEventType(Constant.DEFINE_EVENT_TYPE_LIST);
                uploadViewBean.setMd5(md5);
                uploadViewBean.setAlias(alias);
                uploadViewBean.setListId(listId);
                uploadViewBean.setPageClassName(listBean.getPageClassName());

                uploadViewBeans.add(uploadViewBean);
            }
        }

        return uploadViewBeans;
    }

    @Override
    public void showUploadSuccess() {
        FloatingViewManager.getInstance().showFloatingView(IFloatingView.WINDOW_LOADING);
        LoadingView loadingView = (LoadingView) FloatingViewManager.getInstance().getFloatingView(IFloatingView.WINDOW_LOADING);
        if (null == loadingView) {
            MLog.e(TAG, "showUploadSuccess()-> loadingView is null");
            return;
        }
        loadingView.setLoadingOk(getResources().getString(R.string.loading_upload_ok));
    }

    @Override
    public void showUploadFailed() {
        FloatingViewManager.getInstance().showFloatingView(IFloatingView.WINDOW_LOADING);
        LoadingView loadingView = (LoadingView) FloatingViewManager.getInstance().getFloatingView(IFloatingView.WINDOW_LOADING);
        if (null == loadingView) {
            MLog.e(TAG, "showUploadSuccess()-> loadingView is null");
            return;
        }
        loadingView.setLoadingFail(getResources().getString(R.string.loading_upload_fail));
    }

    @Override
    public void showUploading() {
        FloatingViewManager.getInstance().showFloatingView(IFloatingView.WINDOW_LOADING);
        LoadingView loadingView = (LoadingView) FloatingViewManager.getInstance().getFloatingView(IFloatingView.WINDOW_LOADING);
        if (null == loadingView) {
            MLog.e(TAG, "showUploadSuccess()-> loadingView is null");
            return;
        }
        loadingView.setLoadingOk(getResources().getString(R.string.loading_uploading));
    }

    @Override
    public void downloadConfig(boolean isSuccessful) {
        if (!isSuccessful) {
            MLog.e(TAG, "downloadConfig fail! no need update UI");
            return;
        }

        TaskController.getInstance().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                mUploadViewBeans.clear();
                mUploadViewBeans.addAll(getUploadViewBeans());
                mListAdapter.notifyDataSetChanged();
            }
        });
    }
}
