package com.chinapex.android.monitor.view.config;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.text.TextUtils;
import android.view.View;

import com.chinapex.android.monitor.R;
import com.chinapex.android.monitor.bean.ImportConfigBean;
import com.chinapex.android.monitor.callback.IGetAllConfigVersionsCallback;
import com.chinapex.android.monitor.changelistener.IDownloadConfigCallback;
import com.chinapex.android.monitor.changelistener.MonitorListenerController;
import com.chinapex.android.monitor.executor.TaskController;
import com.chinapex.android.monitor.executor.runnable.DownloadConfig;
import com.chinapex.android.monitor.executor.runnable.GetAllConfigVersions;
import com.chinapex.android.monitor.global.MonitorCache;
import com.chinapex.android.monitor.utils.MLog;
import com.chinapex.android.monitor.view.BaseFloatingView;
import com.chinapex.android.monitor.view.FloatingViewManager;
import com.chinapex.android.monitor.view.IFloatingView;
import com.chinapex.android.monitor.view.adapter.ConfigRvAdapter;
import com.chinapex.android.monitor.view.loading.LoadingView;

import java.util.ArrayList;
import java.util.List;

/**
 * @author SteelCabbage
 * @date 2019/05/09
 */
public class ImportConfigView extends BaseFloatingView implements View.OnClickListener, ConfigRvAdapter.OnItemClickListener,
        IGetAllConfigVersionsCallback, IDownloadConfigCallback {

    private static final String TAG = ImportConfigView.class.getSimpleName();
    private RecyclerView mRvImportConfig;
    private List<ImportConfigBean> mImportConfigBeans;
    private ConfigRvAdapter mConfigRvAdapter;
    private String mCurrentSelectedVersion;

    public ImportConfigView(@NonNull Context context, int windowType) {
        super(context, windowType);
        inflate(context, R.layout.layout_window_import_config, this);

        initView();
        initData();
    }

    private void initView() {
        mRvImportConfig = findViewById(R.id.rv_import_config);
        findViewById(R.id.btn_import_config_confirm).setOnClickListener(this);
        findViewById(R.id.btn_import_config_cancel).setOnClickListener(this);
    }

    private void initData() {
        mCurrentSelectedVersion = null;
        mImportConfigBeans = new ArrayList<>();
        TaskController.getInstance().submit(new GetAllConfigVersions(this));

        mConfigRvAdapter = new ConfigRvAdapter(mImportConfigBeans);
        mConfigRvAdapter.setOnItemClickListener(this);
        mRvImportConfig.setLayoutManager(new LinearLayoutManager(MonitorCache.getInstance().getContext(),
                LinearLayoutManager.VERTICAL, false));
        mRvImportConfig.setAdapter(mConfigRvAdapter);
    }

    @Override
    public void show() {
        super.show();
        MonitorListenerController.getInstance().addIDownloadConfigCallback(this);
    }

    @Override
    public void dismiss() {
        MonitorListenerController.getInstance().removeIDownloadConfigCallback(this);
        super.dismiss();
    }

    @Override
    public void getAllConfigVersions(final List<String> versions) {
        if (null == versions || versions.isEmpty()) {
            MLog.e(TAG, "getAllConfigVersions() -> versions is null or empty!");
            return;
        }

        if (null == mImportConfigBeans) {
            MLog.e(TAG, "getAllConfigVersions() -> mImportConfigBeans is null!");
            return;
        }

        TaskController.getInstance().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                int preClearSize = mImportConfigBeans.size();
                mImportConfigBeans.clear();
                mConfigRvAdapter.notifyItemRangeRemoved(0, preClearSize);

                for (String version : versions) {
                    if (TextUtils.isEmpty(version)) {
                        MLog.e(TAG, "getAllConfigVersions() -> version is null or empty!");
                        continue;
                    }

                    ImportConfigBean importConfigBean = new ImportConfigBean();
                    importConfigBean.setVersion(version);
                    mImportConfigBeans.add(importConfigBean);
                }

                mConfigRvAdapter.notifyItemRangeInserted(0, mImportConfigBeans.size());
            }
        });
    }

    @Override
    public void onClick(View v) {
        int viewId = v.getId();
        if (R.id.btn_import_config_confirm == viewId) {
            FloatingViewManager.getInstance().showFloatingView(IFloatingView.WINDOW_LOADING);
            LoadingView loadingView =
                    (LoadingView) FloatingViewManager.getInstance().getFloatingView(IFloatingView.WINDOW_LOADING);
            if (null == loadingView) {
                MLog.e(TAG, "onClick() -> btn_import_config_confirm: loadingView is null!");
                return;
            }

            if (TextUtils.isEmpty(mCurrentSelectedVersion)) {
                loadingView.setLoadingFail(getResources().getString(R.string.loading_select_config));
            } else {
                loadingView.setLoading(getResources().getString(R.string.loading_downloading));
                TaskController.getInstance().submit(new DownloadConfig(mCurrentSelectedVersion));
            }
        } else if (R.id.btn_import_config_cancel == viewId) {
            FloatingViewManager.getInstance().dismissFloatingView(IFloatingView.WINDOW_IMPORT_CONFIG);
        } else {
            MLog.e(TAG, "unknown click!");
        }
    }

    @Override
    public void onItemClick(int position) {
        if (null == mImportConfigBeans) {
            MLog.e(TAG, "onItemClick() -> mImportConfigBeans is null!");
            return;
        }

        ImportConfigBean importConfigBean = mImportConfigBeans.get(position);
        if (null == importConfigBean) {
            MLog.e(TAG, "onItemClick() -> importConfigBean is null!");
            return;
        }

        mCurrentSelectedVersion = importConfigBean.getVersion();
        MLog.i(TAG, "onItemClick() -> mCurrentSelectedVersion is: " + mCurrentSelectedVersion);
    }

    @Override
    public void downloadConfig(final boolean isSuccessful) {
        final LoadingView loadingView =
                (LoadingView) FloatingViewManager.getInstance().getFloatingView(IFloatingView.WINDOW_LOADING);
        if (null == loadingView) {
            MLog.e(TAG, "downloadConfig() -> loadingView is null!");
            return;
        }

        TaskController.getInstance().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                if (isSuccessful) {
                    loadingView.setLoadingOk(getResources().getString(R.string.loading_download_ok));
                    FloatingViewManager.getInstance().dismissFloatingView(IFloatingView.WINDOW_IMPORT_CONFIG);
                } else {
                    loadingView.setLoadingFail(getResources().getString(R.string.loading_download_fail));
                }
            }
        });
    }

}
