package com.chinapex.android.monitor.view.charts;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.DashPathEffect;
import android.graphics.Paint;
import android.graphics.Path;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;

import com.chinapex.android.monitor.R;
import com.chinapex.android.monitor.bean.LineChartPoint;
import com.chinapex.android.monitor.utils.DateUtils;
import com.chinapex.android.monitor.utils.DensityUtil;
import com.chinapex.android.monitor.utils.MLog;

import java.util.ArrayList;
import java.util.List;


/**
 * @author wyhusky
 * @date 2019/1/2
 */
public class LineChart extends View {

    private static final String TAG = LineChart.class.getSimpleName();

    private static final int CLICK_LINE_ID = 0;
    private static final int PV_LINE_ID = 1;

    private static final int DEFAULT_HEIGHT = 250;
    private static final int DEFAULT_WIDTH = 250;
    private static final int MAX_DATA_LENGTH = 7;

    private long[] mDateArray;
    private long[] mClickCountArray;
    private long[] mPvCountArray;


    private int mWidth;
    private int mHeight;

    /**
     * left and right padding
     */
    private float basePadding;

    /**
     * the width between two items
     */
    private float itemWidth;

    /**
     * height of the highest point
     */
    private float maxLineHeight;

    /**
     * height of the lowest point
     */
    private float minLineHeight;

    /**
     * space between maxLineHeight and minLineHeight
     */
    private float mLineSpace;

    /**
     * y position of baseline
     */
    private float mBaseLinePosition;

    /**
     * max of clickCount
     */
    private long mMaxCount;

    /**
     * min of clickCount
     */
    private long mMinCount;

    /**
     * the range between mMaxCount and mMinCount
     */
    private long mRange;

    /**
     * LineChart attrs
     */
    private int mPointColor, mPointStrokeColor, mTextColor, mBaseLineColor, mBrokenLineColor, mDashLineColor;
    private float mPointRadius, mPointStrokeWidth, mTextSize, mBaseLineWidth, mBrokenLineWidth, mDashLineWidth;


    private List<LineChartPoint> mPointsInfoList;

    private Path mPath;

    private Paint mPointPaint, mPointStrokePaint, mBaseLinePaint, mTextPaint, mBrokenLinePaint, mDashLinePaint;

    public LineChart(Context context) {
        this(context, null);
    }

    public LineChart(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public LineChart(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        TypedArray a = context.getTheme().obtainStyledAttributes(attrs,
                R.styleable.LineChart,
                0, 0);

        //define default attr values
        int defPointColor = Color.WHITE;
        int defPointStrokeColor = getResources().getColor(R.color.c_66BB6A);
        float defPointStrokeWidth = DensityUtil.dip2px(context, 5);
        float defPointRadius = DensityUtil.dip2px(context, 3);
        int defTextColor = Color.parseColor("#212121");
        float defTextSize = DensityUtil.sp2px(context, 12);
        int defBaseLineColor = Color.BLACK;
        float defBaseLineWidth = DensityUtil.dip2px(context, 3);
        int defBrokenLineColor = getResources().getColor(R.color.c_66BB6A);
        float defBrokenLineWidth = DensityUtil.dip2px(context, 1);
        int defDashLineColor = Color.BLACK;
        float defDashLineWidth = DensityUtil.dip2px(context, 1);


        mPointColor = a.getColor(R.styleable.LineChart_pointColor, defPointColor);
        mPointStrokeColor = a.getColor(R.styleable.LineChart_pointStrokeColor, defPointStrokeColor);
        mPointRadius = a.getDimension(R.styleable.LineChart_pointRadius, defPointRadius);
        mPointStrokeWidth = a.getDimension(R.styleable.LineChart_pointStrokeWidth, defPointStrokeWidth);
        mTextColor = a.getColor(R.styleable.LineChart_textColor, defTextColor);
        mTextSize = a.getDimension(R.styleable.LineChart_textSize, defTextSize);
        mBaseLineColor = a.getColor(R.styleable.LineChart_baseLineColor, defBaseLineColor);
        mBaseLineWidth = a.getDimension(R.styleable.LineChart_baseLineWidth, defBaseLineWidth);
        mBrokenLineColor = a.getColor(R.styleable.LineChart_brokenLineColor, defBrokenLineColor);
        mBrokenLineWidth = a.getDimension(R.styleable.LineChart_baseLineWidth, defBrokenLineWidth);
        mDashLineColor = a.getColor(R.styleable.LineChart_dashLineColor, defDashLineColor);
        mDashLineWidth = a.getDimension(R.styleable.LineChart_dashLineWidth, defDashLineWidth);

        a.recycle();
        init();
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
        int measureWidthSize = MeasureSpec.getSize(widthMeasureSpec);
        int measureHeightSize = MeasureSpec.getSize(heightMeasureSpec);
        int measureWidthMode = MeasureSpec.getMode(widthMeasureSpec);
        int measureHeightMode = MeasureSpec.getMode(heightMeasureSpec);
        if (measureWidthMode == MeasureSpec.AT_MOST
                && measureHeightMode == MeasureSpec.AT_MOST) {
            setMeasuredDimension(DEFAULT_WIDTH, DEFAULT_HEIGHT);
        } else if (measureWidthMode == MeasureSpec.AT_MOST) {
            setMeasuredDimension(DEFAULT_WIDTH, measureHeightSize);
        } else if (measureHeightMode == MeasureSpec.AT_MOST) {
            setMeasuredDimension(measureWidthSize, DEFAULT_HEIGHT);
        }
    }


    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);

        mWidth = w - getPaddingLeft() - getPaddingRight();
        mHeight = h - getPaddingTop() - getPaddingBottom();

        basePadding = mWidth / 16f;
        itemWidth = (mWidth - 2 * basePadding) / 6f;

        minLineHeight = mHeight - mHeight * 4 / 5;
        maxLineHeight = mHeight - mHeight / 5;
        mLineSpace = maxLineHeight - minLineHeight;
        caculatePointsInfo();
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        drawDateText(canvas);
        drawBaseLine(canvas);
        drawPoints(canvas);
    }

    public void setData(long[] dateArray, long[] clickCountArray, long[] pvCountArray) {
        if (null == dateArray || null == clickCountArray || null == pvCountArray) {
            MLog.e(TAG, "setData()-> params can not be null!");
            return;
        }

        if (dateArray.length == 0 || clickCountArray.length == 0 || pvCountArray.length == 0) {
            MLog.e(TAG, "setData()-> no data in the array!");
            return;
        }

        if (dateArray.length > MAX_DATA_LENGTH) {
            long[] tmp = dateArray;
            dateArray = new long[MAX_DATA_LENGTH];
            System.arraycopy(tmp, 0, dateArray, 0, MAX_DATA_LENGTH);
        }

        if (clickCountArray.length > MAX_DATA_LENGTH) {
            long[] tmp = clickCountArray;
            clickCountArray = new long[MAX_DATA_LENGTH];
            System.arraycopy(tmp, 0, clickCountArray, 0, MAX_DATA_LENGTH);
        }

        if (pvCountArray.length > MAX_DATA_LENGTH) {
            long[] tmp = pvCountArray;
            pvCountArray = new long[MAX_DATA_LENGTH];
            System.arraycopy(tmp, 0, pvCountArray, 0, MAX_DATA_LENGTH);
        }

        this.mDateArray = dateArray;
        this.mClickCountArray = clickCountArray;
        this.mPvCountArray = pvCountArray;

        long minTmp = clickCountArray[0], maxTmp = clickCountArray[0];
        for (Long count : clickCountArray) {
            if (count < 0) {
                continue;
            }
            if (count < minTmp) {
                minTmp = count;
            }
            if (count > maxTmp) {
                maxTmp = count;
            }
        }
        for (Long pv : pvCountArray) {
            if (pv < 0) {
                continue;
            }
            if (pv < minTmp) {
                minTmp = pv;
            }
            if (pv > maxTmp) {
                maxTmp = pv;
            }
        }
        mMaxCount = maxTmp;
        mMinCount = minTmp;
        MLog.d(TAG, "max" + mMaxCount + "min" + mMinCount);
        mRange = mMaxCount - mMinCount;
        mRange = mRange == 0 ? 1 : mRange;

        caculatePointsInfo();
        invalidate();
    }

    public List<LineChartPoint> getmPointsInfoList() {
        return mPointsInfoList;
    }

    private void init() {
        mPath = new Path();
        mPointsInfoList = new ArrayList<>();

        mPointPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mPointPaint.setStyle(Paint.Style.FILL);
        mPointPaint.setColor(mPointColor);

        mPointStrokePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mPointStrokePaint.setStyle(Paint.Style.STROKE);
        mPointStrokePaint.setColor(mPointStrokeColor);
        mPointStrokePaint.setStrokeWidth(mPointStrokeWidth);

        mBaseLinePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mBaseLinePaint.setStyle(Paint.Style.FILL);
        mBaseLinePaint.setColor(mBaseLineColor);
        mBaseLinePaint.setStrokeWidth(mBaseLineWidth);

        mTextPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mTextPaint.setColor(mTextColor);
        mTextPaint.setTextSize(mTextSize);
        mTextPaint.setTextAlign(Paint.Align.CENTER);

        mBrokenLinePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mBrokenLinePaint.setStyle(Paint.Style.STROKE);
        mBrokenLinePaint.setColor(mBrokenLineColor);
        mBrokenLinePaint.setStrokeWidth(mBrokenLineWidth);

        mDashLinePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mDashLinePaint.setStyle(Paint.Style.STROKE);
        mDashLinePaint.setStrokeWidth(mDashLineWidth);
        mDashLinePaint.setColor(mDashLineColor);
        mDashLinePaint.setPathEffect(new DashPathEffect(new float[]{5, 10}, 0));
    }

    private void caculatePointsInfo() {
        if (null == mDateArray) {
            MLog.w(TAG, "caculatePointsInfo()-> mDateArray is null");
            return;
        }
        if (null == mClickCountArray) {
            MLog.w(TAG, "caculatePointsInfo()-> mClickCount is null");
            return;
        }
        if (null == mPvCountArray) {
            MLog.d(TAG, "caculatePointsInfo()-> mPvCountArray is null");
            return;
        }
        int dataLength = mDateArray.length;
        mPointsInfoList.clear();
        float dx, dy;
        for (int i = 0; i < dataLength; i++) {
            dx = basePadding + itemWidth * i;
            dy = mHeight - ((mClickCountArray[i] - mMinCount) * mLineSpace / mRange + minLineHeight);
            LineChartPoint point = LineChartPoint.of(dx, dy, mDateArray[i], mClickCountArray[i], CLICK_LINE_ID);
            mPointsInfoList.add(point);
        }
        for (int i = 0; i < dataLength; i++) {
            dx = basePadding + itemWidth * i;
            dy = mHeight - ((mPvCountArray[i] - mMinCount) * mLineSpace / mRange + minLineHeight);
            LineChartPoint point = LineChartPoint.of(dx, dy, mDateArray[i], mPvCountArray[i], PV_LINE_ID);
            mPointsInfoList.add(point);
        }
    }

    private void drawBaseLine(Canvas canvas) {
        Paint.FontMetrics fontMetrics = mTextPaint.getFontMetrics();
        float fontHeight = fontMetrics.descent - fontMetrics.ascent;
        mBaseLinePosition = mHeight - fontHeight * 3 / 2;

        int startx = (int) basePadding;
        int endx = (int) (mWidth - basePadding);
        int y = (int) mBaseLinePosition;
        canvas.drawLine(startx, y, endx, y, mBaseLinePaint);
    }

    private void drawDateText(Canvas canvas) {
        float dy = mHeight;
        if (null == mDateArray || 0 == mDateArray.length) {
            MLog.w(TAG, "drawDateText()-> mDateArray is empty");
            return;
        }
        for (int i = 0; i < mDateArray.length; i++) {
            LineChartPoint point = mPointsInfoList.get(i);
            canvas.drawText(DateUtils.long2DateFormat2(mDateArray[i]), point.getX(), dy, mTextPaint);
        }
    }

    private void drawPoints(Canvas canvas) {
        // draw brokenLine
        boolean needReset = true;
        for (int i = 0; i < mPointsInfoList.size(); i++) {
            LineChartPoint point = mPointsInfoList.get(i);
            float dx = point.getX();
            float dy = point.getY();
            long value = point.getValue();
            int lineId = point.getLineId();
            if (value < 0) {
                needReset = true;
                continue;
            }
            if (i == mPointsInfoList.size() / 2) {
                needReset = true;
            }
            if (lineId == CLICK_LINE_ID) {
                mBrokenLinePaint.setColor(mBrokenLineColor);
            }
            if (lineId == PV_LINE_ID) {
                mBrokenLinePaint.setColor(getResources().getColor(R.color.c_FFB74D));
            }
            if (needReset) {
                mPath.reset();
                mPath.moveTo(dx, dy);
                needReset = false;
            } else {
                mPath.lineTo(dx, dy);
                canvas.drawPath(mPath, mBrokenLinePaint);
            }
        }

        // draw dashLine
        for (int i = 0; i < mPointsInfoList.size() / 2; i++) {
            LineChartPoint point1 = mPointsInfoList.get(i);
            LineChartPoint point2 = mPointsInfoList.get(mPointsInfoList.size() / 2 + i);
            float dx = point1.getX();
            float dy1 = point1.getY();
            float dy2 = point2.getY();
            long value1 = point1.getValue();
            long value2 = point2.getValue();
            if (value1 < 0 && value2 < 0) {
                continue;
            }
            float dy = dy1 > dy2 ? dy2 : dy1;
            mPath.reset();
            mPath.moveTo(dx, dy);
            mPath.lineTo(dx, mBaseLinePosition);
            canvas.drawPath(mPath, mDashLinePaint);
        }

        // draw points
        for (int i = 0; i < mPointsInfoList.size(); i++) {
            LineChartPoint point = mPointsInfoList.get(i);
            float dx = point.getX();
            float dy = point.getY();
            long value = point.getValue();
            int lineId = point.getLineId();
            if (value < 0) {
                continue;
            }
            if (lineId == CLICK_LINE_ID) {
                mPointPaint.setColor(mPointColor);
                mPointStrokePaint.setColor(mPointStrokeColor);
            }
            if (lineId == PV_LINE_ID) {
                mPointPaint.setColor(Color.WHITE);
                mPointStrokePaint.setColor(getResources().getColor(R.color.c_FFB74D));
            }
            canvas.drawCircle(dx, dy, mPointRadius, mPointStrokePaint);
            canvas.drawCircle(dx, dy, mPointRadius, mPointPaint);
        }
    }
}
