package com.chinapex.android.monitor.view.adapter;

import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseExpandableListAdapter;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.TextView;

import com.bumptech.glide.Glide;
import com.chinapex.android.monitor.R;
import com.chinapex.android.monitor.bean.StatisticsBean;
import com.chinapex.android.monitor.global.Constant;
import com.chinapex.android.monitor.global.MonitorCache;
import com.chinapex.android.monitor.utils.BitmapStorageUtil;
import com.chinapex.android.monitor.utils.MLog;
import com.chinapex.android.monitor.utils.NumberUtils;
import com.chinapex.android.monitor.view.FloatingViewManager;
import com.chinapex.android.monitor.view.IFloatingView;
import com.chinapex.android.monitor.view.snapshot.SnapshotView;

import java.util.List;

/**
 * @author : Seven
 * @date : 2019/1/8
 */
public class StatisticsListAdapter extends BaseExpandableListAdapter implements View.OnClickListener {

    private static final String TAG = StatisticsListAdapter.class.getSimpleName();

    private List<StatisticsBean> mStatisticsBeans;
    private OnItemClickListener mOnItemClickListener;

    /**
     * 条目点击监听器
     */
    public interface OnItemClickListener {
        /**
         * 条目点击回调方法
         *
         * @param position position
         */
        void onItemClick(int position);
    }

    public StatisticsListAdapter(List<StatisticsBean> statisticsBeans) {
        mStatisticsBeans = statisticsBeans;
    }

    public void setOnItemClickListener(OnItemClickListener onItemClickListener) {
        mOnItemClickListener = onItemClickListener;
    }

    @Override
    public int getGroupCount() {
        return null == mStatisticsBeans ? 0 : mStatisticsBeans.size();
    }

    @Override
    public int getChildrenCount(int groupPosition) {
        if (null == mStatisticsBeans) {
            return 0;
        }

        StatisticsBean statisticsBean = mStatisticsBeans.get(groupPosition);
        if (null == statisticsBean) {
            return 0;
        }

        String viewPathMD5 = statisticsBean.getViewPathMD5();
        if (TextUtils.isEmpty(viewPathMD5)) {
            return 0;
        }

        if (viewPathMD5.equals(Constant.OTHERS_VIEW_PATH_MD5)) {
            return 0;
        }

        return 1;
    }

    @Override
    public Object getGroup(int groupPosition) {
        return mStatisticsBeans.get(groupPosition);
    }

    @Override
    public Object getChild(int groupPosition, int childPosition) {
        return childPosition;
    }

    @Override
    public long getGroupId(int groupPosition) {
        return groupPosition;
    }

    @Override
    public long getChildId(int groupPosition, int childPosition) {
        return childPosition;
    }

    @Override
    public boolean hasStableIds() {
        return false;
    }

    @Override
    public boolean isChildSelectable(int groupPosition, int childPosition) {
        return true;
    }

    @Override
    public View getGroupView(int groupPosition, boolean isExpanded, View convertView, ViewGroup parent) {
        StatisticsListAdapter.GroupViewHolder groupViewHolder;
        if (convertView == null) {
            convertView = LayoutInflater.from(parent.getContext()).inflate(R.layout.statistics_list_item_group, parent, false);
            groupViewHolder = new StatisticsListAdapter.GroupViewHolder(convertView);
            convertView.setTag(groupViewHolder);
        } else {
            groupViewHolder = (StatisticsListAdapter.GroupViewHolder) convertView.getTag();
        }

        if (null == mStatisticsBeans) {
            MLog.e(TAG, "getGroupView() -> mUploadViewBeans is null !");
            return convertView;
        }

        StatisticsBean statisticsBean = mStatisticsBeans.get(groupPosition);
        if (null == statisticsBean) {
            MLog.e(TAG, "getGroupView() -> statisticsBean is null !");
            return convertView;
        }

        groupViewHolder.eventSequence.setText(String.valueOf(groupPosition));
        groupViewHolder.evenName.setText(TextUtils.isEmpty(statisticsBean.getEventLabel()) ?
                statisticsBean.getViewPathMD5() : statisticsBean.getEventLabel());
        groupViewHolder.clickCount.setText(String.valueOf(statisticsBean.getClickCount()));
        groupViewHolder.clickProportion.setText(NumberUtils.getFormatPercent(statisticsBean.getClickProportion(), 2));

        groupViewHolder.ibDeleteItem.setTag(groupPosition);
        groupViewHolder.ibDeleteItem.setOnClickListener(this);

        return convertView;
    }

    @Override
    public View getChildView(int groupPosition, int childPosition, boolean isLastChild, View convertView, ViewGroup parent) {
        StatisticsListAdapter.ChildViewHolder childViewHolder;
        if (convertView == null) {
            convertView = LayoutInflater.from(parent.getContext()).inflate(R.layout.statistics_list_item_child, parent, false);
            childViewHolder = new StatisticsListAdapter.ChildViewHolder(convertView);
            convertView.setTag(childViewHolder);
        } else {
            childViewHolder = (StatisticsListAdapter.ChildViewHolder) convertView.getTag();
        }

        if (null == mStatisticsBeans) {
            MLog.e(TAG, "getChildView() -> mUploadViewBeans is null !");
            return convertView;
        }

        StatisticsBean statisticsBean = mStatisticsBeans.get(groupPosition);

        if (null == statisticsBean) {
            MLog.e(TAG, "getChildView() -> statisticsBean is null !");
            return convertView;
        }

        final String snapshotPath = MonitorCache.getInstance().getServerRoot() + Constant.URL_DOWN_SANPSHOT
                    + "?md5=" + statisticsBean.getViewPathMD5() + "&terminal=" + Constant.TERMINAL;
        Glide.with(parent).load(snapshotPath).centerInside().into(childViewHolder.thumbnail);
        convertView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                FloatingViewManager.getInstance().showFloatingView(IFloatingView.WINDOW_SNAPSHOT);
                SnapshotView sv = (SnapshotView) FloatingViewManager.getInstance().getFloatingView(IFloatingView.WINDOW_SNAPSHOT);
                sv.loadImage(snapshotPath);
            }
        });
        return convertView;

    }

    @Override
    public void onClick(View v) {
        if (null == mOnItemClickListener) {
            MLog.e(TAG, "mOnItemClickListener is null!");
            return;
        }

        int tag = (Integer) v.getTag();
        mOnItemClickListener.onItemClick(tag);
        notifyDataSetChanged();
    }

    class GroupViewHolder {

        TextView eventSequence;
        TextView evenName;
        TextView clickCount;
        TextView clickProportion;
        ImageButton ibDeleteItem;

        GroupViewHolder(View itemView) {
            eventSequence = itemView.findViewById(R.id.tv_sequence);
            evenName = itemView.findViewById(R.id.tv_event_name);
            clickCount = itemView.findViewById(R.id.tv_item_click_cnt_value);
            clickProportion = itemView.findViewById(R.id.tv_item_click_proportion_value);
            ibDeleteItem = itemView.findViewById(R.id.ib_delete_item);
        }
    }

    class ChildViewHolder {
        ImageView thumbnail;

        ChildViewHolder(View itemView) {
            thumbnail = itemView.findViewById(R.id.iv_thumbnail_statistic);
        }
    }
}
