/*
 * Copyright (c) 2015 Chimera IoT
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.chimeraiot.android.ble;

import android.bluetooth.BluetoothAdapter;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.util.Log;

/** Bluetooth connection status broadcast. */
public class BluetoothStateReceiver extends BroadcastReceiver {
    /** Log tag. */
    private static final String TAG = BluetoothStateReceiver.class.getSimpleName();
    /** Indicates that broadcast receiver was registered and needs to be unregistered. */
    private boolean isRegistered = false;

    public void register(Context context) {
        final IntentFilter intentFilter = new IntentFilter();
        intentFilter.addAction(BluetoothAdapter.ACTION_STATE_CHANGED);
        context.registerReceiver(this, intentFilter);
        isRegistered = true;
    }

    public void unregister(Context context) {
        if (isRegistered) {
            context.unregisterReceiver(this);
            isRegistered = false;
        }
    }

    @Override
    public void onReceive(final Context context, final Intent intent) {
        final String action = intent.getAction();
        switch (action) {
            case BluetoothAdapter.ACTION_STATE_CHANGED:
                final int state = intent.getIntExtra(BluetoothAdapter.EXTRA_STATE,
                        BluetoothAdapter.STATE_OFF);
                if (BleConfig.isDebugEnabled()) {
                    final boolean enabled = state == BluetoothAdapter.STATE_ON;
                    Log.d(TAG, "bluetooth: " + enabled + " (state " + state + ")");
                }
                onBluetoothStateChanged(state);
                break;
            default:
                break;
        }
    }

    /**
     * Notifies about Bluetooth device state changed.
     * @param state new state.
     */
    public void onBluetoothStateChanged(int state) {
    }

}