/*
 * Copyright (c) 2015 Chimera IoT
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.chimeraiot.android.ble;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.os.Build;
import android.support.annotation.IntRange;

import java.util.Collection;

/** BLE device scanner. */
public interface BleScanner {

    /** Default scan period. */
    long DEFAULT_SCAN_PERIOD = 3000L;
    /** Value of period used to scan for list of ble devices once. */
    long PERIOD_SCAN_ONCE = -1;

    /**
     * Sets scan period.
     * @param scanPeriod scan period for one scan cycle. Scanning will start automatically again on end of period.
     * To scan once only set {@link #PERIOD_SCAN_ONCE} as value.
     */
    void setScanPeriod(@IntRange(from = -1) long scanPeriod);

    /**
     * Sets scan mode.
     * @param scanMode scan mode to scan devices.
     */
    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    void setScanMode(int scanMode);

    /**
     * Indicates whether scanning is in progress.
     * @return {@code true} if scanning is in progress.
     */
    boolean isScanning();

    /** Starts scanning. */
    void start();

    /**
     * Starts scanning for specified devices.
     * @param filter list of device names to search. <b>Please note that this parameter
     *               will be ignored for pre-Lollipop devices.</b>
     */
    void start(Collection<String> filter);

    /** Stops scanning. */
    void stop();

    /** BLE devices scanner listener. */
    interface BleDevicesScannerListener extends BluetoothAdapter.LeScanCallback {
        /** Notifies that scan process started.
         * <br/><strong>Executed on not-UI thread.</strong> */
        void onScanStarted();
        /** Notifies that scan started next iteration.
         * <br/><strong>Executed on not-UI thread.</strong> */
        void onScanRepeat();
        /** Notifies that scan process stopped.
         * <br/><strong>Executed on not-UI thread.</strong> */
        void onScanStopped();

        /** Notifies that device has been found.
         * <br/><strong>Executed on not-UI thread.</strong> */
        @Override
        void onLeScan(BluetoothDevice device, int i, byte[] bytes);
    }

}
